/******************************************************************************/
/*									      */
/*	ctk_generators.hh	    				    	      */
/*									      */
/*	Class declarations for ctk_generators.cpp		      	      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			  	      */
/*									      */
/******************************************************************************/

#ifndef CTK_GENERATORS_HH
#define CTK_GENERATORS_HH

#include "ctk_block.hh"

class ParamFloat;

const Float	PARAM_DEFAULT_WAVE_FREQ	= 0.0;
const Float	PARAM_DEFAULT_WAVE_AMP	= 1.0;
const Float	PARAM_DEFAULT_WAVE_PHASE= 0.0;

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: GeneratorBlock                                           */
/*                                                                            */
/******************************************************************************/

class GeneratorBlock: public SourceBlock {

private:
  
  ParamInt *WIDTH_param;

  int width;
  
public:

  GeneratorBlock(const string &a_name, const string &a_block_type);

  virtual Block* clone(const string &new_name) const=0;

  virtual void reset();

  virtual void build_output_data_descriptors();
  
protected:
  void output(float x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: ConstantBlock                                                */
/*                                                                            */
/******************************************************************************/

class ConstantBlock:public GeneratorBlock {

private:
  
  static const string help_text;
  static const string type_name;

  ParamFloat *X_param;
  float x;
  
public:

  ConstantBlock(const string &a_name);
  virtual ~ConstantBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual void reset();

private:
  virtual void compute();
  
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: WaveBlock                                                */
/*                                                                            */
/******************************************************************************/

class WaveBlock:public GeneratorBlock {

private:
  
  static const string help_text;

protected:

  ParamFloat *FREQ_param;
  ParamFloat *AMP_param;
  ParamFloat *PHASE_param;

  Float current_phase, current_amp_phase;
  Float phase_step, amp_phase_step;
  
  // Precomputed to save time
  Float delta, two_delta;
  Float amplitude, two_amplitude;
  
public:

  WaveBlock(const string &a_name, const string &a_block_type);
  virtual ~WaveBlock(){};
  virtual Block* clone(const string &new_name) const=0;

protected:
  
  virtual void reset();

private:
  virtual void compute()=0;
  
};



/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: ImpulseWaveBlock                                         */
/*                                                                            */
/******************************************************************************/

class ImpulseWaveBlock:public WaveBlock {

  static const string help_text;
  static const string type_name;

public:

  ImpulseWaveBlock(const string &a_name);
  virtual ~ImpulseWaveBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void compute();
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SineWaveBlock                                            */
/*                                                                            */
/******************************************************************************/

class SineWaveBlock:public WaveBlock {

  static const string help_text;
  static const string type_name;

public:

  SineWaveBlock(const string &a_name);
  virtual ~SineWaveBlock(){};

  virtual Block* clone(const string &new_name) const;
  
  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void compute();
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SquareWaveBlock                                          */
/*                                                                            */
/******************************************************************************/

class SquareWaveBlock:public WaveBlock {

  static const string help_text;
  static const string type_name;

public:

  SquareWaveBlock(const string &a_name);
  virtual ~SquareWaveBlock(){};
  
  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void compute();
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: TriangleWaveBlock                                        */
/*                                                                            */
/******************************************************************************/

class TriangleWaveBlock:public WaveBlock {

private:

  static const string help_text;
  static const string type_name;

  Boolean is_rising;

public:

  TriangleWaveBlock(const string &a_name);
  virtual ~TriangleWaveBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:
  virtual void reset();
  virtual void compute();
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SawtoothWaveBlock                                        */
/*                                                                            */
/******************************************************************************/

class SawtoothWaveBlock:public WaveBlock {

  static const string help_text;
  static const string type_name;

public:

  SawtoothWaveBlock(const string &a_name);
  virtual ~SawtoothWaveBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void compute();
  
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: GlottalWaveBlock                                         */
/*                                                                            */
/******************************************************************************/

class GlottalWaveBlock:public WaveBlock {

private:

  static const string help_text;
  static const string type_name;

  ParamFloat *ATTACK_param;
  ParamFloat *SUSTAIN_param;
  ParamFloat *DECAY_param;
  ParamFloat *CLOSED_param;

  float full_on_phase;
  float end_sustain_phase;
  float full_off_phase;

  // Precomputed stuff to speed computation
  float decay_period;   // Phase duration of the decay;
  float amp_over_full_on_phase;
  float amp_over_decay_period;
  
public:

  GlottalWaveBlock(const string &a_name);
  virtual ~GlottalWaveBlock(){};
  
  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:
  virtual void reset();
  virtual void compute();
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: WhiteNoiseBlock                                          */
/*                                                                            */
/******************************************************************************/

class WhiteNoiseBlock:public GeneratorBlock {

private:

  static const string help_text;
  static const string type_name;

  ParamFloat *AMP_param;

  Float amplitude, two_amplitude;

public:

  WhiteNoiseBlock(const string &a_name);
  virtual ~WhiteNoiseBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void reset();
  virtual void compute();

};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: GaussianNoiseBlock                                       */
/*                                                                            */
/******************************************************************************/

class GaussianNoiseBlock:public GeneratorBlock {

private:

  static const string help_text;
  static const string type_name;

  ParamFloat *AMP_param;

  Float amplitude;

public:

  GaussianNoiseBlock(const string &a_name);
  virtual ~GaussianNoiseBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void reset();
  virtual void compute();

};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: PinkNoiseBlock                                           */
/*                                                                            */
/******************************************************************************/

// Filter Coefficients
const Float PinkNoise_A2 = -4.0469475000;
const Float PinkNoise_A3 =  6.3705074735;
const Float PinkNoise_A4 = -4.8224326841;
const Float PinkNoise_A5 =  1.7211909155;
const Float PinkNoise_A6 = -0.2223181410;

const Float PinkNoise_B2 = -3.4673362000;
const Float PinkNoise_B3 =  4.4316658211;
const Float PinkNoise_B4 = -2.4427741968;
const Float PinkNoise_B5 =  0.4607539308;
const Float PinkNoise_B6 =  0.0176931572;

// Factor amplied to amplitude to undo the effect that
// the filtering has on the RMS amplitude value

const Float PINK_NOISE_NORMALISATION_COEFF = 0.5436;

class PinkNoiseBlock:public GeneratorBlock {


private:
  static const string help_text;
  static const string type_name;

  ParamFloat *AMP_param;

  Float amplitude;

  Float y1, y2, y3, y4, y5, y6;
  Float x1, x2, x3, x4, x5, x6;

public:

  PinkNoiseBlock(const string &a_name);
  virtual ~PinkNoiseBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void reset();
  virtual void compute();

};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: RampBlock                                                */
/*                                                                            */
/******************************************************************************/

const Float	PARAM_DEFAULT_RAMP_START	= 0.0;
const Float	PARAM_DEFAULT_RAMP_STEP	        = 1.0;

class RampBlock:public GeneratorBlock {

private:

  static const string help_text;
  static const string type_name;

  ParamFloat *START_param;
  ParamFloat *STEP_param;

  Float data;

public:

  RampBlock(const string &a_name);
  virtual ~RampBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void reset();
  virtual void compute();

};


#endif

/* End of ctk_generators.hh */
