/******************************************************************************/
/*									      */
/*	ctk_CTK_reader.hh	      					      */
/*									      */
/*	Class declarations for ctk_CTK_reader.cpp			      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007	         		      */
/*									      */
/******************************************************************************/

#ifndef CTK_CTK_READER_HH
#define CTK_CTK_READER_HH

#include <map>
#include <fstream>

#include "ctk_block.hh"
#include "ctk_script.hh"

/** Necessary forward declarations */

class CTKReader;


/******************************************************************************/
/*									      */
/*	Local Type Definitions						      */
/*									      */
/******************************************************************************/

const Integer CTK_MAX_INCLUDE_NEST_DEPTH = 20;   // maximum depth for included files
const Integer CTK_MAX_SCRIPT_LINE_SIZE = 1024;

/******************************************************************************/
/*									      */
/*	CLASS NAME: UnresolvedParam	       				      */
/*									      */
/******************************************************************************/

class UnresolvedParam {

public:
  
  string name;
  vector<int> components;
  vector<string> fixed_components;

public:
  
  UnresolvedParam(const string &aname, const vector<int> &acomponents, const vector<string> &afixed_components);
  UnresolvedParam(const string &aname, const string &value);

  int get_max_param_number() const;
  bool is_simple() const;

private:  

  // used during the parsing of the value string to add fixed and unresolved elements to the component lists
  void add_element(const string &element, bool fixed);
  
  friend ostream& operator<< (ostream& out, const UnresolvedParam& x);
};

  
/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKReader	         				      */
/*									      */
/******************************************************************************/

class CTKReader {
private:
  
  typedef vector<CTKScriptElement>::iterator SIt;

  std::istream *ifstr;
  
  //  StringVector lines;
  vector<CTKScriptElement> lines;

  Boolean in_block;
  string help_text;
  
  BlockList block_list;
  Block* current_block;

  int max_param_number;   // Max $X param number used in script

  // block parameters whos exact values are not know until command line parameter values have been substituted
  vector<UnresolvedParam> unresolved_parameters;  
  
  map<string, Integer> op_translator;

  map<string, Block*> constructor_translator;

  string error_string;
  
  //
  //
  //
  
public:
  CTKReader(string file_name, vector<string> arguments=vector<string>());
  ~CTKReader();
  
  Block *get_block(const string &name);
  BlockList get_block_list();

  UInteger get_max_param_number();

  const vector<UnresolvedParam> &get_unresolved_parameters();

  void register_block(Block *b) {
    constructor_translator[b->get_blocktype()]=b;
  }
    
private:

  void script_error(const CTKScriptElement &element);

  void build_op_map();
  void build_constructor_map();

  void process_include_statements();
  
  void insert_file(vector<CTKScriptElement>::iterator &it, string file_name);
  
  void do_argument_substitution(vector <string> arguments);

  Boolean interpret_BLOCK(CTKScriptElement &s);
  Boolean interpret_ENDBLOCK(CTKScriptElement &s);
  Boolean interpret_ADD(CTKScriptElement &s);
  Boolean interpret_CONNECT(CTKScriptElement &s);
  Boolean interpret_PARAMETER(CTKScriptElement &s);
  Boolean interpret_SET(CTKScriptElement &s);
  Boolean interpret_INPUT(CTKScriptElement &s);
  Boolean interpret_OUTPUT(CTKScriptElement &s);
  
  Boolean do_assignment(Block *block, CTKScriptElement &s);

  // Return true if the parameter value contains an unresolved command line parameter e.g. $1, $2 etc
  bool contains_unresolved_param(const string &param_value) const;
};

#endif

/* End of ctk_CTK_reader.hh */
