/******************************************************************************/
/*									      */
/*	ctk_HMM_decoder.hh   			             		      */
/*									      */
/*	Class declarations for ctk_HMM_decoder.cpp	              	      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			     	      */
/*									      */
/******************************************************************************/

// CHANGES Jul 12, 2000: JPB
// NBest search threshold start and step changed from -30.0 and -10.0 to -5.0 and -5.0.
// The finer steps reduce the chance of the NBest search getting stuck in big recursions

#ifndef CTK_HMM_DECODER_HH
#define CTK_HMM_DECODER_HH

#include <sys/types.h>
#include <regex.h> /* Provides regular expression matching */

#include "ctk_block.hh"
#include "ctk_HMM.hh"

class ParamString;
class ParamBool;
class ParamFloat;
class ParamInt;
class ParamEnumerated;

class RecoHypothesis;
class RecoStats;



/******************************************************************************/
/*									      */
/*	CLASS NAME: HMMDecoderBlock		              	       	      */
/*									      */
/******************************************************************************/

class HMMDecoderBlock: public Block {

private:

  vector<vector<HMMFloat> > present_component;
  vector<vector<HMMFloat> > missing_component;

  // Parameters for probability calculation
  ParamBool *max_mixtures_param;
  ParamBool *has_deltas_param;

  // Parameters for HMM decoding
  ParamString *log_file_param;             // Output file for hypothesised transcription
  ParamString *log_file2_param;            // Output file for detailed decoding information
  ParamString *timitfile_param;            // Output file for hypothesis written in TIMIT format
  ParamInt *samples_per_frame_param;   // Only needed when producing TIMIT output
  ParamFloat *word_creation_penalty_param;
  ParamFloat *pruning_beamwidth_param;
  ParamString *hmm_file_param;
  ParamString *noise_hmm_file_param;
  ParamEnumerated *grammar_format_param;
  ParamString *grammar_filename_param;
  ParamString *noise_grammar_filename_param;
  ParamString *label_file_param;
  ParamString *dictionary_param;
  ParamString *first_token_param;
  ParamString *final_token_param;
  ParamString *transcription_param;
  ParamString *silence_param;
  ParamBool *state_path_param;

  ParamString *hypothesis_filter_param;
  ParamBool *use_posteriors_param;
  ParamBool *output_confusions_param;
  ParamBool *dump_parameters_param;
  ParamInt *num_hyp_param;
  ParamBool *display_groups_param;
  ParamString *mask_output_filename_param;
  
  HMMFloat word_creation_penalty;
  
  Boolean use_bounds;
  Boolean use_delta_bounds;
  Boolean max_mixtures;
  Boolean has_deltas;
  Boolean use_deltas;
  Boolean use_posteriors;
  Boolean display_groups;
  string mask_output_filename;
  
  int num_hyp;
  
  string filename;

  FILE *logfile;     // log file for transcription
  FILE *logfile2;    // log file for decoding details
  FILE *timitfile;   // log file for transcription in TIMIT format
  
  vector<RecoHypothesis*> hypotheses;

  list<string> label_list;  // list of HMM model labels
  
  RecoStats *overall_stats;
  
  vector<vector<Integer> > group_buffer;  // For holding the group label mask in the case when a final plot is requested
  vector<vector<Float> > mask_buffer;  // For holding the MD mask in the case when a final plot is requested
  
  regex_t *filter;   //  Regular expression for implementing HYPOTHESIS_FILTER

  string grammar;
  string noise_grammar;

  bool decoder_performs_probability_calculation;
  
protected:

  ParamBool *use_deltas_param; 

  SetOfHMMs *hmms;   // hmms describing the signal
  SetOfHMMs *noise_hmms; // hmms describing the noise - NULL if not used
  
  class Decoder *decoder; 

  //
  //
  //
  
public:

  HMMDecoderBlock(const string &a_name, const string &a_type, bool performs_probability=true);
  virtual ~HMMDecoderBlock();

protected:
  
  virtual Block* clone(const string &new_name) const=0;
  
  virtual void reset();
  virtual void compute()=0;
  virtual void close();
  virtual void close2();
  virtual void close_final();

  virtual void write_hypothesis(RecoHypothesis *hyp, FILE *file);
  
  void set_use_bounds(Boolean x);
  void set_use_delta_bounds(Boolean x);
  
  bool uses_noise_models() const;

  bool get_has_deltas() const {return has_deltas;}
  bool get_use_deltas() const {return use_deltas;}
  
  void process_frame(const CTKVector &frame, CTKVector *mask=NULL, CTKVector *nmask=NULL, CTKVector *lower_bounds=NULL, CTKVector *upper_bounds=NULL, bool use_marginal_normalisation=true);
  
  void process_frame_multisource(const CTKVector &frame, const CTKVector &mask, CTKVector *lower_bounds, CTKVector *upper_bounds, vector<Integer> *mapped_groups, float md_weight, bool soft_multisource=false);
  
  void round_mask(CTKVector *mask, Float round);

  virtual HMMMixture *get_HMM_mixture_prototype() const = 0;


  class Decoder *get_decoder(){return decoder;} 

private:
  
  bool test_file_exists(const string &name) const;

  void print_decoder_details_XML(FILE *fp);

  regex_t *compile_regular_expression(const string &regex) const;
    
  list<string> strip_silence(const list<string> &words, const list<string> &silences);

  FILE *open_logfile(ParamString *file_param, FILE *default_file=stdout);

  void reset_calc_UD_trigger();

  void display_group_mask();
  void save_group_mask(const string &filename);
  
  // Called after new HMMs have been loaded to build suitable decoder
  Decoder *buildDecoder(SetOfHMMs *hmms, ParamString *grammar_filename_param);

  // Called before decoding each utterance to put decoder into good start state
  // State path decoding may be switched on or off, and NBestList length may be set
  void resetDecoder(bool state_path_recording, int num_hypotheses);   
 
  SetOfHMMs *load_hmms(ParamString *hmm_file_param);
};

/******************************************************************************/
/*									      */
/*	CLASS NAME: HMMDecoderOnlyBlock	              	       	      */
/*									      */
/******************************************************************************/

class HMMDecoderOnlyBlock: public HMMDecoderBlock {

  static const string type_name;
  static const string help_text;

public:

  HMMDecoderOnlyBlock(const string &a_name);
  virtual ~HMMDecoderOnlyBlock(){};

protected:

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

  virtual void reset();

  virtual void compute();

private:

  HMMMixture *get_HMM_mixture_prototype() const {
    return new HMMMixtureStandard();
  }
  
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: StandardHMMDecoderBlock	              	       	      */
/*									      */
/******************************************************************************/

class StandardHMMDecoderBlock: public HMMDecoderBlock {

  static const string type_name;
  static const string help_text;

public:

  StandardHMMDecoderBlock(const string &a_name);
  virtual ~StandardHMMDecoderBlock(){};

protected:

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

  virtual void reset();

  virtual void compute();

private:

  virtual void build_output_data_descriptors();

  HMMMixture *get_HMM_mixture_prototype() const {
    return new HMMMixtureStandard();
  }
  
};



/******************************************************************************/
/*									      */
/*	CLASS NAME: BaseMD_HMMDecoderBlock	              	       	      */
/*									      */
/******************************************************************************/

class BaseMD_HMMDecoderBlock: public HMMDecoderBlock {

private:
  
  ParamBool *use_bounds_param;
  ParamBool *use_delta_bounds_param;
  ParamBool *use_erf_table_param;
  ParamBool *use_log_table_param;

public:

  BaseMD_HMMDecoderBlock(const string &a_name, const string &a_type);
  virtual ~BaseMD_HMMDecoderBlock(){};

  virtual void reset();
  virtual void compute()=0;
  virtual void close();
  
  //  virtual void write_hypothesis(RecoHypothesis *, FILE *){};
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: MD_HMMDecoderBlock	              	       	      */
/*									      */
/******************************************************************************/
// Missing data with with input - data, mask and optional bounds

class MD_HMMDecoderBlock: public BaseMD_HMMDecoderBlock {

  static const string type_name;
  static const string help_text;

public:

  MD_HMMDecoderBlock(const string &a_name);
  virtual ~MD_HMMDecoderBlock(){};

protected:
  
  virtual Block* clone(const string &new_name) const;
  
  virtual const string &get_helptext() const {return help_text;}

  virtual void reset();
  virtual void compute();

private:
  
  virtual void build_output_data_descriptors();

  HMMMixture *get_HMM_mixture_prototype() const;

};


/******************************************************************************/
/*									      */
/*	CLASS NAME: MD_PHMMDecoderBlock	              	       	      */
/*									      */
/******************************************************************************/
// Missing data with with input - data, mask and optional bounds

class MD_PHMMDecoderBlock: public BaseMD_HMMDecoderBlock {

  static const string type_name;
  static const string help_text;

public:

  MD_PHMMDecoderBlock(const string &a_name);
  virtual ~MD_PHMMDecoderBlock(){};

protected:
  
  virtual Block* clone(const string &new_name) const;
  
  virtual const string &get_helptext() const {return help_text;}

  virtual void reset();
  virtual void compute();

private:
  
  virtual void build_output_data_descriptors();

  HMMMixture *get_HMM_mixture_prototype() const;

};

/******************************************************************************/
/*									      */
/*	CLASS NAME: BoundsUD_HMMDecoderBlock	              	       	      */
/*									      */
/******************************************************************************/
// Uncertain data -i.e. Fuzzy mask decoder  (Three inputs - data, fuzzy mask, bounds)

class BoundsUD_HMMDecoderBlock: public BaseMD_HMMDecoderBlock {

  static const string type_name;
  static const string help_text;

  ParamFloat *one_zero_rounding_param;

public:

  BoundsUD_HMMDecoderBlock(const string &a_name);
  virtual ~BoundsUD_HMMDecoderBlock(){};

protected:

  virtual Block* clone(const string &new_name) const;
  
  virtual const string &get_helptext() const {return help_text;}

  virtual void reset();
  virtual void compute();

private:
  
  virtual void build_output_data_descriptors();

  HMMMixture *get_HMM_mixture_prototype() const;

};

/******************************************************************************/
/*									      */
/*	CLASS NAME: PMC_HMMDecoderBlock	              	       	      */
/*									      */
/******************************************************************************/
// Uncertain data -i.e. Fuzzy mask decoder  (Three inputs - data, fuzzy mask, bounds)

class PMC_HMMDecoderBlock: public BaseMD_HMMDecoderBlock {

private:
  
  static const string type_name;
  static const string help_text;

  ParamBool *adaptive_param;
  ParamBool *adaptation_uses_paths_param;
  ParamBool *tracking_param;
  ParamFloat *memory_factor_param;
  ParamFloat *bias_param;
  ParamFloat *likelihood_threshold_param;
  
  CTKVector *means;
  CTKVector *variance;
  CTKVector *s;

  bool first_frame;

  float memory_factor;
  float nprob_scale;
  float likelihood_threshold;
  bool adaptation_uses_paths;
  
  vector<int> nvarcount;
  int ncount;
  
public:

  PMC_HMMDecoderBlock(const string &a_name);
  virtual ~PMC_HMMDecoderBlock(){};

protected:

  virtual Block* clone(const string &new_name) const;
  
  virtual const string &get_helptext() const {return help_text;}

  virtual void reset();
  virtual void compute();

private:
  
  virtual void build_output_data_descriptors();

  HMMMixture *get_HMM_mixture_prototype() const;

};

#endif

/* End of ctk_HMM_decoder.hh */
