/******************************************************************************/
/*									      */
/*	ctk_colour_mask.hh   			             		      */
/*									      */
/*	Class declarations for ctk_colour_mask.cpp	               	      */
/*									      */
/*	Author: Jon Barker, Sheffield University              	      	      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007                   		      */
/*									      */
/******************************************************************************/

 
#ifndef CTK_COLOUR_MASK_HH
#define CTK_COLOUR_MASK_HH

#include <deque>
#include <map>

#include "ctk_block.hh"
#include "ctk_group.hh"

/******************************************************************************/
/*									      */
/*	CLASS NAME: ColourMaskBlock		              	       	      */
/*									      */
/******************************************************************************/

// Takes a 1/0 missing data mask and produces a simple group labelling
// A group is identified as areas of connected 1's in the missing data mask
//  - as an option an "onset grouping" rule can be applied to merge groups that
// start at the same time

// The algorithm uses a fixed buffer of a given number of frames, so that it can
// effectively look ahead and assign the same label to groups that start off
// separate but eventually merge together.


const Integer	PARAM_DEFAULT_WINDOW_SIZE	= 5;
const Boolean   PARAM_DEFAULT_ONSET_GROUPING    = false;
const Boolean   PARAM_DEFAULT_OFFSET_GROUPING    = false;
const Integer   PARAM_DEFAULT_MIN_GROUP_SIZE    = 0;      // i.e. no minimum size
const Integer   PARAM_DEFAULT_MAX_GROUP_SIZE    = 0;      // this means no maximum size
const Integer   PARAM_DEFAULT_NUM_SUBBANDS      = 1;
const Boolean   PARAM_DEFAULT_HAS_DELTAS        = false;
const Float     PARAM_DEFAULT_VOICING_THRESHOLD = 0.5;
const Float     PARAM_DEFAULT_DELTA_PITCH_THRESHOLD = 10.0;

class ColourMaskBlock:public Block {

  // help_text string, should be declared for ALL blocks.
  static const string type_name;
  static const string help_text;

private:

  class ParamInt *WINDOW_SIZE_param;
  class ParamBool *ONSET_GROUPING_param;
  class ParamBool *OFFSET_GROUPING_param;
  class ParamInt *MIN_GROUP_SIZE_param;
  class ParamInt *MAX_GROUP_SIZE_param;
  class ParamInt *NUM_SUBBANDS_param;
  class ParamBool *HAS_DELTAS_param;
  class ParamFloat *VOICING_THRESHOLD_param;
  class ParamFloat *DELTA_PITCH_THRESHOLD_param;
  
  //
  //

  int window_size;
  int min_group_size;
  int max_group_size;
  bool onset_grouping;
  bool offset_grouping;
  int num_subbands;
  bool has_deltas;

  float voicing_threshold;
  float delta_pitch_threshold;

  float previous_pitch;
  bool previous_voicing_decision;
  
  int frame_number;
  int new_group_number;

  int width, vector_width;
  
  CTKVector xvlast;
  
  deque<CTKVector *> frame_buffer;

  map<Integer, Group> groups;

  map<Integer, Integer> group_input_label;

public:

  
  ColourMaskBlock(const string &a_name);
  virtual ~ColourMaskBlock();

  virtual Block* clone(const string &new_name) const;
  
  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void reset();

  virtual void compute();

  virtual void flush_buffer();

  virtual void close();

  void propagate_frame(const CTKVector &x0, CTKVector &x1);
  void start_new_groups(CTKVector &x);
  void spread_groups_across_frequency(CTKVector &x);
  void merge_adjacent_groups(CTKVector &x);
  inline void relabel(int old_label, int new_label);
  void offset_detection();
  void label_deltas(CTKVector &x);
  void clean_up(CTKVector &x);
  
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: GroupStartEndDetectionBlock		              	       	      */
/*									      */
/******************************************************************************/

class GroupStartEndDetectionBlock:public Block {

  static const string type_name;
  static const string help_text;

private:
  
  map<Integer, Integer> active_groups;

  class ParamInt *WINDOW_SIZE_param;
  
  int new_group_number;
  int window_size;
  int input_frame_number;
  int output_frame_number;
  int input_width;
  
  deque<CTKVector *> frame_buffer;

public:

  
  GroupStartEndDetectionBlock(const string &a_name);
  virtual ~GroupStartEndDetectionBlock();

  virtual Block* clone(const string &new_name) const;
  
  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void reset();

  virtual void compute();

  virtual void flush_buffer();

  virtual void close();

  void process_1_buffered_frame();
  
  CTKVector *onset_detection(CTKVector *xv0);
  CTKVector *offset_detection();

};
#endif


/* End of ctk_colour_mask.hh */
