/******************************************************************************/
/*									      */
/*	ctk_param.hh	    						      */
/*									      */
/*	Class declarations for ctk_param.cpp				      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		          	      */
/*									      */
/******************************************************************************/

#ifndef CTK_PARAM_HH
#define CTK_PARAM_HH

#include <string>
#include <vector>

#include "ctk_local.hh"

class Param;
class CTKObject;
class Block;


/******************************************************************************/
/*									      */
/*	Local Type Definitions						      */
/*									      */
/******************************************************************************/

const UInteger CTK_PARAM_TYPE_CONST 	= 0001;     // Parameter is read-only (for the user)
const UInteger CTK_PARAM_TYPE_HIDDEN 	= 0002;     // Parameter is not displayed or available 
const UInteger CTK_PARAM_TYPE_TRANSMITTED = 0004;   // Parameter is passed forward to connected blocks
const UInteger CTK_PARAM_TYPE_DEPRECATED = 0010;    // Parameter is recognised but ignored - a warning is issued

typedef vector<Param*>::iterator ParamIt;
typedef vector<Param*>::const_iterator ParamConstIt;

typedef enum {PARAM_BOOL,PARAM_INT, PARAM_FLOAT, PARAM_STRING, PARAM_ENUMERATED, PARAM_VECTOR_BOOL, PARAM_VECTOR_INT, PARAM_VECTOR_FLOAT, PARAM_VECTOR_STRING, PARAM_INVALID_TYPE} ParameterType;

/******************************************************************************/
/*									      */
/*	Non CLASS functions						      */
/*									      */
/******************************************************************************/

string bool_string(bool x);

/******************************************************************************/
/*									      */
/*	CLASS NAME: Validator						      */
/*									      */
/******************************************************************************/

class Validator {

public:

  typedef enum {VLOWER, VUPPER} VLimitType;

private:
  float lower_limit_;
  float upper_limit_;
  bool check_upper_;
  bool check_lower_;

public:
  Validator():lower_limit_(0.0), upper_limit_(0.0), check_upper_(false), check_lower_(false){}
  Validator(float lower_limit, float upper_limit):lower_limit_(lower_limit), upper_limit_(upper_limit){}
  Validator(VLimitType vlimit, float limit) {
    check_upper_=check_lower_=false;
    if (vlimit==VUPPER) {
      upper_limit_=limit; check_upper_=true;
    } else {
      lower_limit_=limit; check_lower_=true;
    }

  }

  ~Validator(){}

  Validator(const Validator &validator) {
    lower_limit_=validator.lower_limit_;
    upper_limit_=validator.upper_limit_;
    check_upper_=validator.check_upper_;
    check_lower_=validator.check_lower_;
  }

  CTKStatus validate(float value){
    if (check_upper_ && value>upper_limit_) return CTK_FAILURE;
    if (check_lower_ && value<lower_limit_) return CTK_FAILURE;
    return CTK_SUCCESS;
  }

};

/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamList						      */
/*									      */
/******************************************************************************/

class ParamList {

private:
  vector<Param*> params;
  CTKObject *objectp;
  Boolean override_flag;

  // Declare but don't define to block implicit constructors
  ParamList(const ParamList&); 
  ParamList& operator=(const ParamList&);
  //
  //
  //

public:
  ParamList(CTKObject *this_block);
  ~ParamList();

  int size() const {return params.size();}

  Boolean register_parameter(Param* paramp, UInteger parameter_flags=0);

  void display(ostream &outfile, const string &this_name) const;
  void display_all(ostream &outfile, const string &space=string()) const;
  void fprintf_XML_all(FILE *fp) const;

  CTKStatus set(const string &name, const string &value);
  CTKStatus set_unresolved_value(const string &name, const string &value);
  CTKStatus set(const string &name, Float value);
  CTKStatus unset(const string &name);
  CTKStatus set_const(const string &name);
  CTKStatus set_hidden(const string &name);
  CTKStatus set_transmitted(const string &name);
  CTKStatus unset_const(const string &name);
  CTKStatus unset_hidden(const string &name);
  CTKStatus unset_transmitted(const string &name);
  CTKStatus set_arg_number(const string &name, Integer arg_number);
    
  Param* make_subparameter(const string &name, const string &parent_param_name);

  void clone(ParamList *cloned_param_list, const vector<Block*> b) const;

  void copy_settings(const ParamList *list_to_copy);

  void clear_transmitted_parameters();   // unset all transmitted parameters

  CTKStatus copy_transmitted_settings(const ParamList *list_to_copy, ParamIt &dodgy_parameter);

  void unlock();
  void relock();

  ParamConstIt begin() const {return params.begin();}
  ParamConstIt end() const {return params.end();}

private:
  ParamIt begin() {return params.begin();}
  ParamIt end() {return params.end();}
  
};



/******************************************************************************/
/*									      */
/*	CLASS NAME: Param						      */
/*									      */
/******************************************************************************/

class Param {
  friend class ParamList;

private:
  // Declared but not defined
  Param(const Param&);
  Param &operator=(const Param&);
  
private:
  Param *subparamp;

  CTKObject *objectp;

  string name;
  
  string helptext;
  string unit;
  
  mutable string fullname;
  mutable string pathname;

  string unresolved_value; // for string values contain command line arguments  e.g. file.$1_$2.$3
  Integer arg_number;   // Command line argument number, or 0 if the parameter is not a command line parameter

  Boolean deprecated_flag;  // True is parameter is no longer recognised in current version of CTK
  Boolean const_flag;     // True if the parameter is not changable by the user
  Boolean transmitted_flag; // True if this is an `transmitted' parameter
  Boolean hidden_flag;    // True if the parameter is hidden from the user
  Boolean dirty_flag;     // True if the parameter has been changed

  Validator *validatorp_;

protected:
  
  mutable string value_string;
  Boolean set_flag;       // True if the parameter has a value 
  Param *baseparamp;
  mutable Boolean dirty_string_flag;  // True if the parameter's Numeric value has changed and the 'value_string' is out of date. As long as this is false the value_string is valid
  
  //
  //
  //
    
public:
  Param(const string &this_name, Param *subparamp=NULL);

  virtual ~Param();

  void set_helptext(string text);
  void set_unit(string a_unit);

  void rename(string text);
  
  string get_unit() const {return unit;}
  string get_helptext() const {return helptext;}
  
  Boolean get_set_flag() const;
  string getname() const;
  string get_subparam_fullname() const;

  virtual string get_value_string() const =0;

  Boolean get_deprecated_flag() const;
  Boolean get_hidden_flag() const;
  Boolean get_const_flag() const;
  Boolean get_transmitted_flag() const;
  Boolean get_dirty_flag() const;
  Boolean am_i_a_baseparam() const;
  
  virtual ParameterType type() const = 0;

  Integer get_arg_number() const;
  void set_arg_number(Integer number);

  bool has_unresolved_value() const;
  string get_unresolved_value() const;
  void set_unresolved_value(const string &value);
  
  virtual void display(ostream &outfile, const string &space=string(), bool detailed=false) const;
  virtual void fprintf_XML(FILE *fp) const;

  CTKStatus install_validator(Validator *validatorp);

protected:

  string get_param_pathname() const;
  string get_param_fullname() const;
 
  virtual Boolean set_from_param(Param *a_param, bool set_properties=true);
  virtual Boolean set_value_from_param(Param *a_param) {return set_from_param(a_param, false);}

  virtual Boolean set_with_string(const string &s, Boolean override_flag);

  virtual void my_set_with_string(const string &s, Boolean override_flag)=0;

  virtual Boolean set_with_float(Float x, Boolean override_flag);

  void invalidate_value_string();  // Mark value string as out of date.
  void set_value_string(const string &s) const;  // Update value string and mark as up to date.
  
  void clean_param_string(string &) const;

  void set_dirty_flag() {dirty_flag=true;}

  CTKStatus validate(int value);
  CTKStatus validate(double value);

  const Validator *validatorp(){return validatorp_;}

private:
  
  Boolean is_my_name(string this_name) const;
  Boolean is_my_fullname(string this_name) const;
  Boolean is_my_pathname(string this_name) const;
  Boolean is_my_basepathname(string this_name) const;

  void set_new_value_string(const string &s);
 
  virtual Param *return_parent_parameter(const string &this_name) =0;

  string get_param_name() const;
  
  string get_param_basepathname() const;

  void set_objectp(CTKObject *this_objectp) {objectp=this_objectp;};
  
  void unset();
  
  void set_deprecated();
  void set_hidden();
  void set_const();
  void set_transmitted();
  void set_dirty();

  void unset_deprecated();
  void unset_hidden();
  void unset_const();
  void unset_transmitted();
  void unset_dirty();

};




/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamInt						      */
/*									      */
/******************************************************************************/


class ParamInt: public Param {

private:
  
  Integer value;
  
  //
  //
  //

public:

  ParamInt(const string &name, Integer this_value, Param *subparamp=NULL);
  ParamInt(const string &name, Param *subparamp=NULL);

  virtual ~ParamInt(){};

  Integer get_value() const;

  string get_value_string() const;

  virtual ParameterType type() const {return PARAM_INT;}

private:

  void my_set_with_string(const string &s, Boolean override_flag);
  Boolean set_with_float(Float x, Boolean override_flag);


  Param *return_parent_parameter(const string &this_name);
};



/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamFloat						      */
/*									      */
/******************************************************************************/


class ParamFloat: public Param {

private:

  Float value;

  //
  //
  //

public:

  ParamFloat(const string &name, Float this_value, Param *subparamp=NULL);
  ParamFloat(const string &name, Param *subparamp=NULL);

  virtual ~ParamFloat(){};

  Float get_value() const;

  string get_value_string() const;

  virtual ParameterType type() const {return PARAM_FLOAT;}

private:

  void my_set_with_string(const string &s, Boolean override_flag);
  Boolean set_with_float(Float x, Boolean override_flag);

  Param *return_parent_parameter(const string &this_name);
};



/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamBool						      */
/*									      */
/******************************************************************************/


class ParamBool: public Param {
  
private:

  Boolean value;
  
  //
  //
  //

public:

  ParamBool(const string &name, Boolean this_value,  Param *subparamp=NULL);
  ParamBool(const string &name, Param *subparamp=NULL);

  virtual ~ParamBool(){};

  Boolean get_value() const;

  string get_value_string() const;

  virtual ParameterType type() const {return PARAM_BOOL;}

private:

  void my_set_with_string(const string &s, Boolean override_flag);
  Boolean set_with_float(Float x, Boolean override_flag);

  Param *return_parent_parameter(const string &this_name);
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamString						      */
/*									      */
/******************************************************************************/

class ParamString: public Param {

private:

  string value;
  //
  //
  //

public:

  ParamString(const string &name, string this_value, Param *subparamp=NULL);
  ParamString(const string &name, Param *subparamp=NULL);

  virtual ~ParamString(){};

  string get_value() const;

  string get_value_string() const;

  virtual ParameterType type() const {return PARAM_STRING;}

private:

  void my_set_with_string(const string &s, Boolean override_flag);

  Param *return_parent_parameter(const string &this_name);

};

/******************************************************************************/
/*							        	      */
/*	CLASS NAME: ParamEnumerated		      			      */
/*									      */
/******************************************************************************/

class ParamEnumerated: public Param {

private:

  string value;
  vector<string> valid_values;
  //
  //
  //

public:

  ParamEnumerated(const string &name, vector<string> valid, const string &this_value, Param *subparamp=NULL);
  ParamEnumerated(const string &name, vector<string> valid, Param *subparamp=NULL);
  ParamEnumerated(const string &name, const char *valid[], const string &this_value, Param *subparamp=NULL);
  ParamEnumerated(const string &name, const char *valid[], Param *subparamp=NULL);

  virtual ~ParamEnumerated(){};

  vector<string> get_valid_values() const;

  string get_value() const;
  
  string get_value_string() const;

  Integer get_index() const;
  
  virtual ParameterType type() const {return PARAM_ENUMERATED;}

private:

  void my_set_with_string(const string &s, Boolean override_flag);

  Param *return_parent_parameter(const string &this_name);

  Boolean is_valid(string name);

  void make_valid_values(const char *values[]);

};


/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamIntVector		       			      */
/*									      */
/******************************************************************************/

class ParamIntVector: public Param {

private:

  vector<int> value;
  
  //
  //
  //

public:

  ParamIntVector(const string &name, const vector<int> &this_value,  Param *subparamp=NULL);
  ParamIntVector(const string &name, Param *subparamp=NULL);

  virtual ~ParamIntVector(){};

  Integer get_value(int index) const;

  vector<int> get_vector() const;

  Integer get_size() const;

  string get_value_string() const;

  virtual ParameterType type() const {return PARAM_VECTOR_INT;}

private:

  void my_set_with_string(const string &s, Boolean override_flag);

  Param *return_parent_parameter(const string &this_name);
};



/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamFloatVector	      				      */
/*									      */
/******************************************************************************/

class ParamFloatVector: public Param {

private:

  vector<Float> value;
  
  //
  //
  //

public:

  ParamFloatVector(const string &name, const vector<Float> &this_value, Param *subparamp=NULL);
  ParamFloatVector(const string &name, Param *subparamp=NULL);

  virtual ~ParamFloatVector(){};

  Float get_value(int index) const;

  vector<Float> get_vector() const;

  Integer get_size() const;

  string get_value_string() const;

  virtual ParameterType type() const {return PARAM_VECTOR_FLOAT;}

private:

  void my_set_with_string(const string &s, Boolean override_flag);

  Param *return_parent_parameter(const string &this_name);
};




/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamBoolVector	      				      */
/*									      */
/******************************************************************************/


class ParamBoolVector: public Param {

private:

  vector<bool> value;
  
  //
  //
  //

public:

  ParamBoolVector(const string &name, const vector<bool> &this_value,  Param *subparamp=NULL);
  ParamBoolVector(const string &name, Param *subparamp=NULL);

  virtual ~ParamBoolVector(){};

  Boolean get_value(int index) const;
  
  vector<bool> get_vector() const;
  
  Integer get_size() const;
  
  string get_value_string() const;

  virtual ParameterType type() const {return PARAM_VECTOR_BOOL;}

private:

  void my_set_with_string(const string &s, Boolean override_flag);

  Param *return_parent_parameter(const string &this_name);
};



/******************************************************************************/
/*									      */
/*	CLASS NAME: ParamStringVector	      				      */
/*									      */
/******************************************************************************/


class ParamStringVector: public Param {

private:

  vector<string> value;
  
  //
  //
  //

public:

  ParamStringVector(const string &name, const vector<string> &this_value,  Param *subparamp=NULL);
  ParamStringVector(const string &name, Param *subparamp=NULL);

  virtual ~ParamStringVector(){};

  string get_value(int index) const;
  
  vector<string> get_vector() const;
  
  Integer get_size() const;
  
  string get_value_string() const;

  virtual ParameterType type() const {return PARAM_VECTOR_BOOL;}

private:

  void my_set_with_string(const string &s, Boolean override_flag);

  Param *return_parent_parameter(const string &this_name);
};


#endif


/* End of ctk_param.hh */
