/******************************************************************************/
/*									      */
/*	ctk_parse.hh	    						      */
/*									      */
/*	Class declarations for ctk_parse.cpp				      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		        	      */
/*									      */
/******************************************************************************/

#ifndef CTK_PARSE_HH
#define CTK_PARSE_HH

#include <iosfwd>

#include "ctk_decoder.hh"

/******************************************************************************/
/*									      */
/*	Parameters							      */
/*									      */
/******************************************************************************/

typedef NodePair NetworkNodePair;
typedef Node NetworkNode;



/******************************************************************************/
/*									      */
/*	CLASS NAME: FiniteStateGrammar	     				      */
/*									      */
/******************************************************************************/

class GrammarNode;

class FiniteStateGrammar {

private:

  GrammarNode *base_node;
  list<GrammarNode *> nodeList;


public:
  
  FiniteStateGrammar(istream &ifstr);
  
  virtual ~FiniteStateGrammar();

  CTKStatus buildNetwork(Decoder &network, NetworkNodePair node_pair);  // Construct a network to represent this grammar 
  
private:
  bool parseAssignment(char *line, char *name, char *value);
  GrammarNode *getNodeNamed(const string &aname);
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: GrammarNode	               				      */
/*									      */
/******************************************************************************/
#define CTK_CHAR_BUFFER_SIZE 1024

class GrammarNode {

private:

  string name;
  list<GrammarNode *> nodeList;
  char a_stop_char;
  bool is_alternative;
  
protected:
  GrammarNode();
  GrammarNode(char *&value, const string &name);
  GrammarNode(const GrammarNode &node);  // copy constructor
  virtual GrammarNode *clone() = 0; //{return new GrammarNode(*this);}

  virtual void setAlternative(bool x){is_alternative=x;}
  

public:

  virtual ~GrammarNode();

  virtual void print(ostream &outfile);
  
  string getname() {return name;}
  bool isNamed(string aname){return aname==name;}

  void expandNode(list<GrammarNode *> aNodeList, int depth=0);
  void simplifyBrackets();
  virtual NetworkNodePair expandNetwork(Decoder &network, NetworkNodePair node_pair, bool join_to_end);

  void countNodes(int &n);

  virtual bool isALeafNode(){return false;}
  virtual bool isALinkNode(){return false;}
  virtual bool isAFactorNode(){return false;}
  virtual bool isAnAlternativeNode() {return is_alternative;}
  virtual bool isASimpleFactorNode(){return false;}
  virtual bool isAnOptionalFactorNode(){return false;}
  virtual bool isARepeatableFactorNode(){return false;}
  virtual bool isAnOptionalRepeatableFactorNode(){return false;}
  
  
protected:

  void parseName(char *&value);
  virtual void parse(char *&value);
  char stop_char(){return a_stop_char;}
  void set_stop_char(char stop_char) {a_stop_char=stop_char;}

  
  void print(ostream &outfile, char start, char end, char space) {
    outfile << start;
    for (list<GrammarNode *>::iterator np=nodeList.begin(); np!=nodeList.end(); ++np) {
      (*np)->print(outfile);
      if (*np!=nodeList.back()) outfile << space;
    }
    outfile << end;
  }
  
private:
  void addNode(GrammarNode *a_node, bool &alternative);
  void grabNodesFrom(GrammarNode *anode);
  void init();
  void deleteRedundantDoubleBrackets();
  void deleteRedundantLeafBrackets();

  // Remove child nodes (but don't delete them)
  void removeChildren(){nodeList.clear();}
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: GrammarLeafNode	               		      	      */
/*									      */
/******************************************************************************/

class GrammarLeafNode: public GrammarNode {

public:
  GrammarLeafNode(char *&vp, char stop_char) {set_stop_char(stop_char); parse(vp);};
  GrammarLeafNode(const GrammarLeafNode &node):GrammarNode(node){}
  
  virtual GrammarLeafNode *clone() {return new GrammarLeafNode(*this);}

  virtual void setAlternative(bool x);
  
  virtual NetworkNodePair expandNetwork(Decoder &network, NetworkNodePair node_pair, bool join_to_end);

  ~GrammarLeafNode(){};
  
  virtual bool isALeafNode(){return true;}

protected:
  virtual void parse(char *&value);

  virtual void print(ostream &outfile) {outfile << getname();}
};



/******************************************************************************/
/*									      */
/*	CLASS NAME: GrammarLinkNode	               		      	      */
/*									      */
/******************************************************************************/

class GrammarLinkNode: public GrammarNode {

public:
  GrammarLinkNode(char *&vp, char stop_char) {set_stop_char(stop_char); parse(vp);};
  GrammarLinkNode(const GrammarLinkNode &node):GrammarNode(node){}
  virtual GrammarLinkNode *clone() {return new GrammarLinkNode(*this);}

  ~GrammarLinkNode(){}
  
  virtual bool isALinkNode(){return true;}

protected:
  virtual void parse(char *&value);
  virtual void print(ostream &outfile) {outfile << getname();}
};



/******************************************************************************/
/*									      */
/*	CLASS NAME: GrammarFactorNode	               		      	      */
/*									      */
/******************************************************************************/

class GrammarFactorNode: public GrammarNode {

private:
  bool is_optional;
  bool is_repeatable;

  //
  //
  //
public:
  virtual ~GrammarFactorNode() = 0;
  
protected:
  GrammarFactorNode(bool optional, bool repeatable):is_optional(optional), is_repeatable(repeatable){}
  GrammarFactorNode(char *&vp, char stop_char, bool optional, bool repeatable): is_optional(optional), is_repeatable(repeatable) {++vp;set_stop_char(stop_char); parse(vp);};
  GrammarFactorNode(const GrammarFactorNode &node):GrammarNode(node), is_optional(node.is_optional), is_repeatable(node.is_repeatable){}

  GrammarFactorNode(char *&value, const string &name):GrammarNode(value, name), is_optional(false), is_repeatable(false){};
  
  virtual NetworkNodePair expandNetwork(Decoder &network, NetworkNodePair node_pair, bool join_to_end);

  virtual bool isAFactorNode(){return true;}

};



/******************************************************************************/
/*									      */
/*	CLASS NAME: GrammarSimpleFactorNode	       		      	      */
/*									      */
/******************************************************************************/

class GrammarSimpleFactorNode: public GrammarFactorNode {

public:
  GrammarSimpleFactorNode():GrammarFactorNode((bool)0, (bool)0) {}
  GrammarSimpleFactorNode(char *&vp):GrammarFactorNode(vp, ')', false, false){}  
  GrammarSimpleFactorNode(const GrammarSimpleFactorNode &node):GrammarFactorNode(node) {}

  GrammarSimpleFactorNode(char *&value, const string &name):GrammarFactorNode(value, name){};

  virtual GrammarSimpleFactorNode *clone() {return new GrammarSimpleFactorNode(*this);}

  virtual ~GrammarSimpleFactorNode() {}  // JON FOR CC

protected:
  void print(ostream &outfile){GrammarNode::print(outfile, '(',')', isAnAlternativeNode()?'|':' ');}
  virtual bool isASimpleFactorNode(){return true;}
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: GrammarOptionalFactorNode	       		      	      */
/*									      */
/******************************************************************************/

class GrammarOptionalFactorNode: public GrammarFactorNode {

public:
  GrammarOptionalFactorNode():GrammarFactorNode(1,0) {}
  GrammarOptionalFactorNode(char *&vp):GrammarFactorNode(vp, ']', true, false){}
  GrammarOptionalFactorNode(const GrammarOptionalFactorNode &node):GrammarFactorNode(node) {}
  virtual GrammarOptionalFactorNode *clone() {return new GrammarOptionalFactorNode(*this);}

  virtual ~GrammarOptionalFactorNode() {}  // JON FOR CC

protected:
  void print(ostream &outfile){GrammarNode::print(outfile, '[',']',isAnAlternativeNode()?'|':' ');}
  virtual bool isAnOptionalFactorNode(){return true;}
};



/******************************************************************************/
/*									      */
/*	CLASS NAME: GrammarRepeatableFactorNode	       		      	      */
/*									      */
/******************************************************************************/

class GrammarRepeatableFactorNode: public GrammarFactorNode {

public:

  GrammarRepeatableFactorNode():GrammarFactorNode(0,1) {}
  GrammarRepeatableFactorNode(char *&vp):GrammarFactorNode(vp, '>', false, true){}
  GrammarRepeatableFactorNode(const GrammarRepeatableFactorNode &node):GrammarFactorNode(node) {}
  virtual GrammarRepeatableFactorNode *clone() {return new GrammarRepeatableFactorNode(*this);}

  virtual ~GrammarRepeatableFactorNode() {} // JON FOR CC

protected:
  void print(ostream &outfile){GrammarNode::print(outfile, '<','>',isAnAlternativeNode()?'|':' ');}
  virtual bool isARepeatableFactorNode(){return true;}
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: GrammarOptionalRepeatableFactorNode	       	      	      */
/*									      */
/******************************************************************************/

class GrammarOptionalRepeatableFactorNode: public GrammarFactorNode {

public:
  GrammarOptionalRepeatableFactorNode():GrammarFactorNode(1,1) {}
  GrammarOptionalRepeatableFactorNode(char *&vp):GrammarFactorNode(vp, '}', true, true){}
  GrammarOptionalRepeatableFactorNode(const GrammarOptionalRepeatableFactorNode &node):GrammarFactorNode(node) {}
  virtual GrammarOptionalRepeatableFactorNode *clone() {return new GrammarOptionalRepeatableFactorNode(*this);}

  virtual ~GrammarOptionalRepeatableFactorNode() {} // JON FOR CC


protected:
  void print(ostream &outfile){GrammarNode::print(outfile, '{','}',isAnAlternativeNode()?'|':' ');}
  virtual bool isAnOptionalRepeatableFactorNode(){return true;}
};


#endif

/* End of ctk_parse.hh */
