/******************************************************************************/
/*									      */
/*	ctk_socket.hh	    						      */
/*									      */
/*	Class declarations for ctk_socket.hh				      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			              */
/*									      */
/******************************************************************************/

#ifndef CTK_SOCKET_HH
#define CTK_SOCKET_HH

#include <string>

#include "ctk_local.hh"
#include "ctk_types.hh"   // For CTKVector and CTKSample
#include "ctk_error.hh"

class Pipe;
class FramePipe;
class FloatPipe;
class DataDescriptor;

//
// Declaration and definition of socket classes
//

typedef enum {OUTPUT_SOCKET_TYPE, INPUT_SOCKET_TYPE, INVALID_SOCKET_TYPE} SocketType;

class Socket;
class InputSocket;
class OutputSocket;

class Block;

typedef vector<Socket*>::iterator SocketIt;
typedef vector<Socket*>::const_iterator SocketConstIt;

typedef vector<InputSocket*>::iterator InputSocketIt;
typedef vector<InputSocket*>::const_iterator InputSocketConstIt;

typedef vector<OutputSocket*>::iterator OutputSocketIt;
typedef vector<OutputSocket*>::const_iterator OutputSocketConstIt;


/******************************************************************************/
/*									      */
/*	CLASS NAME: SocketList	         				      */
/*									      */
/******************************************************************************/

class SocketList {

private:

  // Declared but not defined
  SocketList(const SocketList&);
  SocketList& operator=(SocketList &);
  
  Block *parent_block;
  Boolean configurable;
  Boolean dirty_flag;         // Set to true when NSockets is changed from its default value
  Integer min_num;  // The minimum allowable number of sockets
  Integer max_num;  // The maximum allowable number of sockets
  
protected:

  vector<Socket*> sockets;
  
  //
  //
public:
  
  virtual SocketIt begin() {return sockets.begin();}
  virtual SocketIt end() {return sockets.end();}

  virtual SocketConstIt begin() const {return sockets.begin();}
  virtual SocketConstIt end() const {return sockets.end();}

  SocketList(Block *block);
  virtual ~SocketList();

  Socket* operator[](int n) const;

  virtual void configure(UInteger n)=0;

  int size() const;
  int num_optional() const;
  int num_non_optional() const;
  
  virtual void add(Socket &this_socket, Block *this_block, Socket* this_subsocketp)=0;
  
  Integer get_min_num() const;
  Integer get_max_num() const;

  Boolean is_configurable() const;
  Boolean get_dirty_flag() const;
  
  Boolean set_configurable(Boolean this_configurable, Integer min_num, Integer max_num);

  Integer count_contiguous_connected_sockets();

  virtual SocketType type() const =0;
  
  Boolean set_description(const string &name, const string &desc);
  Boolean set_optional(const string &name);

protected:
  
  Block* get_parent_block();
  void set_dirty_flag() {dirty_flag=true;}
};



/******************************************************************************/
/*									      */
/*	CLASS NAME: InputSocketList	      				      */
/*									      */
/******************************************************************************/

class InputSocketList: public SocketList {

public:
  
  InputSocketList(Block *block=NULL);
  virtual ~InputSocketList(){};
 
  void configure(UInteger n);

  //  Socket* operator[](int n) const;
  
  InputSocket *add_socket(const string &this_name, Block *this_block, Socket* this_subsocketp);

  void add(Socket &this_socket, Block *this_block, Socket* this_subsocketp);

  friend ostream& operator<< (ostream& out, const InputSocketList& sockets);

  virtual SocketType type() const {return INPUT_SOCKET_TYPE;}

};



/******************************************************************************/
/*									      */
/*	CLASS NAME: OutputSocketList	      				      */
/*									      */
/******************************************************************************/

class OutputSocketList: public SocketList {

public:
  
  OutputSocketList(Block *block=NULL);
  virtual ~OutputSocketList(){};
  
  void configure(UInteger n);
 
  //  Socket* operator[](int n) const;

  OutputSocket *add_socket(const string &this_name, Block *this_block, Socket* this_subsocketp);

  void add(Socket &this_socket, Block *this_block, Socket* this_subsocketp);


  friend ostream& operator<< (ostream& out, const OutputSocketList& sockets);

  virtual SocketType type() const {return OUTPUT_SOCKET_TYPE;}
};




/******************************************************************************/
/*									      */
/*	CLASS NAME: Socket	         				      */
/*									      */
/******************************************************************************/


class Socket {
  
private:

  // Declared but not defined
  Socket(const Socket&);
  Socket& operator=(Socket &);
  
  Socket();

  string name;
  string description;
  
  ///
  Block *blockp;
  Block *baseblockp;

  const DataDescriptor *data_desc;
  
  Socket *supersocketp;
  Socket *subsocketp;
  Socket *basesocketp;

  string remote_socket_name;
  
  Boolean is_a_subsocket;

  Boolean is_optional;  // If TRUE then socket doesn't need to be connected (used only for inputs)
  
  FramePipe *pipepF;    // Pipe for sending/receiving frame data
  FloatPipe *pipepS;    // Pipep for sending/receiving sample data

  //
  //
  //
protected:
  Socket *remote_socketp;
  
 public:

  //    Socket(){};
  ///
  Socket(const string &this_name, Block *this_block, Socket* this_subsocketp);

  virtual Socket *clone(Block *this_block, Socket* this_subsocketp) =0;
  

  virtual ~Socket();

  virtual void fire()=0;

  virtual void reset()=0; 

  Boolean is_name(string aname) const;
  Boolean am_i_called(string aname) const;
  Boolean is_my_fullname(string aname) const;

  Boolean optional() const {return is_optional;}
  
  ///
  string get_name() const {return name;}
  string get_description() const {return description;}
  
  string get_socket_name() const;
  string get_socket_fullname() const;

  string get_subsocket_name() const;

  string get_remote_socket_name() const;

  Block *get_remote_baseblock() const;
  Block *get_remote_block() const;

  const Socket *get_remote_socket() const;
  const Socket *get_super_socket() const;
  
  Boolean connected() const;

  Boolean connect(Socket *this_socket, FramePipe *this_pipepF, FloatPipe *this_pipepS);
  Boolean disconnect(Socket *this_socket);

  void set_data_descriptor(const DataDescriptor *this_data_desc);
  const DataDescriptor *get_data_descriptor();
  
  Boolean am_i_a_subsocket() const;

protected:

  void socket_copy(Socket *socket);

  ///

  Socket *get_basesocketp() const;
  
  Pipe *get_pipep() const;
  FloatPipe *get_pipepS() const;
  FramePipe *get_pipepF() const;

  void delete_pipe();

#if defined _DEBUG
 void check_sample(Float sample) const; // DEBUG only -- check sample is a valid float
#endif

public:
  
  virtual UInteger get_nframes() const {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual void make_primed_on_reset() {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual void reset_fire() {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual Boolean has_fired() const {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual void set_end_of_data() const {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual Boolean get_end_of_data() const {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual Boolean data_ready() const {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual Boolean make_connection(Socket *){throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual Boolean break_connection(Socket *){throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}

  virtual Boolean put_back(Integer) const{throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual void set_nframes(Integer) {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}

  virtual Boolean get_sample(Float &) const{throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual Boolean put_sample(Float) const{throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}

  virtual Boolean read_sample(CTKVector &) const {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual Integer flush_sample(CTKVector &) const {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual Boolean write_sample(const CTKVector &, Integer=0, Integer=0) const {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}

  virtual Boolean get_vector(CTKVector* &) const{throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual Boolean put_vector(CTKVector *) const{throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}

  virtual Boolean read_vector(vector<CTKVector*> &) const {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual Integer flush_vector(vector<CTKVector*> &) const {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}
  virtual Boolean write_vector(const vector<CTKVector*> &, Integer=0, Integer=0) const {throw(CTKErrorGCE(__FILE__, __LINE__, "Internal error!"));}


  Block *get_baseblock() const;
  Block *get_block() const;
  

  Boolean set_description(const string &desc);

  void set_optional() {is_optional=true;}
  
  friend ostream& operator<< (ostream& out, const Socket &socket);
}
;


/// OUTPUT and INPUT sockets


/******************************************************************************/
/*									      */
/*	CLASS NAME: OutputSocket	      				      */
/*									      */
/******************************************************************************/

class OutputSocket: public Socket {

friend class InputSocket;

public:

  OutputSocket(const string &this_name, Block *this_block, Socket* this_subsocketp=NULL);
  
  virtual Socket* clone(Block *this_block, Socket* this_subsocketp);

  void copy(OutputSocket *socket);
  
  virtual ~OutputSocket();

  ///
  void set_end_of_data() const;
  ///
  void reset(); 

  void fire();
  
  virtual Boolean make_connection(Socket *this_input_socket);
  virtual Boolean break_connection(Socket *this_input_socket);


  virtual Boolean put_sample(Float x) const; 
  virtual Boolean put_vector(CTKVector *x) const; 
  //  virtual Boolean put_array(CTKArray *x) const;
  
  virtual Boolean write_sample(const CTKVector &out_data, Integer pos=0, Integer n=0) const;   
  virtual Boolean write_vector(const vector<CTKVector*> &x, Integer pos=0, Integer n=0) const;
  //  virtual Boolean write_array(const vector<CTKArray*> &x, Integer pos=0, Integer n=0) const;

};



/******************************************************************************/
/*									      */
/*	CLASS NAME: InputSocket	         				      */
/*									      */
/******************************************************************************/


class InputSocket: public Socket {

friend class OutputSocket;

private:

  ///
  Integer nframes;
  ///
  Boolean fired;
  ///
  Boolean initial_state;

  //
  //
  //

public:

  InputSocket(const string &this_name, Block *this_block, Socket* this_subsocketp=NULL);

  virtual Socket* clone(Block *this_block, Socket* this_subsocketp);

  void copy(InputSocket *socket);
  
  virtual ~InputSocket();

  ///
  void fire();
  ///
  void reset_fire();
  ///
  virtual Boolean has_fired() const;
  ///
  void make_primed_on_reset();
 
  ///
  void reset();

  ///
  Boolean data_ready() const;
  ///
  Boolean get_end_of_data() const;
  ///
  void set_nframes(Integer n) {nframes=n;};


  // Fetch the data descriptor from the feeding output socket  
  const DataDescriptor *fetch_descriptor(const DataDescriptor *default_dd = NULL);



  virtual Boolean put_back(Integer n) const; 
  
  virtual Boolean get_sample(Float &data) const; 
  virtual Boolean get_vector(CTKVector* &data) const; 
  //  virtual Boolean get_array(CTKArray* &data) const; 
 
  virtual Boolean read_sample(CTKVector &in_data) const; 
  virtual Boolean read_vector(vector<CTKVector*> &in_data) const ;
  //  virtual Boolean read_array(vector<CTKArray*> &in_data) const ;

  virtual Integer flush_sample(CTKVector &in_data) const; 
  virtual Integer flush_vector(vector<CTKVector*> &in_data) const; 
  //  virtual Integer flush_array(vector<CTKArray*> &in_data) const; 
 
 


  
protected:

  UInteger get_nframes() const;
}///
;







#endif 



/* End of ctk_socket.hh */
