/******************************************************************************/
/*									      */
/*	ctk_unary_ops.hh	    				    	      */
/*									      */
/*	Class declarations for ctk_unary_ops.cpp		      	      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			              */
/*									      */
/******************************************************************************/

#ifndef CTK_UNARY_OPS_HH
#define CTK_UNARY_OPS_HH

#include <functional>

#include "ctk_block.hh"

template <class _Tp>
struct sigmoid : public binary_function<_Tp,_Tp,_Tp> {
  _Tp operator()(const _Tp& __x, const _Tp& __T) const { return 1.0/(1.0+exp(-__T * __x)); }
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: UnaryOpBlock                                             */
/*                                                                            */
/******************************************************************************/

class UnaryOpBlock:public Block {

public:

  UnaryOpBlock(const string &a_name, const string &a_block_type);
  virtual ~UnaryOpBlock(){};
  virtual Block* clone(const string &new_name) const=0;

protected:
  virtual void reset();

private:
  virtual void compute()=0;
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: P0_UnaryOpBlock                                          */
/*                                                                            */
/******************************************************************************/

class P0_UnaryOpBlock:public UnaryOpBlock {

public:

  P0_UnaryOpBlock(const string &a_name, const string &a_block_type);
  virtual ~P0_UnaryOpBlock(){};
  virtual Block* clone(const string &new_name) const=0;

protected:
  virtual void reset();

  virtual void compute();

  virtual void process(Float &x)=0;
  
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: P1_UnaryOpBlock                                          */
/*                                                                            */
/******************************************************************************/

class P1_UnaryOpBlock:public UnaryOpBlock {

protected:

  class ParamFloat *X_param;

  // Precomputed to save time
  Float param_1;
  Boolean p1_socket_connected;
  Boolean p1_socket_is_sample_data;
  
public:

  P1_UnaryOpBlock(const string &a_name, const string &a_block_type);
  virtual ~P1_UnaryOpBlock(){};
  virtual Block* clone(const string &new_name) const=0;

protected:
  virtual void reset();
  
  virtual void compute();
  
  virtual void process(Float &x, Float p1)=0;
  
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: P2_UnaryOpBlock                                          */
/*                                                                            */
/******************************************************************************/

class P2_UnaryOpBlock:public UnaryOpBlock {

protected:

  class ParamFloat *X1_param;
  class ParamFloat *X2_param;

  // Precomputed to save time
  Float param_1;
  Float param_2;
  
  Boolean p1_socket_connected;
  Boolean p1_socket_is_sample_data;
  Boolean p1_socket_is_frame_data;

  Boolean p2_socket_connected;
  Boolean p2_socket_is_sample_data;
  Boolean p2_socket_is_frame_data;

public:

  P2_UnaryOpBlock(const string &a_name, const string &a_block_type);
  virtual ~P2_UnaryOpBlock(){};
  virtual Block* clone(const string &new_name) const=0;

protected:
  virtual void reset();
  
  virtual void compute();
  
  virtual void process(Float &x, Float p1, Float p2)=0;
  
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: HalfWaveRectifyBlock                                     */
/*                                                                            */
/******************************************************************************/

class HalfWaveRectifyBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  HalfWaveRectifyBlock(const string &a_name);
  
  virtual ~HalfWaveRectifyBlock(){};
  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  virtual void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: FullWaveRectifyBlock                                     */
/*                                                                            */
/******************************************************************************/

class FullWaveRectifyBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  FullWaveRectifyBlock(const string &a_name);
  virtual ~FullWaveRectifyBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: AmpToDBBlock                                             */
/*                                                                            */
/******************************************************************************/

class AmpToDBBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  AmpToDBBlock(const string &a_name);
  virtual ~AmpToDBBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: DBToAmpBlock                                             */
/*                                                                            */
/******************************************************************************/

class DBToAmpBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  DBToAmpBlock(const string &a_name);
  virtual ~DBToAmpBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: Log10Block                                               */
/*                                                                            */
/******************************************************************************/

class Log10Block:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  Log10Block(const string &a_name);
  virtual ~Log10Block(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: LogBlock                                               */
/*                                                                            */
/******************************************************************************/

class LogBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  LogBlock(const string &a_name);
  virtual ~LogBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: ExpBlock                                                 */
/*                                                                            */
/******************************************************************************/

class ExpBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  ExpBlock(const string &a_name);
  virtual ~ExpBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: LogicalNOTBlock                                          */
/*                                                                            */
/******************************************************************************/

class LogicalNOTBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  LogicalNOTBlock(const string &a_name);
  virtual ~LogicalNOTBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SinBlock                                                 */
/*                                                                            */
/******************************************************************************/

class SinBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  SinBlock(const string &a_name);
  virtual ~SinBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: CosBlock                                                 */
/*                                                                            */
/******************************************************************************/

class CosBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  CosBlock(const string &a_name);
  virtual ~CosBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: TanBlock                                                 */
/*                                                                            */
/******************************************************************************/

class TanBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  TanBlock(const string &a_name);
  virtual ~TanBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: AsinBlock                                                 */
/*                                                                            */
/******************************************************************************/

class AsinBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  AsinBlock(const string &a_name);
  virtual ~AsinBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: AcosBlock                                                 */
/*                                                                            */
/******************************************************************************/

class AcosBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  AcosBlock(const string &a_name);
  virtual ~AcosBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: TanBlock                                                 */
/*                                                                            */
/******************************************************************************/

class AtanBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  AtanBlock(const string &a_name);
  virtual ~AtanBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SinhBlock                                                */
/*                                                                            */
/******************************************************************************/

class SinhBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  SinhBlock(const string &a_name);
  virtual ~SinhBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: CoshBlock                                                */
/*                                                                            */
/******************************************************************************/

class CoshBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  CoshBlock(const string &a_name);
  virtual ~CoshBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: TanhBlock                                                */
/*                                                                            */
/******************************************************************************/

class TanhBlock:public P0_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  TanhBlock(const string &a_name);
  virtual ~TanhBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x);
  
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: FloorBlock                                               */
/*                                                                            */
/******************************************************************************/

class FloorBlock:public P1_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  FloorBlock(const string &a_name);
  virtual Block* clone(const string &new_name) const;

protected:

  void process(Float &x, Float p1);
  
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: CeilingBlock                                             */
/*                                                                            */
/******************************************************************************/

class CeilingBlock:public P1_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  CeilingBlock(const string &a_name);
  virtual ~CeilingBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x, Float p1);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: ScaleBlock                                               */
/*                                                                            */
/******************************************************************************/

class ScaleBlock:public P1_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  ScaleBlock(const string &a_name);
  virtual ~ScaleBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x, Float p1);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: OffsetBlock                                              */
/*                                                                            */
/******************************************************************************/

class OffsetBlock:public P1_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  OffsetBlock(const string &a_name);
  virtual ~OffsetBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x, Float p1);
  
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: PowerBlock                                               */
/*                                                                            */
/******************************************************************************/

class PowerBlock:public P1_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  PowerBlock(const string &a_name);
  virtual ~PowerBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x, Float p1);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SigmoidBlock                                             */
/*                                                                            */
/******************************************************************************/

class SigmoidBlock:public P1_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  SigmoidBlock(const string &a_name);
  virtual ~SigmoidBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x, Float p1);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: GreaterThanXBlock                                        */
/*                                                                            */
/******************************************************************************/

class GreaterThanXBlock:public P1_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  GreaterThanXBlock(const string &a_name);
  virtual ~GreaterThanXBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x, Float p1);
  
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: LessThanXBlock                                           */
/*                                                                            */
/******************************************************************************/

class LessThanXBlock:public P1_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  LessThanXBlock(const string &a_name);
  virtual ~LessThanXBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x, Float p1);
  
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: EqualToXBlock                                           */
/*                                                                            */
/******************************************************************************/

class EqualToXBlock:public P1_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  EqualToXBlock(const string &a_name);
  virtual ~EqualToXBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x, Float p1);
  
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: NotEqualToXBlock                                           */
/*                                                                            */
/******************************************************************************/

class NotEqualToXBlock:public P1_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  NotEqualToXBlock(const string &a_name);
  virtual ~NotEqualToXBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x, Float p1);
  
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: FixBlock                                           */
/*                                                                            */
/******************************************************************************/

class FixBlock:public P1_UnaryOpBlock {

  static const string help_text;
  static const string type_name;

public:

  FixBlock(const string &a_name);
  virtual ~FixBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void process(Float &x, Float p1);
  
};



/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: NormalPDFBlock                                           */
/*                                                                            */
/******************************************************************************/

class NormalPDFBlock:public P2_UnaryOpBlock {

private:

  static const string help_text;
  static const string type_name;

  float gconst;
  float log_gconst;
  float varm05;

  ParamBool *LOG_PROBS_param;
  
  bool log_probs;
  bool constant_variance;
  
  
public:

  NormalPDFBlock(const string &a_name);
  virtual ~NormalPDFBlock(){};

  virtual Block* clone(const string &new_name) const;
  
  virtual const string &get_helptext() const {return help_text;}

protected:

  void reset();
  
  void process(Float &x, Float p1, Float p2);

private:
  
  void calc_gconst_varm05(float variance);

};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: ErfBlock                                                 */
/*                                                                            */
/******************************************************************************/

class ErfBlock:public P2_UnaryOpBlock {

private:

  static const string help_text;
  static const string type_name;

  float varconst;

public:

  ErfBlock(const string &a_name);
  virtual ~ErfBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

protected:

  void reset();
  
  void process(Float &x, Float p1, Float p2);
  
};


#endif

/* End of ctk_unary_ops.hh */
