/******************************************************************************/
/*                                                                  	      */
/*     my_blocks.cpp                                                          */
/*                                                                            */
/*     Example 'user-supplied' blocks to illustrate how toolkit is extended   */
/*                                                                            */
/*     Author: Jon Barker, Sheffield University                               */
/*                                                                            */
/*      CTK VERSION 1.3.5  Apr 22, 2007			      	      */
/*                                                                            */
/******************************************************************************/
 
#include "ctk-config.h"

#include "ctk_local.hh"

#include "ctk_block.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"

#include "my_blocks.hh"

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: WindowBlock                                              */
/*                                                                            */
/******************************************************************************/

const string WindowBlock::help_text =
"Applies a windowing function to a frame of data.";

WindowBlock::WindowBlock(const string &a_name):CTKObject(a_name),Block(a_name, "Window") {

  SHIFT_param= new ParamInt("SHIFT", PARAM_DEFAULT_WINDOW_SHIFT);
  WINDOW_param= new ParamFloatVector("WINDOW", PARAM_DEFAULT_WINDOW_WINDOW);

  parameters->register_parameter(SHIFT_param);
  parameters->register_parameter(WINDOW_param);

  make_input_sockets(1);
  make_output_sockets(1);
}

Block* WindowBlock::clone(const string &n) const{
  Block *ablock = new WindowBlock(n.empty()?getname():n);
  ablock->copy_parameters(parameters);
  if (input_sockets->is_configurable()) ablock->set_num_inputs(input_sockets->size());
  if (output_sockets->is_configurable()) ablock->set_num_outputs(output_sockets->size());
  return ablock;
}

void WindowBlock::reset(){
  Block::reset();
  (*input_sockets)[1-1]->set_nframes(WINDOW_param->get_size());
  data.resize(WINDOW_param->get_size());
  
  overlap=WINDOW_param->get_size()-SHIFT_param->get_value();  
  if (overlap<0) throw(0);  
}

void WindowBlock::compute() {
    Float rms=0.0;
    (*input_sockets)[1-1]->read_sample(data);
  
    for (UInteger i=0; i<data.size(); i++)
      rms+=WINDOW_param->get_value(i)*data[i];
    rms/=data.size();
  
    (*output_sockets)[0]->put_sample(rms);
    (*input_sockets)[0]->put_back(overlap);
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: RampArrayBlock                                           */
/*                                                                            */
/******************************************************************************/

const string RampArrayBlock::help_text =
"Generates frames of data of width N. The first frame is filled with the value START, successive frames are incremented by the amount STEP.";

RampArrayBlock::RampArrayBlock(const string &a_name):CTKObject(a_name),SourceBlock(a_name, "RampArray") {

  START_param= new ParamFloat("START", PARAM_DEFAULT_RAMPARRAY_START);
  STEP_param= new ParamFloat("STEP", PARAM_DEFAULT_RAMPARRAY_STEP);
  N_param= new ParamInt("N");

  parameters->register_parameter(START_param);
  parameters->register_parameter(STEP_param);
  parameters->register_parameter(N_param);

  make_output_sockets(1);
}

Block* RampArrayBlock::clone(const string &n) const{
  Block *ablock = new RampArrayBlock(n.empty()?getname():n);
  ablock->copy_parameters(parameters);
  if (input_sockets->is_configurable()) ablock->set_num_inputs(input_sockets->size());
  if (output_sockets->is_configurable()) ablock->set_num_outputs(output_sockets->size());
  return ablock;
}

void RampArrayBlock::reset(){
  SourceBlock::reset();
  data=START_param->get_value();
}

void RampArrayBlock::compute() {
  data+=STEP_param->get_value();
  (*output_sockets)[0]->put_vector(new CTKVector(N_param->get_value(), data));
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: OutputArrayBlock                                         */
/*                                                                            */
/******************************************************************************/

const string OutputArrayBlock::help_text =
"Writes an array to stdout.";

OutputArrayBlock::OutputArrayBlock(const string &a_name):CTKObject(a_name),Block(a_name, "OutputArray") {
  make_input_sockets(1);
}

Block* OutputArrayBlock::clone(const string &n) const{
  Block *ablock = new OutputArrayBlock(n.empty()?getname():n);
  ablock->copy_parameters(parameters);
  if (input_sockets->is_configurable()) ablock->set_num_inputs(input_sockets->size());
  if (output_sockets->is_configurable()) ablock->set_num_outputs(output_sockets->size());
  return ablock;
}

void OutputArrayBlock::reset(){ 
  Block::reset();
}

void OutputArrayBlock::compute() {
  CTKVector *xv;
  
  (*input_sockets)[0]->get_vector(xv);

  cout << getfullname() << ": ";
  for (UInteger i=0; i< xv->size(); i++) {
    cout << (*xv)[i] << " ";
  }
  cout << endl;
  
  delete xv;
}


/* End of my_blocks.cpp */

