/******************************************************************************/
/*									      */
/*	TransformHMM.cpp	    						      */
/*									      */
/*	Transform an HMM and output transformed file   			      */
/*	       								      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include "ctk-config.h"

#include <fstream>

#include "ctk.hh"
#include "ctk_HMM.hh"
#include "ctk_HMM_edit.hh"

void usage() {
  cout << "usage: CTKTransHMM mlist instruction_file hmm_infile [hmm_outfile] " << endl;
  exit(-1);
}

typedef void (*CommandFunction) (istream &, SetOfHMMs *);


void execute_MS(istream &command_file, SetOfHMMs *hmms);
void execute_CT(istream &command_file, SetOfHMMs *hmms);
void build_process_map(map<string, CommandFunction> &process_map);


/******************************************************************************/
/*									      */
/*	Function: main	    						      */
/*									      */
/******************************************************************************/

int main(int argc, char *argv[]) 
{
    
  if (argc==1) {
    cout << CTK_VERSION_STRING << endl;
    exit(-1);
  }

  if (argc==2 && (strcmp(argv[1],"-h")==0||strcmp(argv[1],"-?")==0))
    usage();

  if (argc!=5 && argc!=4)
    usage();

  string label_file_name(argv[1]);
  string instruction_file_name(argv[2]);
  string hmm_infile_name(argv[3]);
  string hmm_outfile_name;
  
  if (argc==5) 
    hmm_outfile_name=argv[4];

  // Build hmm name->label map from label file
  map<string, string> name_label_map;
  map<string, list<string> > dictionary;
  
  read_hmm_label_file(label_file_name, name_label_map);

  // Read HMMs from hmm_infile
  SetOfHMMs *hmms = new SetOfHMMs(hmm_infile_name, "\0", name_label_map, dictionary, new HMMMixtureStandard());

  // Execute commands in instruction file
  map<string, CommandFunction> process_map;
  build_process_map(process_map);
  std::ifstream command_file(instruction_file_name.c_str());
  string instruction;
  while (command_file>>instruction) {
    process_map[instruction](command_file, hmms);
  }
  command_file.close();

  // If output file name supplied then write transformed HMM to file
  if (hmm_outfile_name.size()!=0) {
    std::ofstream ofstr(hmm_outfile_name.c_str());
    if (ofstr==NULL) {
      cerr << "Cannot open file: " << hmm_outfile_name << " for writing" << endl;
      throw(CTKError(__FILE__, __LINE__));
    }  
    ofstr << *hmms;
    ofstr.close();
  }
  
}


/******************************************************************************/
/*									      */
/*	Function: execute_MD   						      */
/*									      */
/******************************************************************************/

void execute_MD(istream &, SetOfHMMs *hmms) {
  // Max Duration - Impose max state duration constraint via the HMM topology.
  //
  // Self transitions are removed, and states are transformed into a skippable
  // sequence of N tied-states where N is the max duration. N is expected to be
  // stored in the max_duration parameter of the states of the HMM prior to
  // transformation.
  
  hmms->edit_HMMs(new HMMEditMaxDuration());
}

/******************************************************************************/
/*									      */
/*	Function: execute_MS   						      */
/*									      */
/******************************************************************************/

void execute_MS(istream &command_file, SetOfHMMs *hmms) {
  // Mixture separation - Split state into parallel mix states with N mixes per state
  int nmixes;
  command_file>>nmixes;
  hmms->edit_HMMs(new HMMEditMixtureSeparation(nmixes));
}
/******************************************************************************/
/*									      */
/*	Function: execute_PT   						      */
/*									      */
/******************************************************************************/

void execute_PT(istream &command_file, SetOfHMMs *hmms) {
  // Prune transitions for which the transition probability is less than a given threshold
  float threshold;
  command_file>>threshold;
  hmms->edit_HMMs(new HMMEditPruneTransitions(threshold));
}

/******************************************************************************/
/*									      */
/*	Function: execute_CT  						      */
/*									      */
/******************************************************************************/

void execute_CT(istream &command_file, SetOfHMMs *hmms) {
  // Count transitions with probability > thresh
  
  float thresh;
  command_file>>thresh;
	
  int ntrans=0;
  for (int i=0; i<hmms->get_num_HMMs(); ++i) {
    const HMM* hmm = hmms->get_HMM(i);
    Transitions trans = hmm->getTrans();
    for (int j=0; j<trans.size(); ++j) {
      for (int k=0; k<trans.size(); ++k) {
	if (trans.get_transition_prob(j,k)>thresh)
	  ++ntrans;
      }
    }
  }

  cerr << "Number of transitions with p>" << thresh << " = " << ntrans << "\n";
}


/******************************************************************************/
/*									      */
/*	Function: build_process_map 	       				      */
/*									      */
/******************************************************************************/

void  build_process_map(map<string, CommandFunction> &process_map) {
  // Prune transitions
  process_map["PT"] = execute_PT;
  // Count transitions
  process_map["CT"] = execute_CT;
  // Mixture separation
  process_map["MS"]= execute_MS;
  // Max Duration
  process_map["MD"]= execute_MD;
}



/* End of CTKScript.cpp */
