/******************************************************************************/
/*									      */
/*	ctk_gui.cpp	      		        			      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*						   		 	      */
/*      CTK VERSION 1.3.5  Apr 22, 2007	           		              */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <fstream>

#include <qlayout.h>
#include <qdir.h>
#include <qapplication.h>
#include <qlineedit.h>
#include <qmenubar.h>
#include <qpainter.h>
#include <qpopupmenu.h>
#include <qpixmap.h>
#include <qmessagebox.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qfiledialog.h>
#include <qstatusbar.h>
#include <qwhatsthis.h>
#include <qprinter.h>
#include <qstylesheet.h>
#include <qscrollview.h>

#include "ctk_shell.hh"

#include "ctk_gui_blockw.hh"
#include "ctk_helpwindow.hh"
#include "ctk_gui_preferences.hh"
#include "ctk_gui_grid.hh"
#include "ctk_gui_centralw.hh"
#include "ctk_gui_undo.hh"

#include "ctk_gui.hh"

#include "images/filesave.xpm"
#include "images/filenew.xpm"
#include "images/fileopen.xpm"
#include "images/fileprint.xpm"
#include "images/find.xpm"
#include "images/start.xpm"
#include "images/stop.xpm"
#include "images/toggleprofile.xpm"
#include "images/undo.xpm"


//const int STATUS_BAR_HEIGHT = 36;
const int STATUS_MESSAGE_DURATION = 2000;
const int WARNING_MESSAGE_DURATION = 2000;

const char * fileNewText = 
"Click this button to open a <em>new file</em>. <br><br>"
"You can also select the <b>New command</b> from the File menu.";
const char * fileOpenText = 
"Click this button to load a <em>file</em>. <br><br>"
"You can also select the <b>Open command</b> from the File menu.";
const char * fileSaveText = "Click this button to save the file you are "
"editing.  You will be prompted for a file name.\n\n"
"You can also select the Save command from the File menu.";
const char * filePrintText = "Click this button to print the file you "
"are editing.\n\n"
"You can also select the Print command from the File menu.";
const char * undoText= "Click this button to undo the last edit.";
const char * toggleProfileText = "Toggles CTK network profiling mode on and off\n";
const char * startText= "Run the system";
const char * stopText="Stop a running system";
const char * findText="Find a block given its name";

PreferencesWidget *CTKMainWindow::preferencesw(NULL);


/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKScrollView	        			      */
/*									      */
/******************************************************************************/

CTKScrollView::CTKScrollView(CentralWidget *parent, bool read_only/*=false*/):QScrollView(parent) {
  layout_frame = new LayoutFrame(parent, this, read_only);
  connect(layout_frame, SIGNAL(scrollBy(int, int)), this, SLOT(scrollBy(int, int)));
  addChild(layout_frame);
}

CTKScrollView::~CTKScrollView() {
  delete layout_frame;
}

LayoutFrame *CTKScrollView::getLayoutFrame() const {
  return layout_frame;
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKMainWindow		        	      	      */
/*									      */
/******************************************************************************/

CTKMainWindow::CTKMainWindow(QWidget* parent/*=0*/, const char* name/*=0*/, const char* ctk_file_name/*=0*/) :
  QMainWindow(parent,name) {

  preferencesw=new PreferencesWidget();
  
  
  resize(600,400);

  trying_to_enter_shell=false;
  
  printer = new QPrinter;
  status_bar = statusBar();

  QFont boldfont; boldfont.setWeight(QFont::Bold);
 
  int id;
  QPixmap newIcon, openIcon, saveIcon, printIcon, undoIcon, toggleProfileIcon, startIcon, stopIcon, findIcon;
  
  newIcon = QPixmap( filenew_xpm );
  openIcon = QPixmap( fileopen_xpm );
  saveIcon = QPixmap( filesave_xpm );
  printIcon = QPixmap( fileprint_xpm );
  undoIcon = QPixmap( undo_xpm );
  toggleProfileIcon = QPixmap( toggleprofile_xpm );
  startIcon = QPixmap( start_xpm );
  stopIcon = QPixmap( stop_xpm );
  findIcon = QPixmap( find_xpm );
  
  QMenuBar* menubar = menuBar();

  file_tools = new QToolBar( this, tr("file operations"));

  QToolButton * fileNewButton
    = new QToolButton( QIconSet(newIcon), tr("New File"), QString::null,
		       this, SLOT(newDoc()), file_tools, tr("new file") ); 
  QToolButton * fileOpenButton
    = new QToolButton( QIconSet(openIcon), tr("Open File"), QString::null,
		       this, SLOT(openCTKFile()), file_tools, tr("open file") ); 
  fileSaveButton
    = new QToolButton( QIconSet(saveIcon), tr("Save File"), QString::null,
		       this, SLOT(saveCTKFile()), file_tools, tr("save file") );
  QToolButton * filePrintButton
    = new QToolButton( QIconSet(printIcon), tr("Print File"), QString::null,
		       this, SLOT(print()), file_tools, tr("print file") );
  QToolButton * toggleProfileButton
    = new QToolButton( QIconSet(toggleProfileIcon), tr("Toggle Profile"), QString::null,
		       this, SLOT(toggleProfile()), file_tools, tr("toggle profile") );
  undoButton
    = new QToolButton( QIconSet(undoIcon), tr("Undo"), QString::null,
		       this, SLOT(undo()), file_tools, tr("undo") );

  startButton
    = new QToolButton( QIconSet(startIcon), tr("Start"), QString::null,
		       this, SLOT(runSystem()), file_tools, tr("start") );

  stopButton
    = new QToolButton( QIconSet(stopIcon), tr("Stop"), QString::null,
		       this, SLOT(stopSystem()), file_tools, tr("stop") );

  findButton
    = new QToolButton( QIconSet(findIcon), tr("Find"), QString::null,
		       this, SLOT(find()), file_tools, tr("find") );

  fileSaveButton->setEnabled(false);  
  undoButton->setEnabled(false);
  stopButton->setEnabled(false);
  

  (void)QWhatsThis::whatsThisButton( file_tools );
  
  QWhatsThis::add( fileNewButton, fileNewText );
  QWhatsThis::add( fileOpenButton, fileOpenText );
  QWhatsThis::add( fileSaveButton, fileSaveText );
  QWhatsThis::add( filePrintButton, filePrintText );
  QWhatsThis::add( toggleProfileButton, toggleProfileText );
  QWhatsThis::add( undoButton, undoText);
  QWhatsThis::add( startButton, startText);
  QWhatsThis::add( stopButton, stopText);
  QWhatsThis::add( findButton, findText);

  //  Set up 'File' pop up menu
  
  fileMenu = new QPopupMenu(this);
  menubar->insertItem( tr("&File"), fileMenu );
  fileMenu->insertItem( tr("&New"), this, SLOT(newDoc()), CTRL+Key_N );
  id=fileMenu->insertItem(openIcon, tr("&Open"), this,  SLOT(openCTKFile()), CTRL+Key_O );
  fileMenu->setWhatsThis( id, fileOpenText );

  fileMenuSaveID=fileMenu->insertItem(saveIcon, tr("&Save"), this,  SLOT(saveCTKFile()), CTRL+Key_S );

  fileMenu->setItemEnabled ( fileMenuSaveID, false );

  fileMenu->setWhatsThis( fileMenuSaveID, fileSaveText );

  fileMenu->insertItem(tr("Save As..."), this,  SLOT(saveAsCTKFile()));

  fileMenu->insertItem(tr("Revert"), this,  SLOT(revertCTKFile()));


  fileMenu->insertSeparator();

  
  id = fileMenu->insertItem( printIcon,tr("&Print"),
			 this, SLOT(print()), CTRL+Key_P );
  fileMenu->setWhatsThis( id, filePrintText );
  fileMenu->insertSeparator();
  fileMenu->insertItem( tr("&Quit"), this,  SLOT(quit()), CTRL+Key_Q );
  
  // Set up 'Edit' pop up menu
  
  editMenu = new QPopupMenu(this);
  menubar->insertItem( tr("Edit"), editMenu );
  editMenu->insertItem( tr("Find"), this,  SLOT(find()));
  editMenu->insertItem( tr("Unselect all"), this,  SLOT(unselectAll()));
  editMenu->insertSeparator();
  editMenuUndoID=editMenu->insertItem( tr("Undo"), this,  SLOT(undo()));
  editMenu->insertSeparator();
  editMenu->insertItem( tr("Preferences..."), this,  SLOT(preferences()));

  editMenu->setItemEnabled ( editMenuUndoID, false );

  // Set up 'Layout' pop up menu

  layoutMenu = new QPopupMenu(this);
  menubar->insertItem( tr("Layout"), layoutMenu );
  layoutMenu->setCheckable(TRUE);
  layoutMenuToggleGridID=layoutMenu->insertItem( tr("Toggle Grid"), this,  SLOT(toggleGrid()));
  layoutMenuToggleSnapID=layoutMenu->insertItem( tr("Toggle Snap"), this,  SLOT(toggleSnap()));


  // Set up 'Run' pop up menu

  
  runMenu = new QPopupMenu(this);
  menubar->insertItem( tr("Run"), runMenu );
  runMenuRunID=runMenu->insertItem( tr("&Run"), this,  SLOT(runSystem()), CTRL+Key_R );
  runMenuRunWithID=runMenu->insertItem( tr("Run with..."), this,  SLOT(runSystemWith()));
  runMenu->insertSeparator();
  runMenuStopID=runMenu->insertItem( tr("Stop"), this, SLOT(stopSystem()) );

  runMenu->setItemEnabled ( runMenuStopID, false );
 
  
  // Set up 'Profile' pop up menu

  
  profileMenu = new QPopupMenu(this);
  menubar->insertItem( tr("Profile"), profileMenu );
  profileMenu->setCheckable(TRUE);
  profileMenuToggleProfileID=profileMenu->insertItem( tr("Profile"), this,  SLOT(toggleProfile()));
  profileMenu->insertItem( tr("Reset"), this,  SLOT(resetProfile()));
  profileMenu->insertItem( tr("Show report..."), this,  SLOT(displayProfileReport()));

#ifdef _HAS_CURSES
  // Set up 'Shell' pop up menu

  // Interactive shell - requires curses
  QPopupMenu* shell = new QPopupMenu(this);
  menubar->insertItem( tr("Shell"), shell );
  shell->insertItem( tr("Shell"), this,  SLOT(enterShell()));
#endif
  
  QPopupMenu * help = new QPopupMenu( this );
  menubar->insertSeparator();
  menubar->insertItem( tr("Help"), help );

  help->insertItem( tr("Help..."), this, SLOT(help()), CTRL+Key_F1);
  help->insertSeparator();
  help->insertItem( tr("About..."), this, SLOT(about()), Key_F1);
  help->insertSeparator();
  help->insertItem( tr("What's This"), this, SLOT(whatsThis()), SHIFT+Key_F1);

  menubar->setSeparator(QMenuBar::InWindowsStyle);

  central=new CentralWidget(this,"CTK");
  setCentralWidget(central);

  connect(central, SIGNAL(warningMessage(const QString &)), this, SLOT(warningMessage(const QString &)) );
  connect(central, SIGNAL(statusMessage(const QString &)), this, SLOT(statusMessage(const QString &)) );
  connect(central, SIGNAL(caughtCTKError(CTKError &, QString)), this, SLOT(handleCTKError(CTKError &, QString)) );
  connect(&UndoController::theUndoController(), SIGNAL(signalUndoStatusHasChanged(bool)), this, SLOT(undoStatusHasChanged(bool)));
	  
  //  connect(central, SIGNAL(readOnlyMessage(bool)), this, SLOT(readOnlyMessage(bool)) );
  connect(central, SIGNAL(dirtyMessage(bool)), this, SLOT(dirtyMessageSlot(bool)) );
  connect(central, SIGNAL(dirtyMessage(bool)), central, SLOT(dirtyMessageSlot(bool)) );
  connect(central, SIGNAL(readOnlyMessage(bool)), central, SLOT(readOnlyMessageSlot(bool)) );

  statusMessage(tr("Ready"));


  connect(preferencesw, SIGNAL(signalBlockSizePreferenceChanged()), central, SLOT(resizeAllBlockWidgets())); 
  connect(preferencesw, SIGNAL(signalGridSizePreferenceChanged(int)), central->getGrid(), SLOT(valueChanged(int))); 
  connect(preferencesw, SIGNAL(signalProfileModePreferenceChanged(int)), central, SLOT(profileModeHasChanged()));
  connect(preferencesw, SIGNAL(signalBlockIconsModePreferenceChanged(bool)), central, SLOT(profileModeHasChanged()));
  connect(preferencesw, SIGNAL(signalBackgroundColourChanged(QColor)), central, SLOT(setRWLayoutBackgroundColour(QColor)));
  connect(preferencesw, SIGNAL(signalBlockColourChanged(QColor)), central, SLOT(setBlockWidgetsColour(QColor)));

  
  if (ctk_file_name!=0 && strlen(ctk_file_name)!=0) {
    ifstream file(ctk_file_name);
    // Change the main window title bar to show file name 
    setCaption(ctk_file_name);
    if (!file) {
      // If the file doesn't exist then
      // save the (empty) CTK system under the given name
      ((CentralWidget*)centralWidget())->saveAsCTKFile(ctk_file_name);
    } else {
      // Else if the file exists just read it
      ((CentralWidget*)centralWidget())->loadCTKFile(ctk_file_name);
    }
  }
  
}

CTKMainWindow::~CTKMainWindow() {

  // Disconnect signals that connect to the message bar - there will be problems if signals arrive at the message bar slots during destruction of the block layouts but after the MainWindow message bar has been destroyed.
  
  disconnect(central, SIGNAL(dirtyMessage(bool)), this, SLOT(dirtyMessageSlot(bool)) );

  disconnect(central, SIGNAL(dirtyMessage(bool)), central, SLOT(dirtyMessageSlot(bool)) );
  disconnect(central, SIGNAL(readOnlyMessage(bool)), central, SLOT(readOnlyMessageSlot(bool)) );
  
  disconnect(central, SIGNAL(warningMessage(const QString &)), this, SLOT(warningMessage(const QString &)) );
  disconnect(central, SIGNAL(statusMessage(const QString &)), this, SLOT(statusMessage(const QString &)) );
  disconnect(central, SIGNAL(caughtCTKError(CTKError &, QString)), this, SLOT(handleCTKError(CTKError &, QString)) );

  disconnect(&UndoController::theUndoController(), SIGNAL(signalUndoStatusHasChanged(bool)), this, SLOT(undoStatusHasChanged(bool)));

  if (preferencesw!=NULL) delete preferencesw;
  
  delete printer;
  delete status_bar;
}

void CTKMainWindow::newDoc() {
  if (((CentralWidget*)centralWidget())->newDoc()) {
    setCaption("   ");
  }
}

void CTKMainWindow::runSystem() {
  startButton->setEnabled(false);
  runMenu->setItemEnabled ( runMenuRunID, false );
  runMenu->setItemEnabled ( runMenuRunWithID, false );
  stopButton->setEnabled(true);
  runMenu->setItemEnabled ( runMenuStopID, true );
  
  ((CentralWidget*)centralWidget())->runSystem();
  
  stopButton->setEnabled(false);
  runMenu->setItemEnabled ( runMenuStopID, false );
  startButton->setEnabled(true);
  runMenu->setItemEnabled ( runMenuRunID, true );
  runMenu->setItemEnabled ( runMenuRunWithID, true );
}

void CTKMainWindow::runSystemWith() {
  startButton->setEnabled(false);
  runMenu->setItemEnabled ( runMenuRunID, false );
  runMenu->setItemEnabled ( runMenuRunWithID, false );
  stopButton->setEnabled(true);
  runMenu->setItemEnabled ( runMenuStopID, true );

  ((CentralWidget*)centralWidget())->runSystemWith();   

  stopButton->setEnabled(false);
  runMenu->setItemEnabled ( runMenuStopID, false );
  startButton->setEnabled(true);
  runMenu->setItemEnabled ( runMenuRunID, true );
  runMenu->setItemEnabled ( runMenuRunWithID, true );
}

void CTKMainWindow::stopSystem() {
  ((CentralWidget*)centralWidget())->stopSystem();
}

void CTKMainWindow::saveCTKFile() {
  QString filename = ((CentralWidget*)centralWidget())->saveCTKFile();

  if (filename.length()>0)   // Don't change caption if save failed
    setCaption(filename);
}

void CTKMainWindow::saveAsCTKFile() {
  QString  filename= ((CentralWidget*)centralWidget())->saveAsCTKFile();
  setCaption(filename);
}

void CTKMainWindow::revertCTKFile() {  // Revert to saved
  ((CentralWidget*)centralWidget())->revertCTKFile();
}

void CTKMainWindow::toggleGrid() {
  layoutMenu->setItemChecked(layoutMenuToggleGridID, !layoutMenu->isItemChecked(layoutMenuToggleGridID));
  ((CentralWidget*)centralWidget())->toggleGrid();
}

void CTKMainWindow::toggleSnap() { 
  layoutMenu->setItemChecked(layoutMenuToggleSnapID, !layoutMenu->isItemChecked(layoutMenuToggleSnapID));
  ((CentralWidget*)centralWidget())->toggleSnap();
}

void CTKMainWindow::toggleProfile() { 
  profileMenu->setItemChecked(profileMenuToggleProfileID, !profileMenu->isItemChecked(profileMenuToggleProfileID));
  ((CentralWidget*)centralWidget())->toggleProfile();
}

void CTKMainWindow::resetProfile() { 
  ((CentralWidget*)centralWidget())->resetProfile();
}

void CTKMainWindow::displayProfileReport() {
  ((CentralWidget*)centralWidget())->displayProfileReport();
}

void CTKMainWindow::undo() {
  ((CentralWidget*)centralWidget())->undo();
}

void CTKMainWindow::find() {
  ((CentralWidget*)centralWidget())->find();
}

void CTKMainWindow::unselectAll() {
  ((CentralWidget*)centralWidget())->unselectAll();
}

void CTKMainWindow::preferences() {
  preferencesw->raise();
  preferencesw->show();
}

void CTKMainWindow::enterShell() {
  // Close down the main window
  trying_to_enter_shell=true;
  hide();
}

void CTKMainWindow::hideEvent ( QHideEvent * e) {
  QMainWindow::hideEvent(e);
  if (trying_to_enter_shell) {
    enterShell_partII();
   }
}

void CTKMainWindow::enterShell_partII() {
  trying_to_enter_shell=false;
  
  // Construct a shell in the window accepting stdout
  CTK_Shell *shell=new CTK_Shell(&cin, true);  // true = interactive shell
  shell->run();
  delete shell;
  
  // The user has quit from the shell, so bring back the GUI window
  showNormal();
  repaint();
}

void CTKMainWindow::print() {

    if ( printer->setup(this) ) {		// printer dialog
	statusMessage(tr("Printing..."));
	show();
	repaint();
	QApplication::flushX ();
	QPixmap pm = QPixmap::grabWindow (winId());
	QPainter p;
	p.begin( printer );// paint on printer
	p.drawPixmap( 10, 10, pm );
	p.end();				// send job to printer
	statusMessage(tr("Printing completed"));
    } else {
	statusMessage(tr("Printing aborted"));
    }
}

void CTKMainWindow::quit() {
  if (((CentralWidget*)centralWidget())->cleanUpBeforeQuitting())
    qApp->quit();
}

void CTKMainWindow::about() {
    QMessageBox::about( this, "CTK GUI Interface 1.0",
			 QString(CTK_VERSION_STRING)
			+ "\nThe RESPITE CASA Toolkit Project\n");

}

void CTKMainWindow::help() {
  QStyleSheetItem* style;
  
  // Modify the application-wide default style sheet to handle
  // some extra HTML gracefully.
  //
  // Ignore any bodytext in <head>...</head>:
  style = new QStyleSheetItem( QStyleSheet::defaultSheet(), "head" );
  style->setDisplayMode(QStyleSheetItem::DisplayNone);
  //
  // Not in default style sheet, just fake it:
  style = new QStyleSheetItem( QStyleSheet::defaultSheet(), "dl" );
  style->setDisplayMode(QStyleSheetItem::DisplayBlock);
  style = new QStyleSheetItem( QStyleSheet::defaultSheet(), "dt" );
  style->setDisplayMode(QStyleSheetItem::DisplayBlock);
  style->setContexts("dl");
  //
  // Many HTML files omit the </p> or </li>, so we add this for efficiency:
  QStyleSheet::defaultSheet()->item("p")->setSelfNesting( FALSE );
  QStyleSheet::defaultSheet()->item("li")->setSelfNesting( FALSE );
  
  QString home;
  
  home = QString(getenv("CTKROOT")) + "/doc/html/index.html";

  ( new HelpWindow(home, home, 0, "CTK help") )->show();

}

int CTKMainWindow::openCTKFile()
{
  QFileDialog::Mode mode = QFileDialog::ExistingFile;
  QString start = QDir::currentDirPath();
  QString caption = "tr(Choose a CTK script)";
  QString filter = "*.ctk";
    
  QFileDialog fd( QString::null, filter, 0, 0, TRUE );
  fd.setMode( mode );
  fd.setCaption( caption );
  fd.setSelection( start );
  if ( fd.exec() == QDialog::Accepted ) {
    QString result = fd.selectedFile();
    if (((CentralWidget*)centralWidget())->loadCTKFile(result)==CTK_SUCCESS) {
      // Change the main window title bar to show file name
      setCaption(result);
    }
    
    return 0;
  } else {
    return 1;
  }
  
}


void CTKMainWindow::warningMessage(const QString &message) {
  if (status_bar!=NULL)
    status_bar->message(message, WARNING_MESSAGE_DURATION);
}

void CTKMainWindow::statusMessage(const QString &message) {
  if (status_bar!=NULL)
    status_bar->message(message, STATUS_MESSAGE_DURATION);
}

void CTKMainWindow::handleCTKError(CTKError &error, QString extra/*="\0"*/) {
  char *buffer = new char[1024];
  error.get_message(buffer);
  cerr << buffer << endl;
  QString message=buffer;
  if (extra.length()!=0)
    message+=(QString(": ")+extra);
  statusBar()->message(message, WARNING_MESSAGE_DURATION);
  delete buffer;
}

 
void CTKMainWindow::dirtyMessageSlot(bool is_dirty) {
  fileSaveButton->setEnabled(is_dirty);
  fileMenu->setItemEnabled ( fileMenuSaveID, is_dirty );
}

void CTKMainWindow::undoStatusHasChanged(bool undo_status) { 
  undoButton->setEnabled(undo_status);
  editMenu->setItemEnabled ( editMenuUndoID, undo_status );
}
 
/******************************************************************************/
/*									      */
/*	CLASS NAME: GetCommandLineModal	                 	      	      */
/*									      */
/******************************************************************************/

GetCommandLineModal::GetCommandLineModal( QWidget *parent, QString name )
  : QDialog( parent, name, TRUE )
{
  argc=0;
  argv=NULL;
  
  QBoxLayout *top_layout = new QVBoxLayout(this,5,5);

  line_edit = new QLineEdit(this);
  
  top_layout->addWidget(line_edit);

  QBoxLayout *row2_layout = new QHBoxLayout(top_layout);
  
  QPushButton *ok, *cancel;
  ok = new QPushButton( "OK", this );
  connect( ok, SIGNAL(clicked()), SLOT(okClicked()) );
  connect( line_edit, SIGNAL(returnPressed()), SLOT(okClicked()) );
  
  cancel = new QPushButton( "Cancel", this );
  connect( cancel, SIGNAL(clicked()), SLOT(reject()) );

  row2_layout->addWidget(ok);
  row2_layout->addWidget(cancel);
  
}

void GetCommandLineModal::okClicked() {
  QString text = line_edit->text();
  free_argv();
  build_argv(text);
  accept();
}

void GetCommandLineModal::free_argv() {
  for (int i=0; i<argc; ++i)
    free (argv[i]);
  free (argv);

  argc=0;
}

void GetCommandLineModal::build_argv(QString text) {
  vector<QString> words;

  QString word;
  QString simple=text.simplifyWhiteSpace();


  words.push_back("CTK");  // First word on the command line must be "CTK"
  words.push_back("UNKNOWN");  // Second word on the command line 

  
  if (simple.length()>0) {  
    for (unsigned int i=0; i<simple.length(); ++i) {
      if (simple[i].isSpace()) {
	words.push_back(word);
	word="\0";
      } else {
	word+=simple[i];
      }
    }
    words.push_back(word);
  }
  
  argc=words.size();
  argv=new char*[argc];
  for (int i=0; i<argc; ++i) {
    argv[i]=strdup(words[i].ascii());
  }
    
}


int GetCommandLineModal::get_argc() const {
  return argc;
}

char **GetCommandLineModal::get_argv() const {
  return argv;
}


/////////////////////


int main( int argc, char** argv )
{
    QApplication app( argc, argv );

#ifdef _HAS_QTAI

    //   QFigureBlock::setApplication(&app);

#endif
    
    QString lang="en";
    QString ctk_file;
    
    if ( argc != 2 )
      ctk_file="\0";
     else 
       ctk_file = argv[1];

    QString lfile = QString(getenv("CTKROOT")) + "/src/GUI/" + "ctk_" + lang + ".qm";

    QFileInfo fi( lfile );
    if ( !fi.exists() ) {
        QMessageBox::warning( 0, "File error",
                  QString("Cannot find translation for language: "+lang+
                  "\n(try 'de', 'en' or 'dk')") );
        return 0;
    }

    QTranslator translator( 0 );
    translator.load( lfile, "." );
    app.installTranslator( &translator );

    CTKMainWindow ctk_main(0, "CTK", ctk_file.ascii());
    
    ctk_main.show();

    QObject::connect(qApp, SIGNAL(lastWindowClosed()), qApp, SLOT(quit()));

    return app.exec();
}

#include "moc_ctk_gui.cpp"

/* End of ctk_gui.cpp */
