/******************************************************************************/
/*									      */
/*	ctk_gui.cpp	      		        			      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*						   		 	      */
/*      CTK VERSION 1.3.5  Apr 22, 2007	           		              */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <algorithm>
#include <string>

#include <qpainter.h>
#include <qcolor.h>
#include <qpen.h>
#include <qsize.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qtextstream.h>
#include <qfile.h>

#include "ctk_block.hh"

#include "ctk_gui_grid.hh"
#include "ctk_gui_centralw.hh"
#include "ctk_gui_profile.hh"
#include "ctk_gui_param.hh"
#include "ctk_gui_preferences.hh"
#include "ctk_gui_undo.hh"

#include "ctk_gui.hh"

#include "ctk_gui_blockw.hh"

extern string  expand_environment_variables(const string &word);

const QColor RONLY_LAYOUT_BGROUND_COLOR = QColor(80, 255, 255);
const QColor GROUPBLOCK_BGROUND_COLOR = QColor(80, 180, 255);

const QColor INNER_TRACK_COLOR = Qt::red;
const QColor OUTER_TRACK_COLOR = Qt::black;
const QColor OPTIONAL_SOCKET_COLOR = Qt::blue;

const QColor SOCKET_CONNECTED_COLOR = INNER_TRACK_COLOR;
const QColor SOCKET_UNCONNECTED_COLOR = Qt::yellow;

const int CONNECTOR_WIDTH = 6;
// Tolerance within block edges within which user is considered to be clicking sockets
const int BLOCK_EDGE_WIDTH = 12;
// Distance by which a user can 'miss' a block and still activate a socket
const int BLOCK_CLICK_TOLERANCE = 10;
const int XDIST_MIN = 10;

// Size of the markings that indicate blocks are selected
const int SELECTED_MARKING_SIZE = 6;

const QSize LAYOUT_SIZE = QSize(2000, 2000);



/******************************************************************************/
/*									      */
/*	CLASS NAME: BlockWidget		        			      */
/*									      */
/******************************************************************************/

BlockWidget::BlockWidget(Block *a_block, LayoutFrame *aframe): QPushButton(a_block->get_blocktype().c_str(), aframe){

  frame=aframe;
  block=a_block;

  resize(CTKMainWindow::preferencesw->getBlockSize());

  socket_number_selected  = -1;

  setBlockText(aframe->getCentral()->getProfileSwitch());

  buildSockets(block->get_input_sockets(), fromBlockList, fromSocketList, fromSubsocketList);
  buildSockets(block->get_output_sockets(), toBlockList, toSocketList, toSubsocketList);

  // Add the full name as a tool tip
  QToolTip::add(this, QRect(BLOCK_EDGE_WIDTH,0,width()-2*BLOCK_EDGE_WIDTH,height()), block->get_blocktype().c_str());

  QWhatsThis::add(this, block->get_helptext().c_str());

  if ((block->get_subblocks()).size()==0)
    setPalette( QPalette( CTKMainWindow::preferencesw->getBlockColour() ) );
  else
    setPalette( QPalette( GROUPBLOCK_BGROUND_COLOR ) );

  paramw=NULL;
  profilew=NULL;
  selected=false;
}


void BlockWidget::buildSockets(const SocketList *sockets, vector<BlockWidget*> &blockList, vector<int> &socketList, vector<bool> &subsocketList) {
  int nsockets = sockets->size();
  
  blockList.resize(nsockets, 0);
  socketList.resize(nsockets, 0);
  
  subsocketList.resize(nsockets, 0);
  
  for (int i=0; i<nsockets; ++i)
    subsocketList[i]=((*sockets)[i]->get_super_socket()!=NULL);
  
  addSocketToolTips(sockets);
}

void BlockWidget::addSocketToolTips(const SocketList *sockets) {
  int nsockets=sockets->size();
  
  if (nsockets==0) return;

  SocketType socket_type = sockets->type();
  int xpos = (socket_type==INPUT_SOCKET_TYPE)?0:width()-BLOCK_EDGE_WIDTH;

  for (int i=0; i<nsockets; ++i) {
    int ypos=((i+1)*height())/(nsockets+1)-BLOCK_EDGE_WIDTH/2;
    QString tip = (*(sockets->begin()+i))->get_description().c_str();
    const Socket *super_socket = (*(sockets->begin()+i))->get_super_socket();
    if (super_socket!=NULL) tip+=(QString(" [")+super_socket->get_name().c_str()+"]");
    QToolTip::add(this, QRect(xpos,ypos,BLOCK_EDGE_WIDTH, BLOCK_EDGE_WIDTH), tip);
  }

}


BlockWidget::~BlockWidget() {

  for(unsigned int i=0; i<fromBlockList.size(); ++i) {
    disconnectInputSocket(i);
  }

  for(unsigned int i=0; i<toBlockList.size(); ++i) {
    disconnectOutputSocket(i);
  }
  
  UndoController::theUndoController().recordDeleteBlockActionForUndo(frame, this);
 
  frame->remove_subblock(getBlock()->getfullname().c_str());

  if (paramw!=NULL) delete paramw;
  if (profilew!=NULL) delete profilew;
}

bool BlockWidget::disconnectOutputSocket(int socket_number) {
  
  BlockWidget **blockpp=&toBlockList[socket_number];
  if (*blockpp==NULL) return false;
  BlockWidget *blockp = *blockpp;
  *blockpp=NULL;
  blockp->disconnectInputSocket(toSocketList[socket_number]);
  update();
  frame->emitHasChanged();

  // Break the connection in the CTK block structure
  QString from_socket_name=getSocketName(socket_number, OUTPUT_SOCKET_TYPE);
  QString to_socket_name=blockp->getSocketName(toSocketList[socket_number], INPUT_SOCKET_TYPE);

  UndoController::theUndoController().recordDisconnectActionForUndo(frame, this, blockp, socket_number, toSocketList[socket_number]);
  frame->disconnect(from_socket_name, to_socket_name);
      
  return true;
}

bool BlockWidget::disconnectInputSocket(int socket_number) {

  BlockWidget **blockpp=&fromBlockList[socket_number];
  if (*blockpp==NULL) return false;
  BlockWidget *blockp = *blockpp;
  *blockpp=NULL;
  blockp->disconnectOutputSocket(fromSocketList[socket_number]);
  update();
  frame->emitHasChanged();
  return true;
}


// Attempt to change block's name
QString BlockWidget::changeBlockName(const QString &new_name) {
  if (new_name.length()>0) { 
    // Must first check that no other block is using this name

#if defined (_DEBUG)
    assert(frame!=NULL);
#endif
    
    for (unsigned int i=0; i<frame->blockWidgetList.size(); ++i) {
      string aname = frame->blockWidgetList[i]->block->getname();
      if (aname==new_name.ascii()) {
	return QString(block->getname().c_str());
      }
    }

    if (new_name==block->getname().c_str()) return  QString(block->getname().c_str());
    
    UndoController::theUndoController().recordBlockNameChangeActionForUndo(frame, this, block->getname().c_str());
    block->change_name(new_name.ascii());
    
    if (paramw!=NULL)          
      paramw->refreshParameters();
   }
  
  return QString(block->getname().c_str());
}


void BlockWidget::mousePressEvent( QMouseEvent *e ) {

  //Explicitly grab the mouse:
  // I'm having to do this as I'm redirecting mousePressEvents when a user narrowly misses a block.
  
  grabMouse();

  
  clicked_point=e->pos();
  
  frame->block_selected=this;
  frame->blockpp_selected=getBlockpAt(clicked_point);

  switch(e->button()) {
  case LeftButton:
    if (e->state()&ShiftButton) {
	selected=!selected;
	update();
    } else if (!pointIsNearLeftSide(clicked_point)&&!pointIsNearRightSide(clicked_point)) {
      blockMoveMode = TRUE;
      block_from_pos=this->pos();
    }  else {
      if (frame->get_read_only()==false) {
	frame->socket_type_selected=socket_type_selected=getSocketTypeAt(clicked_point);
	frame->socket_number_selected=socket_number_selected=getSocketNumberAt(clicked_point);

	if (socket_number_selected!=-1) {
	  bool connected = (socket_type_selected==INPUT_SOCKET_TYPE)?
	    fromBlockList[socket_number_selected]!=NULL:toBlockList[socket_number_selected]!=NULL;
	  
	  if (!connected) {
	    connectMode = TRUE;
	    dragpos = pos()+clicked_point;
	    update();
	  }
	}
	
      }
    }
    break;
  case MidButton:
    break;
  case RightButton:
    // Action occurs on release
    break;
  default:
    break;
  }
}

void BlockWidget::mouseReleaseEvent( QMouseEvent *e ) {
  switch(e->button()) {
  case LeftButton:
    if (connectMode) {
      if (frame->makeConnectionWithBlockAt(pos()+e->pos())==false)
	frame->repaint();  // Repaint if connection has failed to remove the track being dragged
    } else if (blockMoveMode){
      QPoint new_pos=pos()+e->pos()-clicked_point;
      //    if (frame->buttonAllowedAt(frame->block_selected, new_pos))
      if (this->pos()!=block_from_pos) {
	UndoController::theUndoController().recordMoveActionForUndo(frame, this, block_from_pos);
	frame->moveBlockTo(new_pos);
      }
    }
    
    connectMode = FALSE;
    blockMoveMode = FALSE;
    break;
  case MidButton:
    break;
  case RightButton:   // Delete:  Connection or block
    if (frame->get_read_only()==false) {
      frame->block_selected=NULL;
      
      if (pointIsNearLeftSide(e->pos()) || pointIsNearRightSide(e->pos())) {
	if (disconnectSocketAt(e->pos())!=false)
	  parentWidget()->update();
      } else {
	frame->deleteBlock(this);
      }
    }
    break;
  default:
    break;
  }

  if (socket_number_selected!=-1) {
    frame->socket_number_selected=socket_number_selected=-1;
    update();
  }
  
  frame->block_selected=NULL;


  // The explicit releaseMouse() call is needed to match the explicit grabMouse() in mousePressEvent()
  releaseMouse();
}


void BlockWidget::mouseDoubleClickEvent(QMouseEvent *e) {
  switch(e->button()) {
  case LeftButton:
    displayParameterPanel();
    break;
  case MidButton:
    if (profilew==NULL)
      profilew=new BlockProfileWidget(this);
    profilew->raise();
    profilew->show();
    break;
  default:
    break;
  }
}


void BlockWidget::mouseMoveEvent( QMouseEvent *e ) {

  frame->scrollForPoint(pos()+e->pos());
  
  if (blockMoveMode){
    QPoint new_pos=pos()+e->pos()-clicked_point;
    //    if (frame->buttonAllowedAt(frame->block_selected, new_pos))
    frame->moveBlockTo(new_pos);
  } else if (connectMode) {
    
    QPoint connect_point = (socket_type_selected==OUTPUT_SOCKET_TYPE)?
		      outputConnectionPoint(socket_number_selected):inputConnectionPoint(socket_number_selected);

    QPoint *from_pointp, *to_pointp;
    if (socket_type_selected==OUTPUT_SOCKET_TYPE) {
      from_pointp=&connect_point; to_pointp=&dragpos;
    } else {
      from_pointp=&dragpos; to_pointp=&connect_point;
    }
    
    frame->repaint(connectorUpdateRect(*from_pointp, *to_pointp));

    dragpos=pos()+e->pos();

    QPainter *p=frame->painter();
    p->setPen(QPen( red, CONNECTOR_WIDTH/3 ));
    drawConnectorTrack(p, *from_pointp, *to_pointp);
    delete p;
  }
  
}



void BlockWidget::paintEvent(QPaintEvent *) {

  QPainter *p = new QPainter(this);
  //  p->scale(0.5,0.5);

  drawButton(p);
  
  delete p;
}

void BlockWidget::drawButton(QPainter * p) {
  QPushButton::drawButton(p);
  
  int nsockets=toBlockList.size();
  for (int i=0; i<nsockets; ++i)
    drawSocketAt(p, ((i+1)*height())/(nsockets+1), OUTPUT_SOCKET_TYPE, (toBlockList[i]!=NULL)||
		 (socket_number_selected==i && socket_type_selected==OUTPUT_SOCKET_TYPE) ||
		 toSubsocketList[i], false);

  nsockets=fromBlockList.size();
  for (int i=0; i<nsockets; ++i)
    drawSocketAt(p, ((i+1)*height())/(nsockets+1), INPUT_SOCKET_TYPE, (fromBlockList[i]!=NULL)||
		 (socket_number_selected==i && socket_type_selected==INPUT_SOCKET_TYPE) ||
		 fromSubsocketList[i], (*(block->get_input_sockets()))[i]->optional());

  if (selected)
    drawSelectedMarking(p);
  
}

void BlockWidget::displayParameterPanel() {
  if (paramw==NULL)
    paramw=new ParameterListWidget(this, frame->get_read_only());
  paramw->raise();
  paramw->show();
}
 

bool BlockWidget::pointIsNearRightSide(const QPoint &pos) {
  return (pos.x()>(width()-BLOCK_EDGE_WIDTH));
}

bool BlockWidget::pointIsNearLeftSide(const QPoint &pos) {
  return (pos.x()<BLOCK_EDGE_WIDTH);
}


bool BlockWidget::disconnectSocketAt(const QPoint &pos) {

  SocketType socket_type = getSocketTypeAt(pos);
  
  if (socket_type==INVALID_SOCKET_TYPE)  return false;

  int socket_number = getSocketNumberAt(pos);

  if (socket_number==-1) return false;

  if (socket_type==OUTPUT_SOCKET_TYPE) 
    return disconnectOutputSocket(socket_number);
  else
    return disconnectInputSocket(socket_number);

}


SocketType BlockWidget::getSocketTypeAt(const QPoint &pos) {
  SocketType socket_type;
  
  if (pointIsNearLeftSide(pos))
    socket_type = INPUT_SOCKET_TYPE;
  else if (pointIsNearRightSide(pos))
    socket_type = OUTPUT_SOCKET_TYPE;
  else
    socket_type = INVALID_SOCKET_TYPE;

  return socket_type;
}

BlockWidget **BlockWidget::getBlockpAt(const QPoint &pos) {

  int socket_no = getSocketNumberAt(pos);

  if (socket_no==-1) return NULL;
  
  vector<BlockWidget*> &block_list=(getSocketTypeAt(pos)==INPUT_SOCKET_TYPE)?fromBlockList:toBlockList;

  return &block_list[socket_no];
}



int BlockWidget::getSocketNumberAt(const QPoint &pos) {

  SocketType socket_type = getSocketTypeAt(pos);
  if (socket_type==INVALID_SOCKET_TYPE)
    return -1;

  int nsockets=(socket_type==INPUT_SOCKET_TYPE)?fromBlockList.size():toBlockList.size();

  if (nsockets==0) return -1;
  
  return max(0,min(nsockets-1, int(((float)(pos.y()*(nsockets+1.0))/height())+0.5)-1));

}
	  


void BlockWidget::drawSelectedMarking(QPainter *p) {
  
  QBrush brush(OUTER_TRACK_COLOR);
  p->fillRect(0,0, SELECTED_MARKING_SIZE, SELECTED_MARKING_SIZE, brush);
  p->fillRect(width()-SELECTED_MARKING_SIZE, 0, SELECTED_MARKING_SIZE, SELECTED_MARKING_SIZE, brush);
  p->fillRect(0, height()-SELECTED_MARKING_SIZE, SELECTED_MARKING_SIZE, SELECTED_MARKING_SIZE, brush);
  p->fillRect(width()-SELECTED_MARKING_SIZE, height()-SELECTED_MARKING_SIZE, SELECTED_MARKING_SIZE, SELECTED_MARKING_SIZE, brush);

}

void BlockWidget::drawSocketAt(QPainter *p, int ypos, SocketType socket_type, bool is_connected, bool is_optional) {

  const int half_size=CONNECTOR_WIDTH/2;
  const int half_inner_size=CONNECTOR_WIDTH/6;

  int xpos1=(socket_type==INPUT_SOCKET_TYPE)?0:width()-CONNECTOR_WIDTH;
  int xpos2=(socket_type==INPUT_SOCKET_TYPE)?0:width()-CONNECTOR_WIDTH+(half_size-half_inner_size);
  
  QBrush brush1(is_optional?OPTIONAL_SOCKET_COLOR:OUTER_TRACK_COLOR);
  QBrush brush2(is_connected?SOCKET_CONNECTED_COLOR:SOCKET_UNCONNECTED_COLOR);

  p->fillRect(xpos1, ypos-half_size, CONNECTOR_WIDTH, CONNECTOR_WIDTH, brush1);
  p->fillRect(xpos2, ypos-half_inner_size, CONNECTOR_WIDTH-(half_size-half_inner_size), CONNECTOR_WIDTH/3, brush2);
  

}

bool BlockWidget::connectBlockTo(BlockWidget *toBlock, int from_socket_no, int to_socket_no) {
  BlockWidget **socket;

  if ((socket=toBlock->insertConnection(to_socket_no, this,  from_socket_no, OUTPUT_SOCKET_TYPE))!=NULL) {
    if (this->insertConnection(from_socket_no, toBlock,  to_socket_no, INPUT_SOCKET_TYPE)!=NULL) {
      toBlock->update();
      update();
      return true;
    } else {
      // To connection failed. So remove from connection.
      *socket=NULL;
    }
  }
  return false;
}


BlockWidget **BlockWidget::insertConnection(int at_socket_number, BlockWidget *to_block, int to_socket_number, SocketType to_socket_type) {
  vector<BlockWidget*> &block_list = (to_socket_type==INPUT_SOCKET_TYPE?toBlockList:fromBlockList);
  vector<int> &socket_list = (to_socket_type==INPUT_SOCKET_TYPE?toSocketList:fromSocketList);
    
  if (block_list[at_socket_number]!=NULL) return NULL;
  block_list[at_socket_number]=to_block;
  socket_list[at_socket_number]=to_socket_number;
  return &block_list[at_socket_number];
}


void BlockWidget::drawOutputConnections(QPainter *p) {
  for (unsigned int i=0; i<toBlockList.size(); ++i) {
    BlockWidget *to_block = toBlockList[i];
    if (to_block!=NULL) {
      QPoint to_point = to_block->inputConnectionPoint(toSocketList[i]);
      QPoint from_point = outputConnectionPoint(i);
      drawConnector(p, from_point, to_point);
    }
  }
}

void BlockWidget::drawSuperSocketConnections(QPainter *) {

}


void BlockWidget::drawConnector(QPainter *p, const QPoint &from, const QPoint &to) {
  QPen pen1( OUTER_TRACK_COLOR, CONNECTOR_WIDTH );
  QPen pen2( INNER_TRACK_COLOR, CONNECTOR_WIDTH/3 );
  p->setPen(pen1);
  drawConnectorTrack(p, from, to);
  p->setPen(pen2);
  drawConnectorTrack(p, from, to);
}

void BlockWidget::drawConnectorTrack(QPainter *p, const QPoint &from, const QPoint &to) {
  int xdist=to.x()-from.x();
  int ydist=to.y()-from.y();

  int width=p->pen().width();
  int x_offset = (xdist>XDIST_MIN+XDIST_MIN?width/2:-width/2);
  int y_offset = (ydist>0?width/2:-width/2);
  
  p->moveTo(from);

  if (xdist>(XDIST_MIN+XDIST_MIN)) {
    p->lineTo(from.x()+xdist/2+x_offset, from.y());
    p->moveTo(from.x()+xdist/2, from.y());
    p->lineTo(from.x()+xdist/2, to.y()+y_offset);
    p->moveTo(from.x()+xdist/2, to.y());
  } else {
    p->lineTo(from.x()+XDIST_MIN-x_offset, from.y());
    p->moveTo(from.x()+XDIST_MIN, from.y());
    p->lineTo(from.x()+XDIST_MIN, from.y()+ydist/2+y_offset);
    p->moveTo(from.x()+XDIST_MIN, from.y()+ydist/2);
    p->lineTo(to.x()-XDIST_MIN+x_offset, from.y()+ydist/2);
    p->moveTo(to.x()-XDIST_MIN, from.y()+ydist/2);
    p->lineTo(to.x()-XDIST_MIN, to.y()+y_offset);
    p->moveTo(to.x()-XDIST_MIN, to.y());
  }
  
  p->lineTo(to);
}

QRect BlockWidget::getRedrawRectangle() {
  QRect bigrect, rect;
  QPoint pto, pfrom;
  
  for (unsigned int i=0; i<fromBlockList.size(); ++i) {
    BlockWidget *x=fromBlockList[i];
    if (x!=NULL) {
      pto=inputConnectionPoint(i);
      pfrom=x->outputConnectionPoint(fromSocketList[i]);
      rect = connectorUpdateRect(pfrom, pto);
      bigrect=bigrect|rect;
    }
  }

  for (unsigned int i=0; i<toBlockList.size(); ++i) {
    BlockWidget *x=toBlockList[i];
    if (x!=NULL) {
      pfrom=outputConnectionPoint(i);
      pto=x->inputConnectionPoint(toSocketList[i]);
      rect = connectorUpdateRect(pfrom, pto);
      bigrect=bigrect|rect;
    }
  }
  return bigrect;
}


QRect BlockWidget::connectorUpdateRect(QPoint pfrom, QPoint pto) {
  const int xmargin=XDIST_MIN +CONNECTOR_WIDTH;
  const int ymargin=CONNECTOR_WIDTH;
  QRect rect = QRect(pfrom, pto).normalize();
  rect.setCoords(rect.left()-xmargin, rect.top()-ymargin, rect.right()+xmargin, rect.bottom()+ymargin);
  return rect;
}


QPoint BlockWidget::outputConnectionPoint(int socket_number) {
  int nsockets = toBlockList.size();
  int yoffset= ((socket_number+1)*height())/(nsockets+1);
  return pos()+QPoint(width(), yoffset);
}

QPoint BlockWidget::inputConnectionPoint(int socket_number) {
  int nsockets = fromBlockList.size();
  int yoffset= ((socket_number+1)*height())/(nsockets+1);
  return pos()+QPoint(0, yoffset);
}

QString BlockWidget::getSocketName(int socket_number, SocketType socket_type) {

  QString socket_name;
  QString name = QString(getBlock()->getfullname().c_str());
  QString type = (socket_type==INPUT_SOCKET_TYPE?"in":"out");

  
  socket_name.sprintf("%s:%s%d",name.ascii(),type.ascii(),socket_number+1);

  return socket_name;
  
}

void BlockWidget::refreshParameters() {
  if (paramw!=NULL && paramw->isVisible()) {
    paramw->refreshParameters();
  }
}

void BlockWidget::refreshProfile(bool profile_switch) {
  if (profile_switch) {
    if (profilew!=NULL) {
      profilew->refreshProfile();
    }
  }
  setBlockText(profile_switch);
}


void BlockWidget::setBlockText(bool profile_switch) {
  if (profile_switch==false) {
    if (CTKMainWindow::preferencesw->getDisplayIcons()) 
      setPixmap(QPixmap(block->getIcon()) );
    else 
      setText(QString(block->get_blocktype().c_str()));
  } else {
    QString blocktext=QString(block->get_blocktype().c_str())+"\n";
    if ((block->get_subblocks()).size()==0) {
      // Baselevel block - display profile statistics
      switch (CTKMainWindow::preferencesw->getProfileMode()) {
      case PROFILE_DISPLAY_MODE_PERCENTAGE:
	blocktext+=QString("%1%").arg(block->get_profile_process_time_percentage(), 0, 'g', 3);
	break;
      case PROFILE_DISPLAY_MODE_TIME:
	blocktext+=QString("%1").arg(block->get_profile_process_time(), 0, 'g', 3);
	break;
      case PROFILE_DISPLAY_MODE_CALLS:
	blocktext+=QString("%1").arg(block->get_profile_call_count());
	break;
      }
    } else {
      // Intermediate block - can't report statistics in this version
      blocktext+=QString("?");
    }
    setText(blocktext);
  }
}

int BlockWidget::getMinNumInputs() const {
  return block->get_min_num_inputs();
}

int BlockWidget::getMaxNumInputs() const {
  return block->get_max_num_inputs();
}

int BlockWidget::getMinNumOutputs() const {
  return block->get_min_num_outputs();
}

int BlockWidget::getMaxNumOutputs() const {
  return block->get_max_num_outputs();
}


void BlockWidget::setNumOutputs(int n) {

  if (block->get_num_outputs()==n) return;

  if (n<(int)toBlockList.size()) {
    for (int i=n; i<(int)toBlockList.size(); ++i)
      disconnectOutputSocket(i);
  }

  UndoController::theUndoController().recordOutputSocketChangeActionForUndo(getLayoutFrame(),this,block->get_num_outputs());

  block->set_num_outputs(n);
  buildSockets(block->get_output_sockets(), toBlockList, toSocketList, toSubsocketList);
  frame->connectUpBlocks();
  frame->update();
  update();
  
  if (paramw!=NULL)          
    paramw->refreshParameters();
 
}

void BlockWidget::setNumInputs(int n) {
  
  if (block->get_num_inputs()==n) return;

  if (n<(int)fromBlockList.size()) {
    for (int i=n; i<(int)fromBlockList.size(); ++i)
      disconnectInputSocket(i);
  }

  UndoController::theUndoController().recordInputSocketChangeActionForUndo(getLayoutFrame(),this,block->get_num_inputs());

  block->set_num_inputs(n);
  buildSockets(block->get_input_sockets(), fromBlockList, fromSocketList, fromSubsocketList);
  frame->connectUpBlocks();
  frame->update();
  update();
 
  if (paramw!=NULL)          
    paramw->refreshParameters();
}

void BlockWidget::unsetParameter(QString name) {

  try {
    block->unset_parameter(name.ascii());
    frame->emitHasChanged();   // emit signal to indicate frame has changed
  }
  catch (BlockErrorCFP &error) {
    //    frame->emitWarningMessage(QString(tr("unrecognised parameter name: %1.")).arg(name));
    frame->emitCaughtCTKError(error, "\0");
  }
  catch (CTKError &error) {
    //    frame->emitWarningMessage(tr("Invalid value for parameter."));
    frame->emitCaughtCTKError(error, "\0");
  }
  
  if (paramw!=NULL)          
    paramw->refreshParameters();
}


void BlockWidget::setParameterArgNumber(QString name, int value) {

  try {
    block->set_parameter_arg_number(name.ascii(), value);
    frame->emitHasChanged();   // emit signal to indicate frame has changed
  }
  catch (BlockErrorCFP &error) {
    frame->emitCaughtCTKError(error, "\0");
  }
  catch (CTKError &error) {
    frame->emitCaughtCTKError(error, "\0");
  }
  
  if (paramw!=NULL)          
    paramw->refreshParameters();

}

void BlockWidget::setParameter(QString name, QString value) {

  try {
    block->set_parameter(name.ascii(), value.ascii());
    frame->emitHasChanged();   // emit signal to indicate frame has changed
  }
  catch (BlockErrorCFP &error) {
    //    frame->emitWarningMessage(QString(tr("unrecognised parameter name: %1.")).arg(name));
    frame->emitCaughtCTKError(error, "\0");
  }
  catch (CTKError &error) {
    //    frame->emitWarningMessage(tr("Invalid value for parameter."));
    frame->emitCaughtCTKError(error, "\0");
  }
  
  if (paramw!=NULL)          
    paramw->refreshParameters();
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: LayoutFrame		        			      */
/*									      */
/******************************************************************************/


LayoutFrame::LayoutFrame(CentralWidget* a_central, QWidget *parent, bool read_only/*=false*/) :
  QFrame(parent)
{
  is_read_only=read_only;
  main_block=NULL;

  central=a_central;
  
  block_selected=NULL;
  setFrameStyle(QFrame::Box|QFrame::Sunken);

  setPalette( QPalette((is_read_only)? RONLY_LAYOUT_BGROUND_COLOR : CTKMainWindow::preferencesw->getBackgroundColour()) );
  resize(LAYOUT_SIZE);

  block_select_rectangle=NULL;
}

LayoutFrame::~LayoutFrame() {
  
  for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
    delete blockWidgetList[i];
  }
  
  blockWidgetList.resize(0);

  // Note, the CTK block in the view is not deleted. Block deletion is handled by CentralWidget
}

BlockWidget *LayoutFrame::addNewBlock(const QPoint &pos, const QString &blocktype, const QString &blockname) {
  if (main_block==NULL) return NULL;

  Block *new_block = central->getNewBlock(blockname, blocktype);
 
  Block *copy_block = main_block->add_subblock(new_block);

  if (copy_block->get_subblocks().size()!=0) 
    central->addScrollViewForBlock(new_block, true);   // true=> read_only
    
  BlockWidget *block=new BlockWidget(copy_block, this);

  UndoController::theUndoController().recordConstructBlockActionForUndo(this, block);

  block->move(central->getGrid()->snapToGrid(pos));
  block->show();
  blockWidgetList.push_back(block);
  emit hasChanged();

  // Return widget pointer
  return block;

}


void LayoutFrame::mousePressEvent( QMouseEvent *e ) {

  switch(e->button()) {
  case LeftButton:
    if (is_read_only==false) {
      BlockWidget *missed_block;
      if (e->state()&ShiftButton) {
	// Shift-drag a block selection window
	block_select_rectangle = new DragRectangle(this, e->pos());
      } else if ((missed_block=getBlockNear(e->pos(), QSize(BLOCK_CLICK_TOLERANCE, 0)))==NULL)
	// Add block to layout
	addNewBlock(QPoint(e->pos().x()-CTKMainWindow::preferencesw->getBlockSize().width()/2, e->pos().y()-CTKMainWindow::preferencesw->getBlockSize().height()/2));
      else {
	// Very near another block so treat this like a missed block click 
	QPoint pos = e->pos()-missed_block->pos();
	if (e->pos().x()<missed_block->pos().x()) 
	  pos.setX(0);
	else
	  pos.setX(missed_block->width());

        QMouseEvent *me= new QMouseEvent( QEvent::MouseButtonPress, pos, e->button(), e->state() );
	missed_block->mousePressEvent(me);
	delete me;
	
      }
    }
    break;
  case MidButton:
    break;
  case RightButton:
    break;
  default:
    break;
  }
}

void LayoutFrame::mouseReleaseEvent( QMouseEvent *e ) {

  switch(e->button()) {
  case LeftButton:
    if (block_select_rectangle!=NULL) {
      QRect rect=block_select_rectangle->getRect();
      selectBlocksInRectangle(rect);
      delete block_select_rectangle;
      block_select_rectangle=NULL;
      repaint(rect);
    }
    break;
  case MidButton:
    break;
  case RightButton:
    break;
  default:
    break;
  }
}


void LayoutFrame::mouseMoveEvent( QMouseEvent *e ) {
  if (block_select_rectangle!=NULL) 
    block_select_rectangle->updateRectangle(e->pos());

}

  
void  LayoutFrame::moveBlockTo(const QPoint &pos, const string &blockname ) {
  BlockWidget *bwp = findWidgetForBlockNamed(blockname.c_str());
  if (bwp!=NULL) moveBlockTo(pos, bwp);
}

void LayoutFrame::moveBlockTo(const QPoint &a_pos, BlockWidget *block/*=NULL*/ ) {

  QPoint move_vector;
  
  if (block==NULL) block=block_selected;
 
  if (block!=NULL) {

    QPoint pos=central->getGrid()->snapToGrid(a_pos);

    // Only move block if its position has changed
    if (pos!=block->pos()) {  
      QPainter *p = new QPainter(this);
      //      p->scale(0.5,0.5);

      move_vector=pos-block->pos();
      
      if (!block->getSelected())
	moveBlockBy(move_vector, block);
      else {
	for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
	  if (blockWidgetList[i]->getSelected()) {
	    moveBlockBy(move_vector, blockWidgetList[i]);
	  }
	}
      }

      delete p;
    }
  }
}

void LayoutFrame::moveBlockBy(const QPoint &offset, BlockWidget *block) {

  QRect first_rect = block->getRedrawRectangle();  
  block->move(block->pos()+offset);
  QRect second_rect = block->getRedrawRectangle();
  repaint(first_rect|second_rect);
  emit hasChanged();
}


void LayoutFrame::selectBlocksInRectangle(QRect rect) {
  for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
    BlockWidget *bw=blockWidgetList[i];
    QPoint p1=bw->pos();
    QPoint p2=QPoint(p1.x()+bw->width(), p1.y()+bw->height());
    if (rect.contains(p1) && rect.contains(p2)) {
      bw->setSelected(true);
      bw->update();
    }
  }

}

bool LayoutFrame::makeConnectionWithBlockAt(const QPoint &pos ) {

  BlockWidget *from_block, *to_block;
  int from_socket_number, to_socket_number;

  BlockWidget *with_block=findBlockAt(pos);

  if (with_block==NULL) return false;
  
  int with_socket_number=with_block->getSocketNumberAt(pos-with_block->pos());
  SocketType with_socket_type=with_block->getSocketTypeAt(pos-with_block->pos());

  // Return if mouse not pointing to a valid with socket
  if (with_socket_number==-1 || socket_number_selected==-1)
    return false;
  
  // Cannot connect input to input or output to output
  if (with_socket_type==socket_type_selected)
    return false;
  
  // Note - you can't connect a block to itself
  if (with_block!=NULL && with_block!=block_selected && socket_type_selected!=INVALID_SOCKET_TYPE) {

    if (socket_type_selected==OUTPUT_SOCKET_TYPE) {
      from_block=block_selected; to_block=with_block;
      from_socket_number=socket_number_selected;
      to_socket_number=with_socket_number;
    } else {
      from_block=with_block; to_block=block_selected;
      from_socket_number=with_socket_number;
      to_socket_number=socket_number_selected;
    }

    if (connectBlockFromTo(from_block, to_block, from_socket_number, to_socket_number))
      return true;
    
  }
  return false;
}

bool LayoutFrame::disconnectBlockOutputSocket(const string &blockname, int output_socket_number) {
  BlockWidget *block = findWidgetForBlockNamed(blockname.c_str());

  if (block!=NULL) {
    bool status= block->disconnectOutputSocket(output_socket_number);
    update();
    return status;
  }
  
  return false;
    
}

bool LayoutFrame::connectBlockFromTo(const string &from_blockname, const string &to_blockname, int from_socket_number, int to_socket_number) {
  BlockWidget *fromblock = findWidgetForBlockNamed(from_blockname.c_str());
  BlockWidget *toblock = findWidgetForBlockNamed(to_blockname.c_str());

  if (fromblock!=NULL & toblock!=NULL) 
    return connectBlockFromTo(fromblock, toblock, from_socket_number, to_socket_number);

  return false;
}

bool LayoutFrame::connectBlockFromTo(BlockWidget *from_block, BlockWidget *to_block, int from_socket_number, int to_socket_number) {
  
  if (from_block->connectBlockTo(to_block, from_socket_number, to_socket_number)) {
    
    UndoController::theUndoController().recordConnectActionForUndo(this, from_block, from_socket_number);
    // Make the connection in the CTK block structure
    QString from_socket_name=from_block->getSocketName(from_socket_number, OUTPUT_SOCKET_TYPE);
    QString to_socket_name=to_block->getSocketName(to_socket_number, INPUT_SOCKET_TYPE);
    main_block->connect(from_socket_name.ascii(), to_socket_name.ascii());
    
    QRect first_rect = from_block->getRedrawRectangle();
    QRect second_rect = to_block->getRedrawRectangle();
    
    repaint(first_rect|second_rect);
    from_block->update();
    to_block->update();
    emit hasChanged();
    return true;
  } else
    return false;
}



void LayoutFrame::paintEvent(QPaintEvent *) {
  QPainter *p = new QPainter(this);
  //  p->scale(0.5,0.5);

  for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
    blockWidgetList[i]->update();
  }
  central->getGrid()->drawGrid(p, visibleRect());
  
  for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
    blockWidgetList[i]->drawOutputConnections(p);
    blockWidgetList[i]->drawSuperSocketConnections(p);
  }

  if (block_select_rectangle!=NULL) {
    p->setPen(QPen(Qt::red, 0, Qt::SolidLine));
    p->drawRect(block_select_rectangle->getRect());
  }
  delete p;
}



void LayoutFrame::drawContents ( QPainter * p) {
  //  p->scale(0.5,0.5);
  QFrame::drawContents(p);
}

bool LayoutFrame::buttonAllowedAt(BlockWidget *block, QPoint pos) {
  return (getBlockNear(pos, block->size(), block)==NULL);
}

BlockWidget *LayoutFrame::getBlockNear(QPoint pos, QSize margin, BlockWidget *ignore_this_block/*=NULL*/) {
  BlockWidget *test_block;
  
  for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
    if ((test_block=blockWidgetList[i])==ignore_this_block) continue;  
    QPoint bpos=test_block->pos();
    QRect badrect= QRect(bpos.x()-margin.width(), bpos.y()-margin.height(),
			 2*margin.width()+test_block->width(), 2*margin.height()+test_block->height());
    if (badrect.contains(pos)) return test_block;
  }
  return NULL;
}


void  LayoutFrame::deleteBlock(const string &blockname ) {

  BlockWidget *bwp = findWidgetForBlockNamed(blockname.c_str());
  if (bwp!=NULL) deleteBlock(bwp);

}

void LayoutFrame::deleteBlock(BlockWidget *block) {

  blockWidgetList.erase(std::find(blockWidgetList.begin(), blockWidgetList.end(), block));

  delete block;

  update();

  if (blockWidgetList.size()!=0)
    emit hasChanged();
}


void LayoutFrame::setMainBlock(Block *a_main_block, QString posfilename/*="\0"*/) {

  main_block = a_main_block;
  
  const BlockList blocks=main_block->get_subblocks();

  // Construct block widgets
  for (unsigned int i=0; i<blocks.size(); ++i) {
    BlockWidget *block=new BlockWidget(blocks[i], this);
    block->move(i*30, i*30);
    blockWidgetList.push_back(block);
  }

  // Position block widgets
  // First from the block's own file...
  QString filename=expand_environment_variables(main_block->get_file_name()).c_str();
  if (filename.length()!=0)
    setBlockPositionsFromFile(filename);
  
  // ... then from posfilename if it was supplied.
  if (posfilename.length()!=0) 
    setBlockPositionsFromFile(posfilename);
  
  // Connect up widgets according to CTK block connections
  connectUpBlocks();

  // Display block widgets
  for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
    blockWidgetList[i]->show();
  }

  update();

}



BlockWidget *LayoutFrame::findWidgetForBlockNamed(const QString &name) {
  for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
    BlockWidget *block = blockWidgetList[i];
    if (QString(block->getBlock()->getfullname().c_str())==name)
      return block;
  }
  return NULL;
}

void LayoutFrame::setBlockPositionsFromFile(const QString &filename) {
  QFile file(filename);

  char name[255];
  int x, y;
  
  file.open(IO_ReadOnly);
  QString line;
  
  while (file.readLine(line, 512)>0) {
    QString simple_line = line.simplifyWhiteSpace();
    if (simple_line.left(3)!="#!#") continue;
    if (sscanf(simple_line.ascii(),"#!# %s %d %d\n", name, &x, &y)==3) {
      BlockWidget *block=findWidgetForBlockNamed(name);
      if (block!=NULL)
	block->move(x,y);
    }
  }
}

BlockWidget *LayoutFrame::findBlockAt(const QPoint &pos) {

  for (int i=blockWidgetList.size()-1; i>=0; --i) {
    QRect blockBoundary= QRect(blockWidgetList[i]->pos(),blockWidgetList[i]->size());
    if (blockBoundary.contains(pos)) return blockWidgetList[i];
  }

  return NULL;
}


void LayoutFrame::connectUpBlocks() {
  BlockWidget *blockw;
  OutputSocket *socket;
  const Block *block;

  for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
    blockw=blockWidgetList[i];
    block=blockw->getBlock();
    const OutputSocketList *output_sockets=block->get_output_sockets();
    for (int from_socket_number=0; from_socket_number<output_sockets->size(); ++from_socket_number) {
      socket=(OutputSocket*)(*output_sockets)[from_socket_number];
      if (socket->connected()) {
	const Socket *rsocket=socket->get_remote_socket(); 
	Block *rblock=rsocket->get_block();
	BlockWidget *toBlockw= findWidgetForBlockNamed(rblock->getfullname().c_str());

	if (toBlockw==NULL) {
	  //	  CTKMainWindow::warningMessage(QString( "Cannot find block named: %1" ).arg(rblock->getfullname()));
	} else {
	  int to_socket_number =  QString(socket->get_remote_socket_name().c_str()).right(1).toInt()-1;
	  blockw->connectBlockTo(toBlockw, from_socket_number, to_socket_number);
	}
      }
    }
  }

}

bool LayoutFrame::writeBlockPositionsToFile(QString file_name/*="\0"*/) {

  if (file_name.length()==0)
    file_name=main_block->get_file_name().c_str();
  
  QFile file(file_name);
  if (file.open(IO_ReadWrite|IO_Append)) {
    QTextStream fstr(&file);
    for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
      BlockWidget *block = blockWidgetList[i];
      fstr << "#!# " << block->getBlock()->getfullname().c_str() << " " << block->pos().x() << " " << block->pos().y() << "\n";
    }
    
    file.close();
    return true; 
  } else {
    emit warningMessage(QString(tr("Cannot open file: %1\n")).arg(file_name));
    return false;
  }
}


void LayoutFrame::resizeBlockWidgets(QSize blocksize) {
  for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
    blockWidgetList[i]->resize(blocksize);
  }
  update();
}

// Set colour of base-block widgets
void LayoutFrame::setBlockWidgetsColour(QColor col) {
  for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
    // only set the colour of widgets representing base-blocks, not intermediate `scripted' blocks
    if (blockWidgetList[i]->getBlock()->get_subblocks().size()==0)
      blockWidgetList[i]->setPalette(QPalette(col));
  }
  update();
}

void LayoutFrame::unselectAll() {
  for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
    blockWidgetList[i]->setSelected(false);
    blockWidgetList[i]->update();
  }
  update();
}

void LayoutFrame::refreshParameterWidgets() {
  for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
    blockWidgetList[i]->refreshParameters();
  }
}

void LayoutFrame::refreshProfileWidgets(bool profile_switch) {
  for (unsigned int i=0; i<blockWidgetList.size(); ++i) {
    blockWidgetList[i]->refreshProfile(profile_switch);
  }
}


QString  LayoutFrame::getname() const {
  if (main_block==NULL)
    return QString("\0");
  else
    return QString(main_block->get_blocktype().c_str());
}

// This method is called by BlockWidget during a disconnect, so that the disconnect
// is mirrored in the CTK block structure
void LayoutFrame::disconnect(QString from_socket_name, QString to_socket_name) {

  main_block->disconnect(from_socket_name.ascii(), to_socket_name.ascii());
}

// This method is called by the BlockWidget distructor, so that the block deletion
// is mirrored in the CTK block structure
void LayoutFrame::remove_subblock(QString block_name) {
  main_block->remove_subblock(block_name.ascii());
}


QPainter *LayoutFrame::painter() {
  QPainter *painter= new QPainter(this);
  //  painter->scale(0.5,0.5);
  return painter;
}

void LayoutFrame::scrollForPoint(const QPoint &point) {
  if (visibleRect().contains(point)) return;

  QRect vr=visibleRect();
  
  if (point.x()>vr.right()) {
    emit scrollBy(point.x()-vr.right(),0);
  } else if (point.x()<vr.left()) {
    emit scrollBy(point.x()-vr.left(),0);
  }
 
  if (point.y()>vr.bottom()) {
    emit scrollBy(0, point.y()-vr.bottom());
  } else if (point.y()<vr.top()) {
    emit scrollBy(0, point.y()-vr.top());
  }


}

#include "moc_ctk_gui_blockw.cpp"

/* End of ctk_gui_blockw.cpp */
