/******************************************************************************/
/*									      */
/*	ctk_gui.hh	        					      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007  			      */
/*									      */
/******************************************************************************/

#ifndef CTK_GUI_BLOCKW_HH
#define CTK_GUI_BLOCKW_HH

#include <vector>
#include <string>

#include <qframe.h>
#include <qpoint.h>
#include <qpushbutton.h>
#include <qrect.h>
#include <qstring.h>

#include <ctk_socket.hh>

class QMouseEvent;
class LayoutFrame;
class BlockWidget;
class CentralWidget;
class ParameterListWidget;
class BlockProfileWidget;
class PreferencesWidget;
class QWidget;
class Block;

/******************************************************************************/
/*									      */
/*	CLASS NAME: DragRectangle	        			      */
/*									      */
/******************************************************************************/

// A rectangle that can be dragged about

class DragRectangle {

private:
  
  QWidget *canvas_;
  
  QPoint p_;
  QRect rect_;
  
public:
  
  DragRectangle(QWidget *a_widget, QPoint pos):canvas_(a_widget), p_(pos), rect_(pos, pos) {
  }

  ~DragRectangle() {
  }
  
  void updateRectangle(QPoint pos) {
    QRect oldrect=rect_;
    rect_=QRect(p_, pos);
    rect_=rect_.normalize();
    canvas_->repaint(rect_|oldrect);
  }

  QRect getRect() {
    return rect_;
  }
  

  
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: BlockWidget		        			      */
/*									      */
/******************************************************************************/


class BlockWidget: public QPushButton {
  Q_OBJECT
  
private:
  
  QPoint clicked_point;
  QPoint block_from_pos;
  
  bool blockMoveMode;
  bool connectMode;
  
  LayoutFrame *frame;

  vector<BlockWidget*> toBlockList;
  vector<int> toSocketList;
  vector<bool> toSubsocketList;
  
  vector<BlockWidget*> fromBlockList;
  vector<int> fromSocketList;
  vector<bool> fromSubsocketList;
  
  Block *block;
  ParameterListWidget *paramw;
  BlockProfileWidget *profilew;
  
  int socket_number_selected;
  SocketType socket_type_selected;

  QPoint dragpos;

  bool selected;
  
public:
  
  BlockWidget(Block *, LayoutFrame *);

  ~BlockWidget();

  LayoutFrame *getLayoutFrame() const {return frame;}
  
  void mousePressEvent(QMouseEvent *e);
  void mouseReleaseEvent(QMouseEvent *e );
  void mouseMoveEvent(QMouseEvent *e );
  void mouseDoubleClickEvent(QMouseEvent *e );
  
  void paintEvent(QPaintEvent *e );
  

  bool connectBlockTo(BlockWidget *toBlock, int from_socket_no, int to_socket_no);
  
  void drawOutputConnections(QPainter *p);
  void drawSuperSocketConnections(QPainter *p);

  QRect getRedrawRectangle();

  int getSocketNumberAt(const QPoint &pos);
  SocketType getSocketTypeAt(const QPoint &pos);

  const Block *getBlock(){return block;}
  
  QString getSocketName(int socket_number, SocketType socket_type);

  void refreshParameters();
  void refreshProfile(bool profile_switch);

  void setParameter(QString name, QString value);
  void unsetParameter(QString name);

  void setParameterArgNumber(QString, int value);
  
  bool disconnectOutputSocket(int socket_number);
  bool disconnectInputSocket(int socket_number);

  void clearSelected(){selected=false;}
  bool getSelected() const {return selected;}
  void setSelected(bool bselect) {selected=bselect;}

  void displayParameterPanel();
  
  // Attempt to change block name - returns name after change
  QString changeBlockName(const QString &new_name);

protected:
  
  virtual void drawButton(QPainter *p);
  

public slots:

  int getMinNumInputs() const;
  int getMaxNumInputs() const;
  int getMinNumOutputs() const;
  int getMaxNumOutputs() const;
  void setNumInputs(int n);
  void setNumOutputs(int n);
  
  
private:
  
  BlockWidget **insertConnection(int at_socket_number, BlockWidget* to_block, int to_socket_number, SocketType to_socket_type);
  
  QPoint outputConnectionPoint(int socket_number);
  QPoint inputConnectionPoint(int socket_number);

  void drawConnector(QPainter *p, const QPoint &from, const QPoint &to);
  void drawConnectorTrack(QPainter *p, const QPoint &from, const QPoint &to);
  void drawSelectedMarking(QPainter *p);
  
  void drawSocketAt(QPainter *p, int ypos, SocketType socket_type, bool is_connected, bool is_optional);
  bool disconnectSocketAt(const QPoint &pos);

  bool pointIsNearLeftSide(const QPoint &pos);
  bool pointIsNearRightSide(const QPoint &pos);

  BlockWidget **getBlockpAt(const QPoint &pos);

  void buildSockets(const SocketList *sockets, vector<BlockWidget*> &blockList, vector<int> &socketList, vector<bool> &subsocketList);

  void addSocketToolTips(const SocketList *sockets);

  QRect connectorUpdateRect(QPoint pfrom, QPoint pto);

  void setBlockText(bool profile_switch);
  
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: LayoutFrame		        			      */
/*									      */
/******************************************************************************/

class LayoutFrame : public QFrame {
  Q_OBJECT

  friend class Block;
  friend bool BlockWidget::disconnectOutputSocket(int socket_number);
  friend BlockWidget::~BlockWidget();
 
private:
  

  bool is_read_only;
  
  Block *main_block;
  
  CentralWidget *central;

  DragRectangle *block_select_rectangle;

public:
  // tut tut!
  vector<BlockWidget*> blockWidgetList;  
  int socket_number_selected;
  BlockWidget *block_selected;
  BlockWidget **blockpp_selected;
  SocketType socket_type_selected;

  //
  //
  //
  
public:
  
  LayoutFrame(CentralWidget* central, QWidget *parent, bool read_only = false);
  ~LayoutFrame();
    
  void mousePressEvent( QMouseEvent *e );
  void mouseReleaseEvent( QMouseEvent *e );
  void mouseMoveEvent( QMouseEvent *e );
  
  void moveBlockTo(const QPoint &pos, const string &blockname );
  void moveBlockTo(const QPoint &pos, BlockWidget *block=NULL );

  void selectBlocksInRectangle(QRect rect);
 
  bool makeConnectionWithBlockAt(const QPoint &pos );
  void paintEvent(QPaintEvent *);
  
  bool buttonAllowedAt(BlockWidget *block, QPoint pos);
  BlockWidget *getBlockNear(QPoint pos, QSize margin, BlockWidget *ignore_this_block=NULL);

  BlockWidget *addNewBlock(const QPoint &pos, const QString &blocktype="\0", const QString &blockname="\0");
  
  void deleteBlock(const string &blockname);
  void deleteBlock(BlockWidget *block);
  
  void setMainBlock(Block *a_main_block, QString posfilename="\0");

  void set_dirty(bool state = true);

  void connectUpBlocks();

  QPainter *painter();

  // Returns true for success or else false
  bool writeBlockPositionsToFile(QString file_name="\0");

  void unselectAll(); // Unselected all currently selected blocks

  void refreshParameterWidgets();
  void refreshProfileWidgets(bool profile_switch);
  
  const Block *get_main_block() const {return main_block;}
  
  QString getname() const;
    
  bool get_read_only() const {return is_read_only;}

  void scrollForPoint(const QPoint &point);
  
  void emitHasChanged() {emit hasChanged();}
  void emitWarningMessage(const QString &message) {emit warningMessage(message);}
  void emitCaughtCTKError(CTKError &error, QString extra="\0") {emit caughtCTKError(error, extra);}

  bool connectBlockFromTo(const string &from_blockname, const string &to_blockname, int from_socket_number, int to_socket_number);
  bool connectBlockFromTo(BlockWidget *from_block, BlockWidget *to_block, int from_socket_number, int to_socket_number);

  bool disconnectBlockOutputSocket(const string &blockname, int output_socket_number);

  void resizeBlockWidgets(QSize blocksize);
  void setBlockWidgetsColour(QColor col);

  BlockWidget *findWidgetForBlockNamed(const QString &name);

  CentralWidget *getCentral(){return central;};
  
protected:
  
  void drawContents ( QPainter * p);
  
  signals:
  
  void hasChanged();
  void clicked(int,int);
  void warningMessage(const QString &message);
  void statusMessage(const QString &message);
  void caughtCTKError(CTKError &error, QString extra);

  void scrollBy(int dx, int dy);

private:

  BlockWidget *findBlockAt(const QPoint &pos);

  void setBlockPositionsFromFile(const QString &filename);

  void disconnect(QString from_socket_name, QString to_socket_name);
  void remove_subblock(QString block_name);

  void moveBlockBy(const QPoint &offset, BlockWidget *block=NULL);

};


 
#endif


/* End of ctk_gui_blockw.hh */
