/******************************************************************************/
/*									      */
/*	ctk_gui_cemtralw.cpp	      		        			      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*						   		 	      */
/*      CTK VERSION 1.3.5  Apr 22, 2007	           		              */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <qlayout.h>
#include <qtooltip.h>
#include <qcombobox.h>
#include <qframe.h>
#include <qinputdialog.h>
#include <qlabel.h>
#include <qwhatsthis.h>
#include <qmessagebox.h>
#include <qfiledialog.h>
#include <qtabwidget.h>
#include <qstring.h>
#include <qregexp.h>

#include "ctk_block_headers.hh"

#include "ctk_CTK_reader.hh"
#include "ctk_CTK_writer.hh"
#include "ctk_command_line.hh"
#include "ctk_ro_file.hh"

#include "ctk_gui.hh"
#include "ctk_gui_undo.hh"
#include "ctk_gui_grid.hh"
#include "ctk_gui_blockw.hh"
#include "ctk_gui_preferences.hh"

#include "ctk_gui_centralw.hh"

const char * BLOCK_NAME = "block_";

const char * blocktypeButtonText = "Use this pull down list to select\n"
"the CTK block type for the next block\n"
"to be created.";
const char * dirtyStatusText = "This is the `needs-saving' status indicator.\n"
"A + sign will be shown if the layout has been changed since it was last saved.";
const char * readOnlyStatusText = "This is the `read-only' status indicator.\n"
"A % sign will be shown if the frame being viewed is read-only.";

/******************************************************************************/
/*									      */
/*	CLASS NAME: CentralWidget		        	      	      */
/*									      */
/******************************************************************************/


CentralWidget::CentralWidget(QWidget* parent/*=0*/, const char* name/*=0*/) : QWidget(parent,name), main_block(NULL), is_dirty(0), script(NULL) {

  blocktypes=new QStringList();
  
  build_constructor_map();
  buildBlocktypesList();

  QBoxLayout *top_layout = new QVBoxLayout(this);
  blocktype_button = new QComboBox( FALSE, this, tr("Block Types") );
  blocktype_button->insertStringList(*blocktypes);
  QToolTip::add(blocktype_button, tr("Current block type"));
  QWhatsThis::add(blocktype_button, blocktypeButtonText);
  
  connect(blocktype_button, SIGNAL(activated(const QString &)), SLOT(setBlockType(const QString &)) );

  QBoxLayout *row1_layout = new QHBoxLayout(top_layout,5);

  row1_layout->addWidget(blocktype_button);
  row1_layout->addStretch(10);
  
  QFont boldfont; boldfont.setWeight(QFont::Bold);

  dirty_status = new QLabel(this);
  dirty_status->setFrameStyle( QFrame::Panel | QFrame::Raised );
  dirty_status->setFont( boldfont );
  dirty_status->setFixedWidth(20);
  dirty_status->setAlignment( AlignCenter );
  QToolTip::add(dirty_status, tr("`Needs saving' indicator"));
  QWhatsThis::add(dirty_status, dirtyStatusText);
  
  read_only_status = new QLabel(this);
  read_only_status->setFrameStyle( QFrame::Panel | QFrame::Raised );
  read_only_status->setFont( boldfont );
  read_only_status->setFixedWidth(20);
  read_only_status->setAlignment( AlignCenter );
  QToolTip::add(read_only_status, tr("`Read only' indicator"));
  QWhatsThis::add(read_only_status, readOnlyStatusText);

  row1_layout->addWidget(dirty_status);
  row1_layout->addWidget(read_only_status);

  row1_layout->addSpacing(5);
  
  QBoxLayout *row2_layout = new QHBoxLayout(top_layout);


  tp = new QTabWidget( this );
  row2_layout->addWidget(tp);
  connect(tp, SIGNAL(selected(const QString &)), this, SLOT(changeTab(const QString &)) );

  grid=new Grid();

  newDoc();
  
  //  tp = new QTabWidget( this );
  //  tp->addTab( new QPushButton(this), "one" );
  //  tp->addTab( new QPushButton(this), "two" );
  //  tp->addTab( new QPushButton(this), "three" );
  //  row2_layout->addWidget(tp);
  
  top_layout->activate();

  UndoController::theUndoController().setCentralWidget(this);

  // Default state of CTK profiling switch
  profile_switch=false;
  
}

CentralWidget::~CentralWidget() {

  disconnect(this);

  if (script!=NULL) delete script;
  if (grid!=NULL) delete grid;
}

void CentralWidget::paintEvent(QPaintEvent *e) {
  QWidget::paintEvent(e);
  layout_frame->paintEvent(e);
}

LayoutFrame *CentralWidget::selectLayoutPanel(const QString &name) {

  vector<LayoutFrame*>::iterator fp;
  
  for (fp=layout_list.begin(); fp!=layout_list.end(); ++fp) {
    if ((*fp)->getname()==name) break;
  }

  if (fp==layout_list.end()) {
    // Layout was not found
    return NULL;
  } else {
    // How do I raise the tab?
    return *fp;
  }
}

void CentralWidget::changeTab(const QString &name) {
  layout_frame=((CTKScrollView*)tp->currentPage())->getLayoutFrame();
  emit readOnlyMessage(layout_frame->get_read_only());
    
  setBlockType(blocktype_button->currentText());
}

LayoutFrame *CentralWidget::addScrollViewForBlock(Block *block, bool read_only, bool show_page/*=false*/) {

  QString type= block->get_blocktype().c_str();

  // Return immediately if there is already a ScrollView for this blocktype
  for (unsigned int i=0; i<layout_list.size(); ++i)
    if (type==QString(layout_list[i]->get_main_block()->get_blocktype().c_str()))
      return layout_list[i];


  CTKScrollView *vp = new CTKScrollView(this, read_only);
  tp->addTab( vp, type );
  
  LayoutFrame * frame=vp->getLayoutFrame();

  connect(frame, SIGNAL(warningMessage(const QString &)), this, SLOT(emitWarningMessage(const QString &)) );
  connect(frame, SIGNAL(statusMessage(const QString &)), this, SLOT(emitStatusMessage(const QString &)) );
  connect(frame, SIGNAL(caughtCTKError(CTKError &, QString)), this, SLOT(emitCaughtCTKError(CTKError &, QString)) );

  connect(frame, SIGNAL(hasChanged()), this, SLOT(setDirty()) );

  if (main_block!=NULL)
    // Set the main block - constructs the block layout.
    // Either use the main_block's file to look for block positions for all blocks in heirarchy...
    frame->setMainBlock(block, main_block->get_file_name().c_str());
  else
    // ... or if no main_block exists get block positions from the file in which each block is defined. 
    frame->setMainBlock(block);
  
  layout_list.push_back(frame);
  
  if (show_page) tp->showPage(vp);
   
  return vp->getLayoutFrame();
}

void CentralWidget::emitStatusMessage(const QString &message) {
  emit statusMessage(message);
}

void CentralWidget::emitWarningMessage(const QString &message) {
  emit warningMessage(message);
}

void CentralWidget::emitCaughtCTKError(CTKError &error, QString extra) {
  emit caughtCTKError(error, extra);
}

bool CentralWidget::cleanUpBeforeQuitting() {
  if (is_dirty) {
    switch( QMessageBox::information( this, tr("CTK"), tr("Save changes before quitting?"),
				      tr("Yes and quit"), tr("No and quit"), tr("Continue"),
				      0, 1 ) ) {
    case 0:
      saveCTKFile();
      return true;
    case 1:
      return true;
    case 2:
      return false;
      //      break;
    }
  }
  
  return true;
}

void CentralWidget::removeAllLayouts(QTabWidget *tabw) {
  
  QWidget *w;

  // For each layer remove the main_block from the constructor table
  for (unsigned int i=0; i<layout_list.size(); ++i) {
    constructor_translator.erase(layout_list[i]->get_main_block()->get_blocktype());
  }
  
  layout_list.resize(0);
  
  // Rebuild block selection list and button
  buildBlocktypesList();
  refreshBlocktypeButton();
  
  while ((w=tabw->currentPage())!=0) {
    tabw->removePage(w);
    delete w;
  }

  clearDirty();
}


void CentralWidget::refreshBlocktypeButton() {
  blocktype_button->clear();
  blocktype_button->insertStringList(*blocktypes);
  blocktype_button->update();
}



CTKStatus CentralWidget::loadCTKFile(const QString &block_file_name, Boolean called_from_revert/*=false*/) {

  CTKReader *new_script;
  
  try {
    new_script = new CTKReader(block_file_name.ascii());
  } catch (CTKError &error)  {
    emit caughtCTKError(error, "\0");
    return CTK_FAILURE;
  }
  
  // Add any intermediate blocks loaded to the block type button
  const BlockList block_list = new_script->get_block_list();
  makeBlocksAvailable(block_list);
  
  Block *new_main_block=new_script->get_block("main");
  if (new_main_block==NULL) {delete new_script; return CTK_FAILURE;}

  if (script!=NULL) delete script;
  script = new_script;
  
  if (is_dirty) {
    if (!called_from_revert) {
      switch( QMessageBox::information( this, tr("CTK"), tr("Save the old document first?"),
					tr("Save Now"), tr("Cancel"), tr("Continue"),
					0, 1 ) ) {
      case 0:
	saveCTKFile();
	break;
      case 1:
      case 2:
      default:
	break;
      }
    } else {
      switch( QMessageBox::information( this, tr("CTK"), tr("Revert from file on disk?"),
					tr("Yes"), tr("No"),
					0, 1 ) ) {
      case 0:   // If yes then continue
	break;  
      case 1:  // If No  then return
      default:
	return CTK_FAILURE;
	//	break;
      }
    }
  }
  
  try {
    new_main_block->initialise();
  }
  catch (CTKError &error) {
    emit caughtCTKError(error, "\0");
  }

  removeAllLayouts(tp);
  if (main_block!=NULL) 
    delete main_block;
  
  setMainBlock(new_main_block);
  
  // Add each ScrollView, but only draw the first one
  for (BlockList::const_reverse_iterator bp=block_list.rbegin(); bp!=block_list.rend(); ++bp) {
    (*bp)->autoconnect();
    bool read_only = ((*bp)->get_blocktype()!="main");
    addScrollViewForBlock(*bp, read_only, bp==block_list.rbegin() ); 
  }
  
  changeTab("main");

  // Reposition the undo list
  if (called_from_revert)
    UndoController::theUndoController().revertList();
  else
    UndoController::theUndoController().clearList();
  
  return CTK_SUCCESS;

}

void CentralWidget::setMainBlock(Block *new_main_block) {
  main_block=new_main_block;

  // set block_no to the lowest unused block number
  const BlockList blocks=main_block->get_subblocks();
  bool ok;
  block_no=1;
  for (unsigned int i=0; i<blocks.size(); ++i) {
    QString this_name = blocks[i]->getname().c_str();
    this_name.replace(QRegExp(BLOCK_NAME), QString("\0"));  
    int this_block_no=this_name.toInt(&ok);
    if (ok &&this_block_no>=block_no) block_no=this_block_no+1;
  }
  
}

QString CentralWidget::getUniqueBlockName() {
  return QString(BLOCK_NAME)+QString("%1").arg(block_no++);
}


Grid *CentralWidget::getGrid() const {return grid;}

QString CentralWidget::saveCTKFile() {
  if (main_block==NULL) return QString("\0");
  
  QString block_file_name = main_block->get_file_name().c_str();
  if (block_file_name=="\0") {
    return saveAsCTKFile();
  } else {
    if (saveWithFileName(block_file_name))
      return block_file_name;
    else
      return QString("\0");  // Return an empty string if the save has failed
  }
}


QString CentralWidget::saveAsCTKFile(const QString &file_name/*=0*/) {
  QString fn=file_name;

  if (fn==0 || fn.isEmpty())
    fn = QFileDialog::getSaveFileName( QString::null, QString::null,
					     this );
  
  if ( !fn.isEmpty() ) {
    if (saveWithFileName(fn)) {
      main_block->change_file_name(fn.ascii(), main_block->get_file_name());
      return fn;
    } else
      return QString("\0"); // return an empty string if save fails
  } else {
    emit statusMessage(tr("Saving aborted"));
    return QString(main_block->get_file_name().c_str()); // Return the old name
  }

}

bool CentralWidget::saveWithFileName(const QString &file_name) {
 
  // Write file without CTK_GUI comments
  CTKWriter x;

  try {
    x.set_CTKScript_path(CTKMainWindow::preferencesw->getCTKScriptPath());
    x.save(main_block, string(file_name.ascii()));

    
    // Add CTK_GUI block positioning comments
    for (unsigned int i=0; i<layout_list.size(); ++i) {
      if (layout_list[i]->writeBlockPositionsToFile(file_name)==false) {
	emit warningMessage(QString(tr("Failed to save to file: %1")).arg(file_name));
	return false;
      }
    }

    // Make a note of this save on the undo list
    UndoController::theUndoController().markLastSave();
    clearDirty();

    return true;

  }
  catch (CTKError &error) {
    emit warningMessage(QString(tr("Failed to save to file: %1")).arg(file_name));
    return false;
  }
}

CTKStatus CentralWidget::revertCTKFile() {  // Revert to saved
  if (main_block->get_file_name().size()==0) {
    emit warningMessage(QString(tr("Cannot revert - system has not been saved")));
    return CTK_FAILURE;
  } else
    return loadCTKFile(main_block->get_file_name().c_str(),
		true);   // true - indicates that the load is being called as part of a revert
}

  
void CentralWidget::toggleGrid() { 
  grid->toggle();
  layout_frame->update();
}

void CentralWidget::toggleSnap() { 
  grid->toggleSnap();
  layout_frame->update();
}

void CentralWidget::toggleProfile() {
  profile_switch=!profile_switch;
  profileModeHasChanged();
}

void CentralWidget::resetProfile() {
  main_block->reset_all_profile_statistics();
  profileModeHasChanged();
}

void CentralWidget::profileModeHasChanged() {
  for (unsigned int i=0; i<layout_list.size(); ++i) 
    layout_list[i]->refreshProfileWidgets(profile_switch);
}

void CentralWidget::displayProfileReport() {
  main_block->write_profiling_report(cout);
}

void CentralWidget::undo() { 
  UndoController::theUndoController().undo();
}

void CentralWidget::find() {
  // Look for a block with a user supplied name and highlight the corresponding
  // widget by selecting it.
  
  bool ok = FALSE;
  QString text = QInputDialog::getText( tr("Find Block"), tr( "Enter block name:" ), QLineEdit::Normal, QString::null, &ok, this );
  if ( !ok || text.isEmpty() )
    // user entered nothing or pressed cancel
    return;
  
  // user entered something and pressed ok
  
  text=QString("main:")+text;   // Add the "main:" prefix to construct the blocks full name.
  
  BlockWidget *w= layout_frame->findWidgetForBlockNamed(text);
  if (w==NULL) {
    //Block could not be found
    warningMessage(QString(tr("Block could not be found.")));
  } else {
    //Block was found
    unselectAll();
    w->setSelected(true);
    w->update();
    layout_frame->update();
  }

} 

// Unselect all currently selected blocks
void CentralWidget::unselectAll() {
  for (unsigned int i=0; i<layout_list.size(); ++i) {
    layout_list[i]->unselectAll();
  }
}

bool CentralWidget::newDoc() {

  if (is_dirty) {
    switch( QMessageBox::information( this, tr("CTK"),
                                      tr("The document has been changed since "
					 "the last save."),
                                      tr("Save Now"), tr("Cancel"), tr("Continue"),
                                      0, 1 ) ) {
    case 0:
      saveCTKFile();
      break;
    case 1:
    default:
      return false;
    case 2:
      break;
    }
  }
  
  removeAllLayouts(tp);
  if (main_block!=NULL) delete main_block;
  
  setMainBlock(new Block("main", "main"));
  
  layout_frame=addScrollViewForBlock(main_block,
				     false, // not read only
				     true); // display this view
  
  connect(grid, SIGNAL(valueChanged()), layout_frame, SLOT(repaint()));
  
  layout_frame->update();

  changeTab("\0");

  return true;
}

CTKStatus CentralWidget::saveAndReload() {
  // Save and reload to ensure script has been read from a file
  QString name=saveCTKFile();
  if (name.isEmpty()) return CTK_FAILURE;

  CTKStatus status = revertCTKFile();

  // Unfortunately this process invalidates the undo list, so clear it.
  UndoController::theUndoController().clearList();
  
  return status;
}
  

void CentralWidget::runSystem() {

  // Cannot use command line parameters unless the system has been loaded from a script file
  if (is_dirty) {
    warningMessage(QString(tr("Warning - system is being saved first. ")));
    if (saveAndReload()==CTK_FAILURE) return;
  }

  if (script!=NULL && script->get_max_param_number()>0) {
    runSystemWith();
  } else {
    Integer argc=2;
    char **argv=new char*[argc];
    argv[0]=strdup("CTK");
    argv[1]=strdup("UNKNOWN");
    
    runSystemNow(2, argv);
  }
}


// Run system with command line parameters
void CentralWidget::runSystemWith() {

  // Cannot use command line parameters unless the system has been loaded from a script file
  if (is_dirty) {
    warningMessage(QString(tr("Warning - system is being saved first. ")));
    if (saveAndReload()==CTK_FAILURE) return;
  }
  
  GetCommandLineModal m(this, tr("Run with..."));

  if (m.exec()) {

    int argc=m.get_argc();
    char **argv=m.get_argv();

    runSystemNow(argc, argv);
  }
  
}

void CentralWidget::stopSystem() {
  main_block->set_user_interrupt();
  cerr << "USER INTERRUPT REQUESTED\n";
}

void CentralWidget::runSystemNow(int argc, char **argv) {

  try {
    
    CTKCommandLine command_line(string("-param NULL -S NULL"), argc, (const char **)argv);
    
    if (argc>0) {
      
      if (command_line.option_is_present("param")) {
  	do {
  	  CTKScriptElement param = command_line.get_option_argument("param");
  	  string param_name, param_value;
  	  param.interpret_assignment(param_name, param_value);
  	  main_block->set_parameter(param_name.c_str(), param_value.c_str());
  	} while (command_line.next_occurrence("param"));
      }
    }

    // Attempt to initialise - if initialisation fails then return having done nothing
    if (main_block->initialise()==false)
      return;

    
    for (unsigned int i=0; i<layout_list.size(); ++i)
      layout_list[i]->connectUpBlocks();
    layout_frame->update();
    
    command_line.reset_to_first_argument_list();
    Boolean mode=ReadOnceFile::interactive_mode_;

    if (script!=NULL) {
      // Can only process command line parameters if script exists
      // i.e. Cannot use 'Run With' in the GUI unless the system has been read in from a file.
      do {
	int n1, n2;
	// Set parameters according to the current line of command arguments
	if ((n1=(command_line.argument_count()-1))!=(n2=script->get_max_param_number())) {
	  cerr << "Command script requires up to " << n2 << " command line parameters. "<< n1 << " were supplied." << endl;
	emit warningMessage(QString(tr("System requires up to %1 parameters. Only %2 were supplied.")).arg(n1).arg(n2));
	  throw(CTKError(__FILE__, __LINE__));
	}
	
	setParametersFromCommandline(*main_block, command_line, *script);
	
	main_block->process(profile_switch);
	ReadOnceFile::interactive_mode_=false;
      } while (command_line.next_argument_list());
    } else
      // Process without command line parameters
      main_block->process(profile_switch);
    
    ReadOnceFile::interactive_mode_=mode;
    
    main_block->close_final_all();
    
    for (unsigned int i=0; i<layout_list.size(); ++i) {
      layout_list[i]->refreshParameterWidgets();
      layout_list[i]->refreshProfileWidgets(profile_switch);
    }
  }
  
  catch (CTKError &error) {
    emit caughtCTKError(error, tr("Failed to run"));
  }
  
}


void CentralWidget::setBlockType(const QString &a_blocktype) {
  if (a_blocktype.isEmpty()) return;
  pulldown_blocktype=a_blocktype;
}

Block *CentralWidget::getNewBlock(const QString &blockname/*="\0"*/, const QString &blocktype/*="\0"*/) {

  QString str;
  if (blockname.isEmpty())
    str = getUniqueBlockName();
  else
    str = blockname;

  Block *new_block;
  
  if (blocktype.isEmpty())
    new_block = constructor_translator[pulldown_blocktype.ascii()]->clone(str.ascii());
  else
    new_block = constructor_translator[blocktype.ascii()]->clone(str.ascii());

  return new_block;
}


extern void my_translation_table(map<string, Block*> &constructor_translator);

void CentralWidget::build_constructor_map() {
  
  //
  //  These bits of code build the mapping between script block name and the block class
  //
  
  #include "sys/ctk_sys_translation_table.cpp"

  //
  //
  //  Call to user-provided function that supplies translations for user-provided blocks
  //

  my_translation_table(constructor_translator);
  
   
}


void CentralWidget::buildBlocktypesList() {
  
  blocktypes->clear();
  map<std::string, Block*>::iterator map_it;
  
  for (map_it=constructor_translator.begin(); map_it!=constructor_translator.end(); ++map_it) {
    std::string block_type= map_it->first;
    blocktypes->append(QString(block_type.c_str()));
  }

  blocktypes->sort();

}

void CentralWidget::makeBlocksAvailable(const BlockList &block_list) {

  // Note, blocks called `main' are not made available as subblocks
  for (BlockList::const_iterator bp=block_list.begin(); bp!=block_list.end(); ++bp) {
    if (string((*bp)->getname())!=string("main")) {
      constructor_translator[std::string((*bp)->getname())]=(*bp);
    }
  }

  buildBlocktypesList();

  refreshBlocktypeButton();
  
}


void CentralWidget::closeEvent( QCloseEvent* ce )
{
    if ( !is_dirty) {
        ce->accept();
        return;
    }
    
    switch( QMessageBox::information( this, tr("CTK"),
                                      "The document has been changed since "
                                      "the last save.",
                                      tr("Save Now"), tr("Cancel"), tr("Leave Anyway"),
                                      0, 1 ) ) {
   case 0:
        saveCTKFile();
        ce->accept();
        break;
    case 1:
    default: // just for sanity
        ce->ignore();
        break;
    case 2:
        ce->accept();
        break;
    }
}

void CentralWidget::setDirty() {
  is_dirty=true;
  emit dirtyMessage(true);
}

void CentralWidget::clearDirty() {
  is_dirty=false;
  emit dirtyMessage(false);
}

  
void CentralWidget::dirtyMessageSlot(bool is_dirty) {
  dirty_status->setText(is_dirty?"+":" ");
}

void CentralWidget::readOnlyMessageSlot(bool is_read_only) {
  read_only_status->setText(is_read_only?"%":" ");
}

void CentralWidget::resizeAllBlockWidgets() {

  QSize block_size=CTKMainWindow::preferencesw->getBlockSize();
  for (unsigned int i=0; i<layout_list.size(); ++i) {
    layout_list[i]->resizeBlockWidgets(block_size);
  }
}


// Set background colour of Read/Write layout frames
void CentralWidget::setRWLayoutBackgroundColour(QColor col) {
  for (unsigned int i=0; i<layout_list.size(); ++i) {
    if (!layout_list[i]->get_read_only()) {
      layout_list[i]->setPalette(QPalette(col));
      layout_list[i]->update();
    }
  }
}

void CentralWidget::setBlockWidgetsColour(QColor col) {
  for (unsigned int i=0; i<layout_list.size(); ++i) {
    layout_list[i]->setBlockWidgetsColour(col);
  }
}


#include "moc_ctk_gui_centralw.cpp"

/* End of ctk_gui_centralw.cpp */
