/******************************************************************************/
/*									      */
/*	ctk_gui_param.cpp	      					      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		        	      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include "ctk_gui_param.hh"

#include <algorithm>

#include <qhbox.h>
#include <qspinbox.h>
#include <qwhatsthis.h>
#include <qlabel.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qcheckbox.h> 
#include <qcombobox.h>
#include <qtabwidget.h>

#include "ctk_block.hh"

#include <ctk_param.hh>

#include "boost/compose.hpp"

#include "ctk_gui_blockw.hh"
#include "ctk_gui_undo.hh"

const int MIN_TEXT_EDIT_WIDTH = 800;   // Minimum initial size for the edittable text box


/******************************************************************************/
/*									      */
/*	CLASS NAME: ParameterWidget		      			      */
/*									      */
/******************************************************************************/

ParameterWidget::ParameterWidget(const Param *param, QWidget *parent, ParameterListWidget *paramlistw):QFrame(parent), param_(param), paramlistw_(paramlistw) {

  QString param_helptext=QString("<b>")+param_->getname().c_str()+"</b>:<p>"+param_->get_helptext().c_str();
  QWhatsThis::add(this, param_helptext);

  top_layout_ = new QHBoxLayout(this,5);


  // make label into a set `Command line parameter' push button
  label_ = new QPushButton(QString(param_->getname().c_str())+QString(" : "), this);
  label_->setToggleButton(true);
  int param_number=param_->get_arg_number();
    
  if (param_number!=0) {
    label_->setOn(true);
    label_->setText(QString(param_->getname().c_str())+QString("=$%1 :").arg(param_->get_arg_number()));
  }
  connect(label_, SIGNAL(toggled(bool)), SLOT(paramButtonPushed(bool)))  ;


  // If param is const then the button is inactive
  if (param_->get_const_flag()) {
    label_->setFlat(true);
    label_->setEnabled(false);
  }
  
  top_layout_->addWidget(label_);

}

QString ParameterWidget::getParamString() {
  if (param_->get_set_flag()) {
    if (param_->has_unresolved_value())
      return QString(param_->get_unresolved_value().c_str());
    else
      return QString(param_->get_value_string().c_str());
  } else
    return QString("                 ");
}

void ParameterWidget::addToWidget(QWidget *w) {
  top_layout_->addWidget(w);
}

void ParameterWidget::paintEvent(QPaintEvent *e) {
  refreshWithValueFromCTK();
  QFrame::paintEvent(e);
}


const Param *ParameterWidget::getParam(){return param_;}

void ParameterWidget::paramButtonPushed(bool state) {

  if (state) {
    GetIntegerModal m(this,"Enter command-line argument number: $");
    if (m.exec()) {
      int x = m.getInt();
      getParamW()->setParameterArgNumber(getParam()->getname().c_str(), x);
      label_->setText(QString(param_->getname().c_str())+QString("=$%1 :").arg(param_->get_arg_number()));
    } else {
      // User cancelled operation
      label_->setOn(false);
    }
  } else {
    getParamW()->setParameterArgNumber(getParam()->getname().c_str(), 0);
    label_->setText(QString(param_->getname().c_str())+QString(" : "));
  }

}


/******************************************************************************/
/*									      */
/*	CLASS NAME: GetIntegerModal      		      		      */
/*									      */
/******************************************************************************/


GetIntegerModal::GetIntegerModal( QWidget *parent, QString name )
  : QDialog( parent, name, TRUE )
{
  QBoxLayout *top_layout = new QVBoxLayout(this,5);

  QHBoxLayout *row1 = new QHBoxLayout();
  row1->addWidget(new QLabel(name, this));
  row1->addWidget(spin_=new QSpinBox(1, 99, 1, this));
  top_layout->addLayout(row1);

  QPushButton *ok, *cancel;
  QHBoxLayout *row2 = new QHBoxLayout();
  row2->addWidget(ok=new QPushButton("OK", this));
  row2->addWidget(cancel=new QPushButton("Cancel", this));
  top_layout->addLayout(row2);

  connect( ok, SIGNAL(clicked()), SLOT(accept()) );
  connect( cancel, SIGNAL(clicked()), SLOT(reject()) );
}


int GetIntegerModal::getInt() const {
  return spin_->value();
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: ParameterLineEditWidget		      		      */
/*									      */
/******************************************************************************/

ParameterLineEditWidget::ParameterLineEditWidget(const Param *param, QWidget *parent, ParameterListWidget *param_list_widget, bool read_only/*=false*/):ParameterWidget(param, parent, param_list_widget) {

  QString value_string = getParamString();
  
  line_edit_ = new ParameterEdit(value_string, this);

  if (line_edit_->width()<MIN_TEXT_EDIT_WIDTH) line_edit_->resize(MIN_TEXT_EDIT_WIDTH, line_edit_->height());
  
  addToWidget(line_edit_);
  addToWidget(new QLabel(getParam()->get_unit().c_str(), this ));
  connect( line_edit_, SIGNAL(returnPressed()), this, SLOT(setNewValue()) );
  connect( line_edit_, SIGNAL(parameterLostFocus()), this, SLOT(setNewValue()) );

  line_edit_->setEnabled(!read_only && !param->get_const_flag());

}


	   
void ParameterLineEditWidget::refreshWithValueFromCTK() {
  if (line_edit_!=NULL) {
    QString value=getParamString();
    line_edit_->setText(value);
  }
}


void ParameterLineEditWidget::setNewValue() {

  QString text=line_edit_->text().stripWhiteSpace();
  
  if (text.ascii()==getParam()->get_value_string()) return;

  UndoController::theUndoController().recordParameterChangeActionForUndo(getParamW()->getBlockW()->getLayoutFrame(),
									 getParamW()->getBlockW(),
									 getParam()->getname().c_str(),
									 getParam()->get_value_string().c_str()
									 );
  if (text.isEmpty()==false ) {
    getParamW()->setParameter(getParam()->getname().c_str(), text);
  } else {
   getParamW()->unsetParameter(getParam()->getname().c_str());
  }
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: ParameterComboWidget		      		      */
/*									      */
/******************************************************************************/

ParameterComboWidget::ParameterComboWidget(const Param *param, QWidget *parent, ParameterListWidget *param_list_widget, bool read_only/*=false*/):ParameterWidget(param, parent, param_list_widget) {

  QString value_string = getParamString();
  
  combo_box_ = new QComboBox(FALSE, this);
  QStringList *options=getValidStrings();
  combo_box_->insertStringList(*options);
  combo_box_->setEnabled(!read_only && !param->get_const_flag());
  addToWidget(combo_box_);
  connect( combo_box_, SIGNAL(activated(const QString &)), this, SLOT(setNewValue(const QString &)) );
  delete options;

}

QStringList *ParameterComboWidget::getValidStrings() {
  QStringList *list = new QStringList();
  
  vector<std::string> valid_values = ((ParamEnumerated*)getParam())->get_valid_values();
  for (unsigned int i=0; i<valid_values.size(); ++i) {
    list->append(QString(valid_values[i].c_str()));
  }

  return list;
    
}
	   
void ParameterComboWidget::refreshWithValueFromCTK() {
  if (getParam()->get_set_flag() && combo_box_!=NULL) {
    combo_box_->setCurrentItem(((ParamEnumerated*)getParam())->get_index());
  }
}


void ParameterComboWidget::setNewValue(const QString &value) {

  if (value.ascii()==getParam()->get_value_string()) return;
  
  UndoController::theUndoController().recordParameterChangeActionForUndo(getParamW()->getBlockW()->getLayoutFrame(),
									 getParamW()->getBlockW(),
									 getParam()->getname().c_str(),
									 getParam()->get_value_string().c_str()
									 );
  getParamW()->setParameter(getParam()->getname().c_str(), value);
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: ParameterCheckBoxWidget		      		      */
/*									      */
/******************************************************************************/

ParameterCheckBoxWidget::ParameterCheckBoxWidget(const Param *param, QWidget *parent, ParameterListWidget *param_list_widget, bool read_only/*=false*/):ParameterWidget(param, parent, param_list_widget) {

  QString value_string = getParamString();
  

  check_box_ = new QCheckBox(this);
  check_box_->setFocusPolicy(QWidget::StrongFocus);
  
  addToWidget(check_box_);
  connect( check_box_, SIGNAL(clicked()), this, SLOT(setNewValue()) );
  check_box_->setEnabled(!read_only && !param->get_const_flag());
 
}

void ParameterCheckBoxWidget::refreshWithValueFromCTK() {
  if (getParam()->get_set_flag()) {
    bool value = ((ParamBool*)getParam())->get_value();
    check_box_->setChecked(value);
  }
}

void ParameterCheckBoxWidget::setNewValue() {
  UndoController::theUndoController().recordParameterChangeActionForUndo(getParamW()->getBlockW()->getLayoutFrame(),
									 getParamW()->getBlockW(),
									 getParam()->getname().c_str(),
									 getParam()->get_value_string().c_str()
									 );
  getParamW()->setParameter(getParam()->getname().c_str(), (check_box_->isChecked())?"Yes":"No");
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: ParameterListWidget		      			      */
/*									      */
/******************************************************************************/


ParameterListWidget::ParameterListWidget(BlockWidget *this_block, bool read_only/*=false*/):QFrame(), blockw_(this_block), block_(this_block->getBlock()), input_socketw_(0), output_socketw_(0), is_read_only_(read_only) {

  block_=this_block->getBlock();
  
  Boolean needs_new_panel=false;
  Integer item_number=0;
  Integer page_number=1;
  
  QBoxLayout *top_layout = new QVBoxLayout(this,5);
  
  // Display block's type
  top_layout->addWidget(new QLabel(QString(tr("Type: "))+block_->get_blocktype().c_str(),this));

  // Display block's name 
  QHBoxLayout *h_layout = new QHBoxLayout(top_layout,5);
  h_layout->addWidget(new QLabel(QString(tr("Name: ")), this));
  h_layout->addWidget(name_edit_=new ParameterEdit(QString(block_->getname().c_str()), this));
  connect( name_edit_, SIGNAL(returnPressed()), this, SLOT(setNewBlockName()) );
  connect( name_edit_, SIGNAL(parameterLostFocus()), this, SLOT(setNewBlockName()) );


  
  // Display block's parameters
  
  param_list_ = block_->get_parameters();

  // Count the number of parameters excluding those that are hidden or deprecated
  Integer nitems=count_if(param_list_->begin(), param_list_->end(),
			  boost::compose_f_gx_hx(logical_and<bool>(),
						 boost::compose_f_gx(logical_not<Boolean>(), mem_fun(&Param::get_hidden_flag)),
						 boost::compose_f_gx(logical_not<Boolean>(), mem_fun(&Param::get_deprecated_flag)))
			  );

  QBoxLayout *layout=top_layout;

  ParameterWidget *p;
  QTabWidget *tw=NULL;
  QFrame *qw=NULL;
  QFrame *param_parent=NULL;

  if (nitems>MAX_PARAMS_PER_PANEL) {
    // The multi page case - Parameters on tabbed pages
    needs_new_panel=true;
    tw=new QTabWidget(this);
  } else {
    // The single page case - Single page enclosed in a visible frame
    param_parent = new QFrame(this);
    param_parent->setFrameStyle(QFrame::Panel | QFrame::Plain); // Draw box around parameters
    param_parent->setLineWidth(2); 
    top_layout->addWidget(param_parent);  // Add the parameter frame to the top layout
    // Make a new layout within the frame - set a border so the frame boundary is visible
    layout=new QVBoxLayout(param_parent, 3); 
  }
  
  for (ParamConstIt param_it=param_list_->begin(); param_it!=param_list_->end(); ++param_it) {

    const Param* param=*param_it;
    // Don't display hidden or deprecated parameters
    if (!param->get_hidden_flag() && !param->get_deprecated_flag()) {

      if (item_number==MAX_PARAMS_PER_PANEL) {
	tw->addTab(qw, QString("Page %1").arg(page_number++));
	needs_new_panel=true;
      }
      
      if (needs_new_panel) {
	param_parent=qw=new QFrame(tw);
	layout = new QVBoxLayout(qw);
	needs_new_panel=false;
	item_number=0;
      }

      switch (param->type()) {
      case PARAM_BOOL:
	layout->addWidget(p=new ParameterCheckBoxWidget(param, param_parent, this, read_only));
	break;
      case PARAM_ENUMERATED:
	layout->addWidget(p=new ParameterComboWidget(param, param_parent, this, read_only));
	break;
      default:
	layout->addWidget(p=new ParameterLineEditWidget(param, param_parent, this, read_only));
	break;
      }
      paramw_list_.push_back(p);
    
      ++item_number;
    }
  }

  if (nitems>MAX_PARAMS_PER_PANEL) {
    if (item_number>0)
      tw->addTab(qw, QString("Page %1").arg(page_number++));
    top_layout->addWidget(tw);
  }

  // Display block's input and output socket details
  
  if (block_->get_output_sockets()->is_configurable() && read_only==false) 
    top_layout->addWidget(makeSocketController(OUTPUT_SOCKET_TYPE, block_->get_output_sockets()->size()));
  
  if (block_->get_input_sockets()->is_configurable() && read_only==false)
    top_layout->addWidget(makeSocketController(INPUT_SOCKET_TYPE, block_->get_input_sockets()->size()));

  
  // Add a `close' button

  QPushButton *close_button;
  top_layout->addWidget(close_button=new QPushButton(tr("Close"), this));
  connect( close_button, SIGNAL(clicked()), this, SLOT(hide()));
 
}


void ParameterListWidget::paintEvent(QPaintEvent *e) {
  name_edit_->setText(block_->getname().c_str());
  QFrame::paintEvent(e);
}

void ParameterListWidget::setNewBlockName() {
  // Get new name from parameter window
  QString newname = name_edit_->text().stripWhiteSpace();

  // Attempt to change to the new name via the block's widget
  QString aname = blockw_->changeBlockName(newname);

  // Rewrite the returned name back to parameter window
  name_edit_->setText(aname);
}

void ParameterListWidget::refreshParameters() {

  for (unsigned int i=0; i<paramw_list_.size(); ++i)
    paramw_list_[i]->update();
  if (input_socketw_!=NULL) input_socketw_->update();
  if (output_socketw_!=NULL) output_socketw_->update();
  update();  
}

void ParameterListWidget::setParameterArgNumber(QString name, int number) {
  blockw_->setParameterArgNumber(name, number);
}

void ParameterListWidget::setParameter(QString name, QString value) {
  blockw_->setParameter(name,value);
}

void ParameterListWidget::unsetParameter(QString name) {
  blockw_->unsetParameter(name);
}


QWidget *ParameterListWidget::makeSocketController(SocketType socket_type, int nsockets) {

  QHBox *top_layout = new QHBox(this);
  top_layout->setSpacing(5);

  new QLabel(socket_type==OUTPUT_SOCKET_TYPE?QString(tr("Number of Outputs")):QString(tr("Number of Inputs")), top_layout);

  QSpinBox *value;

  if (socket_type==INPUT_SOCKET_TYPE) 
    value=input_socketw_ = new QSpinBox(blockw_->getMinNumInputs(), blockw_->getMaxNumInputs(), 1, top_layout);
  else 
    value=output_socketw_ = new QSpinBox(blockw_->getMinNumOutputs(), blockw_->getMaxNumOutputs(), 1, top_layout);
		       
  value->setValue(nsockets);

  if (socket_type==OUTPUT_SOCKET_TYPE)
    connect( value, SIGNAL( valueChanged( int ) ), this, SLOT( setNumOutputs( int ) ) );
  else 
    connect( value, SIGNAL( valueChanged( int ) ), this, SLOT( setNumInputs( int ) ) );

  return top_layout;
  
}

void ParameterListWidget::setNumInputs(int n) {
  blockw_->setNumInputs(n);
}

void ParameterListWidget::setNumOutputs(int n) {
  blockw_->setNumOutputs(n);
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: ParameterEdit		      			      */
/*									      */
/******************************************************************************/


ParameterEdit::ParameterEdit ( QWidget * parent, const char * name/*=0*/): QLineEdit(parent, name){
  setFocusPolicy(QWidget::StrongFocus);

};

ParameterEdit::ParameterEdit ( const QString & contents, QWidget * parent, const char * name/*=0*/): QLineEdit(contents,parent,name){
  setFocusPolicy(QWidget::StrongFocus);
};

void  ParameterEdit::focusOutEvent ( QFocusEvent * ) {
  emit parameterLostFocus();
}



#include "moc_ctk_gui_param.cpp"

/* End of ctk_gui_param.cpp */
