/******************************************************************************/
/*									      */
/*	ctk_gui_preferences.cpp	      					      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		                   	      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <stdlib.h>
#include <fstream>

#include "ctk_gui_preferences.hh"

#include <qbutton.h>
#include <qslider.h>
#include <qlabel.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include <qstring.h>
#include <qlayout.h>
#include <qtabwidget.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qcolordialog.h>
#include <qcolor.h>
#include <qtoolbutton.h>

#include "ctk_CTK_writer.hh"

/******************************************************************************/
/*									      */
/*	CLASS NAME: PreferencesWidget		      			      */
/*									      */
/******************************************************************************/


const QSize MIN_BLOCK_SIZE = 		QSize(40,40);
const QSize MAX_BLOCK_SIZE = 		QSize(120,70);

const int MIN_GRID_SIZE =		10;
const int MAX_GRID_SIZE = 		50;
const int GRID_SIZE_STEP =		5;

const QColor RWRITE_LAYOUT_BGROUND_COLOR = QColor(20, 255, 255);
const QColor BASEBLOCK_BGROUND_COLOR = QColor(180, 80, 255);

const int DEFAULT_BLOCK_WIDTH = 	80;
const int DEFAULT_BLOCK_HEIGHT =        55;
const int DEFAULT_GRID_SIZE =		20;
const int DEFAULT_PROFILE_MODE =        PROFILE_DISPLAY_MODE_PERCENTAGE;
const bool DEFAULT_CTKSCRIPT_PATH_PREF = CTKSCRIPT_PATH_CAN_BE_CTKROOT;
const bool DEFAULT_DISPLAY_ICONS_PREF =      false;

PreferencesWidget::PreferencesWidget():QFrame() {
 
  QBoxLayout *top_layout = new QVBoxLayout(this,5);
  
  QWidget *general_tabw = new QFrame(this);
  QWidget *profile_tabw = new QFrame(this);
  QWidget *save_tabw = new QFrame(this);
  QWidget *colour_tabw = new QFrame(this);
  
  QTabWidget *tw=new QTabWidget(this);
  top_layout->addWidget(tw);

  tw->addTab(general_tabw, "General");
  tw->addTab(profile_tabw, "Profile");
  tw->addTab(save_tabw, "Save");
  tw->addTab(colour_tabw, "Colour");
  
  drawGeneralTab(general_tabw);
  drawProfileTab(profile_tabw);
  drawSaveTab(save_tabw);
  drawColourTab(colour_tabw);
  
  QBoxLayout *row_layout = new QHBoxLayout(5);
  QPushButton *save_button=new QPushButton(tr("Save"), this);
  QPushButton *revert_button=new QPushButton(tr("Revert"), this);
  QPushButton *default_button=new QPushButton(tr("Default"), this);
  
  connect( save_button, SIGNAL(clicked()), this, SLOT(savePreferences()));
  connect( revert_button, SIGNAL(clicked()), this, SLOT(revertToSaved()));
  connect( default_button, SIGNAL(clicked()), this, SLOT(revertToDefault()));

  row_layout->addWidget(save_button);
  row_layout->addWidget(revert_button);
  row_layout->addWidget(default_button);
  top_layout->addLayout(row_layout);
  
  QPushButton *close_button=new QPushButton(tr("Close"), this);
  top_layout->addWidget(close_button);
  connect( close_button, SIGNAL(clicked()), this, SLOT(hide()));

  revertToDefault();
  revertToSaved();
}



void PreferencesWidget::drawGeneralTab(QWidget *tabw) {
   QBoxLayout *top_layout = new QVBoxLayout(tabw,5);

   block_width_pref_w_= new CTKSliderWidget(MIN_BLOCK_SIZE.width(), MAX_BLOCK_SIZE.width(), 1, DEFAULT_BLOCK_WIDTH, "Block Width", tabw);
   top_layout->addWidget(block_width_pref_w_);
   connect(block_width_pref_w_, SIGNAL(valueChanged(int)), this, SLOT(blockWidthChanged(int)));

   block_height_pref_w_= new CTKSliderWidget(MIN_BLOCK_SIZE.height(), MAX_BLOCK_SIZE.height(), 1, DEFAULT_BLOCK_HEIGHT, "Block Height", tabw);
   top_layout->addWidget(block_height_pref_w_);
   connect(block_height_pref_w_, SIGNAL(valueChanged(int)), this, SLOT(blockHeightChanged(int)));


  grid_size_pref_w_= new CTKSliderWidget(MIN_GRID_SIZE, MAX_GRID_SIZE, GRID_SIZE_STEP, DEFAULT_GRID_SIZE, "Grid Size", tabw);
   top_layout->addWidget(grid_size_pref_w_);
   connect(grid_size_pref_w_, SIGNAL(valueChanged(int)), this, SLOT(gridSizeChanged(int)));

   block_icons_mode_pref_w_ = new QCheckBox(tabw, "displayIconsCheckBox" );
  block_icons_mode_pref_w_->setAutoResize(true);
  block_icons_mode_pref_w_->setText("Display block icons");
  
  block_icons_mode_pref_w_->setChecked(DEFAULT_DISPLAY_ICONS_PREF);

  top_layout->addWidget(block_icons_mode_pref_w_);

  connect(block_icons_mode_pref_w_, SIGNAL(toggled(bool)), SLOT(blockIconsModeChanged(bool)) );

}


void PreferencesWidget::drawProfileTab(QWidget *tabw) {

  QBoxLayout *top_layout = new QVBoxLayout(tabw,5);
  
  profile_mode_pref_w_ = new QButtonGroup ( 1, QButtonGroup::Horizontal, tr("Display"), tabw);
			     
  new QRadioButton(tr("Percentage"), profile_mode_pref_w_);
  new QRadioButton(tr("Time"), profile_mode_pref_w_);
  new QRadioButton(tr("Calls"), profile_mode_pref_w_);

  profile_mode_pref_w_->setButton(DEFAULT_PROFILE_MODE);
  
  top_layout->addWidget(profile_mode_pref_w_);

  connect( profile_mode_pref_w_, SIGNAL(clicked(int)), SLOT(profileModeChanged(int)) );
  
}

void PreferencesWidget::drawSaveTab(QWidget *tabw) {
  
  new QVBoxLayout(tabw,5);

  ctkscript_path_pref_w_ = new QCheckBox(tabw, "pathModeCheckBox" );
  ctkscript_path_pref_w_->setAutoResize(true);
  ctkscript_path_pref_w_->setText("CTKScript path always $CTKLOCAL");

  ctkscript_path_pref_w_->setChecked(DEFAULT_CTKSCRIPT_PATH_PREF);
  
  connect(ctkscript_path_pref_w_, SIGNAL(toggled(bool)), SLOT(pathModeChanged(bool)) );

}

void PreferencesWidget::drawColourTab(QWidget *tabw) {
  
  QBoxLayout *top_layout = new QVBoxLayout(tabw,5);

  background_colour_pref_w_ = new QPushButton(tabw);
  background_colour_pref_w_->setAutoResize(true);
  background_colour_pref_w_->setText("Background");
  background_colour_pref_w_->setToggleButton(false);
  top_layout->addWidget(background_colour_pref_w_);

  connect(background_colour_pref_w_, SIGNAL(released()), SLOT(selectBackgroundColour()) );

  block_colour_pref_w_ = new QPushButton(tabw);
  block_colour_pref_w_->setAutoResize(true);
  block_colour_pref_w_->setText("Block");
  background_colour_pref_w_->setToggleButton(false);
  top_layout->addWidget(block_colour_pref_w_);

  connect(block_colour_pref_w_, SIGNAL(released()), SLOT(selectBlockColour()) );
}

///

void PreferencesWidget::selectBackgroundColour() {
  QColor col = QColorDialog::getColor(getBackgroundColour());
  if (col.isValid())
    setBackgroundColour(col);
}

void PreferencesWidget::selectBlockColour() {
  QColor col = QColorDialog::getColor(getBlockColour());
  if (col.isValid())
    setBlockColour(col);
}

///


void PreferencesWidget::setBlockHeight(int value) {
  block_height_pref_w_->setValue(value);
  emit signalBlockSizePreferenceChanged();
}

void PreferencesWidget::setBlockWidth(int value) {
  block_width_pref_w_->setValue(value);
  emit signalBlockSizePreferenceChanged();
}

void PreferencesWidget::setGridSize(int value) {
  grid_size_pref_w_->setValue(value);
  emit signalGridSizePreferenceChanged(value);
}

void PreferencesWidget::setPathMode(bool value) {
  ctkscript_path_pref_w_->setChecked(value);
}

void PreferencesWidget::setBlockIconsMode(bool value) {
  block_icons_mode_pref_w_->setChecked(value);
  emit signalBlockIconsModePreferenceChanged(value);
}

void PreferencesWidget::setProfileMode(int value) {
  profile_mode_pref_w_->setButton(value);
  emit signalProfileModePreferenceChanged(value);
}

void PreferencesWidget::setBackgroundColour(QColor value) {
  background_colour_pref_w_->setPalette(QPalette(value));
  emit signalBackgroundColourChanged(value);
}

void PreferencesWidget::setBlockColour(QColor value) {
  block_colour_pref_w_->setPalette(QPalette(value));
  emit signalBlockColourChanged(value);
}


void PreferencesWidget::blockHeightChanged(int) {
  emit signalBlockSizePreferenceChanged();
}

void PreferencesWidget::blockWidthChanged(int) {
  emit signalBlockSizePreferenceChanged();
}

void PreferencesWidget::gridSizeChanged(int value) {
  emit signalGridSizePreferenceChanged(value);
}

void PreferencesWidget::pathModeChanged(bool) {

}

void PreferencesWidget::profileModeChanged(int mode) {
  emit signalProfileModePreferenceChanged(mode);
}

void PreferencesWidget::blockIconsModeChanged(bool mode) {
  emit signalBlockIconsModePreferenceChanged(mode);
}

void PreferencesWidget::backgroundColourChanged(QColor col) {
  emit signalBackgroundColourChanged(col);
}

void PreferencesWidget::blockColourChanged(QColor col) {
  emit signalBlockColourChanged(col);
}


////

QSize PreferencesWidget::getBlockSize() const {
  return QSize(block_width_pref_w_->value(), block_height_pref_w_->value());
}

int PreferencesWidget::getBlockWidth() const {
  return block_width_pref_w_->value();
}

int PreferencesWidget::getBlockHeight() const {
  return block_height_pref_w_->value();
}

int PreferencesWidget::getGridSize() const {
  return grid_size_pref_w_->value();
}

bool PreferencesWidget::getDisplayIcons() const {
  return block_icons_mode_pref_w_->isChecked();
}

int PreferencesWidget::getProfileMode() const {
  return profile_mode_pref_w_->id(profile_mode_pref_w_->selected());
}

int PreferencesWidget::getCTKScriptPath() const {
  return ctkscript_path_pref_w_->isChecked();
}

QColor PreferencesWidget::getBackgroundColour() const {
  return background_colour_pref_w_->backgroundColor();
}

QColor PreferencesWidget::getBlockColour() const {
  return block_colour_pref_w_->backgroundColor();
}


///

CTKStatus PreferencesWidget::savePreferences() const {
  // Expand $HOME environment variable
  char *cfname  = getenv("HOME");
  if (cfname==NULL) return CTK_FAILURE;

  // Open file $HOME/.ctk
  string fname = cfname;
  fname+="/.ctk";
  
  std::ofstream ofstr(fname.c_str());
  if (!ofstr) return CTK_FAILURE;

  ofstr << "BLOCK_WIDTH " << getBlockWidth() << "\n";
  ofstr << "BLOCK_HEIGHT " << getBlockHeight() << "\n";
  ofstr << "GRID_SIZE " << getGridSize() << "\n";
  ofstr << "PROFILE_MODE " << getProfileMode() << "\n";
  ofstr << "DISPLAY_ICONS " << getDisplayIcons() << "\n";
  ofstr << "CTKSCRIPT_PATH " << getCTKScriptPath() << "\n";
  QColor col=getBackgroundColour();
  ofstr << "BACKGROUND_COLOUR " << col.red() << " " << col.green() << " " << col.blue() << "\n";
  col=getBlockColour();
  ofstr << "BLOCK_COLOUR " << col.red() << " " << col.green() << " " << col.blue() << "\n";
  
  return CTK_SUCCESS;
}


// Revert preference settings to those saved in the user's $HOME/.ctk file
CTKStatus PreferencesWidget::revertToSaved() {

  // Expand $HOME environment variable
  char *cfname  = getenv("HOME");
  if (cfname==NULL) return CTK_FAILURE;

  // Open file $HOME/.ctk
  string fname = cfname;
  fname+="/.ctk";
  
  std::ifstream ifstr(fname.c_str());
  if (!ifstr) return CTK_FAILURE;
  
  // Read parameter/value pairs from .ctk file
  string param, value;
  int x_int, r_int, g_int, b_int;
  
  while (ifstr) {
    ifstr >> param;
    if (param=="BLOCK_WIDTH") { 
      ifstr >> x_int;
      setBlockWidth(x_int);
    }
    else if (param=="BLOCK_HEIGHT") {
      ifstr >> x_int;
      setBlockHeight(x_int);
    }
    else if (param=="GRID_SIZE") {
      ifstr >> x_int;
      setGridSize(x_int);
    }
    else if (param=="PROFILE_MODE") {
      ifstr >> x_int;
      setProfileMode(x_int);
    }
    else if (param=="DISPLAY_ICONS") {
      ifstr >> x_int;
      setBlockIconsMode(x_int);
    }
    else if (param=="CTKSCRIPT_PATH") {
      ifstr >> x_int;
      setPathMode(x_int);
    }
    else if (param=="BACKGROUND_COLOUR") {
      ifstr >> r_int >> g_int >> b_int;
      setBackgroundColour(QColor(r_int,g_int,b_int));
    }
    else if (param=="BLOCK_COLOUR") {
      ifstr >> r_int >> g_int >> b_int;
      setBlockColour(QColor(r_int,g_int,b_int));
    }
  }
  
  ifstr.close();

  update();
  
  return CTK_SUCCESS;
}
  
void PreferencesWidget::revertToDefault() {
  setBlockWidth(DEFAULT_BLOCK_WIDTH);
  setBlockHeight(DEFAULT_BLOCK_HEIGHT);
  setGridSize(DEFAULT_GRID_SIZE);
  setProfileMode(DEFAULT_PROFILE_MODE);
  setBlockIconsMode(DEFAULT_DISPLAY_ICONS_PREF);
  setPathMode(DEFAULT_CTKSCRIPT_PATH_PREF);
  setBackgroundColour(RWRITE_LAYOUT_BGROUND_COLOR);
  setBlockColour(BASEBLOCK_BGROUND_COLOR);
  
  update();
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKSliderWidget		      			      */
/*									      */
/******************************************************************************/



CTKSliderWidget::CTKSliderWidget(int min, int max, int step, int value, const char *label, QWidget *w): QFrame(w, label) {
   QHBoxLayout *top_layout = new QHBoxLayout(this,5);

   top_layout->addWidget(new QLabel(label, this));
   top_layout->addWidget(value_text_=new QLabel(QString("%1").arg(value), this));
   
   top_layout->addWidget(slider_=new QSlider(min, max, step, value, QSlider::Horizontal, this, label));

   connect(slider_, SIGNAL(valueChanged(int)), this, SLOT(setValue(int)));
}

int CTKSliderWidget::value() const {
    return slider_->value();
}

void CTKSliderWidget::setValue(int value) {
  slider_->setValue(value);
  value_text_->setText(QString("%1").arg(value));
  emit valueChanged(value);
}

#include "moc_ctk_gui_preferences.cpp"

/* End of ctk_gui_preferences.cpp */
