/*****************************************************************************/
/*						                    	     */
/*	ctk_gui_undo.cpp	      				    	     */
/*						      	      	             */
/*	Author: Jon Barker, Sheffield University		             */
/*					      		      	             */
/*      CTK VERSION 1.3.5  Apr 22, 2007	           		      */
/*			     				      		     */
/*****************************************************************************/

#include "ctk-config.h"

#include <qwidget.h>
#include <qtimer.h>

#include "ctk_local.hh"

#include "ctk_gui_centralw.hh"
#include "ctk_gui_blockw.hh"
#include "ctk_gui_undo.hh"


/******************************************************************************/
/*							  		      */
/*	CLASS NAME: UndoController                          	      	      */
/*					      		      		      */
/******************************************************************************/

UndoController::UndoController():central_(0), in_record_mode_(true), last_save_pos_(0) {
  clearList();
}


UndoController& UndoController::theUndoController() {
  static UndoController uc;
  return uc;
}

void UndoController::setCentralWidget(CentralWidget *central) {
  central_=central;
}

// Reposition the list at the last save marker -
void UndoController::revertList() {
  
  if (last_save_pos_<undo_list_.size())
    clearList(last_save_pos_);
  // The list can't be repositioned forward ... so just clear all the undo's
  else {
    clearList(0);
  }
  
}

void UndoController::clearList(unsigned int n/*=0*/) {

  bool stuff_to_undo = (undo_list_.size()>0);
  
  while (undo_list_.size()>n) {
    delete undo_list_.back();
    undo_list_.pop_back();
  }

  // If question of whether or not there is stuff to undo has changed then send signal...
  if ((undo_list_.size()>0) != stuff_to_undo)
    emit signalUndoStatusHasChanged(undo_list_.size()>0);

}

void UndoController::recordMoveActionForUndo(LayoutFrame *frame, BlockWidget *block, const QPoint &pos) {
  if (in_record_mode_) {
    undo_list_.push_back(new UndoMoveRecord(frame, block, pos));
    if (undo_list_.size()==1)
      emit signalUndoStatusHasChanged(true);
  }
}

void UndoController::recordDisconnectActionForUndo(LayoutFrame *frame, BlockWidget *fromBlock, BlockWidget *toBlock, int from_socket_number, int to_socket_number) {
  if (in_record_mode_) {
    undo_list_.push_back(new UndoDisconnectRecord(frame, fromBlock, toBlock, from_socket_number, to_socket_number));
    if (undo_list_.size()==1)
      emit signalUndoStatusHasChanged(true);
  }
    
}

void UndoController::recordConnectActionForUndo(LayoutFrame *frame, BlockWidget *fromBlock, int from_socket_number) {
  if (in_record_mode_) {
    undo_list_.push_back(new UndoConnectRecord(frame, fromBlock, from_socket_number));
    if (undo_list_.size()==1)
      emit signalUndoStatusHasChanged(true);
  }
}

void UndoController::recordDeleteBlockActionForUndo(LayoutFrame *frame, BlockWidget *block) {
  if (in_record_mode_) {
    undo_list_.push_back(new UndoDeleteBlockRecord(frame, block));
    if (undo_list_.size()==1)
      emit signalUndoStatusHasChanged(true);
  }
}

void UndoController::recordConstructBlockActionForUndo(LayoutFrame *frame, BlockWidget *block) {
  if (in_record_mode_) {
    undo_list_.push_back(new UndoConstructBlockRecord(frame, block));
    if (undo_list_.size()==1)
      emit signalUndoStatusHasChanged(true);
  }
}

void UndoController::recordParameterChangeActionForUndo(LayoutFrame *frame, BlockWidget *block, QString param_name, QString param_value) {
  if (in_record_mode_) {
    undo_list_.push_back(new UndoParameterChangeRecord(frame, block, param_name, param_value));
    if (undo_list_.size()==1)
      emit signalUndoStatusHasChanged(true);
  }
}

void UndoController::recordInputSocketChangeActionForUndo(LayoutFrame *frame, BlockWidget *block, int nsockets) {
  if (in_record_mode_) {
    undo_list_.push_back(new UndoInputSocketChangeRecord(frame, block, nsockets));
    if (undo_list_.size()==1)
      emit signalUndoStatusHasChanged(true);
  }
}

void UndoController::recordOutputSocketChangeActionForUndo(LayoutFrame *frame, BlockWidget *block, int nsockets) {
  if (in_record_mode_) {
    undo_list_.push_back(new UndoOutputSocketChangeRecord(frame, block, nsockets));
    if (undo_list_.size()==1)
      emit signalUndoStatusHasChanged(true);
  }
}

void UndoController::recordBlockNameChangeActionForUndo(LayoutFrame *frame, BlockWidget *block, QString blockname) {
  if (in_record_mode_) {
    undo_list_.push_back(new UndoBlockNameChangeRecord(frame, block, blockname));
    if (undo_list_.size()==1)
      emit signalUndoStatusHasChanged(true);
  }
}

// Sets the saved point to the end of the undo list
void UndoController::markLastSave() {
  if (in_record_mode_)
    last_save_pos_=undo_list_.size();
}

void UndoController::undo(bool ignore_undo_action/*=false*/) {
  if (undo_list_.size()>0) {
    Boolean saved_value = in_record_mode_;
    in_record_mode_=false;

    // Get the record from the end of the list, trigger the undo and delete the record
    UndoRecord *undo_record = undo_list_.back();
    undo_list_.pop_back();
    if (!ignore_undo_action)
      undo_record->undo(central_);
    delete undo_record;

    // Set the dirty flag according to relation to save pos
    if (undo_list_.size()==last_save_pos_) 
      central_->clearDirty();
    else
      central_->setDirty();
    
    in_record_mode_=saved_value;

    if (undo_list_.size()==0)
      emit signalUndoStatusHasChanged(false);
    
  } else
    central_->emitWarningMessage("Nothing to undo");
}

/******************************************************************************/
/*							      		      */
/*	CLASS NAME: UndoRecord                                                */
/*					      		      		      */
/******************************************************************************/

UndoRecord::UndoRecord(LayoutFrame *frame):QObject(), layout_frame_name_(frame->get_main_block()->get_blocktype().c_str()){
}  

void UndoRecord::undo(CentralWidget *) {
  // cerr << "Action cannot be undone" << endl;
}


/******************************************************************************/
/*							      		      */
/*	CLASS NAME: UndoMoveRecord                         	      	      */
/*					      		      		      */
/******************************************************************************/

UndoMoveRecord::UndoMoveRecord(LayoutFrame *layoutFrame, BlockWidget *ablock, const QPoint &apos):UndoRecord(layoutFrame),blockname_(ablock->getBlock()->getfullname()), old_pos_(apos) {
};

void UndoMoveRecord::undo(CentralWidget *central) {
  //  cerr << "undo move\n";
  LayoutFrame *layout_frame=central->selectLayoutPanel(layoutFrameName());
  layout_frame->moveBlockTo(old_pos_, blockname_);
  
}

/******************************************************************************/
/*							      		      */
/*	CLASS NAME: UndoDisconnectRecord                              	      */
/*					      		      		      */
/******************************************************************************/

UndoDisconnectRecord::UndoDisconnectRecord(LayoutFrame *layoutFrame, BlockWidget *fromBlock, BlockWidget *toBlock, int from_number, int to_number):UndoRecord(layoutFrame), from_block_name_(fromBlock->getBlock()->getfullname().c_str()), to_block_name_(toBlock->getBlock()->getfullname().c_str()), from_socket_number_(from_number), to_socket_number_(to_number) {
};

void UndoDisconnectRecord::undo(CentralWidget *central) {
  //  cerr << "undo disconnect\n";
  // Reconnect the blocks
  LayoutFrame *layout_frame=central->selectLayoutPanel(layoutFrameName());
  layout_frame->connectBlockFromTo(from_block_name_.ascii(), to_block_name_.ascii(), from_socket_number_, to_socket_number_);
}

/******************************************************************************/
/*							     		      */
/*	CLASS NAME: UndoConnectRecord                         	      	      */
/*					      		      		      */
/******************************************************************************/

UndoConnectRecord::UndoConnectRecord(LayoutFrame *layoutFrame, BlockWidget *fromBlock, int from_number):UndoRecord(layoutFrame), from_block_name_(fromBlock->getBlock()->getfullname().c_str()), from_socket_number_(from_number) {
};

void UndoConnectRecord::undo(CentralWidget *central) {
  //  cerr << "undo connect\n";
  // Disonnect the blocks
  LayoutFrame *layout_frame=central->selectLayoutPanel(layoutFrameName());
  layout_frame->disconnectBlockOutputSocket(from_block_name_.ascii(), from_socket_number_);
}


/******************************************************************************/
/*							      		      */
/*	CLASS NAME: UndoDeleteBlockRecord                              	      */
/*					      		      		      */
/******************************************************************************/


// Note - block->getname() is used rather than block->getfullname()  - as this is what is required for layoutFrame->addNewBlock()

UndoDeleteBlockRecord::UndoDeleteBlockRecord(LayoutFrame *layoutFrame, BlockWidget *block):UndoRecord(layoutFrame), block_name_(block->getBlock()->getname().c_str()), block_type_(block->getBlock()->get_blocktype().c_str()), pos_(block->pos()) {

  // Save blocks parameter values
  const ParamList *param_list = block->getBlock()->get_parameters();
  for (ParamConstIt param_it=param_list->begin(); param_it!=param_list->end(); ++param_it) {
    // Ignore deprecated parameters
    if (!(*param_it)->get_deprecated_flag()) {
      param_name_.push_back((*param_it)->getname().c_str());
      param_value_.push_back((*param_it)->get_value_string().c_str());
    }
  }

  ninputs_=block->getBlock()->get_num_inputs();
  noutputs_=block->getBlock()->get_num_outputs();
  
}

void UndoDeleteBlockRecord::undo(CentralWidget *central) {
  //  cerr << "undo delete\n";
  // Reconstruct the block
  LayoutFrame *layout_frame=central->selectLayoutPanel(layoutFrameName());

  BlockWidget *new_blockw=layout_frame->addNewBlock(pos_, block_type_, block_name_);

  // Restore saved parameter values
  for (unsigned int i=0; i<param_name_.size(); i++) 
    new_blockw->setParameter(param_name_[i], param_value_[i]);

  new_blockw->setNumInputs(ninputs_);
  new_blockw->setNumOutputs(noutputs_);
  
}  
  
/******************************************************************************/
/*							      		      */
/*	CLASS NAME: UndoConstructBlockRecord                          	      */
/*					      		      		      */
/******************************************************************************/

UndoConstructBlockRecord::UndoConstructBlockRecord(LayoutFrame *layoutFrame, BlockWidget *block):UndoRecord(layoutFrame), block_name_(block->getBlock()->getfullname().c_str()) {
};

void UndoConstructBlockRecord::undo(CentralWidget *central) {
  //  cerr << "undo construct\n";
  // Delete the block
  LayoutFrame *layout_frame=central->selectLayoutPanel(layoutFrameName());
  layout_frame->deleteBlock(block_name_.ascii());

}


/******************************************************************************/
/*							      		      */
/*	CLASS NAME: UndoParameterChangeRecord                              	      */
/*					      		      		      */
/******************************************************************************/


// Note - block->getname() is used rather than block->getfullname()  - as this is what is required for layoutFrame->addNewBlock()

UndoParameterChangeRecord::UndoParameterChangeRecord(LayoutFrame *layoutFrame, BlockWidget *ablockw, QString aparam_name, QString aparam_value):UndoRecord(layoutFrame), blockw_(ablockw), block_name_(ablockw->getBlock()->getname().c_str()),  param_name_(aparam_name), param_value_(aparam_value) {

}

void UndoParameterChangeRecord::undo(CentralWidget *) {
  //  cerr << "undo param change\n";
  blockw_->displayParameterPanel();

  // // Can't get this delay to work...
  //  QTimer *timer = new QTimer(this);
  //  connect( timer, SIGNAL(timeout()), this, SLOT(makeChange()) );
  //  timer->start(500, TRUE );                 // 0.5 second delay before  makeChange();

  makeChange();
}

void UndoParameterChangeRecord::makeChange() {
  blockw_->setParameter(param_name_, param_value_);
}


/******************************************************************************/
/*							      		      */
/*	CLASS NAME: UndoInputSocketChangeRecord                              	      */
/*					      		      		      */
/******************************************************************************/

UndoInputSocketChangeRecord::UndoInputSocketChangeRecord(LayoutFrame *layoutFrame, BlockWidget *blockw, int nsockets):UndoRecord(layoutFrame), blockw_(blockw), nsockets_(nsockets) {
}

void UndoInputSocketChangeRecord::undo(CentralWidget *) {
  //  cerr << "undo input socket change change\n";
  blockw_->setNumInputs(nsockets_);
}

/******************************************************************************/
/*							      		      */
/*	CLASS NAME: UndoOutputSocketChangeRecord                              	      */
/*					      		      		      */
/******************************************************************************/

UndoOutputSocketChangeRecord::UndoOutputSocketChangeRecord(LayoutFrame *layoutFrame, BlockWidget *blockw, int nsockets):UndoRecord(layoutFrame), blockw_(blockw), nsockets_(nsockets) {
}

void UndoOutputSocketChangeRecord::undo(CentralWidget *) {
  //  cerr << "undo output socket change change\n";
  blockw_->setNumOutputs(nsockets_);
}

/******************************************************************************/
/*							      		      */
/*	CLASS NAME: UndoBlockNameChangeRecord                              	      */
/*					      		      		      */
/******************************************************************************/

UndoBlockNameChangeRecord::UndoBlockNameChangeRecord(LayoutFrame *layoutFrame, BlockWidget *ablockw, QString ablockname):UndoRecord(layoutFrame), blockw_(ablockw), blockname_(ablockname) {
}

void UndoBlockNameChangeRecord::undo(CentralWidget *) {
  //  cerr << "undo block name change change\n";
  blockw_->changeBlockName(blockname_);
}
#include "moc_ctk_gui_undo.cpp"

/* End of ctk_gui_undo.cpp */
