/******************************************************************************/
/*			      				      		      */
/*	ctk_gui_undo.hh	              				      	      */
/*	      						      		      */
/*	Author: Jon Barker, Sheffield University	         	      */
/*							      		      */
/*      CTK VERSION 1.3.5  Apr 22, 2007	           		      */
/*				      			      		      */
/******************************************************************************/

#ifndef CTK_GUI_UNDO_HH
#define CTK_GUI_UNDO_HH

#include <vector>
#include <string>

#include <qstring.h>
#include <qpoint.h>

#include "ctk_local.hh"

class CentralWidget;
class BlockWidget;
class ParameterListWidget;

class UndoRecord;

/******************************************************************************/
/*				       			      		      */
/*	CLASS NAME: UndoController                          	      	      */
/*				      			      		      */
/******************************************************************************/

class UndoController: public QObject  {
Q_OBJECT

private:

  CentralWidget *central_;
  Boolean in_record_mode_;
  unsigned int last_save_pos_;        // Marks the position of the last save
  vector<UndoRecord*> undo_list_;
  
signals:
  void signalUndoStatusHasChanged(bool x);
  
  //
  //
  
public:
  
  static UndoController &theUndoController();

  void setCentralWidget(CentralWidget *acentral);
  
  // Clear list back to last save_marker
  void revertList();

  // Clear list back to position n
  void clearList(unsigned int n=0);
  
  void recordMoveActionForUndo(class LayoutFrame *frame, class BlockWidget *block, const QPoint &apos);
  void recordDisconnectActionForUndo(class LayoutFrame *frame, BlockWidget *fromBlock, BlockWidget *toBlock, int from_socket_number, int to_socket_number);
  void recordConnectActionForUndo(class LayoutFrame *frame, BlockWidget *fromBlock, int from_socket_number);
  void recordDeleteBlockActionForUndo(class LayoutFrame *frame, BlockWidget *block);
  void recordConstructBlockActionForUndo(class LayoutFrame *frame, BlockWidget *block);
  void recordParameterChangeActionForUndo(class LayoutFrame *frame, BlockWidget *block, QString param_name, QString param_value);
  void recordInputSocketChangeActionForUndo(class LayoutFrame *frame, BlockWidget *block, int nsockets);
  void recordOutputSocketChangeActionForUndo(class LayoutFrame *frame, BlockWidget *block, int nsockets);
  void recordBlockNameChangeActionForUndo(class LayoutFrame *frame, BlockWidget *block, QString block_name);
  
  // Sets the saved point to the end of the undo list . i.e. undoing back to this point will clear the 'Needs Saving' flag.
  
  void markLastSave();
  
  void undo(bool ignore_undo_action=false);
  void removeLast();
  
private:

  UndoController();


};

  

/******************************************************************************/
/*				       			     		      */
/*	CLASS NAME: UndoRecord                          	      	      */
/*				      			      		      */
/******************************************************************************/

class UndoRecord: public QObject  {
private:
  QString layout_frame_name_;  // Name of the layout in which the change occurred

protected:
  QString layoutFrameName() const {return layout_frame_name_;}
  
public:
  
  UndoRecord(LayoutFrame *aframe);
  UndoRecord(const UndoRecord &arecord):QObject(),layout_frame_name_(arecord.layout_frame_name_){};
  
  virtual ~UndoRecord(){}

  virtual void undo(CentralWidget *central);
};


/******************************************************************************/
/*				       			      		      */
/*	CLASS NAME: UndoMoveRecord                          	      	      */
/*				      			      		      */
/******************************************************************************/

class UndoMoveRecord: public UndoRecord {
private:

  string blockname_;
  QPoint old_pos_;

public:
  UndoMoveRecord(class LayoutFrame *frame, class BlockWidget *ablock, const QPoint &apos);
  UndoMoveRecord(const UndoMoveRecord &arecord):UndoRecord(arecord),blockname_(arecord.blockname_),old_pos_(arecord.old_pos_){};
  
  virtual ~UndoMoveRecord(){}

  virtual void undo(CentralWidget *central);
};

/******************************************************************************/
/*				       			     		      */
/*	CLASS NAME: UndoDisconnectRecord                       	      	      */
/*				      			      		      */
/******************************************************************************/

class UndoDisconnectRecord: public UndoRecord {
private:

  QString from_block_name_;
  QString to_block_name_;
  int from_socket_number_;
  int to_socket_number_;
  

public:
  UndoDisconnectRecord(class LayoutFrame *frame, BlockWidget *fromBlock, BlockWidget *toBlock, int from_socket_number, int to_socket_number);
  virtual ~UndoDisconnectRecord(){}

  virtual void undo(CentralWidget *central);
};

/******************************************************************************/
/*				       			      		      */
/*	CLASS NAME: UndoConnectRecord                       	      	      */
/*				      			      		      */
/******************************************************************************/

class UndoConnectRecord: public UndoRecord {
private:

  QString from_block_name_;
  int from_socket_number_;
  

public:
  UndoConnectRecord(class LayoutFrame *frame, BlockWidget *fromBlock, int from_socket_number);
  virtual ~UndoConnectRecord(){}

  virtual void undo(CentralWidget *central);
};


/******************************************************************************/
/*				       			      		      */
/*	CLASS NAME: UndoDeleteBlockRecord                       	      */
/*				      			      		      */
/******************************************************************************/

class UndoDeleteBlockRecord: public UndoRecord {
private:

  vector<QString> param_name_;
  vector<QString> param_value_;
  int ninputs_, noutputs_;
  QString block_name_;
  QString block_type_;
  QPoint pos_;
  
public:
  UndoDeleteBlockRecord(class LayoutFrame *frame, BlockWidget *block);
  virtual ~UndoDeleteBlockRecord(){}

  virtual void undo(CentralWidget *central);
};

/******************************************************************************/
/*				       			      		      */
/*	CLASS NAME: UndoConstructBlockRecord                       	      */
/*				      			      		      */
/******************************************************************************/

class UndoConstructBlockRecord: public UndoRecord {
private:

  QString block_name_;

public:
  UndoConstructBlockRecord(class LayoutFrame *frame, BlockWidget *block);
  virtual ~UndoConstructBlockRecord(){}
  
  virtual void undo(CentralWidget *central);
};

/******************************************************************************/
/*				       			      		      */
/*	CLASS NAME: UndoParameterChangeRecord                       	      */
/*				      			      		      */
/******************************************************************************/

class UndoParameterChangeRecord: public UndoRecord {
  Q_OBJECT
  
private:

  BlockWidget *blockw_;
  QString block_name_;
  QString param_name_;
  QString param_value_;

public:
  
  UndoParameterChangeRecord(class LayoutFrame *frame, BlockWidget *block, QString aparam_name, QString aparam_value);
  ~UndoParameterChangeRecord(){}
  
  void undo(CentralWidget *central);

private slots:
void makeChange();

};

/******************************************************************************/
/*				       			      		      */
/*	CLASS NAME: UndoInputSocketChangeRecord                       	      	      */
/*				      			      		      */
/******************************************************************************/

class UndoInputSocketChangeRecord: public UndoRecord {
private:

  BlockWidget *blockw_;
  int nsockets_;
  

public:
  UndoInputSocketChangeRecord(class LayoutFrame *frame, BlockWidget *fromBlock, int nsockets);
  virtual ~UndoInputSocketChangeRecord(){}

  virtual void undo(CentralWidget *central);
};


/******************************************************************************/
/*				       			      		      */
/*	CLASS NAME: UndoOutputSocketChangeRecord                       	      	      */
/*				      			      		      */
/******************************************************************************/

class UndoOutputSocketChangeRecord: public UndoRecord {
private:

  BlockWidget *blockw_;
  int nsockets_;
  

public:
  UndoOutputSocketChangeRecord(class LayoutFrame *frame, BlockWidget *fromBlock, int nsockets);
  virtual ~UndoOutputSocketChangeRecord(){}

  virtual void undo(CentralWidget *central);
};


/******************************************************************************/
/*				       			      		      */
/*	CLASS NAME: UndoBlockNameChangeRecord                       	      	      */
/*				      			      		      */
/******************************************************************************/

class UndoBlockNameChangeRecord: public UndoRecord {
private:

  BlockWidget *blockw_;
  QString blockname_;
  

public:
  UndoBlockNameChangeRecord(class LayoutFrame *frame, BlockWidget *fromBlock, QString blockname);
  virtual ~UndoBlockNameChangeRecord(){}

  virtual void undo(CentralWidget *central);
};


#endif

/* End of ctk_gui_undo.hh */
