/******************************************************************************/
/*									      */
/*	ctk_helpwindow.cpp	      		      			      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*						   			      */
/*      CTK VERSION 1.3.5  Apr 22, 2007	           		      */
/*									      */
/******************************************************************************/

#include "ctk_helpwindow.hh"

#include <stdlib.h>

#include <qtextbrowser.h>
#include <qframe.h>
#include <qstatusbar.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qmessagebox.h>
#include <qfiledialog.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qobjectlist.h>

#include <ctype.h>

/******************************************************************************/
/*									      */
/*	CLASS NAME: HelpWindow		        			      */
/*									      */
/******************************************************************************/

HelpWindow::HelpWindow( const QString& home, const QString& path, QWidget* parent, const char *name )
    : QMainWindow( parent, name, WDestructiveClose ), path_combo_( 0 ), selected_URL_(),
      path_( QFileInfo( home).dirPath( TRUE ), "*.htm*" )
{
    QString imagedir = QString(getenv("CTKROOT")) + "/src/GUI/images/";

    readHistory();
    readBookmarks();

    file_list_ = path_.entryList();

    browser_ = new QTextBrowser( this );
    browser_->mimeSourceFactory()->setFilePath( path );
    browser_->setFrameStyle( QFrame::Panel | QFrame::Sunken );
    connect( browser_, SIGNAL( textChanged() ),
             this, SLOT( textChanged() ) );

    setCentralWidget( browser_ );

    if ( !home.isEmpty() )
        browser_->setSource( home );

    connect( browser_, SIGNAL( highlighted( const QString&) ),
             statusBar(), SLOT( message( const QString&)) );

    resize( 640,700 );

    QPopupMenu* file = new QPopupMenu( this );
    file->insertItem( tr("&New Window"), this, SLOT( newWindow() ), ALT | Key_N );
    file->insertItem( tr("&Open File"), this, SLOT( openFile() ), ALT | Key_O );
    file->insertSeparator();
    file->insertItem( tr("&Close"), this, SLOT( close() ), ALT | Key_Q );

    QPopupMenu* go = new QPopupMenu( this );
    backward_id_ = go->insertItem( QPixmap(imagedir+"back.xpm"),
                                 tr("&Backward"), browser_, SLOT( backward() ),
                                 ALT | Key_Left );
    forward_id_ = go->insertItem( QPixmap(imagedir+"forward.xpm"),
                                tr("&Forward"), browser_, SLOT( forward() ),
                                ALT | Key_Right );
    go->insertItem( QPixmap(imagedir+"home.xpm"), tr("&Home"), browser_, SLOT( home() ) );

    QPopupMenu* help = new QPopupMenu( this );
    help->insertItem( tr("&About ..."), this, SLOT( about() ) );

    hist_menu_ = new QPopupMenu( this );
    QStringList::Iterator it = history_.begin();
    for ( ; it != history_.end(); ++it )
        history_map_[ hist_menu_->insertItem( *it ) ] = *it;
    connect( hist_menu_, SIGNAL( activated( int ) ),
             this, SLOT( histChosen( int ) ) );

    book_menu_ = new QPopupMenu( this );
    book_menu_->insertItem( tr( "Add Bookmark" ), this, SLOT( addBookmark() ) );
    book_menu_->insertSeparator();

    QStringList::Iterator it2 = bookmarks_.begin();
    for ( ; it2 != bookmarks_.end(); ++it2 )
        bookmark_map_[ book_menu_->insertItem( *it2 ) ] = *it2;
    connect( book_menu_, SIGNAL( activated( int ) ),
             this, SLOT( bookmChosen( int ) ) );

    menuBar()->insertItem( tr("&File"), file );
    menuBar()->insertItem( tr("&Go"), go );
    menuBar()->insertItem( tr( "History" ), hist_menu_ );
    menuBar()->insertItem( tr( "Bookmarks" ), book_menu_ );
    menuBar()->insertSeparator();
    menuBar()->insertItem( tr("&Help"), help );

    menuBar()->setItemEnabled( forward_id_, FALSE);
    menuBar()->setItemEnabled( backward_id_, FALSE);
    connect( browser_, SIGNAL( backwardAvailable( bool ) ),
             this, SLOT( setBackwardAvailable( bool ) ) );
    connect( browser_, SIGNAL( forwardAvailable( bool ) ),
             this, SLOT( setForwardAvailable( bool ) ) );


    QToolBar* toolbar = new QToolBar( this );
    addToolBar( toolbar, "Toolbar");
    QToolButton* button;

    button = new QToolButton( QPixmap(imagedir+"back.xpm"), tr("Backward"), "", browser_, SLOT(backward()), toolbar );
    connect( browser_, SIGNAL( backwardAvailable(bool) ), button, SLOT( setEnabled(bool) ) );
    button->setEnabled( FALSE );
    button = new QToolButton( QPixmap(imagedir+"forward.xpm"), tr("Forward"), "", browser_, SLOT(forward()), toolbar );
    connect( browser_, SIGNAL( forwardAvailable(bool) ), button, SLOT( setEnabled(bool) ) );
    button->setEnabled( FALSE );
    button = new QToolButton( QPixmap(imagedir+"home.xpm"), tr("Home"), "", browser_, SLOT(home()), toolbar );

    toolbar->addSeparator();

    path_combo_ = new QComboBox( TRUE, toolbar );
    connect( path_combo_, SIGNAL( activated( const QString & ) ),
             this, SLOT( pathSelected( const QString & ) ) );
    toolbar->setStretchableWidget( path_combo_ );
    setRightJustification( TRUE );

    path_combo_->insertItem( home );
    path_combo_->installEventFilter( this );
    QObjectList *l = queryList( "QLineEdit" );
    if ( l && l->first() )
        ( (QLineEdit*)l->first() )->installEventFilter( this );

    browser_->setFocus();
}


void HelpWindow::setBackwardAvailable( bool b)
{
    menuBar()->setItemEnabled( backward_id_, b);
}

void HelpWindow::setForwardAvailable( bool b)
{
    menuBar()->setItemEnabled( forward_id_, b);
}


void HelpWindow::textChanged()
{
    if ( browser_->documentTitle().isNull() )
        setCaption( browser_->context() );
    else
        setCaption( browser_->documentTitle() ) ;

    selected_URL_ = caption();
    if ( !selected_URL_.isEmpty() && path_combo_ ) {
        path_ = QDir( QFileInfo( selected_URL_ ).dirPath( TRUE ), "*.htm*" );
        file_list_ = path_.entryList();
        bool exists = FALSE;
        int i;
        for ( i = 0; i < path_combo_->count(); ++i ) {
            if ( path_combo_->text( i ) == selected_URL_ ) {
                exists = TRUE;
                break;
            }
        }
        if ( !exists ) {
            path_combo_->insertItem( selected_URL_, 0 );
            path_combo_->setCurrentItem( 0 );
            history_map_[ hist_menu_->insertItem( selected_URL_ ) ] = selected_URL_;
        } else
            path_combo_->setCurrentItem( i );
        selected_URL_ = QString::null;
    }
}

HelpWindow::~HelpWindow()
{
    history_.clear();
    QMap<int, QString>::Iterator it = history_map_.begin();
    for ( ; it != history_map_.end(); ++it )
        history_.append( *it );

    QFile f( QDir::currentDirPath() + "/.history" );
    f.open( IO_WriteOnly );
    QDataStream s( &f );
    s << history_;
    f.close();

    bookmarks_.clear();
    QMap<int, QString>::Iterator it2 = bookmark_map_.begin();
    for ( ; it2 != bookmark_map_.end(); ++it2 )
        bookmarks_.append( *it2 );

    QFile f2( QDir::currentDirPath() + "/.bookmarks" );
    f2.open( IO_WriteOnly );
    QDataStream s2( &f2 );
    s2 << bookmarks_;
    f2.close();
}

void HelpWindow::about()
{
    QMessageBox::about( this, "CTK Help",
                        "CTK Help V1.0"
        );
}


void HelpWindow::openFile()
{
    QString fn = QFileDialog::getOpenFileName( QString::null, QString::null, this );
    if ( !fn.isEmpty() )
        browser_->setSource( fn );
}

void HelpWindow::newWindow()
{
    ( new HelpWindow(browser_->source(), "CTK help") )->show();
}

void HelpWindow::pathSelected( const QString &path )
{
    browser_->setSource( path );
    path_ = QDir( QFileInfo( path ).dirPath( TRUE ), "*.htm*" );
    file_list_ = path_.entryList();
    QMap<int, QString>::Iterator it = history_map_.begin();
    bool exists = FALSE;
    for ( ; it != history_map_.end(); ++it ) {
        if ( *it == path ) {
            exists = TRUE;
            break;
        }
    }
    if ( !exists )
        history_map_[ hist_menu_->insertItem( path ) ] = path;
}

bool HelpWindow::eventFilter( QObject * o, QEvent * e )
{
    QObjectList *l = queryList( "QLineEdit" );
    if ( !l || !l->first() )
        return FALSE;

    QLineEdit *lined = (QLineEdit*)l->first();

    if ( ( o == path_combo_ || o == lined ) &&
         e->type() == QEvent::KeyPress ) {

        if ( isprint(((QKeyEvent *)e)->ascii()) ) {
            if ( lined->hasMarkedText() )
                lined->del();
            QString nt( lined->text() );
            nt.remove( 0, nt.findRev( '/' ) + 1 );
            nt.truncate( lined->cursorPosition() );
            nt += (char)(((QKeyEvent *)e)->ascii());

            QStringList::Iterator it = file_list_.begin();
            while ( it != file_list_.end() && (*it).left( nt.length() ) != nt )
                ++it;

            if ( !(*it).isEmpty() ) {
                nt = *it;
                int cp = lined->cursorPosition() + 1;
                int l = path_.canonicalPath().length() + 1;
                lined->validateAndSet( path_.canonicalPath() + "/" + nt, cp, cp, l + nt.length() );
                return TRUE;
            }
        }
    }

    return FALSE;
}

void HelpWindow::readHistory()
{
    if ( QFile::exists( QDir::currentDirPath() + "/.history" ) ) {
        QFile f( QDir::currentDirPath() + "/.history" );
        f.open( IO_ReadOnly );
        QDataStream s( &f );
        s >> history_;
        f.close();
        while ( history_.count() > 20 )
            history_.remove( history_.begin() );
    }
}

void HelpWindow::readBookmarks()
{
    if ( QFile::exists( QDir::currentDirPath() + "/.bookmarks" ) ) {
        QFile f( QDir::currentDirPath() + "/.bookmarks" );
        f.open( IO_ReadOnly );
        QDataStream s( &f );
        s >> bookmarks_;
        f.close();
    }
}

void HelpWindow::histChosen( int i )
{
    if ( history_map_.contains( i ) )
        browser_->setSource( history_map_[ i ] );
}

void HelpWindow::bookmChosen( int i )
{
    if ( bookmark_map_.contains( i ) )
        browser_->setSource( bookmark_map_[ i ] );
}

void HelpWindow::addBookmark()
{
    bookmark_map_[ book_menu_->insertItem( caption() ) ] = caption();
}


#include "moc_ctk_helpwindow.cpp"

/* End of ctk_gui.cpp */
