/******************************************************************************/
/*									      */
/*	ctk_FFT.cpp	    		      			              */
/*									      */
/*	Block for applying an FFT to a frame                                  */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*							     		      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		        	      */
/*									      */
/******************************************************************************/
 
#include "ctk-config.h"

#include <cmath>
#include <vector>
 
#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_dsp.hh"
#include "ctk_data_descriptor.hh"
#include "ctk_socket.hh"
#include "ctk_FFT.hh"


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: FFTBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string FFTBlock::help_text = FFT_BLOCK_HELP_TEXT;
const string FFTBlock::type_name = "FFT";

FFTBlock::FFTBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {
  make_input_sockets(1);
  make_output_sockets(1);
}
 
Block* FFTBlock::clone(const string &n) const{
  Block *ablock = new FFTBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void FFTBlock::reset() {
  Block::reset();

  // Check validity of input data
  const DataDescriptor *dd=(*input_sockets)[0]->get_data_descriptor();
  Integer nDims=dd->get_n_dimensions();
  if (nDims!=1) {
    cerr << "Error in network at block: " << getname() << endl;
    cerr << "Attempting to apply FFT to ";
    if (nDims==0)
      cerr << "sample data" << endl;
    else
      cerr << nDims << "-dimensional frame data." << endl;
    cerr << "FFT can only be applied to 1 dimensional frame data." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  if (inputs_are_all_sample_data) {
    cerr << "Error in network at block: " << getname() << endl;
    cerr << "Attempting to apply FFT to sample data" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
  // Check validity of input data
  if ((dd->get_dimension("TIME")==NULL) && (dd->get_dimension("FREQUENCY")==NULL)) {
    cerr << "Error in network at block: " << getname() << endl;
    cerr << "FFT can only be applied to 1 dimensional TIME or FREQUENCY data." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
}

void FFTBlock::build_output_data_descriptors() {
  const DataDescriptor *idd = (*input_sockets)[0]->get_data_descriptor();
  DataDescriptor *dd = new DataDescriptor(*idd);
  
  const DimensionDescriptor *dimension;
  string input_name, output_name;
  
  // Reciprocate AXIS and rename appropriately
  if ((dimension=idd->get_dimension(input_name="TIME"))!=NULL) {
    output_name="FREQUENCY";
  } else {
    dimension=idd->get_dimension(input_name="FREQUENCY");
    output_name="TIME";
  }
  CTKVector axis=dimension->get_axis();

  outframe_size=(int)pow(2.0,int(log(float(axis.size()))/log(2.0)+0.5));
  must_pad= (outframe_size!=axis.size());
  
  CTKVector axis2(outframe_size/2);
  for (UInteger i=0; i<axis2.size(); ++i) axis2[i]=i;
  Float scale=1.0/(axis[1]*axis2.size());
  for (CTKVector::iterator fp=axis2.begin(); fp<axis2.end(); ++fp)
    (*fp)*=scale;
  dd->replace_dimension(input_name, output_name, axis2);   
 
  // Pass this descriptor to all the output socket
  (*output_sockets)[0]->set_data_descriptor(dd);
   
}

void FFTBlock::compute() {

  CTKVector *a;
  
  (*input_sockets)[0]->get_vector(a);

  if (must_pad) {
    CTKVector *new_a = new CTKVector(outframe_size);
    for (UInteger i=0; i<(*a).size(); ++i)
      (*new_a)[i]=(*a)[i];
    delete a;
    a=new_a;
  }

      
  Integer hsize=a->size()/2;

  real.resize(hsize);
  imag.resize(hsize);
  
  realft(*a, real, imag);
  delete a; 
 
  CTKVector *b=new CTKVector(hsize); 
  Float *rp=&real[0], *ip=&imag[0], *bp=&(*b)[0];
  Float *bpend=bp+hsize;
  for (; bp!=bpend; ++ip, ++rp, ++bp)
    *bp=sqrt((*rp) * (*rp) + (*ip)*(*ip));
  (*output_sockets)[0]->put_vector(b);
}



/* End of ctk_FFT.cpp */
 
