/******************************************************************************/
/*									      */
/*	ctk_binary_ops.cpp	      					      */
/*									      */
/*	Binary Op Blocks - these are a class of block that take a two	      */
/*       inputs and transform them to produce a single output.                */
/*            X(t)=F(Y1(t), Y2(t))                                            */           
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			              */
/*									      */
/******************************************************************************/

#include "ctk_local.hh"

#include "ctk_error.hh"

#include "ctk_param.hh"
#include "ctk_socket.hh"

#include "ctk_binary_ops.hh"

const Float CTK_BIG_NUMBER = 1.0e+40;

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: BinaryOpBlock                                            */
/*                                                                            */
/******************************************************************************/

BinaryOpBlock::BinaryOpBlock(const string &a_name, const string &a_block_type):Block(a_name, a_block_type) {
  make_output_sockets(1);
}

void BinaryOpBlock::reset() {
  Block::reset();
  
  // Check the inputs all have the same shape
  if (input_shape_check()==false) {
    cerr << "BinaryOpBlock:: Inputs have varying dimensionality" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
}
    
/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: P0_BinaryOpBlock                                          */
/*                                                                            */
/******************************************************************************/

P0_BinaryOpBlock::P0_BinaryOpBlock(const string &a_name, const string &a_block_type):BinaryOpBlock(a_name, a_block_type) {
  make_input_sockets(2);
}

void P0_BinaryOpBlock::reset() {
  BinaryOpBlock::reset();
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: P1_BinaryOpBlock                                          */
/*                                                                            */
/******************************************************************************/
P1_BinaryOpBlock::P1_BinaryOpBlock(const string &a_name, const string &a_block_type):BinaryOpBlock(a_name, a_block_type) {

  make_input_sockets(2);

  X_param= new ParamFloat("X");
  X_param->set_helptext("The operation's parameter.");
  parameters->register_parameter(X_param);
}

void P1_BinaryOpBlock::reset() {
  BinaryOpBlock::reset();
  param_1=X_param->get_value();
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: DividerBlock                                             */
/*                                                                            */
/******************************************************************************/

const string DividerBlock::help_text =DIVIDER_BLOCK_HELP_TEXT ;
const string DividerBlock::type_name = "Divider";

DividerBlock::DividerBlock(string a_name):CTKObject(a_name),P0_BinaryOpBlock(a_name, type_name) {
  input_sockets->set_description("in1", "numerator");
  input_sockets->set_description("in2", "denominator");
}

Block* DividerBlock::clone(const string &n) const{
  Block *ablock = new DividerBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void DividerBlock::compute() {
  if (inputs_are_all_sample_data) {
    // For dividing sample data
    Float x, y, z=0.0;
    (*input_sockets)[0]->get_sample(x);
    (*input_sockets)[1]->get_sample(y);
    z = (y==0.0)?CTK_BIG_NUMBER:x/y;
    (*output_sockets)[0]->put_sample(z);

  } else {
    // For dividing frame data
    CTKVector *xv0, *xv1;
    
    (*input_sockets)[0]->get_vector(xv0);
    (*input_sockets)[1]->get_vector(xv1);
    
    for (CTKVector::iterator fp1=xv0->begin(), fp2=xv1->begin(); fp1<xv0->end(); ++fp1, ++fp2)
      *fp1/=*fp2;    
    
    delete xv1;
    
    (*output_sockets)[0]->put_vector(xv0);
  }
  
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: ComparatorBlock                                          */
/*                                                                            */
/******************************************************************************/

const string ComparatorBlock::help_text = COMPARATOR_BLOCK_HELP_TEXT;
const string ComparatorBlock::type_name = "Comparator";

ComparatorBlock::ComparatorBlock(string a_name):CTKObject(a_name),P1_BinaryOpBlock(a_name, type_name) {
  input_sockets->set_description("in1", "+ve terminal");
  input_sockets->set_description("in2", "-ve terminal");
  
  X_param->set_helptext("The amount by which in1 must exceed in2 for the output to be true.");
}

Block* ComparatorBlock::clone(const string &n) const{
  Block *ablock = new ComparatorBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void ComparatorBlock::compute() {
  if (inputs_are_all_sample_data) {
    // For sample data
    CTKSample sample1, sample2;
    (*input_sockets)[0]->get_sample(sample1);
    (*input_sockets)[1]->get_sample(sample2);
    CTKSample sample=(sample1>sample2+param_1);
    (*output_sockets)[0]->put_sample(sample);
  } else {
    // For frame data
    CTKVector *xv1, *xv2;
    (*input_sockets)[0]->get_vector(xv1);
    (*input_sockets)[1]->get_vector(xv2);
    for (Float *f1p=&(*xv1)[0], *f2p=&(*xv2)[0]; f1p<&(*xv1)[xv1->size()]; ++f1p, ++f2p)
      *f1p=(*f1p>*f2p+param_1);
    (*output_sockets)[0]->put_vector(xv1);
    delete xv2;
  }
}



/* End of ctk_binary_ops.cpp */
