/******************************************************************************/
/*									      */
/*	ctk_colour_mask.cpp	        	      		              */
/*									      */
/*	Applies simple grouping to a 1/0 missing data  mask                   */
/*									      */
/*	Author: Jon Barker, Sheffield University            		      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007     	         	      */
/*									      */
/******************************************************************************/

#include <cmath>
#include <vector>
#include <deque>
#include <map>

#include <algorithm>

#include "ctk_local.hh"


#include "ctk_socket.hh"  
#include "ctk_data_descriptor.hh"
#include "ctk_param.hh"    

#include "ctk_group.hh"
#include "ctk_colour_mask.hh"  


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: ColourMaskBlock                                         */
/*                                                                            */
/******************************************************************************/

const string ColourMaskBlock::type_name = "ColourMask";
const string ColourMaskBlock::help_text = COLOUR_MASK_BLOCK_HELP_TEXT;


ColourMaskBlock::ColourMaskBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {
  previous_voicing_decision=false;
  previous_pitch=0.0;
  
  make_input_sockets(3);
  input_sockets->set_description("in1", "mask");
  input_sockets->set_description("in2", "pitch estimate");
  input_sockets->set_description("in3", "voicing estimate");
  input_sockets->set_optional("in2");
  input_sockets->set_optional("in3");

  make_output_sockets(1);
  output_sockets->set_description("in1", "group labels");

  // Set up WINDOW_SIZE param
  WINDOW_SIZE_param= new ParamInt("WINDOW_SIZE", PARAM_DEFAULT_WINDOW_SIZE);
  WINDOW_SIZE_param->install_validator(new Validator(Validator::VLOWER, 1.0));
  WINDOW_SIZE_param->set_helptext("The size of the window that colouring algorithm operates on.");
  WINDOW_SIZE_param->set_unit("frames");
  parameters->register_parameter(WINDOW_SIZE_param);

  // Set up ONSET_GROUPING param
  ONSET_GROUPING_param= new ParamBool("ONSET_GROUPING", PARAM_DEFAULT_ONSET_GROUPING);     
  ONSET_GROUPING_param->set_helptext("If set ON then groups starting at the same frame will be merged.");
  parameters->register_parameter(ONSET_GROUPING_param);

  // Set up OFFSET_GROUPING param
  OFFSET_GROUPING_param= new ParamBool("OFFSET_GROUPING", PARAM_DEFAULT_OFFSET_GROUPING);     
  OFFSET_GROUPING_param->set_helptext("If set ON then groups ending at the same frame will be merged with the later starting group inheritting the group label of the earlier starting group. Note, merging will only occur if the later starting group is fully contained within the buffered grouping window.");
  parameters->register_parameter(OFFSET_GROUPING_param);

  // Set up MIN_GROUP_SIZE param
  MIN_GROUP_SIZE_param= new ParamInt("MIN_GROUP_SIZE", PARAM_DEFAULT_MIN_GROUP_SIZE);     
  MIN_GROUP_SIZE_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  MIN_GROUP_SIZE_param->set_helptext("Groups that have a size  of less than MIN_GROUP_SIZE (measured in terms of the number of pixels) will be merged into the forced-foreground group.");
  MIN_GROUP_SIZE_param->set_unit("pixels");
  parameters->register_parameter(MIN_GROUP_SIZE_param);

  // Set up MAX_GROUP_SIZE param
  MAX_GROUP_SIZE_param= new ParamInt("MAX_GROUP_SIZE", PARAM_DEFAULT_MAX_GROUP_SIZE);     
  MAX_GROUP_SIZE_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  MAX_GROUP_SIZE_param->set_helptext("If a group grows to reach MAX_GROUP_SIZE then it will be forced to end at the current frame, and in the next frame a new group will be spawned. i.e. groups that would otherwise be larger than MAX_GROUP_SIZE are effectively cut into smaller segments.");
  MAX_GROUP_SIZE_param->set_unit("pixels");
  parameters->register_parameter(MAX_GROUP_SIZE_param);

  // Set up NUM_SUBBANDS param
  NUM_SUBBANDS_param= new ParamInt("NUM_SUBBANDS", PARAM_DEFAULT_NUM_SUBBANDS);
  NUM_SUBBANDS_param->install_validator(new Validator(Validator::VLOWER, 1.0));
  NUM_SUBBANDS_param->set_helptext("This parameter defines the number of subbands (by default 1) into which the incoming data is cut before group labelling commences. Group labelling then operates independently in each band. <p> Note, group labels will still be unique across all the subbands.");
  parameters->register_parameter(NUM_SUBBANDS_param);

  // Set up HAS_DELTAS param
  HAS_DELTAS_param= new ParamBool("HAS_DELTAS", PARAM_DEFAULT_HAS_DELTAS);
  HAS_DELTAS_param->set_helptext("If set ON then the incoming vectors are considered to be split into two halves: the lower half representing the data mask, and the upper half representing the delta feature mask (or whatever) derived from the data mask. Grouping is performed on the lower stream, and then elements set in the upper stream inherit the labels of the corresponding points in the lower stream.");
  parameters->register_parameter(HAS_DELTAS_param);

  // set up VOICING_THRESHOLD param
  VOICING_THRESHOLD_param = new ParamFloat("VOICING_THRESHOLD", PARAM_DEFAULT_VOICING_THRESHOLD);
  VOICING_THRESHOLD_param->set_helptext("Frame is label to be voiced is voicing input is greater than threshold, else it is labeled unvoiced. Groups will be broken at voicing boundaries.");
  parameters->register_parameter(VOICING_THRESHOLD_param);

  // set up DELTA_PITCH_THRESHOLD param
  DELTA_PITCH_THRESHOLD_param = new ParamFloat("DELTA_PITCH_THRESHOLD", PARAM_DEFAULT_DELTA_PITCH_THRESHOLD);
  DELTA_PITCH_THRESHOLD_param->set_helptext("A change of pitch greater than the threshold will cause groups to be broken.");
  parameters->register_parameter(DELTA_PITCH_THRESHOLD_param);
  
}


ColourMaskBlock::~ColourMaskBlock() {
}

Block* ColourMaskBlock::clone(const string &n) const{
  Block *ablock = new ColourMaskBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}


void ColourMaskBlock::reset() {
  Block::reset();  

  groups.clear();
  frame_number=0;
  new_group_number=1;
  
  window_size=WINDOW_SIZE_param->get_value();
  onset_grouping=ONSET_GROUPING_param->get_value();
  offset_grouping=OFFSET_GROUPING_param->get_value();
  min_group_size=MIN_GROUP_SIZE_param->get_value();
  max_group_size=MAX_GROUP_SIZE_param->get_value();
  num_subbands=NUM_SUBBANDS_param->get_value();
  has_deltas=HAS_DELTAS_param->get_value();
  voicing_threshold=VOICING_THRESHOLD_param->get_value();
  delta_pitch_threshold=DELTA_PITCH_THRESHOLD_param->get_value();
  
}

void ColourMaskBlock::compute() {

  CTKVector *xv0;
  CTKSample voicing, pitch;
  bool voicing_decision = true;
  bool break_groups = false;
  
  (*input_sockets)[0]->get_vector(xv0);

  CTKVector special;

  // Process voicing input (if connected)
  if ((*input_sockets)[2]->connected()) {
    (*input_sockets)[2]->get_sample(voicing);
    voicing_decision=(voicing>voicing_threshold);

    if (voicing_decision!=previous_voicing_decision) {
      break_groups=true; // voicing break
      //    cerr << "voicing break\n";
    }
    previous_voicing_decision=voicing_decision;
  }
  
  // Process pitch input (if connected)
  if ((*input_sockets)[1]->connected()) {
    (*input_sockets)[1]->get_sample(pitch);

    if (fabs(pitch-previous_pitch)>delta_pitch_threshold) {
      if (voicing_decision) {
	break_groups=true;// pitch break
	// cerr << "pitch break\n";
      }
    }
    previous_pitch=pitch;
  }
  
  vector_width = xv0->size();
  
  width=(xv0->size()/num_subbands);

  if (has_deltas)
    xv0->insert(xv0->begin()+xv0->size()/2, 0);
  
  if (num_subbands>1) {
    // Make subbands by inserting 0's at the subband boundaries
    if (has_deltas) width=width/2;
    for (Integer i=num_subbands-1; i>0; --i) 
      xv0->insert(xv0->begin()+i*width, 0);
  }

  special.resize(xv0->size());

  // Mark all incoming data as -ve
  for (CTKVector::iterator xp = xv0->begin(), specialp=special.begin(); xp!=xv0->end(); ++xp, ++specialp) {
    *specialp = (*xp==-1);
    if (*xp<0) *xp=0;
    *xp= -*xp;
  }

  ++frame_number;

  if (frame_number>1) {
    if (break_groups) {
      // Previous groups are not propagated i.e. a new set of groups will start
    } else {
      // Propagate groups from previous frame
      propagate_frame(xvlast, *xv0);
      spread_groups_across_frequency(*xv0);
    }
  }

  start_new_groups(*xv0);

  frame_buffer.push_back(xv0);

  merge_adjacent_groups(*xv0);

  if (frame_number>1 && frame_buffer.size()>1) {
    offset_detection();
  }
      
  for (CTKVector::iterator xp = xv0->begin(), specialp=special.begin(); xp!=xv0->end(); ++xp, ++specialp) {
    if (*specialp) *xp=-1; 
  }
  
  xvlast=*xv0;
  
  if (frame_number>window_size) {
    CTKVector *xv= frame_buffer[0];

    clean_up(*xv);
       
    (*output_sockets)[0]->put_vector(xv);
    frame_buffer.pop_front();
  }
  
}




void ColourMaskBlock::flush_buffer() {

  // Clear internal buffer onto output
  
  while (frame_buffer.size()>0) {
    clean_up(*frame_buffer[0]);
    (*output_sockets)[0]->put_vector(frame_buffer[0]);
    frame_buffer.pop_front();
  }
  
}


void ColourMaskBlock::close() {

  Block::close(); 

}

void ColourMaskBlock::propagate_frame(const CTKVector &x0, CTKVector &x1) {

  CTKVector::iterator xp1=x1.begin();

  int channel_number=1;
  
  for (CTKVector::const_iterator xp0 = x0.begin(); xp0!=x0.end(); ++xp0, ++xp1, ++channel_number) {
    if ((int)*xp0!=0 && *xp1<0.0 && (groups[(int)*xp0].area()<max_group_size || max_group_size==0)) {
      if (groups[(int)*xp0].label()==(int)*xp1) { // Only if input labels match
	*xp1=*xp0;
	groups[(int)*xp1].add_point(frame_number, channel_number);
      }
    }
  }
  
}

// Process the mask frame, x, spreading groups across frequency
void ColourMaskBlock::spread_groups_across_frequency(CTKVector &x) {

  int old_x=(int)x[0];

  int channel_number=1;
  
  for (CTKVector::iterator xp = x.begin()+1; xp!=x.end(); ++xp, ++channel_number) {
    if (*xp<0.0 &&  (int)old_x !=0) {
      if (groups[old_x].label()==(int)*xp) {
	*xp=old_x;
	groups[old_x].add_point(frame_number, channel_number);
      }
    }
    old_x=(int)*xp;
  }

  old_x=(int)x.back();

  channel_number=x.size();
  
  for (CTKVector::reverse_iterator xp = x.rbegin()+1; xp!=x.rend(); ++xp, --channel_number) {
    if (*xp<0.0 && old_x !=0) {
      if (groups[old_x].label()==(int)*xp) {
	*xp=old_x;
	groups[old_x].add_point(frame_number, channel_number);
      }
    }
    old_x=(int)*xp;
  }
  
}

void ColourMaskBlock::start_new_groups(CTKVector &x) {
  
  Boolean in_group=false;
  float last_label=0.0;
  map<int, int> group_label_map;
  int group_label=0;
  int channel_number=1;
  
  for (CTKVector::iterator xp = x.begin(); xp!=x.end(); ++xp, ++channel_number) {

    if (*xp<0.0) {
      if ((onset_grouping==false && in_group==false) || (group_label=group_label_map[(int)*xp])==0) {
	++new_group_number;
	groups[new_group_number]=Group(0, (int)*xp);
	group_label_map[(int)*xp]=new_group_number;
	group_label=new_group_number;
      } 
      
      groups[new_group_number].add_point(frame_number, channel_number);
      in_group=true;
      last_label=*xp;
      *xp=group_label;;
      
    } else {
      in_group=false;
    }
    
  }

}

void ColourMaskBlock::merge_adjacent_groups(CTKVector &x) {

  int old_x=(int)x[0];
  
  for (CTKVector::iterator xp = x.begin()+1; xp!=x.end(); ++xp) {

    if ((old_x!=*xp) && (*xp!=0) && (old_x!=0)) {
      if (groups[(int)*xp].label()==groups[old_x].label()) {
	// Group that started first inherits newer group
	if (groups[old_x].start_frame()<groups[(int)*xp].start_frame())
	  relabel((int)*xp, old_x);
	else
	  relabel(old_x, (int)*xp);
      }
    }

    old_x=(int)*xp;
  }
}


void ColourMaskBlock::relabel(int old_label, int new_label) {

  // Don't merge if it will make a group bigger than the maximum allowable size
  if ((max_group_size>0) && (groups[new_label].area()+groups[old_label].area()>max_group_size))
    return;

  int this_frame=frame_number-frame_buffer.size(); // frame number for start of frame buffer
  for (deque<CTKVector *>::iterator xvp = frame_buffer.begin(); xvp!=frame_buffer.end(); ++xvp) {

    int channel_number=1;
    for (CTKVector::iterator xp = (*xvp)->begin(); xp!=(*xvp)->end(); ++xp, ++channel_number) {
      if (*xp==old_label) {
	groups[new_label].add_point(this_frame, channel_number);
	groups[old_label].remove_point(this_frame, channel_number);
      }
      ++this_frame;
    }
    
    replace((*xvp)->begin(), (*xvp)->end(), old_label, new_label);
  }
  
}

// Find groups that have offset.
// For groups that are fully contained with the frame buffer:
//   Apply offset grouping if requested.
//   Groups below the threshold size can be assigned to a common "fixed foreground" group
//   (this reduces the number of groups and simplifies the decoding without perhaps significantly altering the result) 

void ColourMaskBlock::offset_detection() {

  map<Integer, Boolean> offsets;

  CTKVector *this_frame = *(frame_buffer.end()-1);
  CTKVector *prev_frame = *(frame_buffer.end()-2);

  // Find all integers occurring in vector prev_frame which don't occurr in this_frame;

  for (CTKVector::iterator ip=prev_frame->begin(); ip!=prev_frame->end(); ++ip)
    offsets[(int)*ip]=true;

  for (CTKVector::iterator ip=this_frame->begin(); ip!=this_frame->end(); ++ip)
    offsets[(int)*ip]=false;

  for (map<Integer, Boolean>::iterator ip=offsets.begin(); ip!=offsets.end(); ++ip) {
    if (ip->second) {
      if (groups[ip->first].start_frame()>=frame_number-window_size) {
	if (groups[ip->first].area()<=min_group_size) {
	  // If the group is below the min_group_size then reassign its points to the  "-1" group
	  relabel(ip->first, -1);
	  ip->second=false;
	}
      } else
	(ip->second=false);
    }
  }

  if (offset_grouping) {
    // Merge all remaining offsetting groups that haven't been assigned to the "-1" group
    // (But only if their input labels matched)

    Boolean found_first_group=false;
    Integer first_group_number=0;
    bool repeat_loop = true;
    
    while (repeat_loop) { // Repeat this loop all the time there are unconsidered offsetting groups remaining
      repeat_loop=false;
      
      for (map<Integer, Boolean>::iterator ip=offsets.begin(); ip!=offsets.end(); ++ip) {
	if (ip->second) {
	  if (!found_first_group) {
	    found_first_group=true;
	    first_group_number=ip->first;
	    ip->second=false;
	    repeat_loop=true;
	  } else {
	    // Check input labels match before merging groups
	    if (groups[ip->first].label()==groups[first_group_number].label()) {
	      relabel(ip->first, first_group_number);
	      ip->second=false;
	    }
	  } // if-else
	}  // if
      } // for

    } // while (repeat_loop);
    
  } // if
  
}


// Relabel deltas groups with the group labels of their non-delta features 
void ColourMaskBlock::label_deltas(CTKVector &x) {

  //  static CTKVector lastx;
  
  Integer half_width=x.size()/2;
  for (Integer i=0; i<half_width; ++i) {
    if (x[i+half_width]!=0) {
      x[i+half_width]=x[i];
    }
  }
  
  
}


void ColourMaskBlock::clean_up(CTKVector &xv) {

  // Remove 0 inserted between features and deltas
  if (has_deltas) {
    xv.erase(xv.begin()+vector_width/2 + ((num_subbands>1)*(num_subbands-1)));
  }
  // Remerge subbands by stripping out the inserted 0's
  if (num_subbands>1) {
    for (Integer i=num_subbands-1; i>0; --i) {
      xv.erase(xv.begin()+i*width +i -1);
    }
  }
  
  // Mark deltas with the group labels of their corresponding non-delta features
  if (has_deltas)
    label_deltas(xv);  
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: GroupStartEndDetection                                   */
/*                                                                            */
/******************************************************************************/


const string GroupStartEndDetectionBlock::type_name = "GroupStartEndDetection";
const string GroupStartEndDetectionBlock::help_text = CUT_GROUPS_BLOCK_HELP_TEXT;


GroupStartEndDetectionBlock::GroupStartEndDetectionBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {


  make_input_sockets(1);
  input_sockets->set_description("in1", "group labels");

  make_output_sockets(2);
  output_sockets->set_description("out1", "groups starting");
  output_sockets->set_description("out2", "groups ending");

  // Set up WINDOW_SIZE param
  WINDOW_SIZE_param= new ParamInt("WINDOW_SIZE", PARAM_DEFAULT_WINDOW_SIZE);
  WINDOW_SIZE_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  WINDOW_SIZE_param->set_helptext("The size (in frames) of the start-end detection look-ahead window.");
  parameters->register_parameter(WINDOW_SIZE_param);

}


GroupStartEndDetectionBlock::~GroupStartEndDetectionBlock() {
}

Block* GroupStartEndDetectionBlock::clone(const string &n) const{
  Block *ablock = new GroupStartEndDetectionBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}


void GroupStartEndDetectionBlock::reset() {
  Block::reset();  

  input_width=(*input_sockets)[0]->get_data_descriptor()->get_storage();

  active_groups.clear();
  input_frame_number=output_frame_number=0;
  new_group_number=1;
  window_size=WINDOW_SIZE_param->get_value();
}

void GroupStartEndDetectionBlock::compute() {
  CTKVector *xv0;
  
  (*input_sockets)[0]->get_vector(xv0);

  ++input_frame_number;

  frame_buffer.push_back(xv0);
  
  if (input_frame_number>window_size) {
    process_1_buffered_frame();
  }

}

  
void GroupStartEndDetectionBlock::close() {
  Block::close(); 
}

void GroupStartEndDetectionBlock::flush_buffer() {

  // Clear internal buffer onto output
  
  while (frame_buffer.size()>0) {
    process_1_buffered_frame();
  }

}


void GroupStartEndDetectionBlock::process_1_buffered_frame() {

  CTKVector *start_labels, *end_labels;

  start_labels=onset_detection(frame_buffer[0]);  
  end_labels=offset_detection();
    
  (*output_sockets)[0]->put_vector(start_labels);
  (*output_sockets)[1]->put_vector(end_labels);
  
  delete frame_buffer[0];       
  frame_buffer.pop_front();
  
}

// Find onsets in the given frame 
CTKVector *GroupStartEndDetectionBlock::onset_detection(CTKVector *xv0) {

  CTKVector *start_labels=new CTKVector();
  
  for (CTKVector::iterator xp = xv0->begin(), xp_end=xv0->end(); xp!=xp_end; ++xp) {
    int label = (int)*xp;
    if (label>0) {  // Only use labels 1 and up
      int active_code = active_groups[label];
      if (active_code!=1) {
	if (active_code==0)  // Starting (for the first time) 
	  start_labels->push_back(label);
	else if (active_code==-1)  { // Restarting
	  start_labels->push_back(label);
	  //	  cerr << "RESTARTING !!!\n";
	}
	active_groups[label]=1;
      } 
    }
  }

  start_labels->resize(input_width);
  
  return start_labels;

}


// Look for offsets - given active group list and the lookahead buffer
CTKVector *GroupStartEndDetectionBlock::offset_detection() {

  map<int, bool> offsets;

  CTKVector *end_labels=new CTKVector();

  // Find active groups and record them as potential offsets
  for (map<Integer, Integer>::iterator ip=active_groups.begin(); ip!=active_groups.end(); ++ip) {
    if (ip->second==1) offsets[ip->first]=true;
  }

  // Check look ahead buffer for evidence of groups continuations - they are not offsets if evidence is found
  for (deque<CTKVector *>::iterator fp = frame_buffer.begin(), fp_end=frame_buffer.end(); fp!=fp_end; ++fp) {
    for (CTKVector::iterator xp = (*fp)->begin(), xp_end=(*fp)->end(); xp!=xp_end; ++xp) {
      if (offsets[(int)*xp])
	offsets[(int)*xp]=false;
    }
  }

  // Offsets are groups for which no evidence of continuation was found in the lookahead buffer
  for (map<Integer, bool>::iterator ip=offsets.begin(); ip!=offsets.end(); ++ip) {
    if (ip->second) {
      end_labels->push_back(ip->first);
      active_groups[ip->first]=-1; 
    }
  }

  end_labels->resize(input_width);

  return end_labels;
}



/* End of ctk_colour_mask.cpp */
 
