/******************************************************************************/
/*									      */
/*	ctk_decimator.cpp	 	       			              */
/*									      */
/*	Block for decimation                                                  */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		         	      */
/*									      */
/******************************************************************************/
 
#include "ctk-config.h"

#include <cmath>
#include <vector>
 
#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"
#include "ctk_decimator.hh"


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: DecimatorBlock                                           */
/*                                                                            */
/******************************************************************************/

const string DecimatorBlock::help_text = DECIMATOR_BLOCK_HELP_TEXT;
const string DecimatorBlock::type_name = "Decimator";

//DecimatorBlock::DecimatorBlock(const string &a_name):CTKObject(a_name),Block(a_name, DecimatorBlock_type_name) {
DecimatorBlock::DecimatorBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {

  make_input_sockets(1);
  make_output_sockets(1);
  
  // Set up MULTIPLE parameter
  MULTIPLE_param= new ParamInt("MULTIPLE");
  MULTIPLE_param->set_helptext("An integer multiple by which to decimate the input signal. <p> Either MULTIPLE or OUTPUT_RATE may be set, but not both.");
  MULTIPLE_param->install_validator(new Validator(Validator::VLOWER, 2.0));
  parameters->register_parameter(MULTIPLE_param);

  // Set up OUTPUT_RATE parameter
  OUTPUT_RATE_param= new ParamFloat("OUTPUT_RATE");
  OUTPUT_RATE_param->set_helptext("The desired output sample rate.<p> The input is decimated by the integer multiple which produces an output sample rate closest to the desired value. <p>Either MULTIPLE or OUTPUT_RATE may be set, but not both.");
  OUTPUT_RATE_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(OUTPUT_RATE_param);

}


Block* DecimatorBlock::clone(const string &n) const{
  Block *ablock = new DecimatorBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void DecimatorBlock::reset() {
  Block::reset();

  Boolean multiple_param_set=MULTIPLE_param->get_set_flag(); 
  Boolean output_param_set=OUTPUT_RATE_param->get_set_flag();
  
  if (multiple_param_set && output_param_set) {
    cerr << "Error in decimator block, " << getfullname() << ", set either MULTIPLE or OUTPUT_RATE parameter, not both" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
  if (multiple_param_set) {
    start_count=MULTIPLE_param->get_value();
    unlock_parameters();
    set_parameter("SAMPLE_RATE", sample_rate_param->get_value()/start_count);
    relock_parameters();
  } else if (output_param_set) {
    start_count=(int)(sample_rate_param->get_value()/OUTPUT_RATE_param->get_value());
    unlock_parameters();
    set_parameter("SAMPLE_RATE", sample_rate_param->get_value()/start_count);
    relock_parameters();
  } else {
    cerr << "Error in decimator block, " << getfullname() << ", neither MULTIPLE nor OUTPUT_RATE have been set" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }


  count=start_count;

  if (start_count<1) {
    cerr << "Error in decimator block, " << getfullname() << ", can only decimate by an integer multiple" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
}


void DecimatorBlock::compute() {

  if (inputs_are_all_sample_data) {
    // For decimating sample data
    CTKSample sample;
    (*input_sockets)[0]->get_sample(sample);
    
    --count;
    
    if (count==0) {
      (*output_sockets)[0]->put_sample(sample);
      count=start_count;
    }
    
  }else {
    // For decimating frame data

    CTKVector *xv0;
    (*input_sockets)[0]->get_vector(xv0);
    
    --count;
    
    if (count==0) {
      (*output_sockets)[0]->put_vector(xv0);
      count=start_count;
    }
    else delete xv0;
  }
  
}

void DecimatorBlock::close() {
  unlock_parameters();
  unset_parameter("SAMPLE_RATE");
  relock_parameters();
  Block::close();
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: UpsampleBlock                                           */
/*                                                                            */
/******************************************************************************/

const string UpsampleBlock::type_name = "Upsample";
const string UpsampleBlock::help_text = UPSAMPLE_BLOCK_HELP_TEXT;

UpsampleBlock::UpsampleBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {

  make_input_sockets(1);
  make_output_sockets(1);
  
  // Set up MULTIPLE parameter
  MULTIPLE_param= new ParamInt("MULTIPLE");
  MULTIPLE_param->set_helptext("An integer multiple by which to increase the sampling rate. <p> Either MULTIPLE or OUTPUT_RATE may be set, but not both.");
  MULTIPLE_param->install_validator(new Validator(Validator::VLOWER, 2.0));
  parameters->register_parameter(MULTIPLE_param);

  // Set up OUTPUT_RATE parameter
  OUTPUT_RATE_param= new ParamFloat("OUTPUT_RATE");
  OUTPUT_RATE_param->set_helptext("The desired output sample rate.<p> The input is upsampled by the integer multiple which produces an output sample rate closest to the desired value. <p>Either MULTIPLE or OUTPUT_RATE may be set, but not both.");
  OUTPUT_RATE_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(OUTPUT_RATE_param);

}


Block* UpsampleBlock::clone(const string &n) const{
  Block *ablock = new UpsampleBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void UpsampleBlock::reset() {
  Block::reset();

  Boolean multiple_param_set=MULTIPLE_param->get_set_flag(); 
  Boolean output_param_set=OUTPUT_RATE_param->get_set_flag();
  
  if (multiple_param_set && output_param_set) {
    cerr << "Error in upsample block, " << getfullname() << ", set either MULTIPLE or OUTPUT_RATE parameter, not both" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
  if (multiple_param_set) {
    multiple=MULTIPLE_param->get_value();
    unlock_parameters();
    set_parameter("SAMPLE_RATE", sample_rate_param->get_value()*multiple);
    relock_parameters();
  } else if (output_param_set) {
    multiple=(int)(OUTPUT_RATE_param->get_value()/sample_rate_param->get_value());
    unlock_parameters();
    set_parameter("SAMPLE_RATE", sample_rate_param->get_value()*multiple);
    relock_parameters();
  } else {
    cerr << "Error in upsample block, " << getfullname() << ", neither MULTIPLE nor OUTPUT_RATE have been set" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  if (multiple<1) {
    cerr << "Error in upsample block, " << getfullname() << ", can only upsample by an integer multiple" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
}


void UpsampleBlock::compute() {

  if (inputs_are_all_sample_data) {
    // For decimating sample data
    CTKSample sample;
    (*input_sockets)[0]->get_sample(sample);
    
    for (int i=0; i<multiple; ++i) 
      (*output_sockets)[0]->put_sample(sample);
    
  }else {
    // For decimating frame data

    CTKVector *xv0;
    (*input_sockets)[0]->get_vector(xv0);
    
    (*output_sockets)[0]->put_vector(xv0);    
    for (int i=1; i<multiple; ++i) 
      (*output_sockets)[0]->put_vector(new CTKVector(*xv0));
  }
  
}

void UpsampleBlock::close() {
  unlock_parameters();
  unset_parameter("SAMPLE_RATE");
  relock_parameters();
  Block::close();
}



/* End of ctk_decimator.cpp */
 
