/******************************************************************************/
/*									      */
/*	ctk_differentiator.cpp	    	       			              */
/*									      */
/*	Block for making delta coefficients                		      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			              */
/*									      */
/******************************************************************************/
 
#include "ctk-config.h"

#include <cmath>
#include <vector>
 
#include "ctk_local.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_differentiator.hh"

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: FrameBuffer                                     */
/*                                                                            */
/******************************************************************************/

FrameBuffer::FrameBuffer() {
  size_ = 0;
  count_ = 0;
  buffer_.resize(0);
}

FrameBuffer::~FrameBuffer() {
  reset(0);
}

int FrameBuffer::width() const {
  return (count_>0)?buffer_[size_-1]->size():0;
}

int FrameBuffer::size() const {
  return size_;
}

const CTKVector & FrameBuffer::operator[](int i) const {
  return *(buffer_[i]);
}

void FrameBuffer::reset(int asize) {
  for (int i=0; i<size_; ++i)
    delete buffer_[i];

  count_ = 0;
  size_=asize;
  buffer_.resize(size_, 0);
}

void FrameBuffer::insert(CTKVector *vp) {
  delete buffer_[0];
  for (Integer i=0; i<size_-1; ++i) {
    buffer_[i] = buffer_[i+1];
  }
  buffer_[size_-1]=vp;
  ++count_;
}  

bool FrameBuffer::full() const {
  return (count_>=size_);
}

bool FrameBuffer::just_full() const {
  return (count_==size_);
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: DeltaBlock                                               */
/*                                                                            */
/******************************************************************************/

const string DeltaBlock::type_name = "Delta";
const string DeltaBlock::help_text = DELTA_BLOCK_HELP_TEXT;

DeltaBlock::DeltaBlock(const string &a_name):CTKObject(a_name), Block(a_name, type_name) {
  DELTA_WIN_param= new ParamInt("DELTA_WIN", PARAM_DEFAULT_DELTA_WIN);
  DELTA_WIN_param->set_helptext("Size of the regression window used for the delta computation. <p> This must be an odd integer.");
  DELTA_WIN_param->install_validator(new Validator(Validator::VLOWER, 1.0));
  parameters->register_parameter(DELTA_WIN_param);

  make_input_sockets(1);
  make_output_sockets(1);
}
 
Block* DeltaBlock::clone(const string &n) const{
  Block *ablock = new DeltaBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void DeltaBlock::reset() {
  Block::reset();

  frame_buffer.reset(DELTA_WIN_param->get_value()*2+1);  
}

void DeltaBlock::compute() {
  process_input_vector((*input_sockets)[0], (*output_sockets)[0], frame_buffer, construct_delta_vector); 
}


void DeltaBlock::flush_buffer() {

  // Padding at end of signal

  int win_size=DELTA_WIN_param->get_value();
  for (Integer i=0; i<win_size; ++i) 
    (*output_sockets)[0]->put_vector(construct_delta_vector(frame_buffer));
}

void DeltaBlock::close() {

  frame_buffer.reset(0);
  
  Block::close();
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: DeltaMaskBlock                                           */
/*                                                                            */
/******************************************************************************/
//
// Block for converting a missing data mask into the `strict' mask that should be used
// for the derivatives of the data

const string DeltaMaskBlock::type_name = "DeltaMask";
const string DeltaMaskBlock::help_text = DELTA_MASK_BLOCK_HELP_TEXT;

DeltaMaskBlock::DeltaMaskBlock(const string &a_name):CTKObject(a_name), Block(a_name, type_name) {
  
  DELTA_WIN_param= new ParamInt("DELTA_WIN", PARAM_DEFAULT_DELTA_WIN);
  DELTA_WIN_param->set_helptext("Size of the regression window used for the delta computation. <p> This must be an odd integer.");
  DELTA_WIN_param->install_validator(new Validator(Validator::VLOWER, 1.0));
  parameters->register_parameter(DELTA_WIN_param);

  make_input_sockets(1);
  make_output_sockets(1);
}
 
Block* DeltaMaskBlock::clone(const string &n) const{
  Block *ablock = new DeltaMaskBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void DeltaMaskBlock::reset() {

  Block::reset();

  frame_buffer.reset(DELTA_WIN_param->get_value()*2+1);
  
}

void DeltaMaskBlock::compute() {

  process_input_vector((*input_sockets)[0], (*output_sockets)[0], frame_buffer, construct_delta_mask_vector);

}


void DeltaMaskBlock::flush_buffer() {

  // Padding at end of signal
  
  int win_size=DELTA_WIN_param->get_value();
  for (Integer i=0; i<win_size; ++i) 
    (*output_sockets)[0]->put_vector(construct_delta_mask_vector(frame_buffer));
}

void DeltaMaskBlock::close() {

  frame_buffer.reset(0);
  
  Block::close();
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: BoundedDeltaBlock                                        */
/*                                                                            */
/******************************************************************************/

const string BoundedDeltaBlock::type_name = "DeltaBounds";
const string BoundedDeltaBlock::help_text = BOUNDED_DELTA_BLOCK_HELP_TEXT;


BoundedDeltaBlock::BoundedDeltaBlock(const string &a_name):CTKObject(a_name), Block(a_name, type_name) {
  
  DELTA_WIN_param= new ParamInt("DELTA_WIN", PARAM_DEFAULT_DELTA_WIN);
  DELTA_WIN_param->set_helptext("Size of the regression window used for the delta computation. <p> This must be an odd integer.");
  DELTA_WIN_param->install_validator(new Validator(Validator::VLOWER, 1.0));
  parameters->register_parameter(DELTA_WIN_param);

  make_input_sockets(4);
  make_output_sockets(4);
  
  input_sockets->set_description("in1", "Static Features");
  
  input_sockets->set_description("in2", "Missing Data Mask");
  input_sockets->set_optional("in2");

  input_sockets->set_description("in3", "Lower Bounds");
  input_sockets->set_optional("in3");
  
  input_sockets->set_description("in4", "Upper Bounds");
  input_sockets->set_optional("in4");
  
  output_sockets->set_description("out1", "Delta Features");
  output_sockets->set_description("out2", "Missing Delta Data Mask");
  output_sockets->set_description("out3", "Lower Bounds for Deltas");
  output_sockets->set_description("out4", " Upper Bounds for Deltas");

}
 
Block* BoundedDeltaBlock::clone(const string &n) const{
  Block *ablock = new BoundedDeltaBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void BoundedDeltaBlock::reset() {
  Block::reset();
  
  int win_size=DELTA_WIN_param->get_value()*2+1;
  data_buffer.reset(win_size);
  mask_buffer.reset(win_size);
  lower_bound_buffer.reset(win_size);
  upper_bound_buffer.reset(win_size);
}


void BoundedDeltaBlock::compute() {

  process_input_vector((*input_sockets)[0], (*output_sockets)[0], data_buffer, construct_delta_vector); 

  if ((*input_sockets)[1]->connected())
    process_input_vector((*input_sockets)[1], (*output_sockets)[1], mask_buffer, construct_delta_mask_vector);

  CTKVector *lbp, *ubp;
  
  if ((*input_sockets)[2]->connected())
    (*input_sockets)[2]->get_vector(lbp);
  
  if ((*input_sockets)[3]->connected())
    (*input_sockets)[3]->get_vector(ubp);

  if ((*input_sockets)[2]->connected() && (*input_sockets)[3]->connected()) {

    lower_bound_buffer.insert(lbp);
    upper_bound_buffer.insert(ubp);
    
    if (lower_bound_buffer.full()) {
      construct_delta_bounds(data_buffer, mask_buffer, lower_bound_buffer, upper_bound_buffer, delta_lbp, delta_ubp);
  
      if (lower_bound_buffer.just_full()) {
	int win_size=(lower_bound_buffer.size()-1)/2;
	for (Integer i=0; i<win_size; ++i) {
	  (*output_sockets)[2]->put_vector(new CTKVector (*delta_lbp));
	  (*output_sockets)[3]->put_vector(new CTKVector (*delta_ubp));
	}
      }

      (*output_sockets)[2]->put_vector(delta_lbp);
      (*output_sockets)[3]->put_vector(delta_ubp);
    }
  } else {
    if ((*input_sockets)[2]->connected())
      delete lbp;
    if ((*input_sockets)[3]->connected())
      delete ubp;
  }

}


void BoundedDeltaBlock::flush_buffer() {

  // Padding at end of signal
  
  int win_size=DELTA_WIN_param->get_value();

  for (Integer i=0; i<win_size; ++i) {
    (*output_sockets)[0]->put_vector(construct_delta_vector(data_buffer));
    if ((*input_sockets)[1]->connected())
      (*output_sockets)[1]->put_vector(construct_delta_mask_vector(mask_buffer));
  }

  if ((*input_sockets)[2]->connected() && (*input_sockets)[3]->connected()) {
    CTKVector *delta_lbp, *delta_ubp;
    construct_delta_bounds(data_buffer, mask_buffer, lower_bound_buffer, upper_bound_buffer, delta_lbp, delta_ubp);
    for (Integer i=0; i<win_size-1; ++i) {
      (*output_sockets)[2]->put_vector(new CTKVector (*delta_lbp));
      (*output_sockets)[3]->put_vector(new CTKVector (*delta_ubp));
    }
    (*output_sockets)[2]->put_vector(delta_lbp);
    (*output_sockets)[3]->put_vector(delta_ubp);
  }
}

void BoundedDeltaBlock::close() {

  data_buffer.reset(0);
  mask_buffer.reset(0);
  lower_bound_buffer.reset(0);
  upper_bound_buffer.reset(0);

  Block::close();
}



///
///

void process_input_vector(Socket *input_socket,
			  Socket *output_socket,
			  FrameBuffer &buffer,
			  CTKVector* process(const FrameBuffer &) ) {
  CTKVector *xv0;
  
  input_socket->get_vector(xv0);

  buffer.insert(xv0);

  if (buffer.full()) {
    CTKVector *out_vector = process(buffer);
  
    if (buffer.just_full()) {
      int win_size=(buffer.size()-1)/2;
      for (Integer i=0; i<win_size; ++i) {
	output_socket->put_vector(new CTKVector (*out_vector));
      }
    }
    
    output_socket->put_vector(out_vector);
  }
  
}


CTKVector *construct_delta_vector(const FrameBuffer &buffer) {

  int width=buffer.width();
  int win_size=(buffer.size()-1)/2;

  float denom=calc_delta_denominator(win_size);
  
  CTKVector *new_vector= new CTKVector(width);
  
  for (Integer i=0; i< width; ++i) {
    (*new_vector)[i]=0.0;
    for (Integer j=1; j<=win_size; ++j) {
      (*new_vector)[i]+= j * (buffer[win_size+j][i] - buffer[win_size-j][i]);
    }
    (*new_vector)[i]*=denom;
  }
  return new_vector;
}

CTKVector *construct_delta_mask_vector(const FrameBuffer &buffer) {
  
  int width=buffer.width();
  int win_size=(buffer.size()-1)/2;
  float element;
  
  CTKVector *new_vector = new CTKVector(width);

  for (Integer i=0; i< width; ++i) {
    element=buffer[win_size-1][i];
    (*new_vector)[i]=element;
    for (Integer j=1; j<=win_size; ++j) {
      if ((buffer[win_size+j][i]!=element) || (buffer[win_size-j][i]!=element))
	(*new_vector)[i]=0;
    }
  }

  return new_vector;
}


void construct_delta_bounds(const FrameBuffer &data_buffer, const FrameBuffer &mask_buffer,
			    const FrameBuffer &lower_bound_buffer, const FrameBuffer &upper_bound_buffer,
			    CTKVector * &delta_lbp, CTKVector * &delta_ubp) {
  

  int width=data_buffer.width();
  int win_size=(data_buffer.size()-1)/2;
  
  delta_lbp = new CTKVector(width);
  delta_ubp = new CTKVector(width);

  float denom=calc_delta_denominator(win_size);

  for (Integer i=0; i< width; ++i) {
    (*delta_lbp)[i]=0.0;
    (*delta_ubp)[i]=0.0;
    for (Integer j=1; j<=win_size; ++j) {
      if (mask_buffer[win_size+j][i]) {
	(*delta_lbp)[i]+= j * data_buffer[win_size+j][i];
	(*delta_ubp)[i]+= j * data_buffer[win_size+j][i];
      } else {
	(*delta_lbp)[i]+= j * lower_bound_buffer[win_size+j][i];
	(*delta_ubp)[i]+= j * upper_bound_buffer[win_size+j][i];
      }

      if (mask_buffer[win_size-j][i]) {
	(*delta_lbp)[i]-= j * data_buffer[win_size+j][i];
	(*delta_ubp)[i]-= j * data_buffer[win_size+j][i];
      } else {
	(*delta_lbp)[i]-= j * upper_bound_buffer[win_size+j][i];
	(*delta_ubp)[i]-= j * lower_bound_buffer[win_size+j][i];
      }      
    }
    
    (*delta_lbp)[i]*=denom;
    (*delta_ubp)[i]*=denom;
  }
}

float calc_delta_denominator(int win_size) {
  float denom=0.0;
  for (Integer i=1; i<=win_size; ++i)
    denom+=(i*i);
  denom=0.5/denom;

  return denom;
}


/* End of ctk_differentiator.cpp */
 
