/******************************************************************************/
/*									      */
/*	ctk_file.cpp	    						      */
/*									      */
/*	Implementation for CTKFile class hierarchy			      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		        	      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <cmath>
#include <sys/types.h>
#include <sys/stat.h>

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"
#include "ctk_block.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_file.hh"


/******************************************************************************/

enum {IEEE_LE, IEEE_BE};

/* Return the byte order of the host machine */
int GetHostByteOrder()
{
   int x = 1;
   return *(char *)&x == 1? IEEE_LE: IEEE_BE;
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKFile						      */
/*									      */
/******************************************************************************/

static const char *VALID_BYTE_ORDERS[] = {"BE","LE","\0"};
const char *PARAM_DEFAULT_BYTE_ORDER       = "BE";

CTKFile::CTKFile(): frame_buffer_chan1(0), frame_buffer_chan2(0) {
  
  sample_index=0;
  swap_bytes=false;
  

  // Set up FILE_NAME parameter
  filename_param = new ParamString("FILE_NAME");
  filename_param->set_helptext("The path and filename of the file.");
  parameters->register_parameter(filename_param);

  // Set up FILE_PATH parameter
  file_path_param = new ParamString("FILE_PATH");
  file_path_param->set_helptext("An optional path that is prefixed to the filename");
  parameters->register_parameter(file_path_param);

  // Set up FILE_EXTENSION parameter
  file_extension_param = new ParamString("FILE_EXTENSION");
  file_extension_param->set_helptext("An optional extension that is added to the filename separated by a '.'.");
  parameters->register_parameter(file_extension_param);

  // Set up HEADER_SIZE parameter
  headersize_param = new ParamInt("HEADER_SIZE", 0);
  headersize_param->set_helptext("The size of the file header measured in bytes");
  headersize_param->set_unit("bytes");
  headersize_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(headersize_param, CTK_PARAM_TYPE_HIDDEN| CTK_PARAM_TYPE_CONST);

  // Set up BYTE_ORDER parameter
  byte_order_param= new ParamEnumerated("BYTE_ORDER", VALID_BYTE_ORDERS, PARAM_DEFAULT_BYTE_ORDER);
  byte_order_param->set_helptext("Specifies the byte order of the data in the file. Either BE (big endian) or LE (little endian)");
  parameters->register_parameter(byte_order_param, CTK_PARAM_TYPE_HIDDEN| CTK_PARAM_TYPE_CONST);
  
  // Set up NUMBER_OF_CHANNELS parameter
  num_channels_param = new ParamInt("NUMBER_OF_CHANNELS");
  num_channels_param->set_helptext("Number of channels: i.e. 1=mono-, 2=stereo-, 4=quadro-phonic.");
  num_channels_param->install_validator(new Validator(Validator::VLOWER, 1.0));
  parameters->register_parameter(num_channels_param, CTK_PARAM_TYPE_HIDDEN | CTK_PARAM_TYPE_CONST);

  // Set up NUMBER_OF_SAMPLES parameter
  num_samples_param = new ParamInt("NUMBER_OF_SAMPLES");
  num_samples_param->set_helptext("Number of samples the file contains.");
  num_samples_param->set_unit("samples");
  num_samples_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(num_samples_param, CTK_PARAM_TYPE_HIDDEN | CTK_PARAM_TYPE_CONST);

  // Set up BYTES_PER_SAMPLE parameter
  bytes_per_sample_param = new ParamInt("BYTES_PER_SAMPLE");
  bytes_per_sample_param->set_helptext("Number of bytes per sample");
  bytes_per_sample_param->set_unit("bytes");
  bytes_per_sample_param->install_validator(new Validator(1.0, 8.0));
  parameters->register_parameter(bytes_per_sample_param, CTK_PARAM_TYPE_HIDDEN | CTK_PARAM_TYPE_CONST);

  // Set up SAMPLES_PER_FRAME parameter
  samples_per_frame_param = new ParamInt("SAMPLES_PER_FRAME", 1);
  samples_per_frame_param->set_helptext("Number of samples composing a single frame. <p> e.g. For a 64 channel ratemap file this will be 64, for a simple audio file it will be 1");
  samples_per_frame_param->install_validator(new Validator(Validator::VLOWER, 1.0));
  parameters->register_parameter(samples_per_frame_param, CTK_PARAM_TYPE_HIDDEN | CTK_PARAM_TYPE_CONST);

  // Set up MULAW_ENCODING parameter
  mulaw_param = new ParamBool("MULAW_ENCODING", 0);
  mulaw_param->set_helptext("If set ON then Mu-law encoding is applied to the file.");
  parameters->register_parameter(mulaw_param, CTK_PARAM_TYPE_HIDDEN | CTK_PARAM_TYPE_CONST);

  // Set up FLOATING_POINT parameter
  floating_point_param = new ParamBool("FLOATING_POINT", 0);
  floating_point_param->set_helptext("If set ON the file is stored in floating point format, else it is stored in integer format.");
  parameters->register_parameter(floating_point_param, CTK_PARAM_TYPE_HIDDEN | CTK_PARAM_TYPE_CONST);
  
  // Set up CHECK_VALID parameter
  check_valid_param = new ParamBool("CHECK_VALID", 1);
  check_valid_param->set_helptext("If set ON then an error will be generated if a NaN is detected in the input data.");
  parameters->register_parameter(check_valid_param, CTK_PARAM_TYPE_HIDDEN | CTK_PARAM_TYPE_CONST);
  
  fd=NULL;
}



CTKFile::~CTKFile(){
  close_file();
};

void CTKFile::reset() {
  sample_index=0;
  swap_bytes=false;

  if (byte_order_param->get_set_flag()) {
    swap_bytes = (
		  ((byte_order_param->get_value()=="BE") && (GetHostByteOrder()==IEEE_LE)) ||
		  ((byte_order_param->get_value()=="LE") && (GetHostByteOrder()==IEEE_BE))
		  );
  }
  
}

string CTKFile::getfilename() const {
  string file_name;

  if (file_path_param->get_set_flag()) {
    file_name=file_path_param->get_value();
    if (file_name.size()!=0) file_name+=(string("/"));
  }

  file_name+=filename_param->get_value();
  
  if (file_extension_param->get_set_flag()) {
    string extension=file_extension_param->get_value();
    if (extension.size()!=0) (file_name+=(string(".")))+=extension;
  }

  return file_name;
}

void CTKFile::swap_2bytes(Integer16 &x) const {
  char cc, *c;
  c=(char*)&x;
  cc=c[0]; c[0]=c[1]; c[1]=cc;
}

void CTKFile::swap_4bytes(Integer32 &x) const {
  char cc, *c;
  c=(char*)&x;
  cc=c[0]; c[0]=c[3]; c[3]=cc;
  cc=c[1]; c[1]=c[2]; c[2]=cc;
}

void CTKFile::close_file() {
  if (fd!=NULL) {
    fclose(fd);
  }
  fd=NULL;
}

Boolean CTKFile::file_is_closed() {
  return (fd==NULL);
}


void CTKFile::allocate_sample_buffer(CTKVector &sample_buffer, Integer storage) {

  sample_buffer.clear();
  sample_buffer.resize(storage);
  
  return;
}  

void CTKFile::allocate_frame_buffer(vector<CTKVector*> &frame_buffer, Integer storage) {

  frame_buffer.clear();
  frame_buffer.resize(storage);
  Integer samples_per_frame=samples_per_frame_param->get_value();
  for (Integer i=0; i<storage; ++i) {
    frame_buffer[i]=new CTKVector(samples_per_frame);
  }
  
  return;
}  

/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKInputFile		      			      */
/*									      */
/******************************************************************************/


CTKInputFile::CTKInputFile(const string &a_name, const string &a_type):SourceBlock(a_name, a_type) {

  //  output_sockets made by children. i.e. may be mono or stereo 

  unlock_parameters();  
  set_parameter("DURATION", 0.0);
  relock_parameters();
}

CTKInputFile::~CTKInputFile() {

};

void CTKInputFile::reset(){
  SourceBlock::reset();
  CTKFile::reset();

}

void CTKInputFile::close(){

  close_file();

  SourceBlock::close();
}

void CTKInputFile::reopen_input_file() {

  string filename=getfilename();

  if (filename.size()==0) {
    if (fd!=stdin) {
      close_file();
      fd=stdin;
    }
  }  else {
    close_file();
    if ((fd=(FILE*)fopen(filename.c_str(),"rb"))==(FILE*)NULL) {
      throw(FileErrorIFOF(__FILE__,__LINE__, filename.c_str()));
    }
  }
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKOutputFile		      			      */
/*									      */
/******************************************************************************/


CTKOutputFile::CTKOutputFile(const string &a_name, const string &a_type): Block(a_name, a_type) {
  // Input socket made my children
  
  // Set up APPEND parameter
  append_param = new ParamBool("APPEND", 0);
  append_param->set_helptext("If true file will be opened in 'append' mode");
  parameters->register_parameter(append_param, CTK_PARAM_TYPE_HIDDEN);
}

CTKOutputFile::~CTKOutputFile() {
};

void CTKOutputFile::reset(){

  Block::reset();
  CTKFile::reset();

  if (append_param->get_value()) {
    reopen_output_file("ab");
  } else {
    reopen_output_file("wb");
    fseek(fd, headersize_param->get_value(), SEEK_SET);
  }
  
}


void CTKOutputFile::reopen_output_file(const char *mode) {

  string a_file_name=getfilename();

  if (a_file_name.size()==0) {
    if (fd!=stdout) {
      close_file();
      fd=stdout;
    }
  } else {
    close_file();
    if ((fd=(FILE*)fopen(a_file_name.c_str(),mode))==(FILE*)NULL) {
    // create directory path and try again
      create_directory_path(a_file_name);
      if ((fd=(FILE*)fopen(a_file_name.c_str(),mode))==(FILE*)NULL) {
	throw(FileErrorOFOF(__FILE__,__LINE__, a_file_name.c_str()));
      }
    }
  }
  
}

void CTKOutputFile::compute() {
  // Default do nothing
}

void CTKOutputFile::close() {
  Block::close();
}


void CTKOutputFile::create_directory_path(const string &pathname) {
  string path;
  char c;
  
  for (unsigned int i=0; i<pathname.size(); ++i) {
    c=pathname[i];
    path+=c;
    if (c=='/') {
      mkdir(path.c_str(), 00755);
    }
  }
}




/******************************************************************************/
/*									      */
/*	CLASS NAME: SampleInputFile		      			      */
/*									      */
/******************************************************************************/

SampleInputFile::SampleInputFile(const string &a_name, const string &a_type):CTKInputFile(a_name, a_type){
    make_output_sockets(CTK_FILE_DEFAULT_NUM_CHANNELS,   // Number of sockets
			true,	 	// Number of sockets is user configurable?
			1,	 	// Minimum number of sockets
			4);		// Maximum number of sockets
  
}

SampleInputFile::~SampleInputFile(){

};

void  SampleInputFile::compute(){

  if (sample_index>=num_samples_param->get_value()) {
    set_eod();
    return;
  }

  if (samples_per_frame_param->get_value()==1) {
    (*output_sockets)[0]->write_sample(sample_buffer_chan1, sample_index, CTK_BUFFER_CHUNK_SIZE);
    if (num_channels_param->get_value()==2) {
      (*output_sockets)[1]->write_sample(sample_buffer_chan2, sample_index, CTK_BUFFER_CHUNK_SIZE);
    }
  } else {
    (*output_sockets)[0]->write_vector(frame_buffer_chan1, sample_index, CTK_BUFFER_CHUNK_SIZE);
    if (num_channels_param->get_value()==2) {
      (*output_sockets)[1]->write_vector(frame_buffer_chan2, sample_index, CTK_BUFFER_CHUNK_SIZE);
    }
  }
  
  //  sample_index+=sample_buffer_chan1.size();
  sample_index+=CTK_BUFFER_CHUNK_SIZE;

};


void SampleInputFile::reset() {

  CTKInputFile::reset();

  if (floating_point_param->get_value() && bytes_per_sample_param->get_set_flag() && bytes_per_sample_param->get_value()!=4) {

    throw(FileErrorGFE(__FILE__,__LINE__, "Input file parameter inconsistency: Floating point input files must be 4 bytes per sample"));
  }
  
  if (read_file_header()!=CTK_SUCCESS) {
    string filename=getfilename();
    throw(FileErrorIFH(__FILE__,__LINE__, filename.c_str()));
  }

  
  sample_rate_param->get_value();  // This just makes sure the sample_rate has been set - an error will be thrown otherwise.
 
  output_channel_setup();
    
  read_sample_data();

}

void SampleInputFile::close() {
  unset_parameter("NUMBER_OF_SAMPLES");

  CTKInputFile::close();
}

void SampleInputFile::build_output_data_descriptors() {

  Integer samples_per_frame=samples_per_frame_param->get_value();
  for (SocketIt osocketp=output_sockets->begin(); osocketp!=output_sockets->end();  ++osocketp) {
    DataDescriptor *dd = new DataDescriptor();
    if (samples_per_frame>1) {
      CTKVector axis(samples_per_frame);
      for (Integer k=0; k<samples_per_frame; ++k) axis[k]=Float(k);
      dd->add_outer_dimension("Y axis", axis);   
    }
    (*osocketp)->set_data_descriptor(dd);
  }
    
}

CTKStatus SampleInputFile::read_file_header() {
  // Default behaviour is to do nothing
  return CTK_SUCCESS;
}

int SampleInputFile::get_num_frames() {
  // Return total number of frames that will be input

  int num_frames;
  
  if (num_samples_param->get_set_flag()) {
    num_frames=num_samples_param->get_value();
  } else {
    num_frames=count_frames_in_file();
    unlock_parameters();
    set_parameter("NUMBER_OF_SAMPLES", num_frames);
    relock_parameters();
  }

  // If the duration parameter is set and it >0 then limit the number of frames
  // to be less than or equal to the requested duration
  int duration_limit;

  if ((duration_limit=get_frames_remaining())>0) {
    if (duration_limit<num_frames)
      num_frames=duration_limit;
  }

  return num_frames;
}


void SampleInputFile::read_sample_data() {
  
  Integer i, j, k;
  Integer num_frames;
  Integer8 *raw_buffer;

  UInteger8 *uint8p;
  Integer8 *int8p;
  Integer16 *int16p;
  Integer32 *int32p;

  bool check_valid= check_valid_param->get_value();
  bool has_nans = false;  // Gets set to true if Nans are detected.
  
  Integer bytes_per_sample;

  // If floating point is set to true, then bytes per sample must be 4
  if (floating_point_param->get_set_flag() && floating_point_param->get_value())
    bytes_per_sample=4;
  else
    bytes_per_sample=bytes_per_sample_param->get_value();

  Integer num_channels=num_channels_param->get_value();

  if (num_channels!=1 && num_channels!=2) {
    cerr << "Can only process mono or stereo signals" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  Integer samples_per_frame=samples_per_frame_param->get_value();
  
  reopen_input_file();
  fseek(fd, headersize_param->get_value(), SEEK_SET);

  num_frames=get_num_frames();
  

  
  raw_buffer= new Integer8[num_frames*samples_per_frame*bytes_per_sample*num_channels];
  uint8p=(UInteger8*)raw_buffer;
  int8p=(Integer8*)raw_buffer;
  int16p=(Integer16*)raw_buffer;
  int32p=(Integer32*)raw_buffer;

  if (samples_per_frame_param->get_value()==1) {
    allocate_sample_buffer(sample_buffer_chan1, num_frames);
    if (num_channels==2)
      allocate_sample_buffer(sample_buffer_chan2, num_frames);
  } else {
    allocate_frame_buffer(frame_buffer_chan1, num_frames);
    if (num_channels==2) {
      allocate_frame_buffer(frame_buffer_chan2, num_frames);
    }
  }

  switch (bytes_per_sample) {
  case 1:
    fread(int8p, 1, num_frames*samples_per_frame*num_channels, fd);
    if (mulaw_param->get_value()) {
      if (samples_per_frame==1) {
	for (i=0, j=0; i<num_frames; ++i) {
	  sample_buffer_chan1[i]=(Float)(Integer16)(CTK_FILE_MULAW_LINEAR[uint8p[j++]]);
	  if (num_channels==2) sample_buffer_chan2[i]=(Float)(Integer16)(CTK_FILE_MULAW_LINEAR[uint8p[j++]]);
	}
      } else {
	for (i=0, j=0; i<num_frames; ++i) {
	  for (k=0; k<samples_per_frame; ++k) {
	    (*frame_buffer_chan1[i])[k]=(Float)(Integer16)(CTK_FILE_MULAW_LINEAR[uint8p[j++]]);
	    if (num_channels==2) (*frame_buffer_chan2[i])[k]=(Float)(Integer16)(CTK_FILE_MULAW_LINEAR[uint8p[j++]]);
	  }
	}
      }
    } else {
      if (samples_per_frame==1) {
	for (i=0, j=0; i<num_frames; ++i) {
	  sample_buffer_chan1[i]=(Float)(int8p[j++]);
	  if (num_channels==2) sample_buffer_chan2[i]=(Float)(int8p[j++]);
	}
      } else {
	for (i=0, j=0; i<num_frames; ++i) {
	  for (k=0; k<samples_per_frame; ++k) {
	    (*frame_buffer_chan1[i])[k]=(Float)(int8p[j++]);
	    if (num_channels==2) (*frame_buffer_chan2[i])[k]=(Float)(int8p[j++]);
	  }
	}
      }
    }
    break;
  case 2:
    fread(int16p, 2, num_frames*samples_per_frame*num_channels, fd);
    if (swap_bytes) 
      for (i=0; i<num_frames*samples_per_frame*num_channels; ++i) swap_2bytes(int16p[i]);
    if (samples_per_frame==1) {
      for (i=0, j=0; i<num_frames; ++i) {
	sample_buffer_chan1[i]=(Float)(int16p[j++]);
	if (num_channels==2) sample_buffer_chan2[i]=(Float)(int16p[j++]);
      }
    } else {
      for (i=0, j=0; i<num_frames; ++i) {
	for (k=0; k<samples_per_frame; ++k) {
	  (*frame_buffer_chan1[i])[k]=(Float)(int16p[j++]);
	  if (num_channels==2) (*frame_buffer_chan2[i])[k]=(Float)(int16p[j++]);
	}
      }
    }
    break;
  case 4:
    fread(int8p, 1, num_frames*samples_per_frame*bytes_per_sample*num_channels, fd);
    if (swap_bytes) 
      for (i=0; i<num_frames*samples_per_frame*num_channels; ++i) swap_4bytes(int32p[i]);
    if (samples_per_frame==1) {
      // 1 Sample per frame
      if (floating_point_param->get_value()) {
	for (i=0, j=0; i<num_frames; ++i) {
	  sample_buffer_chan1[i]=(Float)*(float*)(&int32p[j++]);
	  if (num_channels==2) sample_buffer_chan2[i]=(Float)*(float*)(&int32p[j++]);
	}
	if (check_valid) {
	  has_nans=check_vector(sample_buffer_chan1);
	  if (num_channels==2)
	    has_nans = has_nans || check_vector(sample_buffer_chan2);
	}
      } else {
	for (i=0, j=0; i<num_frames; ++i) {
	  sample_buffer_chan1[i]=(Float)(int32p[j++]);
	  if (num_channels==2) sample_buffer_chan2[i]=(Float)(int32p[j++]);
	}
      }
    } else {
      // >1 Sample per frame
      if (floating_point_param->get_value()) {
	for (i=0, j=0; i<num_frames; ++i) {
	  for (k=0; k<samples_per_frame; ++k) {
	    (*(frame_buffer_chan1[i]))[k]=(Float)*(float*)(&int32p[j++]);
	    if (num_channels==2) (*frame_buffer_chan2[i])[k]=(Float)*(float*)(&int32p[j++]);
	  }
	  if (check_valid) {
	    has_nans = has_nans || check_vector(*frame_buffer_chan1[i]);
	    if (num_channels==2)
	    has_nans = has_nans || check_vector(*frame_buffer_chan2[i]);
	  }
	}
      } else {
	for (i=0, j=0; i<num_frames; ++i) {
	  for (k=0; k<samples_per_frame; ++k) {    
	    (*frame_buffer_chan1[i])[k]=(Float)(int32p[j++]);
	    if (num_channels==2) (*frame_buffer_chan2[i])[k]=(Float)(int32p[j++]);
	  }
	}
      }
  
    }
    break;
  default:
    cerr << "Illegal bytes per sample value" << endl;
    throw(CTKError(__FILE__,__LINE__));
  }

  delete[]raw_buffer;  
     
  close_file(); 

  if (has_nans) {
    string name=getfilename();
    cerr << "NaNs have been detected in input data read from file ("<< name.c_str() << "). Halting execution." << endl;
    throw(CTKError(__FILE__,__LINE__));
  }
}


bool SampleInputFile::check_vector(const CTKVector &data) const {
  // Returns true if any element of data is NaN

  bool has_nans = false;

  for (CTKVector::const_iterator datap=data.begin(), datap_end=data.end(); datap!=datap_end; ++datap)
    has_nans=has_nans || isnan(*datap);

  return has_nans;
}


Integer SampleInputFile::count_frames_in_file() {

  Integer samples_read;
  Integer num_samples=0;
  UInteger bytes_per_sample;


  if (floating_point_param->get_value())
    bytes_per_sample=4;
  else
    bytes_per_sample=bytes_per_sample_param->get_value();
  
  Integer buffer[CTK_INPUT_FILE_CHUNK_SIZE];
  long old_position;

  old_position=ftell(fd);
  fseek(fd, headersize_param->get_value(), SEEK_SET);

  if (bytes_per_sample>sizeof(Integer32)) {
    cerr << "Invalid value for BYTES_PER_SAMPLE" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
  do {
    samples_read=fread(buffer, bytes_per_sample,  CTK_INPUT_FILE_CHUNK_SIZE,  fd);
    num_samples+=samples_read;
  } while (samples_read==CTK_INPUT_FILE_CHUNK_SIZE);
  
  fseek(fd, old_position, SEEK_SET);

  num_samples/=num_channels_param->get_value();
  num_samples/=samples_per_frame_param->get_value();
  return num_samples;
  
}



void SampleInputFile::output_channel_setup() {

  Integer nchans, nconnects;
  
  nconnects=output_sockets->count_contiguous_connected_sockets();

  if (num_channels_param->get_set_flag()) {
    if ((nchans=num_channels_param->get_value()) < nconnects) {
      string filename=getfilename();
      cerr << "Trying to supply " << nconnects << " outputs with a " << nchans<< " channel input file: " << filename.c_str() << endl;
      throw(CTKError(__FILE__,__LINE__));
    }
    set_num_outputs(nchans);
  } else {
    set_num_outputs(nconnects);
    unlock_parameters();
    set_parameter("NUMBER_OF_CHANNELS", nconnects);
    relock_parameters();
  }
    
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: AlienSampleInputFile	       			      */
/*									      */
/******************************************************************************/

const string AlienSampleInputFile::help_text = BINARY_INPUT_FILE_BLOCK_HELP_TEXT;
const string AlienSampleInputFile::type_name = "BinaryInputFile";

AlienSampleInputFile::AlienSampleInputFile(const string &a_name, const string &a_filename): CTKObject(a_name),SampleInputFile(a_name, type_name)  {
  set_parameter("FILE_NAME", a_filename);

  unset_parameter_hidden("NUMBER_OF_CHANNELS");
  unset_parameter_hidden("HEADER_SIZE");
  unset_parameter_hidden("BYTE_ORDER");
  unset_parameter_hidden("BYTES_PER_SAMPLE");
  unset_parameter_hidden("NUMBER_OF_SAMPLES");
  unset_parameter_hidden("MULAW_ENCODING");
  unset_parameter_hidden("SAMPLES_PER_FRAME");
  unset_parameter_hidden("FLOATING_POINT");
  unset_parameter_hidden("CHECK_VALID");

  unset_parameter_const("NUMBER_OF_CHANNELS");
  unset_parameter_const("HEADER_SIZE");
  unset_parameter_const("BYTE_ORDER");
  unset_parameter_const("BYTES_PER_SAMPLE");
  unset_parameter_const("MULAW_ENCODING");
  unset_parameter_const("SAMPLES_PER_FRAME");
  unset_parameter_const("FLOATING_POINT");
  unset_parameter_const("CHECK_VALID");

  unlock_parameters();
  set_parameter("BYTE_ORDER", "BE");
  relock_parameters();

}

AlienSampleInputFile::~AlienSampleInputFile(){

};

Block* AlienSampleInputFile::clone(const string &n) const{
  string filename=getfilename();
  Block *ablock= new AlienSampleInputFile(n.empty()?this->getname():n, filename.c_str());
  return copy_this_block_to(ablock);
}



/******************************************************************************/
/*									      */
/*	CLASS NAME: ASCIISampleInputFile	       			      */
/*									      */
/******************************************************************************/

const string ASCIISampleInputFile::help_text = ASCII_INPUT_FILE_BLOCK_HELP_TEXT;
const string ASCIISampleInputFile::type_name = "ASCIIInputFile";

ASCIISampleInputFile::ASCIISampleInputFile(const string &a_name, const string &a_filename):CTKObject(a_name),SampleInputFile(a_name, type_name) {

  set_parameter("FILE_NAME", a_filename);
  
  unset_parameter_hidden("SAMPLES_PER_FRAME");
  unset_parameter_const("SAMPLES_PER_FRAME");

  
}

ASCIISampleInputFile::~ASCIISampleInputFile(){};

Block* ASCIISampleInputFile::clone(const string &n) const{
  string filename=getfilename();
  Block *ablock= new ASCIISampleInputFile(n.empty()?this->getname():n, filename.c_str());
  return copy_this_block_to(ablock);
}


void ASCIISampleInputFile::read_sample_data() {
  Integer i, n;
  Integer samples_per_frame=samples_per_frame_param->get_value();
  
  Integer num_channels=num_channels_param->get_value();
  
  reopen_input_file();

  n=get_num_frames();
  
  if (samples_per_frame==1) {
    allocate_sample_buffer(sample_buffer_chan1, n);
    if (num_channels==2) allocate_sample_buffer(sample_buffer_chan2, n);
    for (i=0; i<n; ++i) {
      fscanf(fd,"%lf",&sample_buffer_chan1[i]);
      if (num_channels==2) fscanf(fd,"%lf",&sample_buffer_chan2[i]);
    }
  } else {
    allocate_frame_buffer(frame_buffer_chan1, n);
    if (num_channels==2) {
      allocate_frame_buffer(frame_buffer_chan2, n);
    }
    for (i=0; i<n; ++i) {
      for (int j=0;j<samples_per_frame; ++j) {
	fscanf(fd,"%lf",  &(*frame_buffer_chan1[i])[j]);
	if (num_channels==2) fscanf(fd,"%lf",  &(*frame_buffer_chan2[i])[j]);
      }
    }
  }
    
  close_file();
  
}


Integer ASCIISampleInputFile::count_frames_in_file() {

  Integer num_samples=0;
  Float this_sample;
  
  long old_position;

  old_position=ftell(fd);
  rewind(fd);
  
  while (fscanf(fd,"%lf\n",&this_sample)==1) {
    ++num_samples;
  }
  
  fseek(fd, old_position, SEEK_SET);
  
  num_samples=num_samples/num_channels_param->get_value();
  num_samples=num_samples/samples_per_frame_param->get_value();
  
  return num_samples;
  
}

/******************************************************************************/
/*									      */
/*	CLASS NAME: AUSampleInputFile      	      			      */
/*									      */
/******************************************************************************/

const string AUSampleInputFile::help_text = AU_SAMPLE_INPUT_FILE_BLOCK_HELP_TEXT;
const string AUSampleInputFile::type_name = "AUInputFile";

AUSampleInputFile::AUSampleInputFile(const string &a_name, const string &a_filename):CTKObject(a_name),SampleInputFile(a_name, type_name)  {

  set_parameter("FILE_NAME", a_filename);

  unset_parameter_hidden("NUMBER_OF_SAMPLES");
  unset_parameter_hidden("SAMPLE_RATE");
  unset_parameter_hidden("NUMBER_OF_CHANNELS");
  unset_parameter_hidden("BYTES_PER_SAMPLE");
  set_parameter_const("SAMPLE_RATE");

}


AUSampleInputFile::~AUSampleInputFile(){};

Block* AUSampleInputFile::clone(const string &n) const{
  string filename=getfilename();
  Block *ablock= new AUSampleInputFile(n.empty()?this->getname():n, filename.c_str());
  return copy_this_block_to(ablock);
}


CTKStatus AUSampleInputFile::read_file_header() {

  CTKStatus error_status=CTK_SUCCESS;
  Integer32 magic, data_location, data_size, data_format, sample_rate, num_channels;
  
  reopen_input_file();
 

  if (
      (fread(&magic, sizeof(Integer32), 1, fd)!=1) ||
      (fread(&data_location, sizeof(Integer32), 1, fd)!=1) ||
      (fread(&data_size, sizeof(Integer32), 1, fd)!=1) || 
      (fread(&data_format, sizeof(Integer32), 1, fd)!=1) ||
      (fread(&sample_rate, sizeof(Integer32), 1, fd)!=1) ||
      (fread(&num_channels, sizeof(Integer32), 1, fd)!=1)
      )
    error_status=CTK_FAILURE;
  
  // the four byte 'info' field is not read.

  if (error_status==CTK_SUCCESS) {
    unlock_parameters();
    
    char *x=(char*)&magic;
    
    if (x[3]=='.' && x[2]=='s' && x[1]=='n' && x[0]=='d') {
      swap_bytes=true;
      swap_4bytes(data_location);
      swap_4bytes(data_size);
      swap_4bytes(data_format);
      swap_4bytes(sample_rate);
      swap_4bytes(num_channels);
    }
    else if (x[3]=='d' && x[2]=='n' && x[1]=='s' && x[0]=='.')
      swap_bytes=false;
    else {
      error_status=CTK_FAILURE;
    }

    if (num_channels!=1) error_status=CTK_FAILURE; 
    
    set_parameter("HEADER_SIZE", data_location);
    set_parameter("SAMPLE_RATE", sample_rate);
    set_parameter("NUMBER_OF_CHANNELS", num_channels);
    set_parameter("NUMBER_OF_SAMPLES", data_size);
    
    switch (data_format) {
    case CTK_FILE_AU_FORMAT_MULAW:
      set_parameter("BYTES_PER_SAMPLE", 1);
      set_parameter("MULAW_ENCODING", "true");
      break;
    case CTK_FILE_AU_FORMAT_16BIT:  
      set_parameter("BYTES_PER_SAMPLE", 2);
      break;
    default:
      error_status=CTK_FAILURE;
      break;
    }
  
    relock_parameters();
  }
  
  //  if (data_format!=CTK_FILE_AU_FORMAT_16BIT) error_status=1;

  close_file();

  return error_status;
  
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: HTKSampleInputFile      	      			      */
/*									      */
/******************************************************************************/

const string HTKSampleInputFile::type_name = "HTKInputFile";
const string HTKSampleInputFile::help_text = HTK_SAMPLE_INPUT_FILE_BLOCK_HELP_TEXT;

HTKSampleInputFile::HTKSampleInputFile(const string &a_name, const string &a_filename):CTKObject(a_name),SampleInputFile(a_name, type_name)  {

  set_parameter("FILE_NAME", a_filename);

  unset_parameter_hidden("NUMBER_OF_SAMPLES");
  unset_parameter_hidden("SAMPLE_RATE");
  unset_parameter_hidden("NUMBER_OF_CHANNELS");
  unset_parameter_hidden("BYTES_PER_SAMPLE");
  set_parameter_const("SAMPLE_RATE");

  
}

HTKSampleInputFile::~HTKSampleInputFile(){};

Block* HTKSampleInputFile::clone(const string &n) const{
  string filename=getfilename();
  Block *ablock= new HTKSampleInputFile(n.empty()?this->getname():n, filename.c_str());
  return copy_this_block_to(ablock);
}

CTKStatus HTKSampleInputFile::read_file_header() {

  CTKStatus error_status=CTK_SUCCESS;
  Boolean floating_point;
  Integer32  num_samples, sample_period;
  Integer16  bytes_per_sample, bytes_per_frame, parameter_kind, samples_per_frame;
  
  reopen_input_file();
  
  if ((fread(&num_samples, sizeof(Integer32), 1, fd)!=1) ||
      (fread(&sample_period, sizeof(Integer32), 1, fd)!=1) ||
      (fread(&bytes_per_frame, sizeof(Integer16), 1, fd)!=1) ||
      (fread(&parameter_kind, sizeof(Integer16), 1, fd)!=1))
    error_status=CTK_FAILURE;
  
  if (error_status==CTK_SUCCESS) {

    unlock_parameters();
    
    if (GetHostByteOrder()==IEEE_LE) {
      swap_4bytes(num_samples);
      swap_4bytes(sample_period);
      swap_2bytes(bytes_per_frame);
      swap_2bytes(parameter_kind);
      swap_bytes=true;
    }
    
    if (parameter_kind!=CTK_FILE_HTK_WAVEFORM_PARAMETER &&
	parameter_kind!=CTK_FILE_HTK_USER_DEFINED_PARAMETER) error_status=CTK_FAILURE;

    if (parameter_kind==CTK_FILE_HTK_USER_DEFINED_PARAMETER) {
      bytes_per_sample=4;
      floating_point=1;
    } else {
      bytes_per_sample=2;
      floating_point=0;
    }
    samples_per_frame=bytes_per_frame/bytes_per_sample;
    
    set_parameter("HEADER_SIZE", CTK_FILE_HTK_HEADER_SIZE);
    set_parameter("SAMPLE_RATE", 1.0/(sample_period*CTK_FILE_HTK_SAMPLE_PERIOD_UNIT));
    set_parameter("NUMBER_OF_CHANNELS", 1);
    set_parameter("NUMBER_OF_SAMPLES", num_samples);
    set_parameter("BYTES_PER_SAMPLE", bytes_per_sample);
    set_parameter("SAMPLES_PER_FRAME", samples_per_frame);
    set_parameter("FLOATING_POINT", floating_point);
    
    relock_parameters();
  }
  
  close_file();

  return error_status;
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: TIMITSampleInputFile      	      			      */
/*									      */
/******************************************************************************/

const string TIMITSampleInputFile::type_name = "TIMITInputFile";
const string TIMITSampleInputFile::help_text = TIMIT_SAMPLE_INPUT_FILE_BLOCK_HELP_TEXT;

TIMITSampleInputFile::TIMITSampleInputFile(const string &a_name, const string &a_filename):CTKObject(a_name),SampleInputFile(a_name, type_name)  {
  set_parameter("FILE_NAME", a_filename);

  // Set up TIMIT version parameter
  version_param = new ParamInt("TIMIT_VERSION");
  version_param->set_helptext("The TIMIT version number recorded in the TIMIT file header.");
  parameters->register_parameter(version_param, CTK_PARAM_TYPE_CONST);

  unset_parameter_hidden("NUMBER_OF_SAMPLES");
  unset_parameter_hidden("SAMPLE_RATE");
  unset_parameter_hidden("NUMBER_OF_CHANNELS");
  unset_parameter_hidden("BYTES_PER_SAMPLE");
  set_parameter_const("SAMPLE_RATE");


 
}

TIMITSampleInputFile::~TIMITSampleInputFile(){};

Block* TIMITSampleInputFile::clone(const string &n) const{
  string filename=getfilename();
  Block *ablock= new TIMITSampleInputFile(n.empty()?this->getname():n, filename.c_str());
  return copy_this_block_to(ablock);
}


CTKStatus TIMITSampleInputFile::read_file_header() {

  CTKStatus error_status=CTK_SUCCESS;
  Integer16 header_size, version, num_channels, sample_rate;
  Integer32 num_samples;
  
  reopen_input_file();

  if (
      (fread(&header_size, sizeof(Integer16), 1, fd)!=1) ||
      (fread(&version, sizeof(Integer16), 1, fd)!=1) ||
      (fread(&num_channels, sizeof(Integer16), 1, fd)!=1) ||
      (fread(&sample_rate, sizeof(Integer16), 1, fd)!=1) ||
      (fread(&num_samples, sizeof(Integer32), 1, fd)!=1)
      )
    error_status=CTK_FAILURE;
  
  if (error_status==CTK_SUCCESS) {
    unlock_parameters();
    
    if (GetHostByteOrder()==IEEE_BE) {
      swap_2bytes(header_size);
      swap_2bytes(version);
      swap_2bytes(num_channels);
      swap_2bytes(sample_rate);
      swap_4bytes(num_samples);
      swap_bytes=true;
    }
    
    set_parameter("HEADER_SIZE", header_size);
    set_parameter("TIMIT_VERSION", version);
    set_parameter("NUMBER_OF_CHANNELS", 1);
    set_parameter("SAMPLE_RATE", sample_rate);
    set_parameter("NUMBER_OF_SAMPLES", num_samples);
    set_parameter("BYTES_PER_SAMPLE", 2);
    
    relock_parameters();
  }
  
  close_file();

  return error_status;
};

/******************************************************************************/
/*									      */
/*	CLASS NAME: NISTSampleInputFile      	      			      */
/*									      */
/******************************************************************************/

const string NISTSampleInputFile::type_name = "NISTInputFile";
const string NISTSampleInputFile::help_text = NIST_SAMPLE_INPUT_FILE_BLOCK_HELP_TEXT;

NISTSampleInputFile::NISTSampleInputFile(const string &a_name, const string &a_filename):CTKObject(a_name),SampleInputFile(a_name, type_name)  {
  set_parameter("FILE_NAME", a_filename);

  unset_parameter_hidden("CHECK_VALID");
  unset_parameter_const("CHECK_VALID");

  unset_parameter_hidden("HEADER_SIZE");
  unset_parameter_hidden("BYTE_ORDER");
  unset_parameter_hidden("NUMBER_OF_SAMPLES");
  unset_parameter_hidden("SAMPLE_RATE");
  unset_parameter_hidden("NUMBER_OF_CHANNELS");
  unset_parameter_hidden("BYTES_PER_SAMPLE");
  set_parameter_const("SAMPLE_RATE");

}

NISTSampleInputFile::~NISTSampleInputFile(){};

Block* NISTSampleInputFile::clone(const string &n) const{
  string filename=getfilename();
  Block *ablock= new NISTSampleInputFile(n.empty()?this->getname():n, filename.c_str());
  return copy_this_block_to(ablock);
}

CTKStatus NISTSampleInputFile::read_file_header() {

  Integer header_size, num_channels, sample_rate, bytes_per_sample;
  Integer num_samples, byte_order;

  Boolean has_num_channels=false,
          has_sample_rate=false,
          has_bytes_per_sample=false,
          has_num_samples=false,
          has_byte_order=false;
  //          has_sample_coding=false,
  //          has_interleaved=false;
  
  char buffer[CTK_CHAR_BUFFER_SIZE+1] ;
  char word[CTK_CHAR_BUFFER_SIZE+1];
  char format_string[CTK_CHAR_BUFFER_SIZE+1];
  
  CTKStatus error_status = CTK_SUCCESS;
  Boolean status=false;

  reopen_input_file();
  
  sprintf(format_string,"%%%ds %%d\n",CTK_CHAR_BUFFER_SIZE);
  if (fscanf(fd, format_string , word, &header_size)!=2) error_status=CTK_FAILURE;
  
  if (strncmp(word,"NIST",4)!=0) error_status=CTK_FAILURE;

  if (error_status==CTK_SUCCESS) {
    
    do {
      if (fgets(buffer,  CTK_CHAR_BUFFER_SIZE, fd)==NULL) break;
      if (sscanf(buffer, "%s", word)!=1) break;
      status=false;
      if (strcmp(word,"channel_count")==0) {
	status=read_NIST_integer_param(buffer, num_channels);
	has_num_channels=true;
      } else if (strcmp(word,"sample_rate")==0) {
	status=read_NIST_integer_param(buffer, sample_rate);
	has_sample_rate=true;
      } else if (strcmp(word,"sample_n_bytes") == 0) {
	status=read_NIST_integer_param(buffer, bytes_per_sample);
	has_bytes_per_sample=true;
      } else if (strcmp(word,"sample_count") == 0) {
	status=read_NIST_integer_param(buffer, num_samples);
	has_num_samples=true;
      } else if (strcmp(word,"sample_byte_format") == 0) {
   	status=read_NIST_byte_order_line(buffer, byte_order);
	has_byte_order=true;
      } else if (strcmp(word,"sample_coding") == 0) {
    	status=read_NIST_sample_coding_line(buffer);
	//	has_sample_coding=true;
      } else if (strcmp(word,"channels_interleaved") == 0) {
    	status=read_NIST_channels_interleaved_line(buffer);
	//	has_interleaved=true;
      }
      if (status) error_status=CTK_FAILURE;
    } while (strcmp(word,"end_head")!=0);
    
    
    if (error_status==CTK_SUCCESS) {
      
      // If no byte order was specified then we'll take LE as default
      if (!has_byte_order)
	byte_order=IEEE_LE;
      
      swap_bytes = (GetHostByteOrder()!=byte_order);
      unlock_parameters();
      
      set_parameter("HEADER_SIZE", header_size);
      set_parameter("BYTE_ORDER", (byte_order==IEEE_LE)?"LE":"BE");
      if (has_num_channels) set_parameter("NUMBER_OF_CHANNELS", num_channels);
      if (has_sample_rate) set_parameter("SAMPLE_RATE", sample_rate);
      if (has_bytes_per_sample) set_parameter("BYTES_PER_SAMPLE", bytes_per_sample);
      if (has_num_samples) set_parameter("NUMBER_OF_SAMPLES", num_samples);
      
      relock_parameters();
    }
    
  }
  
  close_file();

  return error_status;

}



Boolean NISTSampleInputFile::read_NIST_integer_param(const char *buffer, Integer &param) {
  return (sscanf(buffer,"%*s -i %d",&param)!=1);
}

char *NISTSampleInputFile::read_NIST_string_param(const char *buffer) {
  Integer length;
  char *string_param;
  
  if (sscanf(buffer,"%*s -s%d", &length)!=1) return NULL;

  string_param=new char[length+1];
  
  if (sscanf(buffer, "%*s %*s %s", string_param)!=1) {
    delete[] string_param;
    return NULL;
  }

  return string_param;
}

Boolean NISTSampleInputFile::read_NIST_byte_order_line(const char *buffer, int &byte_order) {
  Boolean error_flag=false;
  char *string_param;

  byte_order=IEEE_LE;
  
  string_param=read_NIST_string_param(buffer);
  if (string_param==NULL) return true;
  
  if (strcmp(string_param,"10")==0)
    byte_order=IEEE_BE;
  else if (strcmp(string_param,"01")==0)
    byte_order=IEEE_LE;
  else error_flag=true;
  
  delete[] string_param;
  
  return error_flag;
}

Boolean NISTSampleInputFile::read_NIST_sample_coding_line(const char *buffer) {
  Boolean error_flag=false;
  char *string_param;

  string_param=read_NIST_string_param(buffer);
  if (string_param==NULL) return true;

  //
  //  Only pcm files are implemented so far.
  //
  
  if (strcmp(string_param,"pcm")!=0)
    error_flag=true;

  delete[] string_param;

  return error_flag;
}

Boolean NISTSampleInputFile::read_NIST_channels_interleaved_line(const char *buffer) {
  Boolean error_flag=false;
  char *string_param;

  
  string_param=read_NIST_string_param(buffer);
  if (string_param==NULL) return true;

  //
  //  NOT IMPLEMENTED YET #####################
  //
  delete[] string_param;

  return error_flag;
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: RESPITEInputFile      	      		      */
/*									      */
/******************************************************************************/

const string RESPITEInputFile::help_text = RESPITE_INPUT_FILE_BLOCK_HELP_TEXT;
const string RESPITEInputFile::type_name = "RESPITEInputFile";

RESPITEInputFile::RESPITEInputFile(const string &a_name, const string &a_filename):CTKObject(a_name),CTKInputFile(a_name, type_name)  {

  make_output_sockets(CTK_FILE_DEFAULT_NUM_CHANNELS,   // Number of sockets
		      true,	 	// Number of sockets is user configurable?
		      1,	 	// Minimum number of sockets
		      8);		// Maximum number of sockets
  
  set_parameter("FILE_NAME", a_filename);

  unset_parameter_hidden("NUMBER_OF_SAMPLES");
  unset_parameter_hidden("SAMPLE_RATE");
  unset_parameter_hidden("NUMBER_OF_CHANNELS");
  unset_parameter_hidden("BYTES_PER_SAMPLE");
  set_parameter_const("SAMPLE_RATE");

}


RESPITEInputFile::~RESPITEInputFile(){};

Block* RESPITEInputFile::clone(const string &n) const{
  string filename=getfilename();
  Block *ablock= new RESPITEInputFile(n.empty()?this->getname():n, filename.c_str());
  return copy_this_block_to(ablock);
}


void RESPITEInputFile::reset() {

  CTKInputFile::reset();

  if (read_file_header()!=CTK_SUCCESS) {
    string filename=getfilename();
    throw(FileErrorIFH(__FILE__,__LINE__, filename.c_str()));
  }

  output_channel_setup();

}

void RESPITEInputFile::output_channel_setup() {

  Integer nchans, nconnects;
  
  nconnects=output_sockets->count_contiguous_connected_sockets();

  if (num_channels_param->get_set_flag()) {
    if ((nchans=num_channels_param->get_value()) < nconnects) {
      string filename=getfilename();
      cerr << "Trying to supply " << nconnects << " outputs with a " << nchans<< " channel input file: " << filename.c_str() << endl;
      throw(CTKError(__FILE__,__LINE__));
    }
    set_num_outputs(nchans);
  } else {
    set_num_outputs(nconnects);
    unlock_parameters();
    set_parameter("NUMBER_OF_CHANNELS", nconnects);
    relock_parameters();
  }
    
}

void RESPITEInputFile::build_output_data_descriptors() {

  int stream=0;
  for (SocketIt osocketp=output_sockets->begin(); osocketp!=output_sockets->end();  ++osocketp) {
    int samples_per_frame= (stream_types[stream]==CTK_FILE_RESPITE_TYPE_SAMPLE?1:frame_sizes[stream]);
    DataDescriptor *dd = new DataDescriptor();
    if (samples_per_frame>1) {
      CTKVector axis(samples_per_frame);
      for (Integer k=0; k<samples_per_frame; ++k) axis[k]=Float(k);
      dd->add_outer_dimension("Y axis", axis);   
    }
    (*osocketp)->set_data_descriptor(dd);

    ++stream;
  }
    
}


CTKStatus RESPITEInputFile::read_file_header() {

  UInteger32 sample_rate;
  float frame_length;
  float frame_shift;
  
  stream_names.resize(0);
  frame_sizes.resize(0);
  stream_types.resize(0);

  CTKStatus error_status=CTK_SUCCESS;
  reopen_input_file();
 

  if (
      (fread(&sample_rate, sizeof(UInteger32), 1, fd)!=1) ||
      (fread(&frame_length, sizeof(float), 1, fd)!=1) ||
      (fread(&frame_shift, sizeof(float), 1, fd)!=1) || 
      (fread(&num_streams, sizeof(UInteger32), 1, fd)!=1)
      )
    error_status=CTK_FAILURE;

  for (UInteger32 stream=0; stream<num_streams; ++stream) {
    char stream_name[CTK_FILE_RESPITE_STREAM_NAME_LENGTH];
    UInteger32 stream_type, frame_size;
    if (
	(fread(stream_name, CTK_FILE_RESPITE_STREAM_NAME_LENGTH, 1, fd)!=1) ||
	(fread(&stream_type, sizeof(UInteger32), 1, fd)!=1) ||
	(fread(&frame_size, sizeof(UInteger32), 1, fd)!=1)
	)
      error_status=CTK_FAILURE;
    stream_names.push_back(string(stream_name));
    stream_types.push_back(stream_type);
    frame_sizes.push_back(frame_size);
  }
  
  unlock_parameters();
  set_parameter("SAMPLE_RATE", sample_rate);
  relock_parameters();

  return error_status;
}

void  RESPITEInputFile::compute(){

  // read a single frame of data
  
  // read time stamp
  float time_stamp;
  
  fread(&time_stamp, sizeof(float), 1, fd);

  if (time_stamp==-1.0) {
    set_eod();
    return;
  }
  
  // read each of streams and send data to sockets

  SocketIt osocketp=output_sockets->begin();
  for (UInteger32 stream=0; stream<num_streams; ++stream) {
    if (osocketp!=output_sockets->end()) {
      int frame_size=frame_sizes[stream];
      if (stream_types[stream]==CTK_FILE_RESPITE_TYPE_SAMPLE) {
	if (frame_size==1) {
	  Integer16 sample;
	  fread(&sample, sizeof(Integer16), 1, fd);
	  (*osocketp)->put_sample((float)sample);
	} else {
	  // For the case where the sample data has been squashed into a single frame
	  Integer16 *buffer= new Integer16[frame_size];
	  fread(buffer, sizeof(Integer16), frame_size, fd);
	  CTKVector *framep = new CTKVector; 
	  copy(&buffer[0], &buffer[frame_size], back_inserter(*framep));
	  
	  (*osocketp)->write_sample(*framep, 0, frame_size);
	  delete[] buffer;
	  delete framep;
	}
      } else {
	// Read and send frame
	float *buffer= new float[frame_size];
	fread(buffer, sizeof(float), frame_size, fd);
	CTKVector *framep = new CTKVector;
	// Copy read data into a CTKVector, implicitly casting from float to Float (probably double) 
	copy(&buffer[0], &buffer[frame_size], back_inserter(*framep));
	(*osocketp)->put_vector(framep);
	delete[] buffer;
      }
      ++osocketp;
    }
  }

}


/******************************************************************************/
/*									      */
/*	CLASS NAME: SampleOutputFile	       	      			      */
/*									      */
/******************************************************************************/



SampleOutputFile::SampleOutputFile(const string &a_name, const string &a_type):CTKOutputFile(a_name, a_type) {

  output_buffer=NULL;

  make_input_sockets(1);
  make_output_sockets(1);
  
  allocate_sample_buffer(sample_buffer_chan1, CTK_OUTPUT_FILE_CHUNK_SIZE);
  
  allocate_frame_buffer(frame_buffer_chan1, CTK_OUTPUT_FILE_CHUNK_SIZE);
}


SampleOutputFile::~SampleOutputFile() {
  if (output_buffer!=NULL) delete[] output_buffer;
}


void SampleOutputFile::reset(){
  CTKOutputFile::reset();

  if (floating_point_param->get_value() && bytes_per_sample_param->get_set_flag() && bytes_per_sample_param->get_value()!=4) {
    throw(FileErrorGFE(__FILE__,__LINE__, "Input file parameter inconsistency: Floating point input files must be 4 bytes per sample"));
  }

  const DataDescriptor *dd=(*input_sockets)[0]->get_data_descriptor();

  samples_per_frame=dd->get_storage();

  // If floating point is set to true, then bytes per sample must be 4
  if (floating_point_param->get_set_flag() && floating_point_param->get_value())
    bytes_per_sample=4;
  else
    bytes_per_sample=bytes_per_sample_param->get_value();
   
  unlock_parameters();
  set_parameter("NUMBER_OF_SAMPLES", 0.0);
  set_parameter("NUMBER_OF_CHANNELS", 1.0);
  relock_parameters();

  sample_index=0;
  samples_written_to_file=0;

  if (!get_append_mode()) {
    reopen_output_file("wb");
    write_file_header();
    close_file();
  }
  
  allocate_output_buffer();
}

void SampleOutputFile::compute(){

  if (file_is_closed()) reopen_output_file("ab");

  if (inputs_are_all_sample_data) {
    (*input_sockets)[0]->get_sample(sample_buffer_chan1[sample_index]);
    if ((*output_sockets)[0]->connected())
      (*output_sockets)[0]->put_sample(sample_buffer_chan1[sample_index]);
    ++sample_index;
	
    if (sample_index==CTK_OUTPUT_FILE_CHUNK_SIZE) write_sample_buffer_to_file();
    
  } else {
    (*input_sockets)[0]->get_vector(frame_buffer_chan1[sample_index]);
    if ((*output_sockets)[0]->connected())
      (*output_sockets)[0]->put_vector(new CTKVector(*frame_buffer_chan1[sample_index]));
    ++sample_index;
    
    if (sample_index==CTK_OUTPUT_FILE_CHUNK_SIZE) write_frame_buffer_to_file();
  }
    
    
}

void SampleOutputFile::close() {
 
  if (file_is_closed())
    reopen_output_file("ab");
  
  if (inputs_are_all_sample_data) {
    sample_index+=(*input_sockets)[0]->flush_sample(sample_buffer_chan1);
    write_sample_buffer_to_file(); // flush internal buffer
  } else {
    sample_index+=(*input_sockets)[0]->flush_vector(frame_buffer_chan1);
    write_frame_buffer_to_file(); // flush internal buffer
  }
    

  unlock_parameters();
  set_parameter("NUMBER_OF_SAMPLES", samples_written_to_file);
  relock_parameters();
  
  reopen_output_file("rb+");
  update_header();   // correct header - i.e 'number of samples' entry
  close_file();

  CTKOutputFile::close();
}

void SampleOutputFile::write_sample_buffer_to_file() {
  Integer i;
  
  if (sample_index==0) return; // buffer already empty

  if (floating_point_param->get_value()) {  // 4 byte floating point output
    
    float *obp=(float*)&output_buffer[0];
    for (i=0; i<sample_index; ++i)
      *obp++=(float)sample_buffer_chan1[i];
    
    if (swap_bytes) {
      Integer8 *x=NULL;
      Integer8 saved;
      obp=(float*)&output_buffer[0];
      for (i=0; i<sample_index; ++i) {
	x=(char*)obp;
        ++obp;
	saved=*(x); *(x)=*(x+3); *(x+3)=saved;
	saved=*(x+1); *(x+1)=*(x+2); *(x+2)=saved;
      }
    }
    
    fwrite(output_buffer, sizeof(float), sample_index, fd);

  } else if (bytes_per_sample==2) {  // 2 byte integer output
    
    Integer16 *obp=(Integer16*)&output_buffer[0];
    for (i=0; i<sample_index; ++i)
      *obp++=(Integer16)sample_buffer_chan1[i];
    
    if (swap_bytes) {
      obp=(Integer16*)&output_buffer[0];
      for (i=0; i<sample_index; ++i)
	swap_2bytes(*obp++);
    }
    
    fwrite(output_buffer, sizeof(Integer16), sample_index, fd);

  } else if (bytes_per_sample==4) { // 4 byte integer output
    
    Integer32 *obp=(Integer32*)&output_buffer[0];
    for (i=0; i<sample_index; ++i)
      *obp++=(Integer32)sample_buffer_chan1[i];
    
    if (swap_bytes) {
      obp=(Integer32*)&output_buffer[0];
      for (i=0; i<sample_index; ++i)
	swap_4bytes(*obp++);
    }
    
    fwrite(output_buffer, sizeof(Integer32), sample_index, fd);

  } else if (bytes_per_sample==1) { // 1 byte integer output

    Integer8 *obp=(Integer8*)&output_buffer[0];
    for (i=0; i<sample_index; ++i)
      *obp++=(Integer8)sample_buffer_chan1[i];
    
    fwrite(output_buffer, sizeof(Integer8), sample_index, fd);
  }

  samples_written_to_file+=sample_index;

  sample_index=0;
}

void SampleOutputFile::write_frame_buffer_to_file(){

  CTKVector **fip=&frame_buffer_chan1[0];
  Integer i;

  if (sample_index==0) return; // buffer already empty
  
  if (floating_point_param->get_value()) {   // 4 byte floating point output
    
    float *obp=(float*)&output_buffer[0];
    for (i=0; i<sample_index; ++i, ++fip) {
      for (Float *pp=&(**fip)[0]; pp!=&(**fip)[samples_per_frame]; ++pp) {
	*obp++=(float)(*pp);
      }
      delete *fip; 
    }
    
    if (swap_bytes) {
      Integer8 *x=NULL;
      Integer8 saved;
      obp=(float*)&output_buffer[0];
      for (i=0; i<samples_per_frame*sample_index; ++i) {
	x=(char*)obp;
        ++obp;
	saved=*(x); *(x)=*(x+3); *(x+3)=saved;
	saved=*(x+1); *(x+1)=*(x+2); *(x+2)=saved;
      }
     }
  
    fwrite(output_buffer, sizeof(float), samples_per_frame*sample_index, fd); 

  } else if (bytes_per_sample==2) {  // 2 byte integer output
    Integer16 *obp=(Integer16*)&output_buffer[0];
    for (i=0; i<sample_index; ++i, ++fip) {
      for (Float *pp=&(**fip)[0]; pp!=&(**fip)[samples_per_frame]; ++pp) {
	*obp++=(Integer16)(*pp);
      }
      delete *fip; 
    }
    
    if (swap_bytes) {
      obp=(Integer16*)&output_buffer[0];
      for (i=0; i<samples_per_frame*sample_index; ++i)
	swap_2bytes(*obp++);
    }
  
    fwrite(output_buffer, sizeof(Integer16), samples_per_frame*sample_index, fd);
    
  } else if (bytes_per_sample==4) {  // 4 byte integer output
    Integer32 *obp=(Integer32*)&output_buffer[0];
    for (i=0; i<sample_index; ++i, ++fip) {
      for (Float *pp=&(**fip)[0]; pp!=&(**fip)[samples_per_frame]; ++pp) {
	*obp++=(Integer32)(*pp);
      }
      delete *fip; 
    }
    
    if (swap_bytes) {
      obp=(Integer32*)&output_buffer[0];
      for (i=0; i<samples_per_frame*sample_index; ++i)
	swap_4bytes(*obp++);
    }
  
    fwrite(output_buffer, sizeof(Integer32), samples_per_frame*sample_index, fd);
    
  } else if (bytes_per_sample==1) {  // 1 byte integer output
    
    Integer8 *obp=(Integer8*)&output_buffer[0];
    for (i=0; i<sample_index; ++i, ++fip) {
      for (Float *pp=&(**fip)[0]; pp!=&(**fip)[samples_per_frame]; ++pp) {
	*obp=(Integer8)(*pp);
      }
      delete *fip; 
    }

    fwrite(output_buffer, sizeof(Integer8), samples_per_frame*sample_index, fd);


  }
  
  samples_written_to_file+=sample_index;

  sample_index=0;
}


void SampleOutputFile::allocate_output_buffer() {

  
  if (output_buffer!=NULL) {
    delete[] output_buffer;
    output_buffer=NULL;
  }
  
  output_buffer = new Integer8[CTK_OUTPUT_FILE_CHUNK_SIZE*samples_per_frame*bytes_per_sample];
  
  return;
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: ASCIISampleOutputFile                                     */
/*									      */
/******************************************************************************/

const string ASCIISampleOutputFile::type_name = "ASCIIOutputFile";
const string ASCIISampleOutputFile::help_text = ASCII_OUTPUT_FILE_BLOCK_HELP_TEXT;

ASCIISampleOutputFile::ASCIISampleOutputFile(const string &a_name, const string &a_filename):CTKObject(a_name), SampleOutputFile(a_name, type_name )  {
  set_parameter("FILE_NAME",a_filename);

  unlock_parameters();
  set_parameter("HEADER_SIZE", 0.0); // Not used - but needs a value
  set_parameter("BYTES_PER_SAMPLE", 0.0); // Not used - but needs a value
  unset_parameter_hidden("APPEND");
  relock_parameters();
}

ASCIISampleOutputFile::~ASCIISampleOutputFile() {};

Block* ASCIISampleOutputFile::clone(const string &n) const{
  string filename=getfilename();
  Block *ablock= new ASCIISampleOutputFile(n.empty()?this->getname():n, filename.c_str());
  return copy_this_block_to(ablock);
}



void ASCIISampleOutputFile::write_sample_buffer_to_file(){
  
  Integer i;

  for (i=0; i<sample_index; ++i) 
    fprintf(fd,"%5.8f\n",sample_buffer_chan1[i]);

  sample_index=0;

};

void ASCIISampleOutputFile::write_frame_buffer_to_file(){
  CTKVector **fip=&frame_buffer_chan1[0];
  Integer N=(**fip).size();
  for (Integer i=0; i<sample_index; ++i, ++fip) {
    for (Float *pp=&(**fip)[0]; pp!=&(**fip)[N]; ++pp) {
      fprintf(fd,"%5.8f ", *pp);
    }
    fprintf(fd,"\n");
    delete *fip; // delete frame
  }
  
  sample_index=0;
};


void ASCIISampleOutputFile::write_file_header() {
  // do nothing
}

void ASCIISampleOutputFile::update_header() {
  // do nothing
}


/******************************************************************************/
/*									      */
/*	CLASS NAME: AlienSampleOutputFile                                     */
/*									      */
/******************************************************************************/

const string AlienSampleOutputFile::type_name = "BinaryOutputFile";
const string AlienSampleOutputFile::help_text = BINARY_OUTPUT_FILE_BLOCK_HELP_TEXT;

AlienSampleOutputFile::AlienSampleOutputFile(const string &a_name, const string &a_filename):CTKObject(a_name), SampleOutputFile(a_name, type_name)  {
  set_parameter("FILE_NAME",a_filename);

  unset_parameter_hidden("SAMPLE_RATE");

  unset_parameter_hidden("BYTE_ORDER");
  unset_parameter_hidden("BYTES_PER_SAMPLE");
  unset_parameter_hidden("FLOATING_POINT");
  
  unset_parameter_const("BYTE_ORDER");
  unset_parameter_const("BYTES_PER_SAMPLE");
  unset_parameter_const("FLOATING_POINT");
  
  unset_parameter_hidden("APPEND");

  unlock_parameters();
  set_parameter("BYTE_ORDER", "BE");
  relock_parameters();

} 


AlienSampleOutputFile::~AlienSampleOutputFile() {};

Block* AlienSampleOutputFile::clone(const string &n) const{
  string filename=getfilename();
  Block *ablock= new AlienSampleOutputFile(n.empty()?this->getname():n, filename.c_str());
  return copy_this_block_to(ablock);
}

void AlienSampleOutputFile::write_file_header() {
  // do nothing
}

void AlienSampleOutputFile::update_header() {
  // do nothing
}



/******************************************************************************/
/*									      */
/*	CLASS NAME: AUSampleOutputFile                                       */
/*									      */
/******************************************************************************/

const string AUSampleOutputFile::type_name = "AUOutputFile";
const string AUSampleOutputFile::help_text = AU_SAMPLE_OUTPUT_FILE_BLOCK_HELP_TEXT;

AUSampleOutputFile::AUSampleOutputFile(const string &a_name, const string &a_filename):CTKObject(a_name), SampleOutputFile(a_name, type_name)  {

  set_parameter("FILE_NAME",a_filename);

  unset_parameter_hidden("SAMPLE_RATE");
  
  unlock_parameters();
  set_parameter("HEADER_SIZE", CTK_FILE_AU_HEADER_SIZE);
  set_parameter("BYTES_PER_SAMPLE", 2);
  relock_parameters();

}


AUSampleOutputFile::~AUSampleOutputFile() {};

Block* AUSampleOutputFile::clone(const string &n) const{
  string filename=getfilename();
  Block *ablock= new AUSampleOutputFile(n.empty()?this->getname():n, filename.c_str());
  return copy_this_block_to(ablock);
}

void AUSampleOutputFile::write_file_header() {

  Integer32  magic_number;
  Integer32  header_size, num_samples;
  Integer32  data_format, sample_rate;
  Integer32  num_channels;
  Integer32 info=0;

  char *mn=(char*)&magic_number;
  mn[0]='.'; mn[1]='s'; mn[2]='n'; mn[3]='d';
  
  header_size=headersize_param->get_value();
  num_samples=num_samples_param->get_value();
  data_format=CTK_FILE_AU_FORMAT_16BIT;
  sample_rate=(Integer32)sample_rate_param->get_value();
  num_channels=num_channels_param->get_value();
    
  if (GetHostByteOrder()==IEEE_BE) {
    swap_4bytes(magic_number);
    swap_4bytes(header_size);
    swap_4bytes(num_samples);
    swap_4bytes(data_format);
    swap_4bytes(sample_rate);
    swap_4bytes(num_channels);
    swap_bytes=true;
  }

  reopen_output_file("wb");

  fwrite(&magic_number, sizeof(Integer32), 1, fd);
  fwrite(&header_size, sizeof(Integer32), 1, fd);
  fwrite(&num_samples, sizeof(Integer32), 1, fd);
  fwrite(&data_format, sizeof(Integer32), 1, fd);
  fwrite(&sample_rate, sizeof(Integer32), 1, fd);
  fwrite(&num_channels, sizeof(Integer32), 1, fd);
  fwrite(&info, sizeof(Integer32), 1, fd);

  close_file();
};


void AUSampleOutputFile::update_header() {

  Integer32 num_samples;

  fseek(fd, CTK_FILE_AU_NUM_SAMPLES_HEADER_POS, SEEK_SET);

  num_samples=num_samples_param->get_value();

  if (swap_bytes)
    swap_4bytes(num_samples);

  fwrite(&num_samples, sizeof(Integer32), 1, fd);

}



/******************************************************************************/
/*									      */
/*	CLASS NAME: RESPITEOutputFile                                       */
/*									      */
/******************************************************************************/

const char *RespiteFile::CTK_FILE_RESPITE_DATA_TYPES[]={"SAMPLE", "FEATURE", "PROBABILITY", "\0"};
const char *RespiteFile::CTK_FILE_RESPITE_DEFAULT_DATA_TYPE        = "FEATURE";

const string RESPITEOutputFile::type_name = "RESPITEOutputFile";
const string RESPITEOutputFile::help_text = RESPITE_OUTPUT_FILE_BLOCK_HELP_TEXT;

RESPITEOutputFile::RESPITEOutputFile(const string &a_name, const string &a_filename):CTKObject(a_name), CTKOutputFile(a_name, type_name)  {


  make_input_sockets(CTK_FILE_DEFAULT_NUM_CHANNELS,   // Number of sockets
		      true,	 	// Number of sockets is user configurable?
		      1,	 	// Minimum number of sockets
		      8);		// Maximum number of sockets

 
  // Set up RESPITE stream name parameter
  name_param = new ParamStringVector("STREAM_NAME");
  name_param->set_helptext("Names of stream to be stored in file header.");
  parameters->register_parameter(name_param);

  // Set up data type parameter
  data_type_param= new ParamEnumerated("DATA_TYPE", RespiteFile::CTK_FILE_RESPITE_DATA_TYPES, RespiteFile::CTK_FILE_RESPITE_DEFAULT_DATA_TYPE);
  data_type_param->set_helptext("Specifies whether the data is sample, feature or probability data");
  parameters->register_parameter(data_type_param);

  set_parameter("FILE_NAME",a_filename);
  
  unset_parameter_hidden("SAMPLE_RATE");
  
  unlock_parameters();
  set_parameter("BYTES_PER_SAMPLE", 2);
  relock_parameters();

}

RESPITEOutputFile::~RESPITEOutputFile() {};

Block* RESPITEOutputFile::clone(const string &n) const{
  string filename=getfilename();
  Block *ablock= new RESPITEOutputFile(n.empty()?this->getname():n, filename.c_str());
  return copy_this_block_to(ablock);
}

void RESPITEOutputFile::reset(){
  CTKOutputFile::reset();

  samples_per_frame=(*input_sockets)[0]->get_data_descriptor()->get_storage();

  if (data_type_param->get_value()==string("SAMPLE"))
    stream_type=CTK_FILE_RESPITE_TYPE_SAMPLE;
  else if (data_type_param->get_value()==string("FEATURE"))
    stream_type=CTK_FILE_RESPITE_TYPE_FEATURE;
  else  if (data_type_param->get_value()==string("PROBABILITY"))
    stream_type=CTK_FILE_RESPITE_TYPE_PROBABILITY;
      
  time_stamp=0.0;
  frame_period=1.0/sample_rate_param->get_value();

  write_file_header();
}

void RESPITEOutputFile::write_file_header() {

  UInteger32 sample_rate=(UInteger32)sample_rate_param->get_value();

  float frame_length=1000.0/sample_rate;  // Frame length in ms
  float frame_shift=frame_length;  // Frame shift in ms
  
  UInteger32  num_streams=input_sockets->size();
	
  //  if (GetHostByteOrder()==IEEE_BE) {
  //    swap_4bytes(sample_rate);
  //    swap_bytes=true;
  //  }

  reopen_output_file("wb");

  fwrite(&sample_rate, sizeof(UInteger32), 1, fd);
  fwrite(&frame_length, sizeof(float), 1, fd);
  fwrite(&frame_shift, sizeof(float), 1, fd);
  fwrite(&num_streams, sizeof(UInteger32), 1, fd);

  for (unsigned int i=0; i<num_streams; ++i) {
    string stream_name=name_param->get_value(i);
    stream_name.resize(CTK_FILE_RESPITE_STREAM_NAME_LENGTH);
    fwrite(stream_name.c_str(), 1, CTK_FILE_RESPITE_STREAM_NAME_LENGTH, fd);
    fwrite(&stream_type, sizeof(UInteger32), 1, fd);
    UInteger32  frame_size = (*input_sockets)[i]->get_data_descriptor()->get_storage();
    fwrite(&frame_size, sizeof(UInteger32), 1, fd);
  }
}

void RESPITEOutputFile::update_header() {
  // do nothing
}



void RESPITEOutputFile::compute(){

  // Write time stamp

  time_stamp+=frame_period;
  
  fwrite(&time_stamp, sizeof(float), 1, fd);
  
  for (SocketIt isocketp=input_sockets->begin(), isocketp_end=input_sockets->end(); isocketp!=isocketp_end;  ++isocketp) {
    CTKVector *frame;
    (*isocketp)->get_vector(frame);
    
    // Cast from Float (probably double) to float and write frame to RESPITE format file
    if (stream_type==CTK_FILE_RESPITE_TYPE_SAMPLE) {
      Integer16 sample;
      for (CTKVector::iterator framep=frame->begin(), framep_end=frame->end(); framep!=framep_end; ++framep) {
	sample=(Integer16)*framep;
	fwrite(&sample, sizeof(Integer16), 1, fd);
      }
    } else {
      vector<float> fframe;
      copy(frame->begin(), frame->end(), back_inserter(fframe));
      for (vector<float>::iterator framep=fframe.begin(), framep_end=fframe.end(); framep!=framep_end; ++framep) 
	fwrite(&(*framep), sizeof(float), 1, fd);
    }

    delete frame;
  }
       
}



void RESPITEOutputFile::close() {

  fwrite(&CTK_FILE_RESPITE_END_MARKER, sizeof(float), 1, fd);

  CTKOutputFile::close();
}




/******************************************************************************/
/*									      */
/*	CLASS NAME: HTKSampleOutputFile                                       */
/*									      */
/******************************************************************************/

const string HTKSampleOutputFile::type_name = "HTKOutputFile";
const string HTKSampleOutputFile::help_text = HTK_SAMPLE_OUTPUT_FILE_BLOCK_HELP_TEXT;

HTKSampleOutputFile::HTKSampleOutputFile(const string &a_name, const string &a_filename):CTKObject(a_name), SampleOutputFile(a_name, type_name)   {

  set_parameter("FILE_NAME",a_filename);

  unset_parameter_hidden("SAMPLE_RATE");

  unlock_parameters();
  set_parameter("HEADER_SIZE", CTK_FILE_HTK_HEADER_SIZE);
  set_parameter("BYTES_PER_SAMPLE", 2);
  relock_parameters();
  
}

HTKSampleOutputFile::~HTKSampleOutputFile() {};


void HTKSampleOutputFile::reset() {

  if (inputs_are_all_sample_data) {
    unlock_parameters();
    set_parameter("BYTES_PER_SAMPLE", 2);
    set_parameter("FLOATING_POINT", 0);
    relock_parameters();
  } else {
    unlock_parameters();
    set_parameter("BYTES_PER_SAMPLE", 4);
    set_parameter("FLOATING_POINT", 1);
    relock_parameters();
  }

  SampleOutputFile::reset();
  
}

Block* HTKSampleOutputFile::clone(const string &n) const{
  string filename=getfilename();
  Block *ablock= new HTKSampleOutputFile(n.empty()?this->getname():n, filename.c_str());
  return copy_this_block_to(ablock);
}

void HTKSampleOutputFile::write_file_header() {

  //  Boolean error_status=false;
  Integer32  num_samples, sample_period;
  Integer16  how_many_bytes_per_frame, parameter_kind;
  int samples_per_frame = (*input_sockets)[0]->get_data_descriptor()->get_storage();
  num_samples=headersize_param->get_value();

  how_many_bytes_per_frame=bytes_per_sample_param->get_value()*samples_per_frame;
  sample_period=(Integer32)(1.0/(sample_rate_param->get_value()*CTK_FILE_HTK_SAMPLE_PERIOD_UNIT));

  if (samples_per_frame==1)
    parameter_kind=CTK_FILE_HTK_WAVEFORM_PARAMETER;
  else
    parameter_kind=CTK_FILE_HTK_USER_DEFINED_PARAMETER;
  
  if (GetHostByteOrder()==IEEE_LE) {
    swap_4bytes(num_samples);
    swap_4bytes(sample_period);
    swap_2bytes(how_many_bytes_per_frame);
    swap_2bytes(parameter_kind);

    swap_bytes=true;
  }

  reopen_output_file("wb");
  
  fwrite(&num_samples, sizeof(Integer32), 1, fd);
  fwrite(&sample_period, sizeof(Integer32), 1, fd);
  fwrite(&how_many_bytes_per_frame, sizeof(Integer16), 1, fd);
  fwrite(&parameter_kind, sizeof(Integer16), 1, fd);

  close_file();
};

void HTKSampleOutputFile::update_header() {
  
  Integer32 num_samples;
  
  fseek(fd, CTK_FILE_HTK_NUM_SAMPLES_HEADER_POS, SEEK_SET);

  
  num_samples=num_samples_param->get_value();
  
  if (swap_bytes)
    swap_4bytes(num_samples);

  fwrite(&num_samples, sizeof(Integer32), 1, fd);
}


/* End of ctk_file.cpp */
