/******************************************************************************/
/*									      */
/*	ctk_file.hh	    						      */
/*									      */
/*	Class declarations for ctk_file.cpp				      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007	           		      */
/*									      */
/******************************************************************************/

#ifndef CTK_FILE_HH
#define CTK_FILE_HH

#include <cstdio>

#include "ctk_block.hh"

class ParamString;
class ParamInt;
class ParamBool;
class ParamEnumerated;

/******************************************************************************/
/*									      */
/*	Local Type Definitions						      */
/*									      */
/******************************************************************************/


const Integer CTK_FILE_DEFAULT_NUM_CHANNELS               = 1;

const Integer CTK_FILE_TIMIT_HEADER_SIZE 		= 12;

const Integer CTK_FILE_AU_NUM_SAMPLES_HEADER_POS 	= 8;
const Integer CTK_FILE_AU_FORMAT_MULAW 			= 1;  
const Integer CTK_FILE_AU_FORMAT_16BIT 			= 3;  
const Integer CTK_FILE_AU_HEADER_SIZE 			= 28;

const Integer CTK_FILE_RESPITE_STREAM_NAME_LENGTH       = 32;
const UInteger32 CTK_FILE_RESPITE_TYPE_SAMPLE           = 1;
const UInteger32 CTK_FILE_RESPITE_TYPE_FEATURE          = 2;
const UInteger32 CTK_FILE_RESPITE_TYPE_PROBABILITY      = 3;
const float      CTK_FILE_RESPITE_END_MARKER            = -1.0;  // timestamp signifying end of data


const Integer CTK_FILE_HTK_NUM_SAMPLES_HEADER_POS 	= 0;
const Integer CTK_FILE_HTK_HEADER_SIZE 			= 12;
const Integer CTK_FILE_HTK_WAVEFORM_PARAMETER 		= 0;
const Integer CTK_FILE_HTK_USER_DEFINED_PARAMETER 		= 9;
const Float   CTK_FILE_HTK_SAMPLE_PERIOD_UNIT 		= 1e-7;

const Integer CTK_INPUT_FILE_CHUNK_SIZE 			= 1024;
const Integer CTK_OUTPUT_FILE_CHUNK_SIZE 			= 1024;

const Integer CTK_BUFFER_CHUNK_SIZE 			= 1024;


const Integer DEFAULT_ASCII_INPUT_NUM_CHANNELS 		= 1;
const Integer DEFAULT_ALIEN_INPUT_NUM_CHANNELS 		= 1;


/******************************************************************************/
/*									      */
/*	Table for mulaw to U16bit-linear conversion	      		      */
/*									      */
/******************************************************************************/

const UInteger16 CTK_FILE_MULAW_LINEAR[] = {
    33280, 34308, 35336, 36364, 37393, 38421, 39449, 40477,
    41505, 42534, 43562, 44590, 45618, 46647, 47675, 48703,
    49474, 49988, 50503, 51017, 51531, 52045, 52559, 53073,
    53587, 54101, 54616, 55130, 55644, 56158, 56672, 57186,
    57572, 57829, 58086, 58343, 58600, 58857, 59114, 59371,
    59628, 59885, 60142, 60399, 60656, 60913, 61171, 61428,
    61620, 61749, 61877, 62006, 62134, 62263, 62392, 62520,
    62649, 62777, 62906, 63034, 63163, 63291, 63420, 63548,
    63645, 63709, 63773, 63838, 63902, 63966, 64030, 64095,
    64159, 64223, 64287, 64352, 64416, 64480, 64544, 64609,
    64657, 64689, 64721, 64753, 64785, 64818, 64850, 64882,
    64914, 64946, 64978, 65010, 65042, 65075, 65107, 65139,
    65163, 65179, 65195, 65211, 65227, 65243, 65259, 65275,
    65291, 65308, 65324, 65340, 65356, 65372, 65388, 65404,
    65416, 65424, 65432, 65440, 65448, 65456, 65464, 65472,
    65480, 65488, 65496, 65504, 65512, 65520, 65528,     0,
    32256, 31228, 30200, 29172, 28143, 27115, 26087, 25059,
    24031, 23002, 21974, 20946, 19918, 18889, 17861, 16833,
    16062, 15548, 15033, 14519, 14005, 13491, 12977, 12463,
    11949, 11435, 10920, 10406,  9892,  9378,  8864,  8350,
     7964,  7707,  7450,  7193,  6936,  6679,  6422,  6165, 
     5908,  5651,  5394,  5137,  4880,  4623,  4365,  4108, 
     3916,  3787,  3659,  3530,  3402,  3273,  3144,  3016, 
     2887,  2759,  2630,  2502,  2373,  2245,  2116,  1988, 
     1891,  1827,  1763,  1698,  1634,  1570,  1506,  1441, 
     1377,  1313,  1249,  1184,  1120,  1056,   992,   927, 
      879,   847,   815,   783,   751,   718,   686,   654,
      622,   590,   558,   526,   494,   461,   429,   397,
      373,   357,   341,   325,   309,   293,   277,   261,
      245,   228,   212,   196,   180,   164,   148,   132,
      120,   112,   104,    96,    88,    80,    72,    64,
       56,    48,    40,    32,    24,    16,    8,      0
    };


/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKFile						      */
/*									      */
/******************************************************************************/

class CTKFile: public virtual CTKObject {

private:

  ParamString *filename_param;
  ParamString *file_path_param;
  ParamString *file_extension_param;

protected:
  
  FILE *fd;
  
  ParamInt *headersize_param;

  ParamEnumerated *byte_order_param;

  ParamInt *num_channels_param;
  ParamInt *num_samples_param;
  ParamInt *bytes_per_sample_param;
  ParamInt *samples_per_frame_param;
  ParamBool *mulaw_param;
  ParamBool *floating_point_param;
  ParamBool *check_valid_param;
  
  CTKVector sample_buffer_chan1;
  CTKVector sample_buffer_chan2;

  vector<CTKVector *> frame_buffer_chan1;
  vector<CTKVector *> frame_buffer_chan2;
  
  Integer sample_index;

  Boolean swap_bytes;

  //
  //
  //
  
public:

  CTKFile();
  
  virtual  ~CTKFile();
  
  virtual Block *clone(const string &n) const=0;
  
  virtual void compute()=0;

  void reset();

protected:
  
  string getfilename() const;
    
  inline void swap_2bytes(Integer16 &x) const;
  inline void swap_4bytes(Integer32 &x) const;

  void close_file();
  Boolean file_is_closed();
  
protected:
  virtual void allocate_sample_buffer(CTKVector &sample_buffer, Integer storage);
  virtual void allocate_frame_buffer(vector<CTKVector*> &frame_buffer, Integer storage);
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKInputFile		      			      */
/*									      */
/******************************************************************************/
class CTKInputFile:  public virtual CTKFile, public SourceBlock{

private:

  
public:
  
  CTKInputFile(const string &a_name, const string &a_type);
  virtual ~CTKInputFile();
  
  virtual Block *clone(const string &n) const =0;

  virtual void compute()=0;
  virtual void reset();   
  virtual void close();

  void reopen_input_file();
};

/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKOutputFile		      			      */
/*									      */
/******************************************************************************/

class CTKOutputFile:  public virtual CTKFile, public Block{

private:

  ParamBool *append_param;
   
public:
  
  CTKOutputFile(const string &a_name, const string &a_type);
  virtual ~CTKOutputFile();
 
  virtual Block *clone(const string &n) const=0;

protected:

  virtual void compute();
  virtual void reset();   
  virtual void close();
  
  void reopen_output_file(const char *mode);

  bool get_append_mode(){return append_param->get_value();}

private:

  void create_directory_path(const string &path);
  
};



//
//
//

/******************************************************************************/
/*									      */
/*	CLASS NAME: SampleInputFile  		      			      */
/*									      */
/******************************************************************************/

class SampleInputFile: public virtual CTKObject, public virtual CTKFile,  public CTKInputFile {

private:
  
  
protected:
  
  //  Boolean data_has_been_read;

  //
  //
  //
  
public:

  SampleInputFile(const string &a_name, const string &a_type);
  virtual ~SampleInputFile();

  virtual Block *clone(const string &n) const =0;

protected:
  
  virtual void compute();  
  virtual void reset();
  virtual void close();

  virtual void build_output_data_descriptors();
  virtual CTKStatus read_file_header();
  virtual int get_num_frames();
  virtual void read_sample_data();
  
private:

  virtual void output_channel_setup();
  virtual bool check_vector(const CTKVector &data) const;
  virtual Integer count_frames_in_file(); 
};



/******************************************************************************/
/*									      */
/*	CLASS NAME: AlienSampleInputFile      	      			      */
/*									      */
/******************************************************************************/


class AlienSampleInputFile:  public virtual CTKObject, public virtual CTKFile,  public SampleInputFile {

  static const string help_text;
  static const string type_name;

public:
  
  AlienSampleInputFile(const string &a_name, const string &file_name=string());
  virtual ~AlienSampleInputFile();
  
  virtual Block *clone(const string &n) const;

  virtual const string &get_helptext() const {return help_text;}
};




/******************************************************************************/
/*									      */
/*	CLASS NAME: ASCIISampleInputFile      	      			      */
/*									      */
/******************************************************************************/


class ASCIISampleInputFile:  public virtual CTKObject, public virtual CTKFile, public SampleInputFile {

  static const string help_text;
  static const string type_name;

public:

  ASCIISampleInputFile(const string &a_name, const string &file_name=string());
  virtual ~ASCIISampleInputFile();
  
  virtual Block *clone(const string &n) const;

  virtual const string &get_helptext() const {return help_text;}

protected:
  
  virtual void read_sample_data();

private:

  virtual Integer count_frames_in_file(); 

};


/******************************************************************************/
/*									      */
/*	CLASS NAME: AUSampleInputFile      	      			      */
/*									      */
/******************************************************************************/


class AUSampleInputFile: public virtual CTKObject, public virtual CTKFile, public SampleInputFile {

  static const string help_text;
  static const string type_name;

public:

  AUSampleInputFile(const string &a_name, const string &file_name=string());
  virtual ~AUSampleInputFile();

  virtual Block *clone(const string &n) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  CTKStatus read_file_header();
  
};

/******************************************************************************/
/*									      */
/*	CLASS NAME: RESPITEInputFile            	      	     	      */
/*									      */
/******************************************************************************/

class RespiteFile {
public:
  static const char *CTK_FILE_RESPITE_DATA_TYPES[];
  static const char *CTK_FILE_RESPITE_DEFAULT_DATA_TYPE;
};

class RESPITEInputFile: public virtual CTKObject, public virtual CTKFile, public CTKInputFile {

  static const string help_text;
  static const string type_name;

  UInteger32 num_streams;
  vector<string> stream_names;
  vector<UInteger32> frame_sizes;
  vector<UInteger32> stream_types;

public:

  RESPITEInputFile(const string &a_name, const string &file_name=string());
  virtual ~RESPITEInputFile();

  virtual Block *clone(const string &n) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void compute();
  virtual void reset();   

  void output_channel_setup();
  void build_output_data_descriptors();
  
  CTKStatus read_file_header();
  
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: HTKSampleInputFile      	      			      */
/*									      */
/******************************************************************************/


class HTKSampleInputFile: public virtual CTKObject, public virtual CTKFile, public SampleInputFile {

  static const string help_text;
  static const string type_name;
  
public:

  HTKSampleInputFile(const string &a_name, const string &file_name=string());
  virtual ~HTKSampleInputFile();
  
  virtual Block *clone(const string &n) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  CTKStatus read_file_header();
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: TIMITSampleInputFile      	      			      */
/*									      */
/******************************************************************************/

class TIMITSampleInputFile: public virtual CTKObject, public virtual CTKFile, public SampleInputFile {

  static const string help_text;
  static const string type_name;

  ParamInt *version_param;

public:

  TIMITSampleInputFile(const string &a_name, const string &file_name=string());
  virtual ~TIMITSampleInputFile();

  virtual Block *clone(const string &n) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  CTKStatus read_file_header();
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: NISTSampleInputFile      	      			      */
/*									      */
/******************************************************************************/


class NISTSampleInputFile: public virtual CTKObject, public virtual CTKFile, public SampleInputFile {

  static const string help_text;
  static const string type_name;

public:

  NISTSampleInputFile(const string &a_name, const string &file_name=string());
  virtual ~NISTSampleInputFile();
  
  virtual Block *clone(const string &n) const;

  virtual const string &get_helptext() const {return help_text;}

private:
  
  CTKStatus read_file_header();
  
  Boolean read_NIST_integer_param(const char *buffer, Integer &param);
  char *read_NIST_string_param(const char *buffer);
  
  Boolean read_NIST_byte_order_line(const char *buffer, Boolean &swap_bytes);
  Boolean read_NIST_sample_coding_line(const char *buffer);
  Boolean read_NIST_channels_interleaved_line(const char *buffer);


};

/******************************************************************************/
/*									      */
/*	CLASS NAME: SampleOutputFile  		      			      */
/*									      */
/******************************************************************************/

class SampleOutputFile: public virtual CTKObject, public virtual CTKFile, public CTKOutputFile {

private:

  Integer samples_written_to_file;

  Integer samples_per_frame;
  Integer bytes_per_sample;
  
  Integer8 *output_buffer;

  //
  //
  //
  
public:

  SampleOutputFile(const string &a_name, const string &a_type);
  virtual ~SampleOutputFile();

  virtual Block *clone(const string &n) const =0;
  virtual void reset();
  virtual void compute();
  virtual void close();

protected:
 
  virtual void write_file_header()=0;
  virtual void write_sample_buffer_to_file();
  virtual void write_frame_buffer_to_file();
  virtual void update_header()=0;
  
  virtual void allocate_output_buffer();

};



/******************************************************************************/
/*									      */
/*	CLASS NAME: ASCIISampleOutputFile  		       		      */
/*									      */
/******************************************************************************/

class ASCIISampleOutputFile: public virtual CTKObject, public virtual CTKFile, public SampleOutputFile {

  static const string help_text;
  static const string type_name;

public:

  ASCIISampleOutputFile(const string &a_name, const string &file_name=string());
  virtual ~ASCIISampleOutputFile();

  virtual Block *clone(const string &n) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void write_file_header();
  virtual void write_sample_buffer_to_file();
  virtual void write_frame_buffer_to_file();
  virtual void update_header();
};

/******************************************************************************/
/*									      */
/*	CLASS NAME: AlienSampleOutputFile  		       		      */
/*									      */
/******************************************************************************/

class AlienSampleOutputFile: public virtual CTKObject, public virtual CTKFile, public SampleOutputFile {

  static const string help_text;
  static const string type_name;

public:

  AlienSampleOutputFile(const string &a_name, const string &file_name=string());
  virtual ~AlienSampleOutputFile();
  
  virtual Block *clone(const string &n) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void write_file_header();
  virtual void update_header();
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: AUSampleOutputFile  		       		      */
/*									      */
/******************************************************************************/

class AUSampleOutputFile: public virtual CTKObject, public virtual CTKFile, public SampleOutputFile {

  static const string help_text;
  static const string type_name;

public:

  AUSampleOutputFile(const string &a_name, const string &file_name=string());
  virtual ~AUSampleOutputFile();
  
  virtual Block *clone(const string &n) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void write_file_header();
  virtual void update_header();
};




/******************************************************************************/
/*									      */
/*	CLASS NAME: RESPITEOutputFile  		       		      */
/*									      */
/******************************************************************************/

class RESPITEOutputFile: public virtual CTKObject, public virtual CTKFile, public CTKOutputFile {

  static const string help_text;
  static const string type_name;

  ParamStringVector *name_param;
  ParamEnumerated *data_type_param;

  float time_stamp;     // Time stamp for current frame
  float frame_period;   // time stamp increment per frame
  Integer samples_per_frame;

  UInteger32 stream_type;

public:

  RESPITEOutputFile(const string &a_name, const string &file_name=string());
  virtual ~RESPITEOutputFile();
  
  virtual Block *clone(const string &n) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  void reset();
  void compute();
  void close();
  
  virtual void write_file_header();

  virtual void update_header();
};



/******************************************************************************/
/*									      */
/*	CLASS NAME: HTKSampleOutputFile  		       		      */
/*									      */
/******************************************************************************/

class HTKSampleOutputFile: public virtual CTKObject, public virtual CTKFile, public SampleOutputFile {

  static const string help_text;
  static const string type_name;

public:

  HTKSampleOutputFile(const string &a_name, const string &file_name=string());
  virtual ~HTKSampleOutputFile();
  
  virtual Block *clone(const string &n) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  void reset();
  
  virtual void write_file_header();
  virtual void update_header();
};




#endif

/* End of ctk_file.hh */
