/******************************************************************************/
/*									      */
/*	ctk_filter.cpp	    	       			              */
/*									      */
/*	Block for applying digital filters               		      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			              */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <cmath>
#include <vector>

#include <functional>
#include <numeric>
#include <algorithm>

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_filter.hh"


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: FilterBlock                                              */
/*                                                                            */
/******************************************************************************/

const string FilterBlock::type_name = "Filter";
const string FilterBlock::help_text = FILTER_BLOCK_HELP_TEXT;


FilterBlock::FilterBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {
 
  make_input_sockets(1);
  make_output_sockets(1);

  // Set up FILENAME param
  FILENAME_param = new ParamString("FILENAME");
  FILENAME_param->set_helptext("Name of file containing A and B coefficients.");
  parameters->register_parameter(FILENAME_param);
  
  // Set up A param
  A_param = new ParamFloatVector("A");
  A_param->set_helptext("The A Coefficients.");
  parameters->register_parameter(A_param);

  // Set up B param
  B_param = new ParamFloatVector("B");
  B_param->set_helptext("The B Coefficients.");
  parameters->register_parameter(B_param);
  
}
 

Block* FilterBlock::clone(const string &n) const{
  Block *ablock = new FilterBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}


void FilterBlock::reset() {
  Block::reset();
  
  if ((A_param->get_set_flag() || B_param->get_set_flag()) && FILENAME_param->get_set_flag()) {
    cerr << "Filter coefficients must be specified using either via the block parameters OR in a file" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  if (FILENAME_param->get_set_flag()) {
    // Read coefficients from a file
    read_filter_coefficients_from_file(FILENAME_param->get_value(), a_coeffs, b_coeffs);
  } else if (A_param->get_set_flag() && B_param->get_set_flag()) {
    a_coeffs=A_param->get_vector();
    b_coeffs=B_param->get_vector();
  } else {
    cerr << "Filter coefficients must be specified using either via the block parameters OR in a file" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  nA = a_coeffs.size();
  nB = b_coeffs.size();

  if (a_coeffs.size()==0 || b_coeffs.size()==.0) {
    cerr << "There must be at least one A and one B coefficient." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
  if (a_coeffs[0]==0.0) {
    cerr << "First denominator filter coefficient must be non-zero." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
  if (a_coeffs[0]!=1.0) {
    transform(a_coeffs.begin(), a_coeffs.end(), a_coeffs.begin(), bind2nd(divides<float>(), a_coeffs[0]));
    transform(b_coeffs.begin(), b_coeffs.end(), b_coeffs.begin(), bind2nd(divides<float>(), a_coeffs[0]));
  }

  // Zero this out so it doesn't contribute to the inner product.
  a_coeffs[0]=0.0;
  
  width=(*input_sockets)[0]->get_data_descriptor()->get_storage();

  xstorage.resize(0); ystorage.resize(0);
  resize_static_array(xstorage, nB, width);
  resize_static_array(ystorage, nA, width);

  // Coefficients are stored in a clever way
  //  [A0 A1 A2 ... An] is stored as [An ... A2 A1 A0 An ... A2 A1 A0]
  // this makes the multiplications in compute more efficient
  reverse(a_coeffs.begin(), a_coeffs.end());
  reverse(b_coeffs.begin(), b_coeffs.end());
  makeXtoXX(a_coeffs);
  makeXtoXX(b_coeffs);

  // Reset storage array indicies and coefficient array offsets
  offset_A=nA;
  offset_B=nB;
  index_A=nA-1;
  index_B=nB-1;
  
}


void FilterBlock::compute() {

  CTKVector *xv0;

  if (inputs_are_all_sample_data) {
    // Sample data is treated as frame data of width 1 
    CTKSample sample;
    (*input_sockets)[0]->get_sample(sample);
    xstorage[0][index_B]=sample;
  }else {
    (*input_sockets)[0]->get_vector(xv0);
    for (int i=0; i<width; ++i) {
      xstorage[i][index_B]=(*xv0)[i];
    }
  }
  
  for (int i=0; i<width; ++i) {    
    ystorage[i][index_A]=
      inner_product(xstorage[i].begin(), xstorage[i].end(), b_coeffs.begin()+offset_B, 0.0)
      -inner_product(ystorage[i].begin(), ystorage[i].end(), a_coeffs.begin()+offset_A, 0.0);
  }
  
  if (inputs_are_all_sample_data)
    (*output_sockets)[0]->put_sample(ystorage[0][index_A]);
  else {
    for (int i=0; i<width; ++i)
      (*xv0)[i]=ystorage[i][index_A];
    (*output_sockets)[0]->put_vector(xv0);
  }

  offset_A-=1; if (offset_A==0) offset_A=nA;
  offset_B-=1; if (offset_B==0) offset_B=nB;

  index_A+=1; if (index_A==nA) index_A=0;
  index_B+=1; if (index_B==nB) index_B=0;
  
}


void FilterBlock::close() {
  Block::close();
}

 

  
void FilterBlock::read_filter_coefficients_from_file(string filename, vector<Float> &A, vector<Float> &B) {

  FILE *fp=fopen(filename.c_str(),"r");

  if (fp==NULL) {
    cerr << "Problem opening filter coefficient file, " << filename.c_str() << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  bool error = read_vector(fp, A) || read_vector(fp, B);

  fclose(fp);

  if (error) {
    cerr << "Problem reading filter coefficients from file, " << filename.c_str() << "." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

}


bool FilterBlock::read_vector(FILE *fp, vector<Float> &x) {
  int n_elements;
  if (fscanf(fp,"%d",&n_elements)!=1) return 1;
  x.resize(n_elements);
  for (int i=0; i<n_elements; ++i)
    if (fscanf(fp,"%lf", &x[i])!=1) return 1;
  return 0;
}
  


/* End of ctk_filter.cpp */
 
