/******************************************************************************/
/*									      */
/*	ctk_framer.cpp	    		      			              */
/*									      */
/*	Block for generating frames from a signal                             */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007	         		      */
/*									      */
/******************************************************************************/
 
#include "ctk-config.h"

#include <cmath>
#include <vector>
#include <algorithm>
#include <functional>

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_dsp.hh"

#include "ctk_param.hh"
#include "ctk_data_descriptor.hh"
#include "ctk_socket.hh"

#include "ctk_framer.hh"

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: FramerBlock                                              */
/*                                                                            */
/******************************************************************************/

const string FramerBlock::help_text = FRAME_BLOCK_HELP_TEXT;
const string FramerBlock::type_name = "Frame";

const char *FramerBlock::PARAM_DEFAULT_WINDOW_SHAPE = "RECTANGLE";

FramerBlock::FramerBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {
 
  make_input_sockets(1);
  make_output_sockets(1);

  // Set up SHIFT parameter
  SHIFT_param= new ParamInt("SHIFT", PARAM_DEFAULT_FRAME_SHIFT);
  SHIFT_param->set_helptext("The number of samples or frames by which to shift the window at each step.");
  SHIFT_param->install_validator(new Validator(Validator::VLOWER, 1.0));
  parameters->register_parameter(SHIFT_param);

  // Set up SIZE parameter
  SIZE_param= new ParamInt("SIZE", PARAM_DEFAULT_FRAME_SIZE);
  SIZE_param->set_helptext("The number of samples or frames to group into a single output frame.");
  SIZE_param->install_validator(new Validator(Validator::VLOWER, 1.0));
  parameters->register_parameter(SIZE_param);

  // Set up SHAPE parameter
  SHAPE_param= new ParamEnumerated("SHAPE", StringArrays::VALID_WINDOW_SHAPES, PARAM_DEFAULT_WINDOW_SHAPE);
  SHAPE_param->set_helptext("The form of shaping window that can be applied to the frame. <p> By setting this to RECTANGLE shaping is effectively turned off.");
  parameters->register_parameter(SHAPE_param);
  
  
}

FramerBlock::~FramerBlock() {
  //  free_storage();   // JONJON - this is causing a crash. Why?
}

Block* FramerBlock::clone(const string &n) const{
  Block *ablock = new FramerBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}


void FramerBlock::reset(){
  Block::reset();

  // Check validity of input data
  if ((*input_sockets)[0]->get_data_descriptor()->get_dimension(axis_name="TIME")!=NULL) {
    if ((*input_sockets)[0]->get_data_descriptor()->get_dimension(axis_name="TIME_2")!=NULL) {
      cerr << "Error in network at block: " << getname() << endl;
      cerr << "Framer cannot be applied to frame data that already possesses a TIME and a TIME_2 axis." << endl;
      throw(CTKError(__FILE__, __LINE__));
    }
  }
  
  Integer winsize=SIZE_param->get_value();

  //  free_storage();   // JONJON - this is causing a crash. Why?
  
  is_rectangle= (SHAPE_param->get_value()==string("RECTANGLE"));
  make_window(SHAPE_param->get_value(), window, SIZE_param->get_value());

  first_call=true;
  
  (*input_sockets)[0]->set_nframes(winsize);

  int shift=SHIFT_param->get_value();

  overlap=winsize-shift;;  

  if (shift==0) {
    cerr << "Invalid value for SHIFT parameter for block: " << getfullname() << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  input_sample_rate=sample_rate_param->get_value();
  unlock_parameters();
  set_parameter("SAMPLE_RATE", sample_rate_param->get_value()/shift);
  relock_parameters();

  if (overlap<0) {
    gap=-overlap;
    overlap=0;
  } else
    gap=0;

  frame.resize(winsize);
  dataS.resize(winsize);
  junkS.resize(gap);
  
  dataF.resize(winsize);
  junkF.resize(gap);
}


void FramerBlock::build_output_data_descriptors() {
  
  const DataDescriptor *idd = (*input_sockets)[0]->get_data_descriptor();
  if (idd==NULL) {
    cerr << "FramerBlock::build_output_data_descriptors - internal error - NULL input data desc" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
  DataDescriptor *dd = new DataDescriptor(*idd);

  // Construct the temporal axis
  Integer winsize=SIZE_param->get_value();
  CTKVector axis(winsize);
  Integer i=0;
  
  for (CTKVector::iterator fp=axis.begin(); fp!=axis.end(); ++fp)
    *fp=(float)++i;

  transform(axis.begin(), axis.end(), axis.begin(), bind2nd(divides<Float>(), input_sample_rate));

  // Add the time axis to whatever axes exist already
  dd->add_outer_dimension(axis_name, axis);   

  // Pass this descriptor to all the output socket
  (*output_sockets)[0]->set_data_descriptor(dd);
  
  
}


void FramerBlock::compute() {

  if (inputs_are_all_sample_data) {
    // For framing sample data

    // If there is a gap between frames then read data for the gap
    // and then destroy it - but not on the first call
    if (!first_call && gap!=0) {
      (*input_sockets)[0]->read_sample(junkS);
    }
    
    (*input_sockets)[0]->read_sample(dataS);

    transform(window.begin(), window.end(), dataS.begin(), frame.begin(), multiplies<Float>());

    if (frame.size()>1)
      (*output_sockets)[0]->put_vector(new CTKVector(frame));
    else
      // The bizarre case where the user has made a frame of size 1
      (*output_sockets)[0]->put_sample(frame[0]);
    
    (*input_sockets)[0]->put_back(overlap);
    
    
  }else {
    // For framing frame data
        
    // If there is a gap between frames then read data for the gap
    // and then destroy it - but not on the first call
    if (!first_call && gap!=0) {
      (*input_sockets)[0]->read_vector(junkF);
      for (UInteger i=0; i<junkF.size(); ++i)
	delete junkF[i];
    }
    
    (*input_sockets)[0]->read_vector(dataF);
    
    Integer n;
    CTKVector *x=new CTKVector(dataF.size()*(n=(dataF[0]->size())));

    Float *op=&(*x)[0];
    Float *w0=&window[0];
    
    for (UInteger i=0; i<dataF.size(); ++i) {
      Float *fp0=&(*dataF[i])[0];
      Float *fpn=&(*dataF[i])[n];
      if (is_rectangle)
	op=copy(fp0, fpn, op);
      else
	op=transform(fp0, fpn, w0, op, multiplies<Float>());
    }
    
    (*output_sockets)[0]->put_vector(x);
    
    (*input_sockets)[0]->put_back(overlap);
    
    // Delete the frames that have not been put back
    for (UInteger i=0; i<dataF.size()-overlap; ++i)
      delete dataF[i];    
  }

  
  if (first_call && gap!=0) {
    // After the first call we will need to have enough data for one frame and a possible gap
    (*input_sockets)[0]->set_nframes(SIZE_param->get_value()+gap);
  }

  first_call=false;
  
}


// Clear any allocated space
void FramerBlock::free_storage() {
  for (UInteger i=0; i<dataF.size(); ++i)
    if (dataF[i]!=NULL) delete dataF[i];
  for (UInteger i=0; i<junkF.size(); ++i)
    if (junkF[i]!=NULL) delete junkF[i];
}

/* End of ctk_framer.cpp */
 
