/******************************************************************************/
/*									      */
/*	ctk_gammatone_bank_if.cpp	      	      	      	              */
/*									      */
/*	                                                                      */
/*									      */
/*	Authors: Jon Barker & Martin Cooke, Sheffield University	      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			      	      */
/*        augmented by Martin to provide instantaneous frequency & AM         */
/*						 			      */
/******************************************************************************/
//
//  ??/10/00 MPC:  augmented to provide instantaneous frequency & AM    
//  12/10/00 JPB:  line added to fix leak occurring when AM connected but not env

#include "ctk-config.h"

#include <cmath>
#include <vector>
 
#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_dsp.hh"
#include "ctk_auditory.hh"
#include "ctk_socket.hh"
#include "ctk_param.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_gammatone_bank.hh"

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: GammatoneBankBlock                                       */
/*                                                                            */
/******************************************************************************/

const string GammatoneBankBlock::type_name = "GammatoneBank";
const string GammatoneBankBlock::help_text = GAMMATONE_BANK_BLOCK_HELP_TEXT;
const string GammatoneBankBlock::PARAM_DEFAULT_SCALE       = "ERB";


GammatoneBankBlock::GammatoneBankBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {
 
  make_input_sockets(1);
  make_output_sockets(4);
  
  output_sockets->set_description("out1", "instantaneous envelope");
  output_sockets->set_description("out2", "basilar membrane displacement");
  output_sockets->set_description("out3", "instantaneous frequency");
  output_sockets->set_description("out4", "AM frequency");

  // Set up NUM_CHANS parameter
  NUM_CHANS_param= new ParamInt("NUM_CHANS", PARAM_DEFAULT_NUM_CHANS);
  NUM_CHANS_param->install_validator(new Validator(Validator::VLOWER, 1.0));
  NUM_CHANS_param->set_helptext("The number of filter channels to employ.");
  parameters->register_parameter(NUM_CHANS_param);

  // Set up LOWER parameter
  LOWER_param= new ParamFloat("LOWER", PARAM_DEFAULT_LOWER);
  LOWER_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  LOWER_param->set_helptext("The centre frequency of the lowest filter.");
  LOWER_param->set_unit("Hz");
  parameters->register_parameter(LOWER_param);

  // Set up UPPER parameter
  UPPER_param= new ParamFloat("UPPER", PARAM_DEFAULT_UPPER);
  UPPER_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  UPPER_param->set_helptext("The centre frequency of the highest filter.");
  UPPER_param->set_unit("Hz");
  parameters->register_parameter(UPPER_param);

  // Set up SCALE parameter
  SCALE_param= new ParamEnumerated("SCALE", StringArrays::VALID_SCALES, PARAM_DEFAULT_SCALE);
  SCALE_param->set_helptext("The frequency scale on which filter centre frequencies will be equidistant.");
  parameters->register_parameter(SCALE_param);
  
  
  bm_connected=false;
  env_connected=false;
  if_connected=false;
  am_connected=false;

}
 
Block* GammatoneBankBlock::clone(const string &n) const{
  Block *ablock = new GammatoneBankBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void GammatoneBankBlock::reset(){
  Block::reset();

  // Check validity of input data
  Integer nDims=(*input_sockets)[0]->get_data_descriptor()->get_n_dimensions();
  if (!inputs_are_all_sample_data) {
    cerr << "Error in network at block: " << getname() << endl;
    cerr << "Attempting to apply GammatoneBank to ";
    cerr << nDims << "-dimensional vector frame data." << endl;
    cerr << "GammatoneBank can only be applied to sample data." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  Integer n=NUM_CHANS_param->get_value();
  samplerate=sample_rate_param->get_value();
  
  a1.resize(n); a2.resize(n); a3.resize(n); a4.resize(n); a5.resize(n);
  p0.resize(n); p1.resize(n); p2.resize(n); p3.resize(n); p4.resize(n);
  q0.resize(n); q1.resize(n); q2.resize(n); q3.resize(n); q4.resize(n);
  cf.resize(n); gain.resize(n); 

  bpa.resize(5); bpa[0]=1.0; bpa[1]= -3.7674; bpa[2]= 5.3369; bpa[3]= -3.3702; bpa[4]= 0.8008;
  bpb.resize(5); bpb[0]=0.0055; bpb[1]= 0.0; bpb[2]= -0.0111; bpb[3]= 0.0; bpb[4]= 0.0055;

  lastam.resize(n); zct0.resize(n);

  env1.resize(n); env2.resize(n); env3.resize(n); env4.resize(n);
  denv0.resize(n); denv1.resize(n); denv2.resize(n); denv3.resize(n); denv4.resize(n); 
  
  sine.resize((int)samplerate);
  cosine.resize((int)samplerate);
  cos_ptr.resize(n);

  Float lower=LOWER_param->get_value();
  Float upper=UPPER_param->get_value();
  
  if (lower < 0.0 || upper < 0.0 || lower > upper || upper > samplerate || lower > samplerate) {
    cerr << "Invalid cutoff frequencies for gammatone bank: " << getfullname() << endl;
    cerr << "LOWER and UPPER must be >= 0.0 and <= samplerate, with LOWER < UPPER" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  Float (*hzToScale)(Float);
  Float (*scaleToHz)(Float);
  
  if (SCALE_param->get_value()==string("ERB")) {
    hzToScale = &hzToERBrate;
    scaleToHz = &ERBrateToHz;
  } else if (SCALE_param->get_value()==string("MEL")) {
    hzToScale = &hzToMel;
    scaleToHz = &melToHz;
  } else if (SCALE_param->get_value()==string("LINEAR")) {
    hzToScale = &hzToHz;
    scaleToHz = &hzToHz;
  } else {
    cerr << "ctk_gammatone_bank_if: Unknown scale type :" << SCALE_param->get_value() << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
  GammaToneFloat scale_space;
  GammaToneFloat lower_scale = hzToScale(LOWER_param->get_value());
  GammaToneFloat upper_scale = hzToScale(UPPER_param->get_value());


  if (n > 1)
    scale_space = (upper_scale-lower_scale)/(NUM_CHANS_param->get_value()-1);
  else
    scale_space = 0.0;    

  two_pi_t = (2.0*M_PI)/samplerate;
  GammaToneFloat fi;
  
  Integer i;
  for (i=0; i < samplerate; ++i) {
    fi=i*two_pi_t;
    sine[i]=sin(fi);
    cosine[i]=cos(fi);
  }

  GammaToneFloat a;
  for (i=0; i<n; ++i) {
    cf[i] = scaleToHz(lower_scale+i*scale_space);
    GammaToneFloat tptbw = two_pi_t*erb(cf[i])*BW_CORRECTION;
    a = exp(-tptbw);
    gain[i] = (tptbw*tptbw*tptbw*tptbw)/3;
    a1[i] = 4.0*a; a2[i] = -6.0*a*a; a3[i] = 4.0*a*a*a;
    a4[i] = -a*a*a*a; a5[i] = 4.0*a*a;
    p0[i]=0.0; p1[i]=0.0; p2[i]=0.0; p3[i]=0.0; p4[i]=0.0;
    q0[i]=0.0; q1[i]=0.0; q2[i]=0.0; q3[i]=0.0; q4[i]=0.0;

    env1[i]=0.0; env2[i]=0.0; env3[i]=0.0; env4[i]=0.0;
    denv0[i]=0.0; denv1[i]=0.0; denv2[i]=0.0; denv3[i]=0.0; denv4[i]=0.0;
    lastam[i]=0.0; zct0[i]=0.0;
  }

  env_connected=(*output_sockets)[0]->connected();
  bm_connected=(*output_sockets)[1]->connected();
  if_connected=(*output_sockets)[2]->connected();
  am_connected=(*output_sockets)[3]->connected();

  sample = 0;
}

void GammatoneBankBlock::build_output_data_descriptors() {
  
  // Construct a DataDescriptor for 1-dimension frequency data
  DataDescriptor *dd = new DataDescriptor();
  dd->add_inner_dimension(string("FREQUENCY"), cf);   

  // Pass this descriptor to all the output channels (i.e. bm, env, if)
  // (Note, need a separate data descriptor object for each socket)
  (*output_sockets)[0]->set_data_descriptor(dd);
  (*output_sockets)[1]->set_data_descriptor(new DataDescriptor(*dd));
  (*output_sockets)[2]->set_data_descriptor(new DataDescriptor(*dd));
  (*output_sockets)[3]->set_data_descriptor(new DataDescriptor(*dd));
  
}


void GammatoneBankBlock::compute() {

  Float fsig;
  GammaToneFloat sig;

  (*input_sockets)[0]->get_sample(fsig);
  sig=(GammaToneFloat)fsig;
  
  Integer n=NUM_CHANS_param->get_value();
  GammaToneFloat this_samplerate=sample_rate_param->get_value();

  GammaToneFloat *a1ip=&a1[0], *a2ip=&a2[0], *a3ip=&a3[0], *a4ip=&a4[0], *a5ip=&a5[0];
  GammaToneFloat *p0ip=&p0[0], *p1ip=&p1[0], *p2ip=&p2[0], *p3ip=&p3[0], *p4ip=&p4[0];
  GammaToneFloat *q0ip=&q0[0], *q1ip=&q1[0], *q2ip=&q2[0], *q3ip=&q3[0], *q4ip=&q4[0];
  
  GammaToneFloat *cos_ptr_ip=&cos_ptr[0];
  Float *cfip=&cf[0];
  Float ccf;

  GammaToneFloat *gain_ip=&gain[0];
  
  GammaToneFloat u0, v0, u1=0.0, v1=0.0;

  GammaToneFloat a1i, a2i, a3i, a4i, a5i;
  GammaToneFloat p1i, p2i, p3i, p4i;
  GammaToneFloat q1i, q2i, q3i, q4i;
  GammaToneFloat cs, sn;
  GammaToneFloat instp, instfreq;
  
  Integer cp;

  CTKVector *bm=0, *env=0, *instf=0, *am=0;

  Float *bmip=0, *envip=0, *instfip=0, *amip=0;
  
  if (bm_connected) {
    bm=new CTKVector(n);
    bmip=&(*bm)[0];
  }

  if ((env_connected) || (am_connected)) {
    env=new CTKVector(n);
    envip=&(*env)[0];
  }

  if (if_connected) {
    instf=new CTKVector(n);
    instfip=&(*instf)[0];
  }

  if (am_connected) {
      am=new CTKVector(n);
      amip=&(*am)[0];
  }
  
  for (Integer i=0; i<n; ++i) {

    ccf = *cfip++;
    *cos_ptr_ip += ccf;
    while ((cp=(int)(*cos_ptr_ip)) >= this_samplerate)
      cp=(int)(*cos_ptr_ip -= this_samplerate);
    
    ++cos_ptr_ip;

    *p0ip = sig*(cs=cosine[cp])+(a1i=*a1ip++)*(p1i=*p1ip)+(a2i=*a2ip++)*(p2i=*p2ip)+(a3i=*a3ip++)*(p3i=*p3ip)+(a4i=*a4ip++)*(p4i=*p4ip); 
    *q0ip = -sig*(sn=sine[cp])+a1i*(q1i=*q1ip)+a2i*(q2i=*q2ip)+a3i*(q3i=*q3ip)+a4i*(q4i=*q4ip); 
   
    u0 = p1i+a1i*p2i+(a5i=*a5ip++)*p3i;
    v0 = q1i+a1i*q2i+a5i*q3i;

    if (if_connected) {
      u1 = p2i+a1i*p3i+a5i*p4i;
      v1 = q2i+a1i*q3i+a5i*q4i;
    }
  
    instp = u0*u0 + v0*v0;
    *p4ip++ = p3i; *p3ip++ = p2i; *p2ip++ = p1i; *p1ip++ = *p0ip++;
    *q4ip++ = q3i; *q3ip++ = q2i; *q2ip++ = q1i; *q1ip++ = *q0ip++;
  

    if (bm_connected)
      *bmip++=(Float)((u0*cs-v0*sn)*(*gain_ip));

    if ((env_connected) || (am_connected)) {
	float e = (Float)(sqrt(instp)*(*gain_ip));
	*envip++=e;
      if (am_connected) {
	  denv0[i] = (bpb[0]*e + bpb[1]*env1[i] + bpb[2]*env2[i] + bpb[3]*env3[i] + bpb[4]*env4[i]) -
	      (bpa[1]*denv1[i] + bpa[2]*denv2[i] + bpa[3]*denv3[i] + bpa[4]*denv4[i]);
	  //*amip++=denv0[i];
	  // estimate AM from ZC interval
	  if ((denv0[i]*denv1[i]) < 0.0) {
//	      // new ZC time
	    float zct=sample-1+(denv1[i]/(denv1[i]-denv0[i]));
//	      // AM freq
	    *amip++ = max(50.0,min(300.0,(this_samplerate/(2*(zct-zct0[i])))));
	    
	    zct0[i] = zct;
	    lastam[i]=(*am)[i];
	   } else {
	      *amip++=lastam[i];
	  }
	  env4[i]=env3[i]; env3[i]=env2[i]; env2[i]=env1[i]; env1[i]=e;
	  denv4[i]=denv3[i]; denv3[i]=denv2[i]; denv2[i]=denv1[i]; denv1[i]=denv0[i];
      }	  	  
    }

    if (if_connected) {
      if (instp < 1.0e-20) {
	instfreq=ccf;
      } else {
	instfreq = (Float)(ccf + ((u1*v0 - u0*v1)/(two_pi_t*instp)));
	if ((instfreq < 10) || (instfreq > 10000)) {
	  instfreq=ccf;
	} 
      }
      *instfip++=(Float)instfreq;
    }
    
    ++gain_ip;

  }
  
  
  if (n>1) {
    // Vector output
    if (env_connected)
	(*output_sockets)[0]->put_vector(env);   
    if (bm_connected)
	(*output_sockets)[1]->put_vector(bm); 
    if (if_connected)
	(*output_sockets)[2]->put_vector(instf);  
    if (am_connected)
	(*output_sockets)[3]->put_vector(am); 
  } else {
    // Sample output - for 1 channel case
    if (env_connected)
	(*output_sockets)[0]->put_sample((*env)[0]);   
    if (bm_connected)
	(*output_sockets)[1]->put_sample((*bm)[0]);
    if (if_connected)
	(*output_sockets)[2]->put_sample((*instf)[0]);
    if (am_connected)
	(*output_sockets)[3]->put_sample((*am)[0]);
  }

  // memory leak: add something which deletes env if am is connected and env isn't
  if (am_connected && !env_connected) delete env;

  ++sample;

} 


/* End of ctk_gammatone_bank_if.cpp */
 
