/******************************************************************************/
/*									      */
/*	ctk_generators.cpp	    			       		      */
/*									      */
/*	Signal generating blocks - sinusoids, white noise etc		      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			              */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <cmath>

#include "ctk_generators.hh"
#include "ctk_local.hh"
#include "ctk_error.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_dsp.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"

const Float    TWO_PI = 2.0 * M_PI;

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: GeneratorBlock                                                */
/*                                                                            */
/******************************************************************************/


GeneratorBlock::GeneratorBlock(const string &a_name, const string &a_block_type):SourceBlock(a_name, a_block_type) {

  // Set up WIDTH parameter
  WIDTH_param= new ParamInt("WIDTH", 1);
  WIDTH_param->install_validator(new Validator(Validator::VLOWER, 1.0));
  WIDTH_param->set_unit("channels");
  WIDTH_param->set_helptext("The size of the output frame");
  parameters->register_parameter(WIDTH_param);

}

void GeneratorBlock::reset() {
  SourceBlock::reset();
  if (get_current_count()==0) {
    cerr << "Input block " << getfullname() << " generates no data!" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  width=WIDTH_param->get_value();
}

void GeneratorBlock::build_output_data_descriptors() {

  for (SocketIt osocketp=output_sockets->begin(); osocketp!=output_sockets->end();  ++osocketp) {
    DataDescriptor *dd = new DataDescriptor();
    if (width>1) {
      CTKVector axis(width);
      for (Integer k=0; k<width; ++k) axis[k]=Float(k);
      dd->add_outer_dimension("X", axis);   
    }
    (*osocketp)->set_data_descriptor(dd);
  }
    
}

void GeneratorBlock::output(float x) {
  if (width==1) 
    (*output_sockets)[0]->put_sample(x);
  else
    (*output_sockets)[0]->put_vector(new CTKVector(width, x));
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: WaveBlock                                                */
/*                                                                            */
/******************************************************************************/


WaveBlock::WaveBlock(const string &a_name, const string &a_block_type):CTKObject(a_name),GeneratorBlock(a_name, a_block_type) {

  // Set up FREQ parameter
  FREQ_param= new ParamFloat("FREQ");
  FREQ_param->set_helptext("The waves frequency measured in Hz.");
  FREQ_param->set_unit("Hz");
  FREQ_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(FREQ_param);

  // Set up AMP parameter
  AMP_param= new ParamFloat("AMP", PARAM_DEFAULT_WAVE_AMP);
  AMP_param->set_helptext("The waves maximum amplitude.");
  AMP_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(AMP_param);

  // Set up PHASE parameter
  PHASE_param= new ParamFloat("PHASE", PARAM_DEFAULT_WAVE_PHASE);
  PHASE_param->set_helptext("The waves starting phase measures in radians.");
  PHASE_param->set_unit("radians");
  parameters->register_parameter(PHASE_param);
  
  make_output_sockets(1);
}

void WaveBlock::reset() {
  GeneratorBlock::reset();
  current_phase=PHASE_param->get_value();
  phase_step = (TWO_PI*FREQ_param->get_value())/sample_rate_param->get_value();
  phase_step -= TWO_PI*(int)(phase_step/TWO_PI);

  // Some time saving precomputation
  amplitude=AMP_param->get_value();
  two_amplitude=2.0*amplitude;
  delta=amplitude/M_PI;
  two_delta=2.0*delta;
  amp_phase_step=delta*phase_step;
  current_amp_phase=delta*current_phase;
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: ConstantBlock                                                */
/*                                                                            */
/******************************************************************************/

const string ConstantBlock::type_name = "Constant";
const string ConstantBlock::help_text = CONSTANT_BLOCK_HELP_TEXT;

ConstantBlock::ConstantBlock(const string &a_name):CTKObject(a_name),GeneratorBlock(a_name, type_name) {

  // Set up X parameter
  X_param= new ParamFloat("X");
  X_param->set_helptext("The value to output");
  parameters->register_parameter(X_param);

  make_output_sockets(1);
}

Block* ConstantBlock::clone(const string &n) const{
  Block *ablock = new ConstantBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void ConstantBlock::reset() {
  GeneratorBlock::reset();

  // Some time saving precomputation
  x=X_param->get_value();
}

void ConstantBlock::compute() {
  output(x);
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: ImpulseWaveBlock                                         */
/*                                                                            */
/******************************************************************************/

const string ImpulseWaveBlock::help_text = IMPULSE_WAVE_BLOCK_HELP_TEXT;
const string ImpulseWaveBlock::type_name = "ImpulseWave";


ImpulseWaveBlock::ImpulseWaveBlock(const string &a_name):CTKObject(a_name),WaveBlock(a_name, type_name) {}

Block* ImpulseWaveBlock::clone(const string &n) const{
  Block *ablock = new ImpulseWaveBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void ImpulseWaveBlock::compute() {
  current_phase+=phase_step;
  if (current_phase>TWO_PI) {
    current_phase-=TWO_PI;
    output(amplitude);
  } else {
    output(0.0);
  }
}
 
/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SineWaveBlock                                            */
/*                                                                            */
/******************************************************************************/

const string SineWaveBlock::type_name = "SineWave";
const string SineWaveBlock::help_text = SINE_WAVE_BLOCK_HELP_TEXT;

SineWaveBlock::SineWaveBlock(const string &a_name):CTKObject(a_name),WaveBlock(a_name, type_name) {}

Block* SineWaveBlock::clone(const string &n) const{
  Block *ablock = new SineWaveBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void SineWaveBlock::compute() {
  current_phase+=phase_step;
  if (current_phase>TWO_PI) current_phase-=TWO_PI;
  output(amplitude*sin(current_phase));
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SquareWaveBlock                                          */
/*                                                                            */
/******************************************************************************/

const string SquareWaveBlock::type_name = "SquareWave";
const string SquareWaveBlock::help_text = SQUARE_WAVE_BLOCK_HELP_TEXT;


SquareWaveBlock::SquareWaveBlock(const string &a_name):CTKObject(a_name),WaveBlock(a_name, type_name) {}

Block* SquareWaveBlock::clone(const string &n) const{
  Block *ablock = new SquareWaveBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void SquareWaveBlock::compute() {
  current_phase+=phase_step;
  if (current_phase>TWO_PI) current_phase-=TWO_PI;
  output(current_phase<M_PI?-amplitude:amplitude);
}




/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: TriangleWaveBlock                                        */
/*                                                                            */
/******************************************************************************/

const string TriangleWaveBlock::type_name = "TriangleWave";
const string TriangleWaveBlock::help_text = TRIANGLE_WAVE_BLOCK_HELP_TEXT;


TriangleWaveBlock::TriangleWaveBlock(const string &a_name):CTKObject(a_name),WaveBlock(a_name, type_name) {}

Block* TriangleWaveBlock::clone(const string &n) const{
  Block *ablock = new TriangleWaveBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void TriangleWaveBlock::reset() {
  WaveBlock::reset();
  amp_phase_step=2.0*amp_phase_step;
  current_amp_phase=2.0*current_amp_phase;
  is_rising=(current_phase<M_PI);
}

void TriangleWaveBlock::compute() {
  current_amp_phase+=amp_phase_step;
  
  if (current_amp_phase>two_amplitude) {
    current_amp_phase-=two_amplitude;
    is_rising=!is_rising;
  }
  
  if (is_rising)
    output(current_amp_phase-amplitude);
  else
    output(amplitude-current_amp_phase);
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SawtoothWaveBlock                                        */
/*                                                                            */
/******************************************************************************/

const string SawtoothWaveBlock::type_name = "SawtoothWave";
const string SawtoothWaveBlock::help_text = SAWTOOTH_WAVE_BLOCK_HELP_TEXT;

SawtoothWaveBlock::SawtoothWaveBlock(const string &a_name):CTKObject(a_name),WaveBlock(a_name, type_name) {}

Block* SawtoothWaveBlock::clone(const string &n) const{
  Block *ablock = new SawtoothWaveBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void SawtoothWaveBlock::compute() {
  current_amp_phase+=amp_phase_step;
  if (current_amp_phase>two_amplitude) current_amp_phase-=two_amplitude;
  
  output(current_amp_phase-amplitude);
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: GlottalWaveBlock                                         */
/*                                                                            */
/******************************************************************************/

const string GlottalWaveBlock::type_name = "GlottalWave";
const string GlottalWaveBlock::help_text = GLOTTAL_WAVE_BLOCK_HELP_TEXT;

GlottalWaveBlock::GlottalWaveBlock(const string &a_name):CTKObject(a_name),WaveBlock(a_name, type_name) {

  // Set up ATTACK parameter
  ATTACK_param= new ParamFloat("ATTACK");
  ATTACK_param->set_helptext("Percentage of cycle over which signal grows from 0 to maximum amplitude.");
  ATTACK_param->set_unit("%");
  ATTACK_param->install_validator(new Validator(0.0, 100.0));
  parameters->register_parameter(ATTACK_param);

  // Set up SUSTAIN parameter
  SUSTAIN_param= new ParamFloat("SUSTAIN");
  SUSTAIN_param->set_helptext("Percentage of cycle over which signal maintains its maximum amplitude.");
  SUSTAIN_param->set_unit("%");
  SUSTAIN_param->install_validator(new Validator(0.0, 100.0));
  parameters->register_parameter(SUSTAIN_param);

  // Set up DECAY parameter
  DECAY_param= new ParamFloat("DECAY");
  DECAY_param->set_helptext("Percentage of cycle over which signal falls from its maximum amplitude to 0.");
  DECAY_param->set_unit("%");
  DECAY_param->install_validator(new Validator(0.0, 100.0));
  parameters->register_parameter(DECAY_param);

  // Set up CLOSED parameter
  CLOSED_param= new ParamFloat("CLOSED");
  CLOSED_param->set_helptext("Percentage of cycle over which signal remains at 0.");
  CLOSED_param->set_unit("%");
  CLOSED_param->install_validator(new Validator(0.0, 100.0));
  parameters->register_parameter(CLOSED_param);

}

Block* GlottalWaveBlock::clone(const string &n) const{
  Block *ablock = new GlottalWaveBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void GlottalWaveBlock::reset() {
  WaveBlock::reset();

  Float sum_phases=ATTACK_param->get_value()+SUSTAIN_param->get_value()
    +DECAY_param->get_value()+CLOSED_param->get_value();
  
  full_on_phase=TWO_PI*ATTACK_param->get_value()/sum_phases;
  end_sustain_phase=full_on_phase+TWO_PI*SUSTAIN_param->get_value()/sum_phases;
  full_off_phase=end_sustain_phase+TWO_PI*DECAY_param->get_value()/sum_phases;

  decay_period=full_off_phase-end_sustain_phase;
  if (full_on_phase!=0)
    amp_over_full_on_phase=amplitude/full_on_phase;
  if (decay_period!=0)
    amp_over_decay_period=amplitude/decay_period;
}

void GlottalWaveBlock::compute() {
  current_phase+=phase_step;
  
  if (current_phase>TWO_PI)
    current_phase-=TWO_PI;

  if (current_phase>full_off_phase){
    output(0.0);
  } else if (current_phase>end_sustain_phase) {
    output(amp_over_decay_period*(full_off_phase-current_phase));
  } else if (current_phase>full_on_phase) {
    output(amplitude);
  } else {
    output(amp_over_full_on_phase*current_phase);
  }
  
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: WhiteNoiseBlock                                          */
/*                                                                            */
/******************************************************************************/

const string WhiteNoiseBlock::type_name = "WhiteNoise";
const string WhiteNoiseBlock::help_text = WHITE_NOISE_BLOCK_HELP_TEXT;

WhiteNoiseBlock::WhiteNoiseBlock(const string &a_name):CTKObject(a_name),GeneratorBlock(a_name, type_name) {

  // Set up AMP parameter
  AMP_param= new ParamFloat("AMP", PARAM_DEFAULT_WAVE_AMP);
  AMP_param->set_helptext("rms amplitude of white noise signal");
  AMP_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(AMP_param);
  
  make_output_sockets(1);
}

void WhiteNoiseBlock::reset() {
  GeneratorBlock::reset();
  amplitude=AMP_param->get_value()*sqrt(3.0);
  // The sqrt(3.0) ensures that the signal has an RMS amplitude of AMP_param
  // Without it AMP_param would just be the width of the uniform distribution
  // And the RMS amplitude would be too small.
  
  two_amplitude=2.0*amplitude;
  srand(0);
}

Block* WhiteNoiseBlock::clone(const string &n) const{
  Block *ablock = new WhiteNoiseBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void WhiteNoiseBlock::compute() {
  Float sample=((Float)rand()/RAND_MAX)*two_amplitude-amplitude;
  output(sample);
}



/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: GaussianNoiseBlock                                       */
/*                                                                            */
/******************************************************************************/

const string GaussianNoiseBlock::type_name = "GaussianNoise";
const string GaussianNoiseBlock::help_text = GAUSSIAN_NOISE_BLOCK_HELP_TEXT;

GaussianNoiseBlock::GaussianNoiseBlock(const string &a_name):CTKObject(a_name),GeneratorBlock(a_name, type_name) {

  // Set up AMP parameter
  AMP_param= new ParamFloat("AMP", PARAM_DEFAULT_WAVE_AMP);
  AMP_param->set_helptext("rms amplitude of Gaussian noise signal");
  AMP_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(AMP_param);
  
  make_output_sockets(1);
}

void GaussianNoiseBlock::reset() {
  GeneratorBlock::reset();
  amplitude=AMP_param->get_value();
  srand(0);
}

Block* GaussianNoiseBlock::clone(const string &n) const{
  Block *ablock = new GaussianNoiseBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void GaussianNoiseBlock::compute() {
  Float sample=random_gaussian(0.0, amplitude);
  output(sample);
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: PinkNoiseBlock                                           */
/*                                                                            */
/******************************************************************************/

// Uses a filter with the following poles and zeros
//>> poles=[.9986823 .9914651 .9580812 .8090598 .2896591]';
//>> zeros=[.9963594 .9808756 .9097290 .6128445 -.0324723]';
//>> 

// These are expressed by the following feed forward and feed-back filter coefficients
//>> [b,a]=zp2tf(zeros,poles,1);
//>> 
//>> b = 1.0000   -3.4673    4.4317   -2.4428    0.4608    0.0177
//>> a= 1.0000   -4.0469    6.3705   -4.8224    1.7212   -0.2223

// The pink noise y(t) is obtained by filtering the white noise x(t):
//
//   a(1)*y(n) = b(1)*x(n) + b(2)*x(n-1) + ... + b(6)*x(n-5)
//                         - a(2)*y(n-1) - ... - a(6)*y(n-5)

const string PinkNoiseBlock::type_name = "PinkNoise";
const string PinkNoiseBlock::help_text = PINK_NOISE_BLOCK_HELP_TEXT;

PinkNoiseBlock::PinkNoiseBlock(const string &a_name):CTKObject(a_name),GeneratorBlock(a_name, type_name) {

  // Set up AMP parameter
  AMP_param= new ParamFloat("AMP", PARAM_DEFAULT_WAVE_AMP);
  AMP_param->set_helptext("The rms amplitude of the pink noise signal.");
  AMP_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(AMP_param);
  
  make_output_sockets(1);
}


void PinkNoiseBlock::reset() {
  GeneratorBlock::reset();
  
  amplitude=AMP_param->get_value();
  amplitude*=PINK_NOISE_NORMALISATION_COEFF;
  srand(0);

  y1=y2=y3=y4=y5=0.0;
  x1=x2=x3=x4=x5=0.0;
}

Block* PinkNoiseBlock::clone(const string &n) const{
  Block *ablock = new PinkNoiseBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void PinkNoiseBlock::compute() {
  y6=y5;y5=y4; y4=y3; y3=y2; y2=y1;
  x6=x5;x5=x4; x4=x3; x3=x2; x2=x1;
  
  x1=random_gaussian(0.0, amplitude);
  y1=x1+PinkNoise_B2*x2+PinkNoise_B3*x3+PinkNoise_B4*x4+PinkNoise_B5*x5+PinkNoise_B6*x6
    -PinkNoise_A2*y2-PinkNoise_A3*y3-PinkNoise_A4*y4-PinkNoise_A5*y5-PinkNoise_A6*y6;
  
  output(y1);
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: RampBlock                                                */
/*                                                                            */
/******************************************************************************/

const string RampBlock::type_name = "Ramp";
const string RampBlock::help_text = RAMP_BLOCK_HELP_TEXT;

RampBlock::RampBlock(const string &a_name):CTKObject(a_name),GeneratorBlock(a_name, type_name) {

  // Set up START parameter
  START_param= new ParamFloat("START", PARAM_DEFAULT_RAMP_START);
  START_param->set_helptext("The signal amplitude at the start of the ramp.");
  parameters->register_parameter(START_param);

  // Set up STEP parameter
  STEP_param= new ParamFloat("STEP", PARAM_DEFAULT_RAMP_STEP);
  STEP_param->set_helptext("The amount by which to increase the signal amplitude at each sample.");
  parameters->register_parameter(STEP_param);
  
  make_output_sockets(1);
}

Block* RampBlock::clone(const string &n) const{
  Block *ablock = new RampBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void RampBlock::reset(){
  GeneratorBlock::reset();
  data=START_param->get_value();
}

void RampBlock::compute() {
  data+=STEP_param->get_value();
  output(data);
}

/* End of ctk_generators.cpp */
