/******************************************************************************/
/*									      */
/*	ctk_haircell.cpp	    	      	         	              */
/*						        		      */
/*	Remove harmonic from vectors - useful for cleaning summary     	      */
/*		autocorrelograms	        			      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			      	      */
/*									      */
/******************************************************************************/
 
#include "ctk-config.h"

#include <cmath>
#include <vector>
 
#include "ctk_local.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"

#include "ctk_haircell.hh"

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: HaircellBlock                                            */
/*                                                                            */
/******************************************************************************/

HaircellBlock::HaircellBlock(const string &a_name, const string &a_block_type):Block(a_name,  a_block_type) {
  make_input_sockets(1);
  make_output_sockets(1);
}

void HaircellBlock::reset() {
  Block::reset();
  
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MeddisHaircellBlock                                      */
/*                                                                            */
/******************************************************************************/

const string MeddisHaircellBlock::type_name = "MeddisHaircell";
const string MeddisHaircellBlock::help_text = MEDDIS_HAIRCELL_BLOCK_HELP_TEXT;

MeddisHaircellBlock::MeddisHaircellBlock(const string &a_name):CTKObject(a_name),HaircellBlock(a_name, type_name) {
}

Block* MeddisHaircellBlock::clone(const string &n) const{
  Block *ablock = new MeddisHaircellBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void MeddisHaircellBlock::reset() {
  HaircellBlock::reset();
  
  dt=1.0/sample_rate_param->get_value();
  
  ymdt=MED_Y*MED_M*dt;    
  xdt=MED_X*dt; 
  ydt=MED_Y*dt;
  lplusrdt=(MED_L+MED_R)*dt;
  rdt=MED_R*dt;
  gdt=MED_G*dt;
  hdt=MED_H;
  kt=MED_G*MED_A/(MED_A+MED_B);
  c=MED_M*MED_Y*kt/(MED_L*kt+MED_Y*(MED_L+MED_R));
  q=c*(MED_L+MED_R)/kt;
  w=c*MED_R/MED_X;
}
 
void MeddisHaircellBlock::compute() {

  if (inputs_are_all_sample_data) {
    // For sample data
    CTKSample sample;
    (*input_sockets)[0]->get_sample(sample);
    (*output_sockets)[0]->put_sample(do_meddis(sample));
  } else {
    // For frame data
    CTKVector *xv0;
    (*input_sockets)[0]->get_vector(xv0);
    for (Float *fp=&(*xv0)[0]; fp<&(*xv0)[xv0->size()]; ++fp)
      *fp=do_meddis(*fp);
    (*output_sockets)[0]->put_vector(xv0);
  }
  
}


Float MeddisHaircellBlock::do_meddis(Float sample) {
  
  Float x=sample+MED_A;
  Float replenish;
  
  if (x > 0.0)
    kt=gdt*(x/(x+MED_B)); 
  else 
    kt=0.0;
 
  if (q < MED_M ) 
    replenish=ymdt-ydt*q; 
  else 
    replenish=0.0;
  
  Float eject=kt*q;
  Float reuptakeandloss=lplusrdt*c; 
  Float reuptake=rdt*c;       
  Float reprocess=xdt*w;

  q = max(0.0,q+replenish-eject+reprocess);
  c = max(0.0,c+eject-reuptakeandloss);
  w = max(0.0,w+reuptake-reprocess);
  return hdt*c;
}


/* End of ctk_haircell.cpp */
