/******************************************************************************/
/*									      */
/*	ctk_harmonic_removal.cpp	    	      		              */
/*						        		      */
/*	Remove harmonic from vectors - useful for cleaning summary     	      */
/*		autocorrelograms	        			      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			      	      */
/*									      */
/******************************************************************************/
 
#include "ctk-config.h"

#include <cmath>
#include <vector>
#include <algorithm>
#include <functional>

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_harmonic_removal.hh"

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: HarmonicRemovalBlock                                     */
/*                                                                            */
/******************************************************************************/

const string HarmonicRemovalBlock::type_name = "HarmonicRemoval";
const string HarmonicRemovalBlock::help_text = HARMONIC_REMOVAL_BLOCK_HELP_TEXT;

HarmonicRemovalBlock::HarmonicRemovalBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {
  make_input_sockets(1);
  make_output_sockets(1);

  // Set up FACTOR parameter
  FACTOR_param= new ParamInt("FACTOR");
  FACTOR_param->set_helptext("The number of stretch multiples that the algorithm employs. <p> Small values, 2, 3 or 4 produce the best results.");
  FACTOR_param->install_validator(new Validator(Validator::VLOWER, 1.0));
  parameters->register_parameter(FACTOR_param);
}

Block* HarmonicRemovalBlock::clone(const string &n) const{
  Block *ablock = new HarmonicRemovalBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void HarmonicRemovalBlock::reset() {
  Block::reset();
  
  // Check validity of input data
  const DataDescriptor *dd=(*input_sockets)[0]->get_data_descriptor();
  Integer nDims=dd->get_n_dimensions();
  
  if (nDims!=1 || inputs_are_all_sample_data) {
    cerr << "Error in network at block: " << getname() << endl;
    cerr << "Attempting to apply HarmonicRemoval to ";
    if (nDims==0 || inputs_are_all_sample_data)
      cerr << "sample data" << endl;
    else
      cerr << nDims << "-dimensional frame data." << endl;
    cerr << "HarmonicRemoval can only be applied to 1 dimensional frame data." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
  maxfactor=FACTOR_param->get_value();

  // Do nothing if vector only has one element or maxfactor less than 2
  do_nothing=(maxfactor<2) || (dd->get_storage()<2);  

}

void HarmonicRemovalBlock::compute() {

  // A heavily optimized version of code borrowed from Guy Brown
  
  CTKVector *xv0;
  (*input_sockets)[0]->get_vector(xv0);
  
  Integer maxfactor_value=FACTOR_param->get_value();
  
  if (!do_nothing) {
    Float gap=0.0, xv0_idx=0.0;
    CTKVector *new_xv0 = new CTKVector(*xv0);
    Integer N=xv0->size();
    CTKVector expanded(N);
    
    for (Integer k=2; k<=maxfactor_value; ++k) {
      Float *ep=&expanded[0];
      Integer idx=-1;
      Float *idx0p=&(*xv0)[0];
      Float *idx1p=idx0p+1;
      for (Integer i=0; i<N; ++i, ++ep) {
	Float xval=(Float)i/k;
	if (idx!=(Integer)xval) {
	  ++idx;
	  xv0_idx=*idx0p;
	  gap=*idx1p-xv0_idx;
	  ++idx0p, ++idx1p;
	}
	*ep=xv0_idx+(xval-idx)*gap;
      }
      transform(new_xv0->begin(), new_xv0->end(), expanded.begin(), new_xv0->begin(), minus<Float>());
    }
    
    for (CTKVector::iterator fp=new_xv0->begin(); fp!=new_xv0->end(); ++fp)
      *fp=max(0.0, *fp);
    
    delete xv0;
    xv0=new_xv0;
  }
  
  (*output_sockets)[0]->put_vector(xv0);
  
}


/* End of ctk_harmonic_removal.cpp */
