/******************************************************************************/
/*									      */
/*	ctk_integrator.cpp	    	       			              */
/*									      */
/*	Block for applying an integrator to a frame                           */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			     	      */
/*									      */
/******************************************************************************/
 
#include "ctk-config.h"

#include <cmath>
#include <vector>
 
#include "ctk_local.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"

#include "ctk_integrator.hh"


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: IntegratorBlock                                          */
/*                                                                            */
/******************************************************************************/

const string IntegratorBlock::type_name = "Integrator";
const string IntegratorBlock::help_text = INTEGRATOR_BLOCK_HELP_TEXT;

IntegratorBlock::IntegratorBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {
 
  make_input_sockets(1);
  make_output_sockets(1);
}
 
Block* IntegratorBlock::clone(const string &n) const{
  Block *ablock = new IntegratorBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void IntegratorBlock::reset() {
  Block::reset();

  sumS=0.0;
  sumF.resize(0);
  frame_width=0;
}


void IntegratorBlock::compute() {

  if (inputs_are_all_sample_data) {
    // For integrating sample data
    CTKSample sample;
    (*input_sockets)[0]->get_sample(sample);
    sumS+=sample;
    (*output_sockets)[0]->put_sample(sumS);
      
  }else {
    // For integrating frame data
    
    CTKVector *xv0;
    
    (*input_sockets)[0]->get_vector(xv0);
    
    if (frame_width==0) 
      sumF.resize(frame_width=xv0->size(),0.0);
  
    Float *dip=&(*xv0)[0], *sip=&sumF[0];
    for (Integer i=0; i<frame_width; ++i, ++sip, ++dip) 
      *dip=(*sip+=*dip);
    
    (*output_sockets)[0]->put_vector(xv0);
  }
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: LeakyIntegratorBlock                                     */
/*                                                                            */
/******************************************************************************/

const string LeakyIntegratorBlock::type_name = "LeakyIntegrator";
const string LeakyIntegratorBlock::help_text =LEAKY_INTEGRATOR_BLOCK_HELP_TEXT;

LeakyIntegratorBlock::LeakyIntegratorBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {
 
  make_input_sockets(1);
  make_output_sockets(1);

  // Set up TIME_CONST parameter
  TIME_CONST_param= new ParamFloat("TIME_CONST", PARAM_DEFAULT_TIME_CONSTANT);
  TIME_CONST_param->set_helptext("The time constant of the leaky exponential decay.");
  TIME_CONST_param->set_unit("ms");
  TIME_CONST_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(TIME_CONST_param);

}
 
Block* LeakyIntegratorBlock::clone(const string &n) const{
  Block *ablock = new LeakyIntegratorBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void LeakyIntegratorBlock::reset() {
  Block::reset();

  decay=exp(-1000.0/(TIME_CONST_param->get_value()*sample_rate_param->get_value()));
  intgain=1.0-decay;
  sumS=0.0;
  sumF.resize(0);
  frame_width=0;
}


void LeakyIntegratorBlock::compute() {

  if (inputs_are_all_sample_data) {
    // For integrating sample data
    CTKSample sample;
    (*input_sockets)[0]->get_sample(sample);
    (sumS*=decay)+=sample;
    (*output_sockets)[0]->put_sample(sumS*intgain);
      
  }else {
    // For integrating frame data
    
    CTKVector *xv0;
    
    (*input_sockets)[0]->get_vector(xv0);
    
    if (frame_width==0) 
      sumF.resize(frame_width=xv0->size(),0.0);
  
    Float *dip=&(*xv0)[0], *sip=&sumF[0];
    for (Integer i=0; i<frame_width; ++i, ++sip, ++dip) 
      *dip=((*sip*=decay)+=*dip)*intgain;
    
    (*output_sockets)[0]->put_vector(xv0);
  }
}

/* End of ctk_integrator.cpp */
 
