/******************************************************************************/
/*									      */
/*	ctk_localisation.cpp	    	      	         	              */
/*						        		      */
/*	Remove harmonic from vectors - useful for cleaning summary     	      */
/*		autocorrelograms	        			      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			      	      */
/*									      */
/******************************************************************************/
 
#include "ctk-config.h"

#include <cmath>
#include <vector>
 
#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_localisation.hh"

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: LocalisationBlock                                        */
/*                                                                            */
/******************************************************************************/

//LocalisationBlock::LocalisationBlock(const string &a_name, const string &a_block_type):CTKObject(a_name),Block(a_name, a_block_type) {
LocalisationBlock::LocalisationBlock(const string &a_name, const string &a_block_type):Block(a_name, a_block_type) {
  make_input_sockets(2);
  make_output_sockets(1);
}

void LocalisationBlock::reset() {
  Block::reset();
  
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: LindemannLocalisationBlock                               */
/*                                                                            */
/******************************************************************************/

const string LindemannLocalisationBlock::type_name = "LindemannLocalisation";
const string LindemannLocalisationBlock::help_text = LINDEMANN_LOCALISATION_BLOCK_HELP_TEXT;

LindemannLocalisationBlock::LindemannLocalisationBlock(const string &a_name):CTKObject(a_name),LocalisationBlock(a_name, type_name) {

  // Set up M parameter
  M_param= new ParamInt("M", DEFAULT_M_PARAM);
  M_param->set_helptext("The Magic number: Pick the right value and this algorithm will work (maybe).");
  parameters->register_parameter(M_param);
}

LindemannLocalisationBlock::~LindemannLocalisationBlock() {
  delete[] phi; delete[] ir; delete[] il; delete[] psi;
  delete[] w1; delete[]w2;
  delete[] r; delete[] l; delete[] k;
}

Block* LindemannLocalisationBlock::clone(const string &n) const{
  Block *ablock = new LindemannLocalisationBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void LindemannLocalisationBlock::reset() {
  LocalisationBlock::reset();

  // Check the inputs all have the same shape
  if (input_shape_check()==false) {
    cerr << "LindemannLocalisationBlock:: Inputs have varying dimensionality" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  // Check validity of input data
  const DataDescriptor *dd=(*input_sockets)[0]->get_data_descriptor();
  channels=dd->get_storage();

  Float mf = 6.0, wf = 0.035, Tinh = 10.0, Tint = 5.0;

  M=M_param->get_value();

  Float Td = 1.0/sample_rate_param->get_value();
  k1 = exp(-Td*1000.0/Tinh);
  k2 = exp(-Td*1000.0/Tint);				

  N = channels*(2*M + 1);
  r = new Float[N];  l = new Float[N]; k = new Float[N];
  w1 = new Float[N];
  w2 = new Float[N];
  phi = new Float[N]; ir = new Float[N];  il = new Float[N]; psi = new Float[N];
  
  
  //	initialise weights, inhibitions etc
  
  Float *rp=r, *lp=l, *kp=k, *w1p=w1, *w2p=w2, *psip=psi, *phip=phi, *ilp=il, *irp=ir;
  for (Integer i=0; i<channels; ++i) {
    for (Integer m = 0; m <= 2*M; ++m) {
      *rp++ = 0.0;
      *lp++ = 0.0;
      *kp++ = 0.0;
      *psip++ = 0.0;
      *phip++ = 0.0;
      *ilp++ = 1.0;
      *irp++ = 1.0;
      *w1p = wf*exp(-m/mf); 
      *w2p = 1.0-*w1p;
      ++w1p; ++w2p;
    }
  }
  
}

void LindemannLocalisationBlock::compute() {
  
  // shift excitation along delay line with appropriate inhibition
  Integer MM=M+M;
  Integer MMp1=MM+1;
  
  Float *lp=&l[0], *lp1p=&l[1], *ilp1p=&il[1];
  Float *rp=&r[N-1], *rm1p=&r[N-1], *irm1p=&ir[N-2];

  CTKVector *outvec = new CTKVector(channels);
    
  for (Integer i=0; i<channels; ++i) {
    for (Integer m = 0; m < MM; ++m) {
      *rp-- = *rm1p-- * *irm1p--; 
      *lp++ = *lp1p++ * *ilp1p++;
    }
    ++lp; ++lp1p; ++ilp1p;
    --rp; --rm1p; --irm1p;
  }
  
  // include new data
  
  if (inputs_are_all_sample_data) {
    (*input_sockets)[0]->get_sample(l[MM]); 
    (*input_sockets)[1]->get_sample(r[0]); 
  } else {
    CTKVector *lvec, *rvec;
    (*input_sockets)[0]->get_vector(lvec); 
    (*input_sockets)[1]->get_vector(rvec); 
    for (Integer i=0; i<channels; ++i) {
      l[MM+(i*MMp1)]=(*lvec)[i];
      r[i*MMp1]=(*rvec)[i];
    }
    delete lvec;
    delete rvec;
  }
  
  // compute cross-correlation and inhibitions
  
  Float *phip=phi, *kp=k, *irp=ir, *ilp=il, *psip=psi;
  Float *w1p=w1, *w2p=w2;
  Float *rw1p=&w1[N-1], *rw2p=&w2[N-1];
  lp=l; rp=r;
  for (Integer i=0; i<channels; ++i) {
    Float sum_psi = 0.0;
    Float sum_m_psi = 0.0;
    for (Integer m = 0; m <= MM; ++m) {
      *phip = *kp + *phip * k1 * (1.0-*kp);
      *kp = (*rp* *w2p + *w1p) * (*lp * *rw2p + *rw1p);
      *irp = (1.0 - *lp)*(1.0 - *phip);
      *ilp = (1.0 - *rp)*(1.0 - *phip);
      //	leaky integrate k to form psi
      *psip = *kp + *psip * k2;
      sum_psi += *psip;
      sum_m_psi += (m-M) * *psip;

      ++phip, ++kp, ++rp, ++lp, ++irp, ++ilp, ++psip;
      ++w1p, ++w2p, --rw1p, --rw2p;
    }
    (*outvec)[i]=(sum_psi>1.0e-10)?sum_m_psi/sum_psi:1000000.0;
  }
  
  if (inputs_are_all_sample_data) {
    (*output_sockets)[0]->put_sample((*outvec)[0]);
    delete outvec;
  }
  else
    (*output_sockets)[0]->put_vector(outvec);
  
}




/* End of ctk_localisation.cpp */
