/******************************************************************************/
/*									      */
/*	ctk_matlab.cpp	 	         			              */
/*									      */
/*	Block for passing data through matlab - either for processing or for  */
/*                          display  					      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*      updated by Sue Harding                                                */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			      	              */
/*									      */
/******************************************************************************/
 
#include "ctk-config.h"

#include <cmath>
#include <vector>
#include <list>
#include <algorithm>

#include "boost/compose.hpp"

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_matlab.hh"



/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MatlabInputBlock                                         */
/*                                                                            */
/******************************************************************************/

#ifdef _HAS_MATLAB

const string MatlabInputBlock::type_name = "MInput";
const string MatlabInputBlock::help_text = MINPUT_BLOCK_HELP_TEXT;

MatlabInputBlock::MatlabInputBlock(const string &a_name):CTKObject(a_name),SourceBlock(a_name, type_name) {
  make_output_sockets(1);

  // Set up NUMBER_OF_SAMPLES parameter
  num_samples_param = new ParamInt("NUMBER_OF_SAMPLES");
  num_samples_param->set_helptext("The number of samples to read from the MATLAB variable. <p> If unset of if set to 0, then all the data stored in the variable is used.");
  num_samples_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(num_samples_param, CTK_PARAM_TYPE_HIDDEN | CTK_PARAM_TYPE_CONST);

  // Set up NAME parameter
  NAME_param = new ParamString("NAME");
  NAME_param->set_helptext("The MATLAB variable which to import.");
  parameters->register_parameter(NAME_param);
}


MatlabInputBlock::~MatlabInputBlock() {
}

Block* MatlabInputBlock::clone(const string &n) const{
  Block *ablock = new MatlabInputBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void MatlabInputBlock::reset() {
  SourceBlock::reset();

  matlab_startup();

  index=0;
}

void MatlabInputBlock::build_output_data_descriptors() {

  mxArray *mx_out;

  string variable_name;
  if (NAME_param->get_set_flag())
    variable_name=NAME_param->get_value();
  else
    variable_name=getname();
  
  if ((mx_out=engGetArray(ep, variable_name.c_str()))==NULL) {
    cerr << "Cannot find a matlab variable called: " << variable_name << endl;
    throw(CTKError(__FILE__, __LINE__));
  };
  
  DataDescriptor *dd = new DataDescriptor();
  const Integer *dimensions=mxGetDimensions(mx_out);
  Integer dim_count=0;
  Boolean second_time=false;
  Integer N=1;

  for (Integer j=mxGetNumberOfDimensions(mx_out)-1; j>=0; j--) {
    char dim_name[100];
    Integer npoints=dimensions[j];
    if (npoints>1) {
      if (second_time) {
	// The first dimension (time) is not passed to the data descriptor
	CTKVector axis(npoints);
	for (Integer k=0; k<npoints; ++k) axis[k]=Float(k);
	sprintf(dim_name,"%d",dim_count++);
	dd->add_outer_dimension(dim_name, axis);
      } else {
	// Time dimension;
	N=npoints;
	second_time=true;
      }
    }
    
  }

  is_sample_data=(dim_count==0);
  (*output_sockets)[0]->set_data_descriptor(dd);
  
  unlock_parameters();
  set_parameter("NUMBER_OF_SAMPLES", N);
  relock_parameters();

  if (is_sample_data) {
    // Load data into sample buffer
    Float *fp;
    sample_buffer.resize(N);
    copy(fp=mxGetPr(mx_out), fp+N, &sample_buffer[0]);
  } else {
    // Load data into vectors and vectors into vector buffer
    Integer storage=dd->get_storage();
    vector_buffer.resize(N);
    Float *fp=mxGetPr(mx_out);
    for (Integer i=0; i<N; ++i) {
      CTKVector *xv=new CTKVector(storage);
      copy(fp, fp+storage, &(*xv)[0]);
      fp+=storage;
      vector_buffer[i]=xv;
    }
  }
  
}


void MatlabInputBlock::compute() {

  if (index>=num_samples_param->get_value()) {
    set_eod();
    return;
  }
  
  if (is_sample_data) {
    (*output_sockets)[0]->write_sample(sample_buffer, index, MATLAB_BUFFER_CHUNK_SIZE);
  } else {
    (*output_sockets)[0]->write_vector(vector_buffer, index, MATLAB_BUFFER_CHUNK_SIZE);
  }
  
  index+=MATLAB_BUFFER_CHUNK_SIZE;

}

#endif



/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MatlabOutputBaseBlock                                    */
/*                                                                            */
/******************************************************************************/

MatlabOutputBaseBlock::MatlabOutputBaseBlock(const string &a_name, const string &a_type):CTKObject(a_name),Block(a_name, a_type) {

  make_input_sockets(1);
  make_output_sockets(1);

  // Set up NAME parameter
  NAME_param= new ParamString("NAME");
  NAME_param->set_helptext("The MATLAB variable name to assign to the data. <p> Note, for the MDisplay block also sets the plot title. <p> If not set then the name of the block itself is used.");
  parameters->register_parameter(NAME_param);

  // Set up DISPLAY parameter
  DISPLAY_param = new ParamBool("DISPLAY", PARAM_DEFAULT_DISPLAY);
  DISPLAY_param->set_helptext("If set ON then a MATLAB plot is generated. If set OFF data passes through the block without starting the MATLAB engine.");
  parameters->register_parameter(DISPLAY_param, CTK_PARAM_TYPE_HIDDEN);

  // Set up SHOW_MATLAB parameter
  // added by Sue 28.5.03
  SHOW_MATLAB_param = new ParamBool("SHOW_MATLAB", false);
  SHOW_MATLAB_param->set_helptext("If set ON then the MATLAB code to be executed is displayed on standard error.");
  parameters->register_parameter(SHOW_MATLAB_param, CTK_PARAM_TYPE_HIDDEN);

  // Set up PLOT parameter
  PLOT_param= new ParamString("PLOT");
  PLOT_param->set_helptext("The MATLAB command used to generate the plot. <p> A '%' symbol is used at the point where the MATLAB variable name will be substituted. e.g. PLOT=""plot(%)""");
  parameters->register_parameter(PLOT_param, CTK_PARAM_TYPE_HIDDEN);


  output_connected=false;
}

MatlabOutputBaseBlock::~MatlabOutputBaseBlock() {
  free_storage();
}

void MatlabOutputBaseBlock::reset() {
  Block::reset();

  free_storage();

#ifdef _HAS_MATLAB
  if (DISPLAY_param->get_value())  // Only start up MATLAB if the display parameter is set
    matlab_startup();
#endif
  
  sample_buffer.resize(0);
  vector_buffer.resize(0);

  output_connected=(*output_sockets)[0]->connected();
}

void MatlabOutputBaseBlock::close() {
  Block::close();
}

void MatlabOutputBaseBlock::compute() {

  if (inputs_are_all_sample_data) {
    CTKSample sample;
    (*input_sockets)[0]->get_sample(sample);
    sample_buffer.push_back(sample);
    if (output_connected) {
      (*output_sockets)[0]->put_sample(sample);
    }
  }else {
    // Store data in vector buffer to be displayed and deleted later
    // and if this is a through block then pass on a pointer
    // to *a copy* of the data
    CTKVector *xv0;
    (*input_sockets)[0]->get_vector(xv0);
    vector_buffer.push_back(xv0);
    if (output_connected)
      (*output_sockets)[0]->put_vector(new CTKVector(*xv0));
  }
  
}

#ifdef _HAS_MATLAB

// Original output_data has been split into make_var_names and output_data
// for MatlabDisplayFrameBlock - Sue
void MatlabOutputBaseBlock::make_var_names() {
  
  // Replace any '_' in blockname with '\_'  i.e. so they don't get treated as subscripts
  // when used in matlab names
  string tname=(NAME_param->get_set_flag()?NAME_param->get_value():getname());
  Integer pos=tname.size();
  while ((pos=tname.rfind('_',pos))!=-1) 
    tname.insert(tname.begin()+pos,'\\');
  title_name=tname.c_str();
  //const char* title_name=tname.c_str();

  // Make MATLAB variable name from NAME:
  // Replace all non alphanumeric with _; Strip all preceding '_' and digits.  
  string mname=(NAME_param->get_set_flag()?NAME_param->get_value():getname());

  // // the following line works with gcc-2.95.1 but not gcc-3.0 ...
  //replace_if(mname.begin(), mname.end(), boost::compose_f_gx(logical_not<bool>(), ptr_fun(std::isalnum)), '_');
  // // Use this clumsier paraphrase instead:
  for (string::iterator mp=mname.begin(),mp_end=mname.end(); mp!=mp_end; ++mp)
    if (!isalnum(*mp)) *mp='_';
  
  
  pos=mname.find_first_not_of("_0123456789");

  // If after cleaning the string is empty then revert to using the block name;
  if (pos==-1)  {
    mname=getname(); pos=0;
  }
 
  //const char* matlab_name=mname.c_str()+pos;
  matlab_name=mname.c_str()+pos;

}

#endif

#ifdef _HAS_MATLAB

void MatlabOutputBaseBlock::output_data(Boolean make_plot) {
  
  char do_axis[1024];
  Integer n_dims=(*input_sockets)[0]->get_data_descriptor()->get_n_dimensions();
  
  if (inputs_are_all_sample_data) {
    // 0-D case (i.e. a 1D plot: time)
    mxArray *mx = mxCreateDoubleMatrix(1, sample_buffer.size(), mxREAL);
    mxSetName(mx, matlab_name.c_str());
    copy(sample_buffer.begin(), sample_buffer.end(), mxGetPr(mx));
    engPutArray(ep, mx);
    
    if (make_plot) {
      int pos;
      string plot_string;
      if (!PLOT_param->get_set_flag() || (pos=(plot_string=PLOT_param->get_value()).find('%'))==-1) {
	plot_string=PARAM_DEFAULT_1D_PLOT;
	pos=plot_string.find('%');
      }

      plot_string.replace(pos, 1, matlab_name); 
      if (get_SHOW_MATLAB_param_value())
        cerr << plot_string << endl;
      engEvalString(ep,plot_string.c_str());
      sprintf(do_axis, "title('%s');", title_name.c_str());
      if (get_SHOW_MATLAB_param_value())
        cerr << do_axis << endl;
      engEvalString(ep, do_axis);
    }
    
    mxDestroyArray(mx);
    
  } else if (n_dims==1) {
    // 1-D frame case  (i.e. a 2D plot: time + 1 other D)
    const DimensionDescriptor *dd=(*input_sockets)[0]->get_data_descriptor()->get_dimension(1);
    string yname=dd->get_name();
    CTKVector yaxis=dd->get_axis();
    Integer N=yaxis.size();

    mxArray *mx = mxCreateDoubleMatrix(N, vector_buffer.size(), mxREAL);
    mxSetName(mx, matlab_name.c_str());

    Float *dest=(Float *)mxGetPr(mx);
    
    for (vector<CTKVector*>::iterator vp=vector_buffer.begin(); vp!=vector_buffer.end(); ++vp) {
      dest=copy(&((**vp)[0]), &((**vp)[N]), dest);
    }
    dest=(Float *)mxGetPr(mx);
    engPutArray(ep, mx);

    if (make_plot) {
      int pos;
      string plot_string;
      if (!PLOT_param->get_set_flag() || (pos=(plot_string=PLOT_param->get_value()).find('%'))==-1) {
	plot_string=PARAM_DEFAULT_2D_PLOT;
	pos=plot_string.find('%');
      }
      plot_string.replace(pos, 1, matlab_name); 
 
      mxDestroyArray(mx);
      
      Float xmin=0, xmax=vector_buffer.size();
      Float ymin=yaxis[0], ymax=yaxis[N-1];
      if (ymin==ymax) {ymin-=0.5; ymax+=0.5;} // This happens if data is really only 1 dimensional;
      sprintf(do_axis,";axis([%f %f %f %f]);ylabel('%s'); xlabel('time');;title('%s');set(h,'XData',[%f %f]); set(h,'YData',[%f %f])",  xmin, xmax, ymin, ymax, yname.c_str(), title_name.c_str(), xmin, xmax, ymin, ymax);
      if (get_SHOW_MATLAB_param_value())
        cerr << plot_string << do_axis << endl;
      engEvalString(ep,(plot_string+do_axis).c_str());
    }

    //    mxDestroyArray(mx);   // JON - there is a problem here

  } else {
    // n-D case
    cerr << "Trying to display data with more than 2 dimensions. Not yet implemented." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
}

#endif

void MatlabOutputBaseBlock::free_storage() {
  for (UInteger i=0; i<vector_buffer.size(); ++i)
    if (vector_buffer[i]!=NULL) delete vector_buffer[i];
  vector_buffer.resize(0);
}

Boolean MatlabOutputBaseBlock::get_DISPLAY_param_value() {
  return DISPLAY_param->get_value();
}

Boolean MatlabOutputBaseBlock::get_SHOW_MATLAB_param_value() {
  return SHOW_MATLAB_param->get_value();
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MatlabDisplayBlock                                       */
/*                                                                            */
/******************************************************************************/

const string MatlabDisplayBlock::type_name = "MDisplay";
const string MatlabDisplayBlock::help_text = MDISPLAY_BLOCK_HELP_TEXT;


MatlabDisplayBlock::MatlabDisplayBlock(const string &a_name):CTKObject(a_name),MatlabOutputBaseBlock(a_name, type_name) {

  // Set up BEFORE_PLOT parameter
  BEFORE_PLOT_param= new ParamString("BEFORE_PLOT");
  BEFORE_PLOT_param->set_helptext("MATLAB commands to be executed immediately before the plot is drawn.");
  parameters->register_parameter(BEFORE_PLOT_param);

  // Set up AFTER_PLOT parameter
  AFTER_PLOT_param= new ParamString("AFTER_PLOT");
  AFTER_PLOT_param->set_helptext("MATLAB commands to be executed immediately after the plot is drawn.");
  parameters->register_parameter(AFTER_PLOT_param);

  unset_parameter_hidden("DISPLAY"); 
  unset_parameter_hidden("SHOW_MATLAB"); 
  unset_parameter_hidden("PLOT"); 
}

MatlabDisplayBlock::~MatlabDisplayBlock() {
}

Block* MatlabDisplayBlock::clone(const string &n) const{
  Block *ablock = new MatlabDisplayBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void MatlabDisplayBlock::reset() {
  MatlabOutputBaseBlock::reset();
}



void MatlabDisplayBlock::close() {

#ifdef _HAS_MATLAB

  if (get_DISPLAY_param_value()) {  // Only display if the DISPLAY param is set
    
    if (BEFORE_PLOT_param->get_set_flag()) {
      if (get_SHOW_MATLAB_param_value())
        cerr << BEFORE_PLOT_param->get_value() << endl;
      engEvalString(ep,(BEFORE_PLOT_param->get_value()).c_str());
    }
    
    make_var_names();  // set up variable names for Matlab 
    output_data(true);  // do Matlab plots (true - means data is displayed)
    
    if (AFTER_PLOT_param->get_set_flag()) {
      if (get_SHOW_MATLAB_param_value())
        cerr << AFTER_PLOT_param->get_value() << endl;
      engEvalString(ep,(AFTER_PLOT_param->get_value()).c_str());
    }
    
  }

#endif

  MatlabOutputBaseBlock::close();
}



/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MatlabDisplayFrameBlock  - Sue Harding                   */
/*                                                                            */
/******************************************************************************/

const string MatlabDisplayFrameBlock::type_name = "MDisplayFrame";
const string MatlabDisplayFrameBlock::help_text = "Display frame (3-D) data using Matlab";

MatlabDisplayFrameBlock::MatlabDisplayFrameBlock(const string &a_name):CTKObject(a_name),MatlabOutputBaseBlock(a_name, type_name) {

  // Set up XAXIS parameter
  XAXIS_param = new ParamString("XAXIS");	// Input axis to use as X axis
  XAXIS_param->set_helptext("The name of the axis to display as the X axis. <p> If left unset then the inner axis is used.");
  parameters->register_parameter(XAXIS_param);

  // Set up FRAMELIST parameter
  FRAMELIST_param = new ParamString("FRAMELIST");	// List of frames to display
  FRAMELIST_param->set_helptext("The numbers of the frames to display (default all frames).");
  FRAMELIST_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(FRAMELIST_param);

  // Set up BEFORE_EACH_FRAME parameter
  BEFORE_EACH_FRAME_param= new ParamString("BEFORE_EACH_FRAME");
  BEFORE_EACH_FRAME_param->set_helptext("MATLAB commands to be executed immediately before each frame is drawn.");
  parameters->register_parameter(BEFORE_EACH_FRAME_param);

  // Set up AFTER_EACH_FRAME parameter
  AFTER_EACH_FRAME_param= new ParamString("AFTER_EACH_FRAME");
  AFTER_EACH_FRAME_param->set_helptext("MATLAB commands to be executed immediately after each frame is drawn.");
  parameters->register_parameter(AFTER_EACH_FRAME_param);

  // Set up BEFORE_PLOT parameter
  BEFORE_PLOT_param= new ParamString("BEFORE_PLOT");
  BEFORE_PLOT_param->set_helptext("MATLAB commands to be executed immediately before the first frame is drawn.");
  parameters->register_parameter(BEFORE_PLOT_param);

  // Set up AFTER_PLOT parameter
  AFTER_PLOT_param= new ParamString("AFTER_PLOT");
  AFTER_PLOT_param->set_helptext("MATLAB commands to be executed immediately after the last frame is drawn.");
  parameters->register_parameter(AFTER_PLOT_param);

  unset_parameter_hidden("DISPLAY");
  unset_parameter_hidden("SHOW_MATLAB");
  unset_parameter_hidden("PLOT");


}

MatlabDisplayFrameBlock::~MatlabDisplayFrameBlock() {
}

Block* MatlabDisplayFrameBlock::clone(const string &n) const{
  Block *ablock = new MatlabDisplayFrameBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}


void MatlabDisplayFrameBlock::reset() {
  MatlabOutputBaseBlock::reset();
}

void MatlabDisplayFrameBlock::close() {

#ifdef _HAS_MATLAB

  if (get_DISPLAY_param_value()) {  // Only display if the DISPLAY param is set

    if (BEFORE_PLOT_param->get_set_flag()) {
      if (get_SHOW_MATLAB_param_value())
        cerr << BEFORE_PLOT_param->get_value() << endl;
      engEvalString(ep,(BEFORE_PLOT_param->get_value()).c_str());
    }

    MatlabOutputBaseBlock::make_var_names();  // set up variable names for Matlab 
    output_data(true);  // do Matlab plots (true - means data is displayed)


    if (AFTER_PLOT_param->get_set_flag()) {
      if (get_SHOW_MATLAB_param_value())
        cerr << AFTER_PLOT_param->get_value() << endl;
      engEvalString(ep,(AFTER_PLOT_param->get_value()).c_str());
    }

  }

#endif

  MatlabOutputBaseBlock::close();
}

#ifdef _HAS_MATLAB

void MatlabDisplayFrameBlock::output_data(Boolean make_plot) {
  
  Integer n_dims=(*input_sockets)[0]->get_data_descriptor()->get_n_dimensions();
  char do_axis[1024];

  if (inputs_are_all_sample_data || n_dims<1)
    {
    // 0-D case (i.e. a 1D plot: time)
    cerr << "Error in network at block: " << getname() << endl;
    cerr << "Use MDisplay for sample data " << endl;
    throw(CTKError(__FILE__, __LINE__));
    }
  else  if (n_dims > 2)
    {
    // n-D case
    cerr << "Error in network at block: " << getname() << endl;
    cerr << "Trying to display data with more than 3 dimensions. Not yet implemented." << endl;
    throw(CTKError(__FILE__, __LINE__));
    }

  ////////////////////////////////////////////////////////////////////////
  // Data may have one or two dimensions, plus time (i.e. data flow)
  // One figure is produced per input frame

  // Prepare the axes
  string xaxisname,yaxisname;
  Float xmin=0.0, xmax=0.0;
  Float ymin=0.0, ymax=0.0;
  Integer Nin=1,Nout=1;	// size of inner and outer dimensions
  string inaxisname;
  string outaxisname;
  Boolean xisinner=true;
  Float inmin=0.0,inmax=0.0,outmin=0.0,outmax=0.0;

  if (n_dims==1) 
    {
    // 1-D frame case  (i.e. one 1D plot per input frame)
    // cf. MDisplay which does an image plot with time as x axis
    const DimensionDescriptor *dd=(*input_sockets)[0]->get_data_descriptor()->get_dimension(1);
    xaxisname=dd->get_name();
    CTKVector xaxis=dd->get_axis();
    Nin=xaxis.size();
    xmin=xaxis[0];
    xmax=xaxis[Nin-1];
    yaxisname="";	// Don't label y axis
    Nout=1;	// Each frame is a 1-D vector
    }

  else // (n_dims==2) 
    {
    // 2-D frame case  (i.e. one 2D plot per input frame)
    // Do one plot per set of x/y values determined from parameters
    // Default is x=inner dim and y=outer dim

    // Assign parameters for each dimension - x, y, figure
    // Get the properties of the inner and outer dimensions
    const DimensionDescriptor *ddin;
    const DimensionDescriptor *ddout;
    ddin=(*input_sockets)[0]->get_data_descriptor()->get_inner_dimension();
    inaxisname=ddin->get_name();
    CTKVector inaxis=ddin->get_axis();
    Nin=inaxis.size();		// get the inner dimension size
    inmin=inaxis[0];
    inmax=inaxis[Nin-1];
    ddout=(*input_sockets)[0]->get_data_descriptor()->get_outer_dimension();
    outaxisname=ddout->get_name();
    CTKVector outaxis=ddout->get_axis();
    Nout=outaxis.size();	// get the outer dimension size
    outmin=outaxis[0];
    outmax=outaxis[Nout-1];

    // If x axis parameter not specified, define inner dimension as default x
    // y axis is the other dimension
    if (XAXIS_param->get_set_flag())
      {
      // x axis parameter is defined - check it is valid
      xaxisname=XAXIS_param->get_value();
      if (xaxisname==outaxisname)
	xisinner=false;
      else
        if (xaxisname!=inaxisname)
          {
          cerr << "Error in network at block: " << getname() << endl;
          cerr << "Cannot find axis named: " << xaxisname << endl;
          throw(CTKError(__FILE__, __LINE__));
          }
      }
    }


  // Sort out other parameters used for the Matlab figure
  // and set up plot strings as necessary
  // Don't bother if no plot is required
  char frameliststr[256];	// input list of frames to display
  list<Integer> framelist;	// processed list of frames to display
  Integer frameno;		// frame counter
  Boolean isflist=false;		// flag=true if frame list was supplied
  string plot_string;		// Main Matlab commands
  char plot_str_before[256]="";		// Matlab commands to precede main
  char plot_str_after[256]="";		// Matlab commands to follow main


  if (make_plot)
    {
    if (n_dims==2)
      {
      // Sort out x and y axis parameters from inner and outer dimensions
      xmin=inmin;
      xmax=inmax;
      ymin=outmin;
      ymax=outmax;
      xaxisname=inaxisname;
      yaxisname=outaxisname;
      if (!xisinner)	// Axes are transposed
        {
        xmin=outmin;
        xmax=outmax;
        ymin=inmin;
        ymax=inmax;
        xaxisname=outaxisname;
        yaxisname=inaxisname;
        }
      }

    if (ymin==ymax) {ymin-=0.5; ymax+=0.5;} // This happens if data is really only 1 dimensional;

    // Get the list of frames to display (default all)
    isflist=(FRAMELIST_param->get_set_flag());
    if (isflist)
      {
      strcpy(frameliststr, (FRAMELIST_param->get_value()).c_str());
      // Parse the comma separated list of frame numbers
      char *nextframe=strtok(frameliststr, ",");
      while (nextframe!= NULL)
        {
        frameno=atoi(nextframe);
        if (frameno > 0)
          framelist.push_back(frameno);	// add frame no. to list
        nextframe=strtok(NULL, ",");
        }
      framelist.sort();
      } // end of frame list processing

  
    // Replace '%' in the plot command with the name of the matrix 
    // If axes are swapped, the matrix is transposed
    int pos;
    if (!PLOT_param->get_set_flag() || (pos=(plot_string=PLOT_param->get_value()).find('%'))==-1) 
      {
      if (n_dims==1)
        plot_string=PARAM_DEFAULT_1D_PLOT;
      else
        plot_string=PARAM_DEFAULT_2D_PLOT;
      pos=plot_string.find('%');
      }
    plot_string.replace(pos, 1, matlab_name); 
  
    if (n_dims==2 && !xisinner)
      plot_string.insert(pos+matlab_name.size(),"'"); // Add "'" to transpose
         
    // Add commands to do before and after plot command for each frame
    if (BEFORE_EACH_FRAME_param->get_set_flag()) 
      sprintf(plot_str_before, "%s;", (BEFORE_EACH_FRAME_param->get_value()).c_str());
    if (AFTER_EACH_FRAME_param->get_set_flag()) 
      sprintf(plot_str_after, "%s;", (AFTER_EACH_FRAME_param->get_value()).c_str());
    
    // Add "drawnow" command to force display of each frame
    strcat(plot_str_after, "drawnow;");

    // Prepare Matlab commands to scale and label the axes
    char scalexdata[256], scaleydata[256];
    // Rescale XData (which may be 2-D)
    sprintf(scalexdata, "oxd=get(h,'XData');minoxd=min(min(oxd));maxoxd=max(max(oxd));nxd=(oxd-minoxd)*(%f-%f)/(maxoxd-minoxd)+%f;set(h,'XData',nxd);", xmax, xmin, xmin);
    if (n_dims==1)
      sprintf(do_axis,";ylabel('%s');xlabel('%s');%s;ax=axis;ax(1:2)=[%f %f];axis(ax);",  yaxisname.c_str(), xaxisname.c_str(), scalexdata, xmin, xmax);
    else 	// (n_dims==2)
      {
      sprintf(scaleydata, "oyd=get(h,'YData');minoyd=min(min(oyd));maxoyd=max(max(oyd));nyd=(oyd-minoyd)*(%f-%f)/(maxoyd-minoyd)+%f;set(h,'YData',nyd);", ymax, ymin, ymin);
      sprintf(do_axis,";ylabel('%s');xlabel('%s');%s;%s;axis([%f %f %f %f]);", yaxisname.c_str(), xaxisname.c_str(), scalexdata, scaleydata, xmin, xmax, ymin, ymax);
      }
    } // end (if make_plot)
  
  // Do the Matlab stuff
  // Create matrix to contain data for each figure
  mxArray *mx = mxCreateDoubleMatrix(Nout, Nin, mxREAL);
  mxSetName(mx, matlab_name.c_str());

  Float *dest=(Float *)mxGetPr(mx);
    
  //////////////////////////////////////
  // Main loop for processing each frame
  //////////////////////////////////////
  list<Integer>::iterator framep=framelist.begin();
  vector<CTKVector*>::iterator vp;
  Integer skipframes=0;

  // Find initial no. of frames to skip if list provided
  if (isflist)
    {
    frameno=*framep;
    skipframes=*framep - 1;
    }
  else
    frameno=1;

  // Process each vector
  vp=vector_buffer.begin(); 
  advance(vp, skipframes);
  while (vp!=vector_buffer.end())
    {
    // Copy the input vector into a temporary location
    dest=copy(&((**vp)[0]), &((**vp)[Nin*Nout]), dest);
    
    // Set up matrix
    dest=(Float *)mxGetPr(mx);
    engPutArray(ep, mx);

    // Display the figure if required
    if (make_plot)
      {
      // Prepare the figure title
      char plot_str_title[80]="";
      sprintf(plot_str_title,"title('%s:%d'); ", title_name.c_str(), frameno);
      // Put together all the commands
      if (get_SHOW_MATLAB_param_value())
        cerr << plot_str_before << plot_string << do_axis << plot_str_title << plot_str_after << endl;
      engEvalString(ep,(plot_str_before+plot_string+do_axis+plot_str_title+plot_str_after).c_str());

      } // end make_plot (if)

    if (isflist)	// frame list has been supplied
      {
      framep++;
      // Stop if all required frames have been displayed
      if (framep==framelist.end())
        break;
      skipframes=*framep - frameno;	// Calculate frames to skip
      }
    else
      skipframes=1;	// Only skip one frame if all required

    advance(vp, skipframes);	// Skip however many frames are required
    frameno+=skipframes;	// current frame no.


    } // end (while)

    //mxDestroyArray(mx);
  
}

#endif


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MatlabOutputBlock                                        */
/*                                                                            */
/******************************************************************************/
#ifdef _HAS_MATLAB

const string MatlabOutputBlock::type_name = "MOutput";
const string MatlabOutputBlock::help_text = MOUTPUT_BLOCK_HELP_TEXT;

MatlabOutputBlock::MatlabOutputBlock(const string &a_name):CTKObject(a_name),MatlabOutputBaseBlock(a_name, type_name) {
}

MatlabOutputBlock::~MatlabOutputBlock() {
}

Block* MatlabOutputBlock::clone(const string &n) const{
  Block *ablock = new MatlabOutputBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void MatlabOutputBlock::reset() {
  MatlabOutputBaseBlock::reset();
}


void MatlabOutputBlock::close() {

  make_var_names();  // make Matlab variables 
  output_data(false);  // false - means data is output to matlab but not displayed

  MatlabOutputBaseBlock::close();
}

#endif





/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MatlabProcessBlock                                       */
/*                                                                            */
/******************************************************************************/

#ifdef _HAS_MATLAB

const string MatlabProcessBlock::type_name = "MProcess";
const string MatlabProcessBlock::help_text =MPROCESS_BLOCK_HELP_TEXT ;

MatlabProcessBlock::MatlabProcessBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {

  make_input_sockets(1, true);
  make_output_sockets(1, true);

  // Set up PROCESS parameter
  PROCESS_param= new ParamString("PROCESS");
  PROCESS_param->set_helptext("A string containg the MATLAB commands to be applied to each frame of input data. <p> Block inputs are stored in MATLAB variables named 'in1', 'in2' etc <p> Outputs should be stored in variable named 'out1', 'out2' etc");
  parameters->register_parameter(PROCESS_param);

  // Set up INITIALISE parameter
  INITIALISE_param= new ParamString("INITIALISE");
  INITIALISE_param->set_helptext("MATLAB commands to run before data flow commences.");
  parameters->register_parameter(INITIALISE_param);

}

Block* MatlabProcessBlock::clone(const string &n) const{
  Block *ablock = new MatlabProcessBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void MatlabProcessBlock::reset() {
  char var_name[100];
  
  Block::reset();

  matlab_startup();
  
  // Make matlab variables for input data
  input_string="(";
  for (Integer i=0; i<input_sockets->size(); ++i) {

    const DataDescriptor *dd=(*input_sockets)[i]->get_data_descriptor();
    Integer nDims=dd->get_n_dimensions();
    Integer N=dd->get_storage();
    vector<Integer> dims=dd->get_dimension_sizes();
    
    if (N==1) {
      mx_in[i] = mxCreateDoubleMatrix(1, 1, mxREAL);
    } else {
      if (nDims==1) {
	mx_in[i] = mxCreateDoubleMatrix(1, dims[0], mxREAL);
      } else if (nDims==2) {
	mx_in[i] = mxCreateDoubleMatrix(dims[0], dims[1], mxREAL);
      }
    }
    sprintf(var_name,"in%d",i+1);
    mxSetName(mx_in[i], var_name);
    input_string+=var_name;
    input_string+=',';
  }
  input_string.erase(input_string.end()-1);
  input_string+=')';

  // Make matlab variables for output data
  output_string="[";
  for (Integer i=0; i<output_sockets->size(); ++i) {

    sprintf(var_name,"out%d",i+1);
    output_string+=var_name;
    output_string+=',';
  }
  output_string.erase(output_string.end()-1);
  output_string+=']';
  
  //  do_process=output_string+'='+PROCESS_param->get_value()+input_string;
  do_initialise=INITIALISE_param->get_value();
  do_process=PROCESS_param->get_value();

  // Manipulate the strings to make all variable names they contain unique
  string uniq(getfullname());
  replace(uniq.begin(), uniq.end(), ':', '_'); // Matlab won't like ':'s so replace them with '_'s
  uniq+='_';
  variable_name_uniquification(do_initialise, do_process, uniq);
  
}

void MatlabProcessBlock::build_output_data_descriptors() {
  // Need to try the command with dummy data and then see what the output data types
  // turn out to be so we can set the output data descriptors appropriately

  for (Integer i=0; i<input_sockets->size(); ++i) {
    engPutArray(ep, mx_in[i]);
  }
  
  if (INITIALISE_param->get_set_flag()) {
      //if (get_SHOW_MATLAB_param_value())
        //cerr << do_initialise << endl;
    engEvalString(ep, do_initialise.c_str());
  }

      //if (get_SHOW_MATLAB_param_value())
        //cerr << do_process << endl;
  engEvalString(ep, do_process.c_str());
  
  for (Integer i=0; i<output_sockets->size(); ++i) {
    mxArray *mx_out;
    char var_name[100];
    sprintf(var_name,"out%d",i+1);
    if ((mx_out=engGetArray(ep, var_name))==NULL) {cerr << "Internal error. MatlabProcessBlock::build_output_data_descriptors:Eek99" << endl; exit(-1);};

    DataDescriptor *dd = new DataDescriptor();

    const Integer *dimensions=mxGetDimensions(mx_out);
    Integer dim_count=1;
    for (Integer j=0; j<mxGetNumberOfDimensions(mx_out); ++j) {
      char dim_name[100];
      Integer npoints=dimensions[j];
      if (npoints>1) {
	CTKVector axis(npoints);
	for (Integer k=0; k<npoints; ++k) axis[k]=Float(k);
	sprintf(dim_name,"%d",dim_count++);
	dd->add_outer_dimension(dim_name, axis);
      }
    }
    (*output_sockets)[i]->set_data_descriptor(dd);

  }

  if (INITIALISE_param->get_set_flag()) {
      //if (get_SHOW_MATLAB_param_value())
        //cerr << do_initialise << endl;
    engEvalString(ep, do_initialise.c_str());
  }
  
}

void MatlabProcessBlock::compute() {

  for (Integer i=0; i<input_sockets->size(); ++i) {
    const DataDescriptor *dd=(*input_sockets)[i]->get_data_descriptor();
    Integer N=dd->get_storage();
    if (N==1) {
      Float xsi;
      (*input_sockets)[i]->get_sample(xsi);
      *mxGetPr(mx_in[i])=xsi;
    } else {
      CTKVector *xvi;
      (*input_sockets)[i]->get_vector(xvi);
      copy(&(*xvi)[0], &(*xvi)[N], mxGetPr(mx_in[i]));
      delete xvi;
    }
    engPutArray(ep, mx_in[i]);
  }

      //if (get_SHOW_MATLAB_param_value())
        //cerr << do_process << endl;
  engEvalString(ep, do_process.c_str());

  for (Integer i=0; i<output_sockets->size(); ++i) {
    mxArray *mx_out;
    
    char var_name[100];
    sprintf(var_name,"out%d",i+1);
    if ((mx_out=engGetArray(ep, var_name))==NULL) {cerr << "Internal Error. MatlabProcessBlock::compute: Eek99" << endl; exit(-1);};
    const DataDescriptor *dd=(*output_sockets)[i]->get_data_descriptor();
    Integer N=dd->get_storage();
    if (N==1) {
      Float xsi;
      xsi=mxGetScalar(mx_out);
      (*output_sockets)[i]->put_sample(xsi);
    } else {
      Float *fp;
      CTKVector *xvi=new CTKVector(N);
      copy(fp=mxGetPr(mx_out), fp+N, &(*xvi)[0]);
      (*output_sockets)[i]->put_vector(xvi);
    }
    mxDestroyArray(mx_out);
  }

  
}


void MatlabProcessBlock::variable_name_uniquification(string &initialise, string &process, string uniq) {
  //
  // Searches for variable names that are assigned in the initialise
  // string and then replaces these names where they occur in the
  // initialise and process strings with a new name made by prefixing
  // the string 'uniq'
  //
  
  Integer length=initialise.size();
  Boolean in_string=false;
  Boolean in_name=false;
  Boolean looking=true;

  string var;
  var.resize(0);
  vector<string> var_list(0);

  // First pass to remove == <= >= ~= 
  string initcopy=initialise;
  remove_unwanted_equals_signs(initcopy);

  // Search for assignments in the initialise string
  for (Integer i=0; i<length; ++i) {
    Character ch=initcopy[i];
    if (ch=='"') {
      in_string=!in_string;
    } else if (!in_string && (ch==';' || ch==',')) {
      in_name=false; looking=true;
      var.resize(0);
    } else if (!in_string && ch=='=') {
      if (var.size()!=0) {
	var_list.push_back(var);
	var.resize(0);
	looking=false;
      }
    } else if (!in_string && (isalnum(ch)||ch=='_')) {
      if (looking) {
	var+=ch;
	in_name=true;
      }
    } else {
      var.resize(0);
    }
  }

  // Replace the variable names with their prefixed names
  for (UInteger i=0; i<var_list.size(); ++i) {
    replace_name(initialise, var_list[i], uniq+var_list[i]);
    replace_name(process, var_list[i], uniq+var_list[i]);
  }  
}


//
// Replace all the <=, ==, >=, ~= with ##. This prevents the '='s
// confusing the rather clunky code that searches for assignments
//

void MatlabProcessBlock::remove_unwanted_equals_signs(string &line) {
  char last=line[0];
  for (UInteger i=1; i<line.size(); ++i) {
    Character ch=line[i];
    if (ch=='=' && (last=='=' || last=='<' || last=='>' || last=='~')) {
      line[i]='#'; line[i-1]='#';
    }
    last=ch;
  }
}

//
// Replace all occurances of from_name with to_name. Not a simple
// substring replacement as it makes sure that from_name is a
// separate token and not a substring embedded in a longer name
//

void MatlabProcessBlock::replace_name(string &line, string from_name, string to_name) {
  Boolean in_string=false;
  Boolean matching=false;
  UInteger match_pos=0;
  Integer start_pos=0;
  
  for (UInteger i=0; i<line.size(); ++i) {
    Character ch=line[i];
    if (ch=='"') {
      in_string=!in_string;
    }else if (!in_string && (isalnum(ch)||ch=='_')) {
      if (match_pos==0) {
	matching=true; start_pos=i;
      }
      if (match_pos>from_name.size())
	matching=false;
      
      if (matching && ch!=from_name[match_pos]) {
	matching=false;
      }
      ++match_pos;      
    } else {
      if (matching && match_pos==from_name.size()) {
	line.replace(start_pos, from_name.size(), to_name);
	i+=(to_name.size()-from_name.size());
	matching=false;
      }
      match_pos=0;
    }
  }
  
  if (matching && match_pos==from_name.size()) {
    line.replace(start_pos, from_name.size(), to_name);
    matching=false;
  }
  
}

#endif  // _HAS_MATLAB

/* End of ctk_matlab.cpp */
 
