/******************************************************************************/
/*									      */
/*	ctk_noise_estimation.cpp	    	      	      	              */
/*						        		      */
/*	Noise estimation and spectral subtraction                             */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			      	      */
/*									      */
/******************************************************************************/
 
#include "ctk-config.h"

#include <cmath>
#include <vector>
#include <algorithm>
#include <functional>
#include <numeric>

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_param.hh"
#include "ctk_socket.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_noise_estimation.hh"

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: NoiseEstimationBlock                                     */
/*                                                                            */
/******************************************************************************/

NoiseEstimationBlock::NoiseEstimationBlock(const string &a_name, const string &a_block_type):Block(a_name,  a_block_type) {
  make_input_sockets(1);
  make_output_sockets(3);

  output_sockets->set_description("out1", "estimated noise");
  output_sockets->set_description("out2", "estimated signal");
  output_sockets->set_description("out3", "estimated noise variance");

  // Set up NFRAMES parameter
  NFRAMES_param= new ParamInt("NFRAMES");
  NFRAMES_param->set_helptext("The number of frames - taken from the beginning of the signal - from which the initial noise estimate is made.");
  NFRAMES_param->install_validator(new Validator(Validator::VLOWER, 1.0));
  parameters->register_parameter(NFRAMES_param);
  
  IGNORE_LARGE_param= new ParamBool("IGNORE_LARGE", 0);
  IGNORE_LARGE_param->set_helptext("If set to true then stationary estimate if made from the mean of the smallest 50% of the values in the first N frames.");
  parameters->register_parameter(IGNORE_LARGE_param);
}

NoiseEstimationBlock::~NoiseEstimationBlock() {
}

void NoiseEstimationBlock::reset() {
  Block::reset();

  
   // Check validity of input data
  const DataDescriptor *dd=(*input_sockets)[0]->get_data_descriptor();
  Integer nDims=dd->get_n_dimensions();
  if (nDims!=1) {
    cerr << "Error in network at block: " << getname() << endl;
    cerr << "Attempting to apply noise estimation to ";
    if (nDims==0)
      cerr << "sample data" << endl;
    else
      cerr << nDims << "-dimensional frame data." << endl;
    cerr << "Noise estimation can only be applied to 1 dimensional frame data." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  
  frame_count=NFRAMES_param->get_value();

  frame_buffer.resize(0);
}

Boolean NoiseEstimationBlock::is_initialising() {return frame_count>0;}

void NoiseEstimationBlock::initialise(CTKVector *noise_estimate, CTKVector *noise_estimate_variance) {

  CTKVector *invec;
  (*input_sockets)[0]->get_vector(invec);
  frame_buffer.push_back(invec);

  frame_count--;

  if (frame_count==0) {
    make_initial_estimate(noise_estimate, noise_estimate_variance);
    output_initial_frames(noise_estimate, noise_estimate_variance);
  } 
}

void NoiseEstimationBlock::make_initial_estimate(CTKVector *noise_estimate, CTKVector *noise_estimate_variance) {

  noise_estimate_variance->clear();
  noise_estimate_variance->resize(frame_buffer[0]->size(), 0.0);

  if (IGNORE_LARGE_param->get_value()) {
    if (calculate_buffer_mean_of_lowest(frame_buffer, noise_estimate)==CTK_FAILURE) {
      cerr << "iternal error - ctk_noise_estimation buffer empty\n";
      throw(CTKError(__FILE__, __LINE__));
    }
  } else {
    if (calculate_buffer_mean(frame_buffer, noise_estimate)==CTK_FAILURE) {
      cerr << "iternal error - ctk_noise_estimation buffer empty\n";
      throw(CTKError(__FILE__, __LINE__));
    }
  }

  
  CTKVector buffer;
  buffer.resize(frame_buffer[0]->size(), 0.0);
  
  for (UInteger i=0; i<frame_buffer.size(); ++i) {
    //  x - mean(x)
    transform(frame_buffer[i]->begin(), frame_buffer[i]->end(), noise_estimate->begin(), buffer.begin(), minus<Float>());

    // (x-mean(x))^2
    transform(buffer.begin(), buffer.end(), buffer.begin(), buffer.begin(), multiplies<Float>());

    // Sigma((x-mean(x))^2)
    transform(noise_estimate_variance->begin(), noise_estimate_variance->end(), buffer.begin(), noise_estimate_variance->begin(), plus<Float>());
  }
  
  // 1/N Sigma((x-mean(x))^2)
  transform(noise_estimate_variance->begin(), noise_estimate_variance->end(), noise_estimate_variance->begin(), bind2nd(divides<Float>(),frame_buffer.size()));
  
}


CTKStatus NoiseEstimationBlock::calculate_buffer_mean(const vector <CTKVector *> &buffer, CTKVector *mean) const {
  
  mean->clear();

  if (buffer.size()==0) return CTK_FAILURE;
  
  mean->resize(buffer[0]->size(), 0.0);

  for (UInteger i=0; i<buffer.size(); ++i) 
    transform(mean->begin(), mean->end(), buffer[i]->begin(), mean->begin(), plus<Float>());
  
  transform(mean->begin(), mean->end(), mean->begin(), bind2nd(divides<Float>(),buffer.size()));

  return CTK_SUCCESS;
}

// Calculate mean of lowest half of values in the buffer
CTKStatus NoiseEstimationBlock::calculate_buffer_mean_of_lowest(const vector <CTKVector *> &buffer, CTKVector *mean) const {
  
  mean->clear();

  if (buffer.size()==0) return CTK_FAILURE;

  int buffer_size=buffer.size();
  int buffer_half_size=buffer_size/2;
  int vector_size=buffer[0]->size();

  for (int i=0; i<vector_size; ++i) {
    vector<Float> tmp_vector;
    for (int j=0; j<buffer_size; ++j) {
      tmp_vector.push_back((*buffer[j])[i]);
    }
    sort(tmp_vector.begin(), tmp_vector.end());
    mean->push_back(accumulate(tmp_vector.begin(), tmp_vector.begin()+buffer_half_size, 0.0)/buffer_half_size);  // JON
  }
  
  return CTK_SUCCESS;
}




void NoiseEstimationBlock::output_initial_frames(CTKVector *noise_estimate, CTKVector *noise_estimate_variance) {
  if (inputs_are_all_sample_data) {
    for (Integer i=0; i<NFRAMES_param->get_value(); ++i) {
      (*output_sockets)[0]->put_sample((*noise_estimate)[0]);
      (*output_sockets)[1]->put_sample((*frame_buffer[i])[0]-(*noise_estimate)[0]);
      (*output_sockets)[2]->put_sample((*noise_estimate_variance)[0]);
    }
  } else {
    for (Integer i=0; i<NFRAMES_param->get_value(); ++i) {
      (*output_sockets)[0]->put_vector(new CTKVector(*noise_estimate));

      transform(frame_buffer[i]->begin(), frame_buffer[i]->end(), noise_estimate->begin(), frame_buffer[i]->begin(), minus<Float>());

      (*output_sockets)[1]->put_vector(frame_buffer[i]);
      (*output_sockets)[2]->put_vector(new CTKVector(*noise_estimate_variance));
    }
  }
  
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: StationaryNoiseEstimationBlock                           */
/*                                                                            */
/******************************************************************************/

const string StationaryNoiseEstimationBlock::type_name = "StationaryNoiseEstimation";
const string StationaryNoiseEstimationBlock::help_text = STATIONARY_NOISE_ESTIMATION_BLOCK_HELP_TEXT;

StationaryNoiseEstimationBlock::StationaryNoiseEstimationBlock(const string &a_name):CTKObject(a_name),NoiseEstimationBlock(a_name, type_name) {
  
}

Block* StationaryNoiseEstimationBlock::clone(const string &n) const{
  Block *ablock = new StationaryNoiseEstimationBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void StationaryNoiseEstimationBlock::reset() {
  NoiseEstimationBlock::reset();

}
 
void StationaryNoiseEstimationBlock::compute() {

  if (is_initialising()) {
    initialise(&noise_estimate, &noise_estimate_variance);
  } else {
    if (inputs_are_all_sample_data) {
      CTKSample x;
      (*input_sockets)[0]->get_sample(x);
      (*output_sockets)[0]->put_sample(noise_estimate[0]);
      (*output_sockets)[1]->put_sample(x-noise_estimate[0]);
      (*output_sockets)[2]->put_sample(noise_estimate_variance[0]);
    } else {
      CTKVector *invec;
      (*input_sockets)[0]->get_vector(invec);
      (*output_sockets)[0]->put_vector(new CTKVector(noise_estimate));

      transform(invec->begin(), invec->end(), noise_estimate.begin(), invec->begin(), minus<Float>());

      (*output_sockets)[1]->put_vector(invec);
      (*output_sockets)[2]->put_vector(new CTKVector(noise_estimate_variance));
    }
  }
  
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: HirschWANoiseEstimationBlock                             */
/*                                                                            */
/******************************************************************************/

const string HirschWANoiseEstimationBlock::type_name = "HirschWANoiseEstimation";
const string HirschWANoiseEstimationBlock::help_text =HIRSCH_WA_NOISE_ESTIMATION_BLOCK_HELP_TEXT ;

HirschWANoiseEstimationBlock::HirschWANoiseEstimationBlock(const string &a_name):CTKObject(a_name),NoiseEstimationBlock(a_name, type_name) {

  // Set up ALPHA parameter
  ALPHA_param= new ParamFloat("ALPHA", PARAM_DEFAULT_HIRSCH_WA_ALPHA);
  ALPHA_param->set_helptext("The alpha parameter employed in the Hirsch algorithm. See Block's documentation.");
  parameters->register_parameter(ALPHA_param);
  

  // Set up BETA parameter
  BETA_param= new ParamFloat("BETA", PARAM_DEFAULT_HIRSCH_WA_BETA);
  BETA_param->set_helptext("The beta parameter employed in the Hirsch algorithm. See Block's documentation.");
  parameters->register_parameter(BETA_param);


}

Block* HirschWANoiseEstimationBlock::clone(const string &n) const{
  Block *ablock = new HirschWANoiseEstimationBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void HirschWANoiseEstimationBlock::reset() {
  NoiseEstimationBlock::reset();

  alpha=ALPHA_param->get_value();
  beta=BETA_param->get_value();
  one_minus_alpha=1.0-alpha;
  
}
 
void HirschWANoiseEstimationBlock::compute() {

  if (is_initialising()) {
    initialise(&noise_estimate, &noise_estimate_variance);
  } else {
    if (inputs_are_all_sample_data) {
      CTKSample x;
      (*input_sockets)[0]->get_sample(x);
      
      // updata noise model
      if (x<=beta*noise_estimate[0])
	noise_estimate[0]=new_noise(noise_estimate[0], x);
      
      (*output_sockets)[0]->put_sample(noise_estimate[0]);
      (*output_sockets)[1]->put_sample(x-noise_estimate[0]);
      (*output_sockets)[2]->put_sample(noise_estimate_variance[0]);
    } else {
      CTKVector *invec;
      (*input_sockets)[0]->get_vector(invec);
      
      // update noise model
      for (Float *fp=&noise_estimate[0], *fp2=&(*invec)[0]; fp<&noise_estimate[noise_estimate.size()]; ++fp, ++fp2)
	if (*fp2<=(beta * *fp))
	  *fp = new_noise(*fp, *fp2);
      
      (*output_sockets)[0]->put_vector(new CTKVector(noise_estimate));
      transform(invec->begin(), invec->end(), noise_estimate.begin(), invec->begin(), minus<Float>());
      (*output_sockets)[1]->put_vector(invec);
      (*output_sockets)[2]->put_vector(new CTKVector(noise_estimate_variance));
    }
  }
  
}


Float HirschWANoiseEstimationBlock::new_noise(Float noise, Float observed) {
  return (alpha * noise + one_minus_alpha * observed);
}



/* End of ctk_noise_estimation.cpp */
