/******************************************************************************/
/*									      */
/*	ctk_process_groups.cpp	        	      		              */
/*									      */
/*      Blocks that operate on group labelings                   */
/*									      */
/*	Author: Jon Barker, Sheffield University            		      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007     	         	      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <cmath>
#include <vector>
#include <sstream>
#include <map>

#include <algorithm>

#include "ctk_local.hh"


#include "ctk_socket.hh"  

#include "ctk_param.hh"    

#include "ctk_process_groups.hh"  


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MergeGroupsBlock                                         */
/*                                                                            */
/******************************************************************************/

const string MergeGroupsBlock::type_name = "MergeGroups";
const string MergeGroupsBlock::help_text = MERGE_GROUPS_BLOCK_HELP_TEXT;

static const char *VALID_MERGE_MODES[] = {"MODE_1", "MODE_2", "MODE_3", "\0"};

enum {MERGE_MODE_1=0, MERGE_MODE_2, MERGE_MODE_3};

static const char *PARAM_DEFAULT_MERGE_MODE = "MODE_1";


MergeGroupsBlock::MergeGroupsBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {

  make_input_sockets(2);
  input_sockets->set_description("in1", "group labels");
  input_sockets->set_description("in2", "group labels");

  make_output_sockets(1);
  output_sockets->set_description("out1", "merged group labels");

  // Set up normalisation mode parameter
  MERGE_MODE_param= new ParamEnumerated("MERGE_MODE", VALID_MERGE_MODES, PARAM_DEFAULT_MERGE_MODE);
  MERGE_MODE_param->set_helptext("The style of merging to employ.");
  parameters->register_parameter(MERGE_MODE_param);

}


MergeGroupsBlock::~MergeGroupsBlock() {
}

Block* MergeGroupsBlock::clone(const string &n) const{
  Block *ablock = new MergeGroupsBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}


void MergeGroupsBlock::reset() {
  Block::reset();  

  label_pair_map.clear();
  next_group_label=1;
  
  merge_mode=MERGE_MODE_param->get_index();

}

void MergeGroupsBlock::compute() {
  CTKVector *xv0, *xv1;
  
  (*input_sockets)[0]->get_vector(xv0);
  (*input_sockets)[1]->get_vector(xv1);

  for (CTKVector::iterator xp0 = xv0->begin(), xp1= xv1->begin(), xp0_end=xv0->end(); xp0!=xp0_end; ++xp0, ++xp1) {
    *xp0=mergeGroups((int)*xp0, (int)*xp1);
  }
  
  (*output_sockets)[0]->put_vector(xv0);

  delete xv1;
  
}


int MergeGroupsBlock::mergeGroups(int group_label1, int group_label2) {
  // If both groups are the background group (0) then return the background group number (0)

  if (group_label1==0 && group_label2==0) return 0;
    
  if (group_label1==-1 || group_label2 == -1)  return -1;
  
  // All other groups combinations are assigned a unique label

  std::ostringstream ostr;

  switch (merge_mode) {
  case MERGE_MODE_1:
    ostr << group_label1 << ":" << group_label2 << std::ends;
    break;
  case MERGE_MODE_2:
    if (group_label1==0)
      ostr << "0:" << group_label2 << std::ends;
    else
      ostr << group_label1 << ":0" << std::ends;
    break;
  case MERGE_MODE_3:
    if (group_label1==0)
      ostr << "0:" << group_label2 << std::ends;
    else if (group_label2==0)
      return 0;
    else
      ostr << group_label1 << ":" << group_label2 << std::ends;
    break;
  }

  string unique_label_string(ostr.str());
  
  int label = label_pair_map[unique_label_string];

  if (label==0) {
    // new pair - so assign new label
    label_pair_map[unique_label_string]=next_group_label;
    label = next_group_label;
    ++next_group_label;
  }
  
  return label;
}


void MergeGroupsBlock::close() {

  Block::close(); 

}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: CutGroupsBlock                                         */
/*                                                                            */
/******************************************************************************/

const string CutGroupsBlock::type_name = "CutGroups";
const string CutGroupsBlock::help_text = CUT_GROUPS_BLOCK_HELP_TEXT;


CutGroupsBlock::CutGroupsBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {

  make_input_sockets(2);
  input_sockets->set_description("in1", "group labels");
  input_sockets->set_description("in2", "boolean cutting signal");

  make_output_sockets(1);
  output_sockets->set_description("out1", "group labels after cutting");

}


CutGroupsBlock::~CutGroupsBlock() {
}

Block* CutGroupsBlock::clone(const string &n) const{
  Block *ablock = new CutGroupsBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}


void CutGroupsBlock::reset() {
  Block::reset();  

  group_map.clear();
  next_group_label=1;

  last_x = -1;
}

void CutGroupsBlock::compute() {
  CTKVector *xv0;
  Float x;
  
  (*input_sockets)[0]->get_vector(xv0);
  (*input_sockets)[1]->get_sample(x);

  bool cut = ((last_x!=-1) && (x!=last_x));  

  last_x=int(x);

  if (cut) {
    group_map.clear();
  }

  for (CTKVector::iterator xp0 = xv0->begin(),  xp0_end=xv0->end(); xp0!=xp0_end; ++xp0) {
    *xp0=relabelGroups((int)*xp0);
  }

  (*output_sockets)[0]->put_vector(xv0);

}

int CutGroupsBlock::relabelGroups(int x) {
  if (x==0 || x==-1) return x;
  int new_x=group_map[x];
  if (new_x==0) {
    group_map[x]=next_group_label;
    new_x = next_group_label;
    ++next_group_label;
  }
  return new_x;
}
  
void CutGroupsBlock::close() {

  Block::close(); 

}


/* End of ctk_process_groups.cpp */
 
