/******************************************************************************/
/*                                                                            */
/*      ctk_qfigure.cpp                                                       */
/*                                                                            */
/*      Matlab style Figure using Qt for matrix display                                                      */
/*                                                                            */
/*      Author:  Ljubomir Josifovski MAR/2001                             */
/*                                                                            */
/*      CTK VERSION 1.3.5  Apr 22, 2007                              */
/*                                                                            */
/******************************************************************************/

#include "ctk-config.h"

#include <cstdio>

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_qfigure.hh"

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: QPlot                                      */
/*                                                                            */
/******************************************************************************/

#ifdef _HAS_QT

QPlot::QPlot( vector<Float> some_data, int width/*=400*/, int height/*=300*/, const char *caption/*="QFigure"*/ )
{
 
  // caption
  QWidget::setCaption( caption );
  
  // size
  QWidget::resize( width,height );

  data=some_data;
                 
  this->setMinimumSize(width, height);

     
}


void QPlot::paintEvent( QPaintEvent * )
{
   QPainter paint( this );

   Float ymin=data[0];
   Float ymax=data[0];

   for (unsigned int i=0; i<data.size(); ++i) {
     Float y=data[i];
     if (y<ymin) ymin=y;
     if (y>ymax) ymax=y;
   }
   
   paint.setWindow(0, 0 , data.size(), 1000);

   float yrange = ymax-ymin;
   if (yrange==0) yrange=1.0;
   
   float scale=1000.0/yrange;
   
   paint.translate(0, -ymin);
		      
   paint.moveTo(0,(int)((data[0]-ymin)*scale));
   for (unsigned int i=1; i<data.size(); ++i) 
     paint.lineTo(i, (int)((ymax-data[i])*scale));
   
}

#endif

/******************************************************************************/
/**/
/*     CLASS NAME: QFigure                                       */
/**/
/******************************************************************************/
#ifdef _HAS_QT

// Matlab's jet colormap

const int QFigure::numColors = 64;
const int QFigure::colormap[numColors][3] = {
      {0, 0, 143}, {0, 0, 159}, {0, 0, 175}, {0, 0, 191}, {0, 0, 207}, {0,
0, 223}, {0, 0, 239}, {0, 0, 255},
      {0, 15, 255}, {0, 31, 255}, {0, 47, 255}, {0, 63, 255}, {0, 79, 255},
{0, 95, 255}, {0, 111, 255}, {0, 127, 255},
      {0, 143, 255}, {0, 159, 255}, {0, 175, 255}, {0, 191, 255}, {0, 207,
255}, {0, 223, 255}, {0, 239, 255}, {0, 255, 255},
      {15, 255, 255}, {31, 255, 239}, {47, 255, 223}, {63, 255, 207}, {79,
255, 191}, {95, 255, 175}, {111, 255, 159}, {127, 255, 143},
      {143, 255, 127}, {159, 255, 111}, {175, 255, 95}, {191, 255, 79},
{207, 255, 63}, {223, 255, 47}, {239, 255, 31}, {255, 255, 15},
      {255, 255, 0}, {255, 239, 0}, {255, 223, 0}, {255, 207, 0}, {255, 191,
0}, {255, 175, 0}, {255, 159, 0}, {255, 143, 0},
      {255, 127, 0}, {255, 111, 0}, {255, 95, 0}, {255, 79, 0}, {255, 63,
0}, {255, 47, 0}, {255, 31, 0}, {255, 15, 0},
      {255, 0, 0}, {239, 0, 0}, {223, 0, 0}, {207, 0, 0}, {191, 0, 0}, {175,
0, 0}, {159, 0, 0}, {143, 0, 0}
   };

// Constructor - use 8 bit image
// Immediatelly construct QImage out of data and keep it - that way it onlyneeds rescaling
// (smoothScale) on paint event.
// Dosen't keep the data - maybe difficult to implelemt extensions (user defcolormap, etc)
// Assume that all vectors are of equal length to the first!
//
QFigure::QFigure(vector<CTKVector*> data, int width /*=400*/, int height/*=300*/, char *caption /*="QFigure"*/)
   : QImage(data.size(), data[0]->size(), 8, numColors) {

   // cols = data[0]->size(), rows = data->size()

   // caption
   QWidget::setCaption( caption );

   // size
   QWidget::resize( width,height );

   // set jet colormap
   for (int i=0; i<numColors; ++i) {
      QImage::setColor( i, qRgb(colormap[i][0], colormap[i][1], colormap[i][2]) );
   }

   // find max and min elems in the data
   float mmin, mmax, range;
   mmin = mmax = (*data[0])[0];
   for (UInteger i=0; i<data.size(); ++i) {
      for (UInteger j=0; j<(*data[i]).size(); ++j) {
         float q = (*data[i])[j];
         if (q < mmin)
            mmin = q;
         if (q > mmax)
            mmax = q;
      }
   }

   range=mmax-mmin;

   if (range==0.0) range=1.0;
   
   // scaling factor data -> colormap index
   float scale = (numColors-1)/range;

   for (int i=0; i < QImage::width(); ++i) {
      for (int j=0; j < QImage::height(); ++j) {
         QImage::setPixel(i, j, (int)(((*data[i])[QImage::height()-1-j]-mmin)*scale));
      }
   }

}

//
// Called when the widget needs to be updated.
//
// 8bit indexed color version
//
void QFigure::paintEvent( QPaintEvent * )
{
   QPainter paint( this );

   // scale the image int new one to map to the widget size
   QImage img1 = smoothScale(QWidget::width(), QWidget::height());

   paint.drawImage(0, 0, img1);
}

#endif







/******************************************************************************/
/**/
/*     CLASS NAME: QFigureBlock                                        */
/**/
/******************************************************************************/

int QFigureBlock::global_figure_number=1;


const string QFigureBlock::type_name = "QFigure";
const string QFigureBlock::help_text = "Poormans eqiuvalent of Matlab's figure for displaying matrices";

vector<QWidget*> QFigureBlock::widget_list(0);

QFigureBlock::QFigureBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {

   make_input_sockets(1);
   make_output_sockets(1);

   output_connected=false;

  // Set up TITLE parameter
   title_param = new ParamString("TITLE", "QFigure");
   title_param->set_helptext("The title of the figure window.");
   parameters->register_parameter(title_param);

  // Set up WIDTH parameter
   width_param = new ParamInt("WIDTH", 400);
   width_param->set_helptext("The width of the figure window.");
   width_param->install_validator(new Validator(Validator::VLOWER, 0.0));
   parameters->register_parameter(width_param);

   // Set up HEIGHT parameter
   height_param = new ParamInt("HEIGHT", 300);
   height_param->set_helptext("The height of the figure window.");
   height_param->install_validator(new Validator(Validator::VLOWER, 0.0));
   parameters->register_parameter(height_param);

   // Set up ACTIVE parameter
   active_param = new ParamBool("ACTIVE", true);
   active_param->set_helptext("The block is inactive (only passess in1 on out1 without displaying anything) if set to No.");
   parameters->register_parameter(active_param);

}

QFigureBlock::~QFigureBlock() {
   free_storage();
}

Block* QFigureBlock::clone(const string &n) const{
   Block *ablock = new QFigureBlock(n.empty()?getname():n);
   return copy_this_block_to(ablock);
}

void QFigureBlock::reset() {

   Block::reset();

   free_storage();

   sample_buffer.resize(0);
   vector_buffer.resize(0);

   output_connected=(*output_sockets)[0]->connected();

   title = title_param->get_value();
   width = width_param->get_value();
   height = height_param->get_value();
   active = active_param->get_value();

   // Assign a new figure number
   figure_number=global_figure_number++;
   
}


void QFigureBlock::compute() {

   if (inputs_are_all_sample_data) {
      CTKSample sample;
      (*input_sockets)[0]->get_sample(sample);
      sample_buffer.push_back(sample);
      if (output_connected) {
         (*output_sockets)[0]->put_sample(sample);
      }
   } else {
      // Store data in vector buffer to be displayed and deleted later
      // and if this is a through block then pass on a pointer
      // to *a copy* of the data
      CTKVector *xv0;
      (*input_sockets)[0]->get_vector(xv0);
      vector_buffer.push_back(xv0);
      if (output_connected)
         (*output_sockets)[0]->put_vector(new CTKVector(*xv0));
   }

}


// On every reset QFigureBlock churns out one new QFigure, and forgets about it.
void QFigureBlock::close() {

#ifdef _HAS_QT
  Integer n_dims=(*input_sockets)[0]->get_data_descriptor()->get_n_dimensions();

   if (active && (vector_buffer.size()>0 || sample_buffer.size()>0)) {

     // -- Make sure there is an active qApp -- 
     // Global pointer qApp keeps track of the QApplication object.
     // There can be only one QApplication object.
     // QApplication object must be constructed before any other QWidget and friends.
     if (qApp == NULL) {
       int argc = 0;
       // Don't worry about the pointer, qApp keeps track of *the* QApplication object
       new QApplication(argc, NULL);
     }
     
     char buf[101];
     snprintf(buf, 100, "%s #%d", title.c_str(), figure_number);
       
     if (inputs_are_all_sample_data) {
    
       // 0-D case (i.e. a 1D plot: time)
       
       QPlot *widget2d=new QPlot(sample_buffer, width, height, buf);
       
       if (widget2d == NULL) {
	 cerr << "Out of memory." << endl;
	 throw(CTKError(__FILE__, __LINE__));
       }
       
       widget2d->show();
       
       
     } else if (n_dims==1) {
       
       // 1-D frame case (i.e. a 2D plot: time + 1 other D)
       
       QFigure *pqf = new QFigure(vector_buffer, width, height, buf);
       
       if (pqf == NULL) {
	 cerr << "Out of memory." << endl;
	 throw(CTKError(__FILE__, __LINE__));
       }
       
       pqf->show();
       
       
     } else {
       
       // n-D case
       cerr << "Trying to display data with more than 2 dimensions. Not yet implemented." << endl;
       throw(CTKError(__FILE__, __LINE__));
       
     }
   }
#endif
    
  Block::close();
    
}


void QFigureBlock::close_final() {

}

void QFigureBlock::free_storage() {

   for (UInteger i=0; i<vector_buffer.size(); ++i) {

      if (vector_buffer[i]!=NULL) {
         delete vector_buffer[i];
      }

   }
   vector_buffer.resize(0);
}

 



/* End of ctk_qfigure.cpp */
