/******************************************************************************/
/*									      */
/*	ctk_simple_blocks.cpp	    	       			              */
/*									      */
/*	Block for various trivial operations                                  */
/*          both sample and frame versions are supplied.          	      */
/*									      */
/*      Blocks: Adder, Tee, Multiplier, Plus, Delay         		      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		         	      */
/*							  		      */
/******************************************************************************/
 
#include "ctk-config.h"

#include <cmath>
#include <vector>

#include <algorithm>

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_simple_blocks.hh"


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MultiInputBaseBlock                                      */
/*                                                                            */
/******************************************************************************/

MultiInputBaseBlock::MultiInputBaseBlock(const string &a_name, const string &a_type):CTKObject(a_name),Block(a_name, a_type) {

  make_input_sockets(2,        // Number of sockets
		     true,     // Number of sockets is user configurable?
		     2);       // Minimum number of sockets

  make_output_sockets(1);

}


Boolean MultiInputBaseBlock::multi_input_shape_check() {
  Boolean flag=true;
  SocketIt socketp;

  sample_sockets.resize(0); frame_sockets.resize(0);
  
  for (socketp=input_sockets->begin(); socketp!=input_sockets->end(); ++socketp) {
    if ((*socketp)->get_data_descriptor()->is_sample_data()) {
      sample_sockets.push_back((InputSocket*)*socketp);
    } else {
      frame_sockets.push_back((InputSocket*)*socketp);
      break;
    }
  }

  // If all sample data return true;
  if (socketp==input_sockets->end()) return true;

  const DataDescriptor *dd1=(*socketp)->get_data_descriptor();

  for (SocketIt socket2p=socketp+1; socket2p!=input_sockets->end(); ++socket2p) {
    const DataDescriptor *dd2=(*socket2p)->get_data_descriptor();
    if (dd2->is_sample_data()==false && dd1->matches_shape_of(dd2)==false) {
      flag=false;
      break;
    }
    if (dd2->is_sample_data()) {
      sample_sockets.push_back((InputSocket*)*socket2p);
    } else {
      frame_sockets.push_back((InputSocket*)*socket2p);
    }
  }

  inputs_are_all_frame_data=(sample_sockets.size()==0);

  return flag;
}


void MultiInputBaseBlock::reset() {
  Block::reset();
  
  // Check the inputs all have the same shape
  if (multi_input_shape_check()==false) {
    cerr << get_blocktype() << ": Inputs have varying dimensionality." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
}



void MultiInputBaseBlock::build_output_data_descriptors() {

  const DataDescriptor *dd;
  
  if (frame_sockets.size()!=0) {
    dd=frame_sockets[0]->get_data_descriptor();
  } else {
    dd=sample_sockets[0]->get_data_descriptor();
  }
  
  (*output_sockets)[0]->set_data_descriptor(new DataDescriptor(*dd));
 
}


void MultiInputBaseBlock::compute() {
  if (inputs_are_all_sample_data) {
    // For handling sample data
    Float x0, xi;
    
    (*input_sockets)[0]->get_sample(x0);
    
    for (vector<Socket*>::iterator socketp=input_sockets->begin()+1; socketp!=input_sockets->end(); ++socketp) {
      (*socketp)->get_sample(xi);
      x0=operate(x0,xi);
    }
    if (has_postprocess())
      x0=postprocess(x0);
    (*output_sockets)[0]->put_sample(x0);

  } else if (inputs_are_all_frame_data) {
    // For handling frame data
    CTKVector *xv0, *xvi;
    
    (*input_sockets)[0]->get_vector(xv0);
    
    for (vector<Socket*>::iterator socketp=input_sockets->begin()+1; socketp!=input_sockets->end(); ++socketp) {
      (*socketp)->get_vector(xvi);
      operate(xv0, xvi);
      delete xvi;
    }

    if (has_postprocess())
      postprocess(xv0);
    
    (*output_sockets)[0]->put_vector(xv0);
    
  } else {
    // For handling the general case where there is a mixture of sample and frame inputs
    
    Float x0, xi;

    // operate on the sample data first
    (sample_sockets)[0]->get_sample(x0);
    
    for (vector<InputSocket*>::iterator socketp=sample_sockets.begin()+1; socketp!=sample_sockets.end(); ++socketp) {
      (*socketp)->get_sample(xi);
      x0=operate(x0,xi);
    }
    
    CTKVector *xv0, *xvi;

    // Apply the combined sample inputs to the first frame input
    (frame_sockets)[0]->get_vector(xv0);
    operate(xv0, x0);

    // Operate on the rest of the frame inputs
    for (vector<InputSocket*>::iterator socketp=frame_sockets.begin()+1; socketp!=frame_sockets.end(); ++socketp) {
      (*socketp)->get_vector(xvi);
      operate(xv0, xvi);
      delete xvi;
    }

    if (has_postprocess())
      postprocess(xv0);
    
    (*output_sockets)[0]->put_vector(xv0);
    
  }
  
}



/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: AdderBlock                                               */
/*                                                                            */
/******************************************************************************/

const string AdderBlock::type_name = "Adder";
const string AdderBlock::help_text = ADDER_BLOCK_HELP_TEXT;

AdderBlock::AdderBlock(const string &a_name):CTKObject(a_name),MultiInputBaseBlock(a_name, type_name) {}

Block* AdderBlock::clone(const string &n) const{
  Block *ablock = new AdderBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

Float AdderBlock::operate(Float x1, Float x2) {return x1+x2;}
void AdderBlock::operate(CTKVector *xv1, CTKVector *xv2) {
 transform(xv1->begin(), xv1->end(), xv2->begin(), xv1->begin(), plus<float>());
}
void AdderBlock::operate(CTKVector *xv1, Float x2) {
 transform(xv1->begin(), xv1->end(), xv1->begin(), bind2nd(plus<float>(), x2));
}


Boolean AdderBlock::has_postprocess() {return false;}
Float AdderBlock::postprocess(Float){return 0.0;}
void AdderBlock::postprocess(CTKVector *){};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MultiplierBlock                                          */
/*                                                                            */
/******************************************************************************/

const string MultiplierBlock::type_name = "Multiplier";
const string MultiplierBlock::help_text = MULTIPLIER_BLOCK_HELP_TEXT;

MultiplierBlock::MultiplierBlock(const string &a_name):CTKObject(a_name),MultiInputBaseBlock(a_name, type_name) {}

Block* MultiplierBlock::clone(const string &n) const{
  Block *ablock = new MultiplierBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

Float MultiplierBlock::operate(Float x1, Float x2) {return x1*x2;}
void MultiplierBlock::operate(CTKVector *xv1, CTKVector *xv2) {
 transform(xv1->begin(), xv1->end(), xv2->begin(), xv1->begin(), multiplies<float>());
}
void MultiplierBlock::operate(CTKVector *xv1, Float x2) {
 transform(xv1->begin(), xv1->end(), xv1->begin(), bind2nd(multiplies<float>(), x2));
}

Boolean MultiplierBlock::has_postprocess() {return false;}
Float MultiplierBlock::postprocess(Float){return 0.0;}
void MultiplierBlock::postprocess(CTKVector *){};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MaxBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string MaxBlock::type_name = "Max";
const string MaxBlock::help_text = MAX_BLOCK_HELP_TEXT;

MaxBlock::MaxBlock(const string &a_name):CTKObject(a_name),MultiInputBaseBlock(a_name, type_name) {}

Block* MaxBlock::clone(const string &n) const{
  Block *ablock = new MaxBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

Float MaxBlock::operate(Float x1, Float x2) {return max(x1,x2);}
void MaxBlock::operate(CTKVector *xv1, CTKVector *xv2) {
 transform(xv1->begin(), xv1->end(), xv2->begin(), xv1->begin(), maximum<float>());
}
void MaxBlock::operate(CTKVector *xv1, Float x2) {
 transform(xv1->begin(), xv1->end(), xv1->begin(), bind2nd(maximum<float>(), x2));
}


Boolean MaxBlock::has_postprocess() {return false;}
Float MaxBlock::postprocess(Float){return 0.0;}
void MaxBlock::postprocess(CTKVector *){};



/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MinBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string MinBlock::type_name = "Min";
const string MinBlock::help_text = MIN_BLOCK_HELP_TEXT;

MinBlock::MinBlock(const string &a_name):CTKObject(a_name),MultiInputBaseBlock(a_name, type_name) {}

Block* MinBlock::clone(const string &n) const{
  Block *ablock = new MinBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

Float MinBlock::operate(Float x1, Float x2) {return min(x1,x2);}
void MinBlock::operate(CTKVector *xv1, CTKVector *xv2) {
 transform(xv1->begin(), xv1->end(), xv2->begin(), xv1->begin(), minimum<float>());
}
void MinBlock::operate(CTKVector *xv1, Float x2) {
 transform(xv1->begin(), xv1->end(), xv1->begin(), bind2nd(minimum<float>(), x2));
}


Boolean MinBlock::has_postprocess() {return false;}
Float MinBlock::postprocess(Float ){return 0.0;}
void MinBlock::postprocess(CTKVector *){};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: LogicalANDBlock                                          */
/*                                                                            */
/******************************************************************************/

const string LogicalANDBlock::type_name = "LogicalAND";
const string LogicalANDBlock::help_text = LOGICAL_AND_BLOCK_HELP_TEXT;

LogicalANDBlock::LogicalANDBlock(const string &a_name):CTKObject(a_name),MultiInputBaseBlock(a_name, type_name) {}

Block* LogicalANDBlock::clone(const string &n) const{
  Block *ablock = new LogicalANDBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

Float LogicalANDBlock::operate(Float x1, Float x2) {return fabs(x1)*fabs(x2);}
void LogicalANDBlock::operate(CTKVector *xv1, CTKVector *xv2) {
  transform(xv1->begin(), xv1->end(), xv2->begin(), xv1->begin(), absmultiplies<float>());
}
void LogicalANDBlock::operate(CTKVector *xv1, Float x2) {
  transform(xv1->begin(), xv1->end(), xv1->begin(), bind2nd(absmultiplies<float>(), x2));
}

Boolean LogicalANDBlock::has_postprocess() {return true;}
Float LogicalANDBlock::postprocess(Float x){return x>0.0;}
void LogicalANDBlock::postprocess(CTKVector *xv){
  transform(xv->begin(), xv->end(), xv->begin(), bind2nd(greater<float>(), 0.0));
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: LogicalORBlock                                           */
/*                                                                            */
/******************************************************************************/

const string LogicalORBlock::type_name = "LogicalOR";
const string LogicalORBlock::help_text = LOGICAL_OR_BLOCK_HELP_TEXT;

LogicalORBlock::LogicalORBlock(const string &a_name):CTKObject(a_name),MultiInputBaseBlock(a_name, type_name) {}

Block* LogicalORBlock::clone(const string &n) const{
  Block *ablock = new LogicalORBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

Float LogicalORBlock::operate(Float x1, Float x2) {return fabs(x1)+fabs(x2);}
void LogicalORBlock::operate(CTKVector *xv1, CTKVector *xv2) { 
  transform(xv1->begin(), xv1->end(), xv2->begin(), xv1->begin(), absplus<float>());
}
void LogicalORBlock::operate(CTKVector *xv1, Float x2) { 
  transform(xv1->begin(), xv1->end(), xv1->begin(), bind2nd(absplus<float>(), x2));
}

Boolean LogicalORBlock::has_postprocess() {return true;}
Float LogicalORBlock::postprocess(Float x){return x>0.0;}
void LogicalORBlock::postprocess(CTKVector *xv){
  transform(xv->begin(), xv->end(), xv->begin(), bind2nd(greater<float>(), 0.0));
};




/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: JoinBlock                                                */
/*                                                                            */
/******************************************************************************/

const string JoinBlock::type_name = "Join";
const string JoinBlock::help_text = JOIN_BLOCK_HELP_TEXT;

JoinBlock::JoinBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {
  make_input_sockets(2,        // Number of sockets
		     true,     // Number of sockets is user configurable?
		     2);       // Minimum number of sockets
  make_output_sockets(1);
}

Block* JoinBlock::clone(const string &n) const{
  Block *ablock = new JoinBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void JoinBlock::reset() {
  Block::reset();
  
  // Check the inputs all have the same shape
  if (inputs_1D_or_less_check()==false) {
    cerr << "JoinBlock:: Inputs must all be 1-D frames or samples." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  output_size=sum_of_input_storage();
}

void JoinBlock::build_output_data_descriptors() {
  
  const DataDescriptor *idd = (*input_sockets)[0]->get_data_descriptor();
  DataDescriptor *dd = new DataDescriptor(*idd);
  
  // Replace axis with one of larger dimension
  const DimensionDescriptor *dimd = dd->get_outer_dimension();

  string a_name;
  CTKVector axis;
  
  if (dimd!=NULL) {
    axis=dimd->get_axis();
    a_name=dimd->get_name();   
    dd->remove_outer_dimension();
  } else {
    a_name=string("1"); 
  }
  
  axis.resize(output_size);
  dd->add_outer_dimension(a_name, axis);
  
  (*output_sockets)[0]->set_data_descriptor(dd);
}

void JoinBlock::compute() {

  CTKVector *outvec = new CTKVector(output_size);
  CTKVector *xvi;
  
  
  Integer index=0;
  for (Integer i=0; i<input_sockets->size(); ++i) {
    if ((*input_sockets)[i]->get_data_descriptor()->is_sample_data()) {
      (*input_sockets)[i]->get_sample((*outvec)[index++]);
    } else {
      (*input_sockets)[i]->get_vector(xvi);
      for (UInteger j=0; j<(*xvi).size(); ++j) {
      	(*outvec)[index++]=(*xvi)[j];
      }
      delete xvi;   
    }
  }
  
  (*output_sockets)[0]->put_vector(outvec);
  
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: OptionBlock                                                */
/*                                                                            */
/******************************************************************************/

const string OptionBlock::type_name = "Option";
const string OptionBlock::help_text = OPTION_BLOCK_HELP_TEXT;

OptionBlock::OptionBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {
  make_input_sockets(2);
  make_output_sockets(1);
  
  input_sockets->set_optional("in2");
}

Block* OptionBlock::clone(const string &n) const{
  Block *ablock = new OptionBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void OptionBlock::inherit_unset_parameters(){ 
  ParamList *plistp;
  Block *blockp;

  if (input_sockets->size()!=0)
    parameters->clear_transmitted_parameters(); // unset all transmitted parameter

  // Only inherit from 1st socket (2nd optional socket is ignored)
  SocketIt socketp=input_sockets->begin();
  
  if ((*socketp)->connected()) {
    blockp=(*socketp)->get_remote_baseblock();
    plistp=blockp->get_parameters();
    
    ParamIt dodgy_parameter;
    CTKStatus status = parameters->copy_transmitted_settings(plistp, dodgy_parameter);
    if (status==CTK_FAILURE) {
      cerr << "Inconsistent transmitted parameters: " << getfullname() << ": " << (*dodgy_parameter)->getname() << endl;
      throw(CTKError(__FILE__, __LINE__));
    }
  }

}

void OptionBlock::reset() {
  Block::reset();

  option_supplied = (*input_sockets)[1]->connected();
  
  if (option_supplied && input_shape_check()==false) {
    cerr << "OptionBlock:: Optional data and standard data are of different type." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
    
}

void OptionBlock::compute() {

  if (inputs_are_all_sample_data) {
    // Sample data case
    Float sample;
    if (option_supplied) 
      (*input_sockets)[1]->get_sample(sample);
    else
      (*input_sockets)[0]->get_sample(sample);

    (*output_sockets)[0]->put_sample(sample);
  } else {
    // Frame data case
    CTKVector *xvi;
    if (option_supplied) {
      delete xvi;
      (*input_sockets)[1]->get_vector(xvi);
    } else
      (*input_sockets)[0]->get_vector(xvi);

    (*output_sockets)[0]->put_vector(xvi);
    
  }
  
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SplitBlock                                               */
/*                                                                            */
/******************************************************************************/

const string SplitBlock::type_name = "Split";
const string SplitBlock::help_text = SPLIT_BLOCK_HELP_TEXT;

SplitBlock::SplitBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {

  make_input_sockets(1);
  make_output_sockets(2,        // Number of sockets
		     true,     // Number of sockets is user configurable?
		     2);       // Minimum number of sockets

  // Set up CUT parameter
  CUT_param= new ParamInt("CUT");
  CUT_param->set_helptext("The index of the channel <b>below which</b> to make a split. <p> The indices start at 1, so the lowest sensible value for CUT is 2. <p> Either CUT or CUTS should be set, but not both."); 
  CUT_param->install_validator(new Validator(Validator::VLOWER, 2.0));
  parameters->register_parameter(CUT_param);

  // Set up CUTS parameter
  CUTS_param= new ParamIntVector("CUTS");
  CUTS_param->set_helptext("A list of channel indices <b>below each of which</b> the input vector is split. <p> Either CUT or CUTS should be set, but not both."); 
  CUTS_param->install_validator(new Validator(Validator::VLOWER, 2.0));
  parameters->register_parameter(CUTS_param);
}

Block* SplitBlock::clone(const string &n) const{
  Block *ablock = new SplitBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void SplitBlock::reset() {
  Block::reset();
  
  // Check the inputs all have the same shape
  if ((*input_sockets)[0]->get_data_descriptor()->get_n_dimensions()!=1) {
    cerr << "SplitBlock:: Input must  be 1-D frame data." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
  if (CUT_param->get_set_flag() && CUTS_param->get_set_flag()) {
    cerr << "SplitBlock:: Should set *either* CUT or CUTS paramters not both!" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  Integer input_width=(*input_sockets)[0]->get_data_descriptor()->get_storage();
  
  if (CUT_param->get_set_flag()) {
    vector<int> cutcols(1);
    cutcols[0]=CUT_param->get_value();
    make_output_widths(cutcols, input_width);
  } else if (CUTS_param->get_set_flag()) {
    vector<int> cutcols = CUTS_param->get_vector();
    sort(cutcols.begin(), cutcols.end());
    make_output_widths(cutcols, input_width);
  } else {
    cerr << "SplitBlock:: Must set either CUT or CUTS parameter." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
  set_num_outputs(output_widths.size());
}



void SplitBlock::build_output_data_descriptors() {
  
  const DataDescriptor *idd = (*input_sockets)[0]->get_data_descriptor();

  const DimensionDescriptor *dimd = idd->get_outer_dimension();
  string a_name=dimd->get_name();
  CTKVector axis=dimd->get_axis();

  Integer from=0, width;
  for (UInteger i=0; i<output_widths.size(); ++i) {
    DataDescriptor *dd = new DataDescriptor();
    width=output_widths[i];
    CTKVector axis2(width);
    copy(&(axis[0])+from, &(axis[0])+from+width, &(axis2[0]));
    from+=width;
    dd->add_outer_dimension(a_name, axis2); 
    (*output_sockets)[i]->set_data_descriptor(dd);
  }
  
}

void SplitBlock::compute() {

  CTKVector *invec;
  CTKVector *outvec;
  
  Integer index=0;
  
  (*input_sockets)[0]->get_vector(invec);
  
  for (UInteger i=0; i<output_widths.size(); ++i) {
    Integer width=output_widths[i];
    outvec=new CTKVector(width);
    for (Integer j=0; j<width; ++j)
      (*outvec)[j]=(*invec)[index++];

    if (width>1) 
      (*output_sockets)[i]->put_vector(outvec);
    else {
      (*output_sockets)[i]->put_sample((*outvec)[0]);
      delete outvec;
    }
    
  }
  
  delete invec;
}


void SplitBlock::make_output_widths(vector<int> cuts, Integer total_width) {

  Integer fromcol=1;

  output_widths.resize(cuts.size()+1);
  for (UInteger i=0; i<cuts.size(); ++i) {
    Integer cutcol=cuts[i];
    if (cutcol<2 || cutcol>total_width) {
      cerr << "SplitBlock:: Cannot cut " << total_width << " column data at column " << cutcol <<"." << endl;
      throw(CTKError(__FILE__, __LINE__));
    }
  
    output_widths[i]=cutcol-fromcol;
    fromcol=cutcol;
  }
  output_widths[cuts.size()]=total_width-fromcol+1;
  
}



/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: TeeBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string TeeBlock::type_name = "Tee";
const string TeeBlock::help_text = TEE_BLOCK_HELP_TEXT;

TeeBlock::TeeBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {
  make_input_sockets(1);
  make_output_sockets(2,        // Number of sockets
		     true,     // Number of sockets is user configurable?
		     2);       // Minimum number of sockets
}

Block* TeeBlock::clone(const string &n) const{
  Block *ablock = new TeeBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void TeeBlock::compute() {
  if (inputs_are_all_sample_data) {
    // For teeing sample data
    Float x;
    (*input_sockets)[0]->get_sample(x);
    for_each(output_sockets->begin(), output_sockets->end(), bind2nd(mem_fun(&Socket::put_sample), x));
  } else {
    // For teeing frame data
    CTKVector *xv0;
    (*input_sockets)[0]->get_vector(xv0);
    for (vector<Socket*>::iterator socketp=output_sockets->begin()+1; socketp<output_sockets->end(); ++socketp) 
      (*socketp)->put_vector(new CTKVector(*xv0));

    // Note it must be assumed that 'put_vector' deletes the vector immediately. So xv0 cannot be
    // pushed until after it has been used to make the copies for the other output sockets
    (*output_sockets)[0]->put_vector(xv0);
  }
}




/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: DelayBlock                                               */
/*                                                                            */
/******************************************************************************/

const string DelayBlock::type_name = "Delay";
const string DelayBlock::help_text = DELAY_BLOCK_HELP_TEXT;

DelayBlock::DelayBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {

  // Set up DELAY parameter
  DELAY_param= new ParamInt("DELAY", PARAM_DEFAULT_DELAY_DELAY);
  DELAY_param->set_helptext("The number of frames by which to delay the data stream.");
  DELAY_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(DELAY_param);

  // Set up DELAY parameter
  REPEAT_FIRST_FRAME_param= new ParamBool("REPEAT_FIRST_FRAME", 0);
  REPEAT_FIRST_FRAME_param->set_helptext("If true then delays by repeating the first frame, else delays by transmitting 0 frames.");
  parameters->register_parameter(REPEAT_FIRST_FRAME_param);

  make_input_sockets(1);
  make_output_sockets(1);
}


Block* DelayBlock::clone(const string &n) const{
  Block *ablock = new DelayBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void DelayBlock::reset(){
  Block::reset();

  first_compute=true;
  delay=DELAY_param->get_value();
  repeat_frame=REPEAT_FIRST_FRAME_param->get_value();

  skip_frames=0;
  if (delay<0) skip_frames=-delay;
  
  if (!repeat_frame && delay>0) {
    if (inputs_are_all_sample_data) {
      // For delaying sample data
      for (int i=0; i<delay; ++i)
	(*output_sockets)[0]->put_sample(0.0);
    } else {
      // For delaying frame data
      Integer input_width=(*input_sockets)[0]->get_data_descriptor()->get_storage();
      for (int i=0; i<delay; ++i) 
	(*output_sockets)[0]->put_vector(new CTKVector(input_width,0.0));	  
    }
    
  }
    
}

void DelayBlock::compute() {


  if (skip_frames==0) {
    if (inputs_are_all_sample_data) {
      // For delaying sample data
      Float x1;
      (*input_sockets)[0]->get_sample(x1);
      if (first_compute) {
	if (repeat_frame) {
	  for (int i=0; i<delay; ++i)
	    (*output_sockets)[0]->put_sample(x1);
	}
      }
      (*output_sockets)[0]->put_sample(x1);
    } else {
      // For delaying frame data
      CTKVector *xv0;
      (*input_sockets)[0]->get_vector(xv0);
      if (first_compute) {
	if (repeat_frame) {
	  for (int i=0; i<delay; ++i) 
	    (*output_sockets)[0]->put_vector(new CTKVector(*xv0));
	}
      }
      (*output_sockets)[0]->put_vector(xv0);
    }
    
  } else {
    CTKVector *xv0;
    (*input_sockets)[0]->get_vector(xv0);
    delete xv0;
    skip_frames--;
  }

  // Next call to compute will not be the first
  first_compute=false;
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: OutputBlock                                              */
/*                                                                            */
/******************************************************************************/

const string OutputBlock::type_name = "Output";
const string OutputBlock::help_text = OUPUT_BLOCK_HELP_TEXT;

OutputBlock::OutputBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {

  make_input_sockets(1);
}

Block* OutputBlock::clone(const string &n) const{
  Block *ablock = new OutputBlock(n.empty()?getname():n);
  ablock->copy_parameters(parameters);
  if (input_sockets->is_configurable()) ablock->set_num_inputs(input_sockets->size());
  if (output_sockets->is_configurable()) ablock->set_num_outputs(output_sockets->size());
  return ablock;
}

void OutputBlock::reset(){
  Block::reset();

  n_dims=(*input_sockets)[0]->get_data_descriptor()->get_n_dimensions();
  dim_sizes=(*input_sockets)[0]->get_data_descriptor()->get_dimension_sizes();
  if (n_dims>1) {
    cerr << "Error in network at block: " << getname() << endl;
    cerr << "Output of n-dimensional frames not yet implemented for N>1." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
}

void OutputBlock::compute() {
  Float x;
  cout << getfullname() << ": ";
  if (inputs_are_all_sample_data) {
    (*input_sockets)[0]->get_sample(x);
    cout << x;
  } else if (n_dims==1) {
    CTKVector *invec;
    (*input_sockets)[0]->get_vector(invec);
    for (Float *pp=&(*invec)[0]; pp!=&(*invec)[dim_sizes[0]]; ++pp) {
      cout << *pp << " ";
    }
  } else {
    cerr << "Generalised N-Dimensional frame output not yet implemented." << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  
  cout << endl;

}


/* End of ctk_simple_blocks.cpp */
