/******************************************************************************/
/*									      */
/*	ctk_simple_blocks.hh   			      			      */
/*									      */
/*	Class declarations for ctk_simple_blocks.cpp	               	      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		        	      */
/*									      */
/******************************************************************************/

#ifndef CTK_SIMPLE_BLOCKS_HH
#define CTK_SIMPLE_BLOCKS_HH

#include <functional>

#include "ctk_block.hh"

class ParamInt;
class ParamIntVector;


template <class _Tp>
struct maximum : public binary_function<_Tp,_Tp,_Tp> {
  _Tp operator()(const _Tp& __x, const _Tp& __y) const { return max(__x,__y); }
};

template <class _Tp>
struct minimum : public binary_function<_Tp,_Tp,_Tp> {
  _Tp operator()(const _Tp& __x, const _Tp& __y) const { return min(__x,__y); }
};

template <class _Tp>
struct absplus : public binary_function<_Tp,_Tp,_Tp> {
  _Tp operator()(const _Tp& __x, const _Tp& __y) const { return fabs(__x) + fabs(__y); }
};


template <class _Tp>
struct absmultiplies : public binary_function<_Tp,_Tp,_Tp> {
  _Tp operator()(const _Tp& __x, const _Tp& __y) const { return fabs(__x) * fabs(__y); }
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MultiInputBaseBlock                                      */
/*                                                                            */
/******************************************************************************/

class MultiInputBaseBlock:public Block {

  Boolean inputs_are_all_frame_data;
  
  vector <class InputSocket*> sample_sockets;
  vector <class InputSocket*> frame_sockets;
  
  //
  //
 //
  
public:

  MultiInputBaseBlock(const string &a_name, const string &a_type);
  virtual ~MultiInputBaseBlock(){};

  virtual Block* clone(const string &new_name) const =0;

  virtual const string &get_helptext() const =0;

private:

  Boolean multi_input_shape_check();
  
  virtual void reset();
  virtual void build_output_data_descriptors();
  virtual void compute();

  virtual Float operate(Float x1, Float x2) =0;
  virtual void operate(CTKVector *xv1, CTKVector *xv2) =0;
  virtual void operate(CTKVector *xv1, Float x2) =0;
  
  virtual Boolean has_postprocess()=0;
  virtual Float postprocess(Float x) = 0;
  virtual void postprocess(CTKVector *xv)=0;
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: AdderBlock                                               */
/*                                                                            */
/******************************************************************************/

class AdderBlock:public MultiInputBaseBlock {

  static const string type_name;
  static const string help_text;

public:

  AdderBlock(const string &a_name);
  virtual ~AdderBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  inline virtual Float operate(Float x1, Float x2);
  inline virtual void operate(CTKVector *xv1, CTKVector *xv2);
  inline virtual void operate(CTKVector *xv1, Float x2);
  
  inline virtual Boolean has_postprocess();
  inline virtual Float postprocess(Float x1);
  inline virtual void postprocess(CTKVector *xv);
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MultiplierBlock                                          */
/*                                                                            */
/******************************************************************************/

class MultiplierBlock:public MultiInputBaseBlock {

  static const string type_name;
  static const string help_text;

public:

  MultiplierBlock(const string &a_name);
  virtual ~MultiplierBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  inline virtual Float operate(Float x1, Float x2);
  inline virtual void operate(CTKVector *xv1, CTKVector *xv2);
  inline virtual void operate(CTKVector *xv1, Float x2);
  
  inline virtual Boolean has_postprocess();
  inline virtual Float postprocess(Float x);
  inline virtual void postprocess(CTKVector *xv);

};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MaxBlock                                                 */
/*                                                                            */
/******************************************************************************/

class MaxBlock:public MultiInputBaseBlock {

  static const string type_name;
  static const string help_text;

public:

  MaxBlock(const string &a_name);
  virtual ~MaxBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  inline virtual Float operate(Float x1, Float x2);
  inline virtual void operate(CTKVector *xv1, CTKVector *xv2);
  inline virtual void operate(CTKVector *xv1, Float x2);
  
  inline virtual Boolean has_postprocess();
  inline virtual Float postprocess(Float x);
  inline virtual void postprocess(CTKVector *xv);
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: MinBlock                                                 */
/*                                                                            */
/******************************************************************************/

class MinBlock:public MultiInputBaseBlock {

  static const string type_name;
  static const string help_text;

public:

  MinBlock(const string &a_name);
  virtual ~MinBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  inline virtual Float operate(Float x1, Float x2);
  inline virtual void operate(CTKVector *xv1, CTKVector *xv2);
  inline virtual void operate(CTKVector *xv1, Float x2);
  
  inline virtual Boolean has_postprocess();
  inline virtual Float postprocess(Float x);
  inline virtual void postprocess(CTKVector *xv);
};


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: LogicalANDBlock                                          */
/*                                                                            */
/******************************************************************************/

class LogicalANDBlock:public MultiInputBaseBlock {

  static const string type_name;
  static const string help_text;

public:

  LogicalANDBlock(const string &a_name);
  virtual ~LogicalANDBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  inline virtual Float operate(Float x1, Float x2);
  inline virtual void operate(CTKVector *xv1, CTKVector *xv2);
  inline virtual void operate(CTKVector *xv1, Float x2); 
  
  inline virtual Boolean has_postprocess();
  inline virtual Float postprocess(Float x1);
  inline virtual void postprocess(CTKVector *xv);
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: LogicalORBlock                                           */
/*                                                                            */
/******************************************************************************/

class LogicalORBlock:public MultiInputBaseBlock {

  static const string type_name;
  static const string help_text;

public:

  LogicalORBlock(const string &a_name);
  virtual ~LogicalORBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  inline virtual Float operate(Float x1, Float x2);
  inline virtual void operate(CTKVector *xv1, CTKVector *xv2);
  inline virtual void operate(CTKVector *xv1, Float x2);
  
  inline virtual Boolean has_postprocess();
  inline virtual Float postprocess(Float x1);
  inline virtual void postprocess(CTKVector *xv);
};



/**************************************************************************************************/

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: JoinBlock                                                */
/*                                                                            */
/******************************************************************************/

class JoinBlock:public Block {

  static const string type_name;
  static const string help_text;

private:
  Integer output_size;
  
public:

  JoinBlock(const string &a_name);
  virtual ~JoinBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void reset();
  virtual void build_output_data_descriptors();
  
  virtual void compute();

};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: OptionBlock                                                */
/*                                                                            */
/******************************************************************************/

class OptionBlock:public Block {

  static const string type_name;
  static const string help_text;

private:
  bool option_supplied;
  
public:

  OptionBlock(const string &a_name);
  virtual ~OptionBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void reset();
  
  virtual void compute();

  // specially overriden inherit method that ignores input from the optional input socket
  virtual void inherit_unset_parameters(); 
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SplitBlock                                               */
/*                                                                            */
/******************************************************************************/

class SplitBlock:public Block {

  static const string type_name;
  static const string help_text;

private:
  vector<int> output_widths;

  ParamInt *CUT_param;
  ParamIntVector *CUTS_param;
  
public:

  SplitBlock(const string &a_name);
  virtual ~SplitBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void reset();
  virtual void build_output_data_descriptors();
  
  virtual void compute();

  void make_output_widths(vector<int> cuts, Integer total_width);
};

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: TeeBlock                                                 */
/*                                                                            */
/******************************************************************************/

class TeeBlock:public Block {

  static const string type_name;
  static const string help_text;

public:

  TeeBlock(const string &a_name);
  virtual ~TeeBlock(){};
  
  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void compute();
  
};



/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: DelayBlock                                               */
/*                                                                            */
/******************************************************************************/

const Integer	PARAM_DEFAULT_DELAY_DELAY	= 0;

class DelayBlock:public Block {

  static const string type_name;
  static const string help_text;

private:

  ParamInt *DELAY_param;
  ParamBool *REPEAT_FIRST_FRAME_param;

  Boolean first_compute;

  int delay;
  int skip_frames;
  bool repeat_frame;
  
public:

  DelayBlock(const string &a_name);
  virtual ~DelayBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void reset();
  virtual void compute();
};



/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: OutputBlock                                              */
/*                                                                            */
/******************************************************************************/

class OutputBlock:public Block {

  static const string type_name;
  static const string help_text;

private:

  Integer n_dims;   // Number of input dimensions
  vector<Integer> dim_sizes;

public:

  OutputBlock(const string &a_name);
  virtual ~OutputBlock(){};

  virtual Block* clone(const string &new_name) const;

  virtual const string &get_helptext() const {return help_text;}

private:

  virtual void reset();
  virtual void compute();
};


#endif


/* End of ctk_simple_blocks.hh */
