/******************************************************************************/
/*									      */
/*	ctk_unary_ops.cpp	      					      */
/*									      */
/*	Unary Op Blocks - these are a class of block that take a single	      */
/*       input and transform it to produce a single output. X(t)=F(Y(t))      */           
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007	         		      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <algorithm>
#include <cmath>

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_param.hh"
#include "ctk_socket.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_unary_ops.hh"

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: UnaryOpBlock                                             */
/*                                                                            */
/******************************************************************************/

UnaryOpBlock::UnaryOpBlock(const string &a_name, const string &a_block_type):Block(a_name, a_block_type) {
  make_output_sockets(1);
}

void UnaryOpBlock::reset() {
  Block::reset();
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: P0_UnaryOpBlock                                          */
/*                                                                            */
/******************************************************************************/

P0_UnaryOpBlock::P0_UnaryOpBlock(const string &a_name, const string &a_block_type):UnaryOpBlock(a_name, a_block_type) {
  make_input_sockets(1);
}

void P0_UnaryOpBlock::reset() {
  UnaryOpBlock::reset();
}


void P0_UnaryOpBlock::compute() {
  if (inputs_are_all_sample_data) {
    // For sample data
    CTKSample sample;
    (*input_sockets)[0]->get_sample(sample);
    process(sample);
    (*output_sockets)[0]->put_sample(sample);
  } else {
    // For frame data
    CTKVector *xv0;
    (*input_sockets)[0]->get_vector(xv0);
    for (CTKVector::iterator fp=xv0->begin(); fp<xv0->end(); ++fp)
      process(*fp);
    (*output_sockets)[0]->put_vector(xv0);
  }
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: P1_UnaryOpBlock                                          */
/*                                                                            */
/******************************************************************************/

P1_UnaryOpBlock::P1_UnaryOpBlock(const string &a_name, const string &a_block_type):UnaryOpBlock(a_name, a_block_type) {

  make_input_sockets(2);
  input_sockets->set_description("in2", "X parameter");
  input_sockets->set_optional("in2");
  
  // Set up X parameter
  X_param= new ParamFloat("X");
  X_param->set_helptext("The operation's parameter.");
  parameters->register_parameter(X_param);

}

void P1_UnaryOpBlock::reset() {
  UnaryOpBlock::reset();
  p1_socket_connected= (*input_sockets)[1]->connected();
  p1_socket_is_sample_data= p1_socket_connected && (*input_sockets)[1]->get_data_descriptor()->is_sample_data();

  if (!p1_socket_connected)
    param_1=X_param->get_value();

  if (p1_socket_connected && !p1_socket_is_sample_data) {
    if (input_shape_check()==false) {
      cerr << "Incompatible data and parameter inputs for block: " << getfullname() << endl;
      throw(CTKError(__FILE__, __LINE__));
      
    }
  }
}


void P1_UnaryOpBlock::compute() {
  if (p1_socket_connected==false) {
    // For scalar P1 taken from PARAMETER
    if (inputs_are_all_sample_data) {
      // For sample data 
      CTKSample sample;
      (*input_sockets)[0]->get_sample(sample);
      process(sample, param_1);
      (*output_sockets)[0]->put_sample(sample);
    } else {
      // For frame data
      CTKVector *xv0;
      (*input_sockets)[0]->get_vector(xv0);
      for (CTKVector::iterator xv0p=xv0->begin(); xv0p<xv0->end(); ++xv0p)
	process(*xv0p, param_1);
      (*output_sockets)[0]->put_vector(xv0);
    }
  } else {
    // For P1 taken from DATA STREAM
    if (inputs_are_all_sample_data) {
      // For sample data and scalar p1
      CTKSample sample, p1;
      (*input_sockets)[0]->get_sample(sample);
      (*input_sockets)[1]->get_sample(p1);
      process(sample, p1);
      (*output_sockets)[0]->put_sample(sample);
    } else if (p1_socket_is_sample_data) {
      // For vector data and scalar p1
      CTKSample p1;
      CTKVector *xv0;
      (*input_sockets)[0]->get_vector(xv0);
      (*input_sockets)[1]->get_sample(p1);
      for (CTKVector::iterator xv0p=xv0->begin(); xv0p<xv0->end(); ++xv0p)
	process(*xv0p, p1);
      (*output_sockets)[0]->put_vector(xv0);
    } else {
      CTKVector *xv0, *pv1;
      // For vector data and vector p1
      (*input_sockets)[0]->get_vector(xv0);
      (*input_sockets)[1]->get_vector(pv1);
      for (CTKVector::iterator xv0p=xv0->begin(), pv1p=pv1->begin(); xv0p<xv0->end(); ++xv0p, ++pv1p)
	process(*xv0p, *pv1p);
      (*output_sockets)[0]->put_vector(xv0);
      delete pv1;
    }

  }
}



/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: P2_UnaryOpBlock                                          */
/*                                                                            */
/******************************************************************************/

P2_UnaryOpBlock::P2_UnaryOpBlock(const string &a_name, const string &a_block_type):UnaryOpBlock(a_name, a_block_type) {

  make_input_sockets(3);
  
  input_sockets->set_description("in2", "X1 parameter");
  input_sockets->set_optional("in2");

  input_sockets->set_description("in3", "X2 parameter");
  input_sockets->set_optional("in3");

  // Set up X1 parameter
  X1_param= new ParamFloat("X1");
  X1_param->set_helptext("The operation's first parameter.");
  parameters->register_parameter(X1_param);

  // Set up X2 parameter
  X2_param= new ParamFloat("X2");
  X2_param->set_helptext("The operation's second parameter.");
  parameters->register_parameter(X2_param);

}


void P2_UnaryOpBlock::reset() {
  UnaryOpBlock::reset();
  
  p1_socket_connected= (*input_sockets)[1]->connected();
  p1_socket_is_sample_data= p1_socket_connected && (*input_sockets)[1]->get_data_descriptor()->is_sample_data();
  p1_socket_is_frame_data= p1_socket_connected && !(*input_sockets)[1]->get_data_descriptor()->is_sample_data();

  p2_socket_connected= (*input_sockets)[2]->connected();
  p2_socket_is_sample_data= p2_socket_connected && (*input_sockets)[2]->get_data_descriptor()->is_sample_data();
  p2_socket_is_frame_data= p2_socket_connected && !(*input_sockets)[2]->get_data_descriptor()->is_sample_data();

  if (!p1_socket_connected)
    param_1=X1_param->get_value();

  if (!p2_socket_connected)
    param_2=X2_param->get_value();

  const DataDescriptor *dd0=(*input_sockets)[0]->get_data_descriptor();

  if (p1_socket_connected && !p1_socket_is_sample_data) {
    const DataDescriptor *dd1=(*input_sockets)[1]->get_data_descriptor();
    if (dd0->matches_shape_of(dd1)==false) {
      cerr << "Incompatible data and first parameter inputs for block: " << getfullname() << endl;
      throw(CTKError(__FILE__, __LINE__));      
    }
  }

  if (p2_socket_connected && !p2_socket_is_sample_data) {
    const DataDescriptor *dd2=(*input_sockets)[2]->get_data_descriptor();
    if (dd0->matches_shape_of(dd2)==false) {
      cerr << "Incompatible data and second parameter inputs for block: " << getfullname() << endl;
      throw(CTKError(__FILE__, __LINE__));      
    }
  }

}


void P2_UnaryOpBlock::compute() {
  if (p1_socket_connected==false && p2_socket_connected==false) {
    // For scalar P1 and P2 taken from PARAMETER
    if (inputs_are_all_sample_data) {
      // For sample data 
      CTKSample sample;
      (*input_sockets)[0]->get_sample(sample);
      process(sample, param_1, param_2);
      (*output_sockets)[0]->put_sample(sample);
    } else {
      // For frame data
      CTKVector *xv0;
      (*input_sockets)[0]->get_vector(xv0);
      for (CTKVector::iterator xv0p=xv0->begin(); xv0p<xv0->end(); ++xv0p)
	process(*xv0p, param_1, param_2);
      (*output_sockets)[0]->put_vector(xv0);
    }
  } else {
    CTKVector *xv0, *pv1=NULL, *pv2=NULL;
    (*input_sockets)[0]->get_vector(xv0);
    
    if (p1_socket_is_frame_data)
      (*input_sockets)[1]->get_vector(pv1);
    else if (p1_socket_is_sample_data) 
      (*input_sockets)[1]->get_sample(param_1);

    if (p2_socket_is_frame_data)
      (*input_sockets)[2]->get_vector(pv2);
    else if (p2_socket_is_sample_data) 
      (*input_sockets)[2]->get_sample(param_2);

    if (pv1!=NULL && pv2!=NULL) {
      // Both parameteres are vectors
      for (CTKVector::iterator xv0p=xv0->begin(), pv1p=pv1->begin(), pv2p=pv2->begin(); xv0p<xv0->end(); ++xv0p, ++pv1p, ++pv2p)
	process(*xv0p, *pv1p, *pv2p);
      delete pv1;
      delete pv2;
    } else if (pv2!=NULL) {
      // P1 is scalar, P2 is vector
      for (CTKVector::iterator xv0p=xv0->begin(),  pv2p=pv2->begin(); xv0p<xv0->end(); ++xv0p, ++pv2p)
	process(*xv0p, param_1, *pv2p);
      delete pv2;
    } else {
      // P1 is vector, P2 is scalar
      for (CTKVector::iterator xv0p=xv0->begin(), pv1p=pv1->begin(); xv0p<xv0->end(); ++xv0p, ++pv1p)
	process(*xv0p, *pv1p, param_2);
      delete pv1;
    }

    (*output_sockets)[0]->put_vector(xv0);
  }
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: HalfWaveRectifyBlock                                     */
/*                                                                            */
/******************************************************************************/

const string HalfWaveRectifyBlock::type_name = "HalfWaveRectify";
const string HalfWaveRectifyBlock::help_text = HALFWAVE_RECTIFY_BLOCK_HELP_TEXT;

HalfWaveRectifyBlock::HalfWaveRectifyBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* HalfWaveRectifyBlock::clone(const string &n) const{
  Block *ablock = new HalfWaveRectifyBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void HalfWaveRectifyBlock::process(Float &x) {
  if (x<0.0) x=0.0;
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: FullWaveRectifyBlock                                     */
/*                                                                            */
/******************************************************************************/

const string FullWaveRectifyBlock::type_name = "FullWaveRectify";
const string FullWaveRectifyBlock::help_text = FULLWAVE_RECTIFY_BLOCK_HELP_TEXT;

FullWaveRectifyBlock::FullWaveRectifyBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* FullWaveRectifyBlock::clone(const string &n) const{
  Block *ablock = new FullWaveRectifyBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void FullWaveRectifyBlock::process(Float &x) {
  if (x<0.0) x=-x;
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: AmpToDBBlock                                             */
/*                                                                            */
/******************************************************************************/

const string AmpToDBBlock::type_name = "AmpToDB";
const string AmpToDBBlock::help_text = AMPTODB_BLOCK_HELP_TEXT;

AmpToDBBlock::AmpToDBBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* AmpToDBBlock::clone(const string &n) const{
  Block *ablock = new AmpToDBBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void AmpToDBBlock::process(Float &x) {
  x=20.0*log10(fabs(x));
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: DBToAmpBlock                                             */
/*                                                                            */
/******************************************************************************/

const string DBToAmpBlock::type_name = "DBToAmp";
const string DBToAmpBlock::help_text = DBTOAMP_BLOCK_HELP_TEXT;

DBToAmpBlock::DBToAmpBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* DBToAmpBlock::clone(const string &n) const{
  Block *ablock = new DBToAmpBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void DBToAmpBlock::process(Float &x) {
  x=pow(10.0,x/20.0);
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: Log10Block                                               */
/*                                                                            */
/******************************************************************************/

const string Log10Block::type_name = "Log10";
const string Log10Block::help_text = LOG10_BLOCK_HELP_TEXT;

Log10Block::Log10Block(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* Log10Block::clone(const string &n) const{
  Block *ablock = new Log10Block(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void Log10Block::process(Float &x) {
  x=log10(x);
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: LogBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string LogBlock::type_name = "Log";
const string LogBlock::help_text = LOG_BLOCK_HELP_TEXT;

LogBlock::LogBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* LogBlock::clone(const string &n) const{
  Block *ablock = new LogBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void LogBlock::process(Float &x) {
  x=log(x);
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: ExpBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string ExpBlock::type_name = "Exp";
const string ExpBlock::help_text = EXP_BLOCK_HELP_TEXT;

ExpBlock::ExpBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* ExpBlock::clone(const string &n) const{
  Block *ablock = new ExpBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void ExpBlock::process(Float &x) {
  x=exp(x);
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: LogicalNOTBlock                                          */
/*                                                                            */
/******************************************************************************/

const string LogicalNOTBlock::type_name = "LogicalNOT";
const string LogicalNOTBlock::help_text = LOGICAL_NOT_BLOCK_HELP_TEXT;

LogicalNOTBlock::LogicalNOTBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* LogicalNOTBlock::clone(const string &n) const{
  Block *ablock = new LogicalNOTBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void LogicalNOTBlock::process(Float &x) {
  x=(x==0.0);
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SinBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string SinBlock::type_name = "Sin";
const string SinBlock::help_text = SIN_BLOCK_HELP_TEXT;

SinBlock::SinBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* SinBlock::clone(const string &n) const{
  Block *ablock = new SinBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void SinBlock::process(Float &x) {
  x=sin(x);
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: CosBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string CosBlock::type_name = "Cos";
const string CosBlock::help_text = COS_BLOCK_HELP_TEXT;

CosBlock::CosBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* CosBlock::clone(const string &n) const{
  Block *ablock = new CosBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void CosBlock::process(Float &x) {
  x=cos(x);
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: TanBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string TanBlock::type_name = "Tan";
const string TanBlock::help_text = TAN_BLOCK_HELP_TEXT;

TanBlock::TanBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* TanBlock::clone(const string &n) const{
  Block *ablock = new TanBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void TanBlock::process(Float &x) {
  x=tan(x);
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: AsinBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string AsinBlock::type_name = "Asin";
const string AsinBlock::help_text = ASIN_BLOCK_HELP_TEXT;

AsinBlock::AsinBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* AsinBlock::clone(const string &n) const{
  Block *ablock = new AsinBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void AsinBlock::process(Float &x) {
  x=asin(x);
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: AcosBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string AcosBlock::type_name = "Acos";
const string AcosBlock::help_text = ACOS_BLOCK_HELP_TEXT;

AcosBlock::AcosBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* AcosBlock::clone(const string &n) const{
  Block *ablock = new AcosBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void AcosBlock::process(Float &x) {
  x=acos(x);
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: AtanBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string AtanBlock::type_name = "Atan";
const string AtanBlock::help_text = ATAN_BLOCK_HELP_TEXT;

AtanBlock::AtanBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* AtanBlock::clone(const string &n) const{
  Block *ablock = new AtanBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void AtanBlock::process(Float &x) {
  x=atan(x);
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SinhBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string SinhBlock::type_name = "Sinh";
const string SinhBlock::help_text = SINH_BLOCK_HELP_TEXT;

SinhBlock::SinhBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* SinhBlock::clone(const string &n) const{
  Block *ablock = new SinhBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void SinhBlock::process(Float &x) {
  x=sinh(x);
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: CoshBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string CoshBlock::type_name = "Cosh";
const string CoshBlock::help_text = COSH_BLOCK_HELP_TEXT;

CoshBlock::CoshBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* CoshBlock::clone(const string &n) const{
  Block *ablock = new CoshBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void CoshBlock::process(Float &x) {
  x=cosh(x);
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: TanhBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string TanhBlock::type_name = "Tanh";
const string TanhBlock::help_text = TANH_BLOCK_HELP_TEXT;

TanhBlock::TanhBlock(const string &a_name):CTKObject(a_name),P0_UnaryOpBlock(a_name, type_name) {}

Block* TanhBlock::clone(const string &n) const{
  Block *ablock = new TanhBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void TanhBlock::process(Float &x) {
  x=tanh(x);
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: FloorBlock                                               */
/*                                                                            */
/******************************************************************************/

const string FloorBlock::type_name = "Floor";
const string FloorBlock::help_text = FLOOR_BLOCK_HELP_TEXT;

FloorBlock::FloorBlock(const string &a_name):CTKObject(a_name),P1_UnaryOpBlock(a_name, type_name) {
  X_param->set_helptext("The value below which to truncate the signal.");
}

Block* FloorBlock::clone(const string &n) const{
  Block *ablock = new FloorBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void FloorBlock::process(Float &x, Float p1) {
  if (x<p1) x=p1;
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: CeilingBlock                                             */
/*                                                                            */
/******************************************************************************/

const string CeilingBlock::type_name = "Ceiling";
const string CeilingBlock::help_text = CEILING_BLOCK_HELP_TEXT;

CeilingBlock::CeilingBlock(const string &a_name):CTKObject(a_name),P1_UnaryOpBlock(a_name, type_name) {
  X_param->set_helptext("The value above which to truncate the signal.");
}

Block* CeilingBlock::clone(const string &n) const{
  Block *ablock = new CeilingBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void CeilingBlock::process(Float &x, Float p1) {
  if (x>p1) x=p1;
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: ScaleBlock                                               */
/*                                                                            */
/******************************************************************************/

const string ScaleBlock::type_name = "Scale";
const string ScaleBlock::help_text = SCALE_BLOCK_HELP_TEXT;

ScaleBlock::ScaleBlock(const string &a_name):CTKObject(a_name),P1_UnaryOpBlock(a_name, type_name) {
  X_param->set_helptext("The value by which to multiply the signal.");
}

Block* ScaleBlock::clone(const string &n) const{
  Block *ablock = new ScaleBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void ScaleBlock::process(Float &x, Float p1) {
  x*=p1;
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: OffsetBlock                                              */
/*                                                                            */
/******************************************************************************/

const string OffsetBlock::type_name = "Offset";
const string OffsetBlock::help_text = OFFSET_BLOCK_HELP_TEXT;


OffsetBlock::OffsetBlock(const string &a_name):CTKObject(a_name),P1_UnaryOpBlock(a_name, type_name) {
  X_param->set_helptext("The value which to add to the signal.");
}

Block* OffsetBlock::clone(const string &n) const{
  Block *ablock = new OffsetBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void OffsetBlock::process(Float &x, Float p1) {
  x+=p1;
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: PowerBlock                                              */
/*                                                                            */
/******************************************************************************/

const string PowerBlock::type_name = "Power";
const string PowerBlock::help_text = POWER_BLOCK_HELP_TEXT;

PowerBlock::PowerBlock(const string &a_name):CTKObject(a_name),P1_UnaryOpBlock(a_name, type_name) {
  X_param->set_helptext("The power by which to raise the signal.");
}

Block* PowerBlock::clone(const string &n) const{
  Block *ablock = new PowerBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void PowerBlock::process(Float &x, Float p1) {
  x=pow(x,p1);
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: SigmoidBlock                                             */
/*                                                                            */
/******************************************************************************/

const string SigmoidBlock::type_name = "Sigmoid";
const string SigmoidBlock::help_text = SIGMOID_BLOCK_HELP_TEXT;

SigmoidBlock::SigmoidBlock(const string &a_name):CTKObject(a_name),P1_UnaryOpBlock(a_name, type_name) {
  X_param->set_helptext("The slope parameter of the sigmoid. i.e y=1.0/(1.0+exp(X*x)");
}

Block* SigmoidBlock::clone(const string &n) const{
  Block *ablock = new SigmoidBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void SigmoidBlock::process(Float &x, Float p1) {
  x=1.0/(1.0+exp(-p1*x));
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: GreaterThanXBlock                                        */
/*                                                                            */
/******************************************************************************/

const string GreaterThanXBlock::type_name = "GreaterThanX";
const string GreaterThanXBlock::help_text = GREATER_THAN_X_BLOCK_HELP_TEXT;

GreaterThanXBlock::GreaterThanXBlock(const string &a_name):CTKObject(a_name),P1_UnaryOpBlock(a_name, type_name) {
  X_param->set_helptext("The value with which to compare the signal.");
}

Block* GreaterThanXBlock::clone(const string &n) const{
  Block *ablock = new GreaterThanXBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void GreaterThanXBlock::process(Float &x, Float p1) {
  x=(x>p1);
}
 
/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: LessThanXBlock                                           */
/*                                                                            */
/******************************************************************************/

const string LessThanXBlock::type_name = "LessThanX";
const string LessThanXBlock::help_text = LESS_THAN_X_BLOCK_HELP_TEXT;

LessThanXBlock::LessThanXBlock(const string &a_name):CTKObject(a_name),P1_UnaryOpBlock(a_name, type_name) {
  X_param->set_helptext("The value with which to compare the signal.");
}

Block* LessThanXBlock::clone(const string &n) const{
  Block *ablock = new LessThanXBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void LessThanXBlock::process(Float &x, Float p1) {
  x=(x<p1);
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: EqualToXBlock                                           */
/*                                                                            */
/******************************************************************************/

const string EqualToXBlock::type_name = "EqualToX";
const string EqualToXBlock::help_text = EQUAL_TO_X_BLOCK_HELP_TEXT;

EqualToXBlock::EqualToXBlock(const string &a_name):CTKObject(a_name),P1_UnaryOpBlock(a_name, type_name) {
  X_param->set_helptext("The value with which to compare the signal.");
}

Block* EqualToXBlock::clone(const string &n) const{
  Block *ablock = new EqualToXBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void EqualToXBlock::process(Float &x, Float p1) {
  x=(x==p1);
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: NotEqualToXBlock                                           */
/*                                                                            */
/******************************************************************************/

const string NotEqualToXBlock::type_name = "NotEqualToX";
const string NotEqualToXBlock::help_text = NOT_EQUAL_TO_X_BLOCK_HELP_TEXT;

NotEqualToXBlock::NotEqualToXBlock(const string &a_name):CTKObject(a_name),P1_UnaryOpBlock(a_name, type_name) {
  X_param->set_helptext("The value with which to compare the signal.");
}

Block* NotEqualToXBlock::clone(const string &n) const{
  Block *ablock = new NotEqualToXBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void NotEqualToXBlock::process(Float &x, Float p1) {
  x=(x!=p1);
}


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: FixBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string FixBlock::type_name = "Fix";
const string FixBlock::help_text = FIX_BLOCK_HELP_TEXT;


FixBlock::FixBlock(const string &a_name):CTKObject(a_name),P1_UnaryOpBlock(a_name, type_name) {
  X_param->set_helptext("The value with which to replace occurrences of NaN.");
}

Block* FixBlock::clone(const string &n) const{
  Block *ablock = new FixBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void FixBlock::process(Float &x, Float p1) {
  if (isnan(x)) {
    x=p1;
  }
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: NormalPDFBlock                                           */
/*                                                                            */
/******************************************************************************/

const string NormalPDFBlock::type_name = "NormalPDF";
const string NormalPDFBlock::help_text = NORMAL_PDF_BLOCK_HELP_TEXT;

NormalPDFBlock::NormalPDFBlock(const string &a_name):CTKObject(a_name),P2_UnaryOpBlock(a_name, type_name) {

  // Set up LOG_PROB parameter
  LOG_PROBS_param= new ParamBool("LOG_PROBS", false);
  LOG_PROBS_param->set_helptext("If true then the block outputs log probabilities rather than probabilities.");
  parameters->register_parameter(LOG_PROBS_param);

  X1_param->rename("MEAN");
  X1_param->set_helptext("The mean of the normal distribution.");

  X2_param->rename("VARIANCE");
  X2_param->set_helptext("The variance of the normal distribution.");

  input_sockets->set_description("in2", "mean");
  input_sockets->set_description("in3", "variance");
  
}

void NormalPDFBlock::reset(){
  P2_UnaryOpBlock::reset();

  log_probs=LOG_PROBS_param->get_value();

  constant_variance=false;
  
  if ((*input_sockets)[2]->connected()==false) {
    constant_variance=true;
    calc_gconst_varm05(param_2);
  }

}
  
Block* NormalPDFBlock::clone(const string &n) const{
  Block *ablock = new NormalPDFBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void NormalPDFBlock::process(Float &x, Float p1, Float p2) {

  Float diff = x-p1;

  if (constant_variance==false) {
    calc_gconst_varm05(p2);  // if variance is changing each frame then these need updating
  }
  
  if (log_probs) 
    x = log_gconst + varm05*diff*diff;
  else
    x=  gconst * exp(varm05 * diff * diff);
}


// Stuff that can be pre-computed if the variance is constant
void NormalPDFBlock::calc_gconst_varm05(float variance) {
  gconst=1.0/(sqrt(2*M_PI*variance));
  if (log_probs)
    log_gconst=log(gconst);
  varm05=-0.5/variance;
}

/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: ErfBlock                                                 */
/*                                                                            */
/******************************************************************************/

const string ErfBlock::type_name = "Erf";
const string ErfBlock::help_text = ERF_BLOCK_HELP_TEXT;

ErfBlock::ErfBlock(const string &a_name):CTKObject(a_name),P2_UnaryOpBlock(a_name, type_name) {

  X1_param->rename("MEAN");
  X1_param->set_helptext("The mean of the normal distribution.");

  X2_param->rename("VARIANCE");
  X2_param->set_helptext("The variance of the normal distribution.");

  input_sockets->set_description("in2", "mean");
  input_sockets->set_description("in3", "variance");
  
}

void ErfBlock::reset(){
  P2_UnaryOpBlock::reset();

  varconst=sqrt(0.5/param_2);
}

Block* ErfBlock::clone(const string &n) const{
  Block *ablock = new ErfBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

void ErfBlock::process(Float &x, Float p1, Float) {
  x=0.5*(1.0+erf((x - p1)*varconst));
}



/* End of ctk_unary_ops.cpp */
