/******************************************************************************/
/*									      */
/*	ctk_acg_pitch.cpp	    	       			              */
/*									      */
/*	computes acg-based pitch colouring                           */
/*									      */
/*	Author: Martin Cooke, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007			      	      */
/*									      */
/******************************************************************************/
// CHANGE LOG
//
// 12/10/00  -  JPB  -  Replaced hardcoded sample rate with value of blocks SAMPLERATE parameter
//
//

#include "ctk-config.h"

#include <cmath>
#include <vector>
 
#include <numeric>
#include <algorithm>

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_dsp.hh"

#include "ctk_param.hh"
#include "ctk_socket.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_acg_pitch.hh"


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: AcgPitchBlock                                          */
/*                                                                            */
/******************************************************************************/

const string AcgPitchBlock::type_name = "AcgPitch";
const string AcgPitchBlock::help_text = "computes ACG-based pitch colouring";

AcgPitchBlock::AcgPitchBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {
  make_input_sockets(1);
  make_output_sockets(2);
  input_sockets->set_description("in1", "ACG");
  output_sockets->set_description("out1", "colouring");
  output_sockets->set_description("out2", "lags");

  // Set up NPITCHES parameter
  NPITCHES_param= new ParamInt("Maximum_pitches", PARAM_DEFAULT_NPITCHES);
  NPITCHES_param->set_helptext("Maximum number of pitches to output per frame");  
  NPITCHES_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(NPITCHES_param);

}
 

void AcgPitchBlock::build_output_data_descriptors() {
    
  const DataDescriptor *idd = (*input_sockets)[0]->get_data_descriptor();
  const DimensionDescriptor *dimension = idd->get_dimension("FREQUENCY");

  if (dimension == NULL) {
    cerr << "Error in AcgPitchBlock: cannot find frequencies descriptor" << endl;
    cerr << "Ensure something generates them earlier in the chain (eg gammatone)." << endl;
    throw(CTKError(__FILE__, __LINE__));
  } 

  cfs = dimension->get_axis();
  DataDescriptor *dd = new DataDescriptor();
  dd->add_inner_dimension(string("FREQUENCY"), cfs);   

  // Pass this descriptor to all the output channels 
  (*output_sockets)[0]->set_data_descriptor(dd);
  (*output_sockets)[1]->set_data_descriptor(new DataDescriptor(*dd));
  
}

void AcgPitchBlock::reset() {
  Block::reset();
  
  n_dims=(*input_sockets)[0]->get_data_descriptor()->get_n_dimensions();
  dim_sizes=(*input_sockets)[0]->get_data_descriptor()->get_dimension_sizes();
   
  // Check validity of input data
  if (n_dims!=2) {
    cerr << "Error in network at block: " << getname() << endl;
    cerr << "Attempting to apply acgPitch to " << n_dims << "-dimensional frames." << endl;
    cerr << "Block requires 2 dimensional data at present." << endl;
   throw(CTKError(__FILE__, __LINE__));
  }

  maxpitches = NPITCHES_param->get_value();
  n_chans = dim_sizes[0];
  n_lags = dim_sizes[1];

  // Get the sample rate of the incoming data - JPB
  fs=sample_rate_param->get_value();
  
  //cerr << "n chans and n lags " << n_chans << " " << n_lags << endl;

}

Block* AcgPitchBlock::clone(const string &n) const{
  Block *ablock = new AcgPitchBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}

int pwf(float cf) {
  if (cf < 150) {
    return 3;
  } else {
    if (cf < 300) {
      return 2;
    } else {
      if (cf < 500) {
	return 1;
      } else
	return 0;
    }
  }
}

void AcgPitchBlock::compute() {
  
  CTKVector *invec;
  (*input_sockets)[0]->get_vector(invec);

  vector<int> colouring(n_chans);
  vector<int> lags(n_chans);
  CTKVector summary(n_lags);
  CTKVector skel(n_lags*n_chans);
  vector<bool> peakat(n_lags*n_chans);

  int colour = 0;

  // initialise

  for (int chan=0; chan < n_chans; chan++) {
    colouring[chan] = 0;
    lags[chan] = 0;
  }

  for (int i=0; i< n_chans*n_lags; i++) {
    peakat[i] = false;
    skel[i] = (*invec)[i]; //  0.0;
  }

  // compute pitch range of lags (75-250 Hz)

  float minf = 75.0;
  float maxf = 250.0;
  int maxlag = min(n_lags, (int) (fs/minf));
  int minlag = (int) (fs/maxf);

  // skeletonise and mark peaks

  for (int l=minlag; l< maxlag; l++) summary[l]=0.0;

  for (int chan=0; chan < n_chans; chan++) {
    int offset = chan*n_lags;
    int halfPeakWidth = pwf(cfs[chan]);

    for (int l=minlag+halfPeakWidth+1; l < maxlag-halfPeakWidth-1; l++) {
      int ol= offset+l;
      if (((*invec)[ol] > (*invec)[ol+1]) && ((*invec)[ol] > (*invec)[ol-1])) {
	// skel[ol] = 1.0; // (*invec)[ol];
	// mark as having a peak in the vicinity based on CF
	for (int p=ol-halfPeakWidth; p <= ol+halfPeakWidth; p++)
	  peakat[p] = true;
	// peakat[ol-1] = true;
	// peakat[ol+1] = true;
	
      }
    }
  }
 
  // compute summary

  for (int l=minlag; l< maxlag; l++) summary[l]=0.0;
  for (int chan=0; chan < n_chans; chan++) {
    int offset = chan*n_lags;
    for (int l=minlag; l< maxlag; l++) {
      summary[l] += skel[offset+l];
    }
  }
 
  // find largest peak
  float maxsum = -1.0;
  int peak = 0;
  for (int s=minlag+1; s < maxlag-1; s++) {
    if ((summary[s] > summary[s-1]) && (summary[s] > summary[s+1]))
      if (summary[s] > maxsum) {
	peak = s;
	maxsum = summary[s];
      }
  }

  //  cerr << "peak at lag: ";
  while ((peak > 0) && (colour < maxpitches)) {
     
    //  cerr << peak << " ";

    colour++;
    int removed = 0;

    // colour channels with peak at that lag
    for (int chan=0; chan < n_chans; chan++) {
      if ((colouring)[chan] == 0) { // channel not yet assigned
	 int offset = chan*n_lags+peak;
	 if (peakat[offset]) {
	   colouring[chan] = colour;
	   //	   cerr << "P";
	   lags[chan] = peak;
	   removed++;
	 }
      }
    }

    
    if (removed > 0) {

      // recompute summary from nonzero channels (if any left)
      for (int l=minlag; l< maxlag; l++) summary[l]=0.0;
      for (int chan=0; chan < n_chans; chan++) {
	if ((colouring)[chan] == 0) {
	  int offset = chan*n_lags;
	  for (int l=minlag; l< maxlag; l++) {
	    summary[l] += skel[offset+l];
	  }
	}
      }

      // find largest peak
      float maxsum = -1.0;
      peak = 0;
      for (int s=minlag+1; s < maxlag-1; s++) {
	if ((summary[s] > summary[s-1]) && (summary[s] > summary[s+1]))
	  if (summary[s] > maxsum) {
	    peak = s;
	    maxsum = summary[s];
	  }
      }

    } else
      peak = 0;
  }
  //  cerr << endl;

					    
  CTKVector *col = new CTKVector(n_chans);
  CTKVector *dat = new CTKVector(n_chans);
 
  for (int i=0; i< n_chans; i++) {
    (*col)[i] = (float) colouring[i];
    (*dat)[i] = (float) lags[i];
  }

  (*output_sockets)[0]->put_vector(col);  
  (*output_sockets)[1]->put_vector(dat);  

  delete invec;
}




/* End of ctk_acg_pitch.cpp */
 
