/******************************************************************************/
/*									      */
/*	ctk_adaptive_noise_estimation.cpp                                       */
/*									      */
/*	Author: Ljubomir Josifovski, Sheffield University             	      */
/*									      */
/*     CTK VERSION 1.3.5  Apr 22, 2007                    		      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <cmath>
#include <cassert>

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"
#include "ctk_data_descriptor.hh"

#include "ctk_adaptive_noise_estimation.hh"

#define ASSERTIT(q) assert(q)

/******************************************************************************/
/* 									       */
/*       CLASS NAME:  	AdaptiveNoiseEstimationBlock	       		      */
/*									      */
/******************************************************************************/

/***************************************************************************
********

AdaptiveNoiseEstimationBlock (implements nest_adaptive function) is based on
the Matlab code:

function [sub,neg,nmu,nvar,adapt,ppres]=snra(noisy,thr,teta)
%
% LBJ 23/JUN/2000

if exist('thr')~=1 | isempty(thr); thr=6.9524; end
if exist('teta')~=1 | isempty(teta); teta=0.5; end
 
[nmu,nvar,adapt]=nest_adaptive(noisy);

nx=1+10^(thr/20);
sqrt2=sqrt(2);
ppres = 0.5+0.5*erfs( (noisy/nx-nmu)./(sqrt2*sqrt(nvar)) ); % P(SNR>X)
neg = (ppres < teta); % P(SNR>X) < teta

sub = noisy-nmu;
neg = neg | (sub<0);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function e=erfs(x)
e=2./(1.+exp(-2.3236*x))-1;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [nmu,nvar,adapt]=nest_adaptive(noisy,thr,teta,alpha,nfr)
%NEST_ADAPTIVE return the mean and the variance of adaptive noise estimate
%
% [nmu,nvar,adapt]=nest_adaptive(noisy,thr,teta,nfr)
%
% In:
%   noisy - the noisy speech
%   thr - threshold in dB (default 0)
%   teta - factor (default 0.7)
%   alpha - "remembering" factor (default 0.995)
%   nfr - use frames 2:nfr+1 for the initial noise estimate (default 10)
%
% Out:
%   nmu - mean of the noise estimate
%   nvar - variance of the noise estimate
%   adapt - when noise estimate was adapted
%
% Assumes magnitude representation.
% Assumes normal distrib for the noise. The noise estimates are
% updated with:
%   mu(t) = alpha*mu(t-1) + (1-alpha)*noisy(t)
%   s(t) = alpha*s(t-1) + (1-alpha)*noisy(t)^2
%   var(t) = s(t) - mu(t)^2
% Updates are independent in each individual channel. The noise
% estimates are updated only if P(SNR<thr)>teta.
%
% LBJ 23/JUN/2000

if exist('thr')~=1 | isempty(thr); thr=-6.9524; end
if exist('teta')~=1 | isempty(teta); teta=0.6; end
if exist('alpha')~=1 | isempty(alpha); alpha=0.995; end
if exist('nfr')~=1 | isempty(nfr); nfr=10; end  

nx=1+10^(thr/20);
sqrt2=sqrt(2);

nmu=zeros(size(noisy));
nvar=zeros(size(noisy));
adapt=zeros(size(noisy));
[framesize,numframes]=size(noisy);

% use average of first nfr frames as initial estimate
m1=sum(noisy(:,2:nfr+1),2)/nfr;
s1=sum(noisy(:,2:nfr+1).^2,2)/(nfr-1);
v1=s1-m1.^2;

nmu(:,1:nfr)=m1(:,ones(1,nfr));
nvar(:,1:nfr)=v1(:,ones(1,nfr));

%for f=nfr+1:numframes
for f=1:numframes
   y = noisy(:,f);
   m =   alpha*m1 + (1-alpha)*y;
   s =   alpha*s1 + (1-alpha)*y.^2;
   v =   s - m.^2;

   p = 0.5-0.5*erf( (y/nx-m)./(sqrt2*sqrt(v)) ); % P(SNR<X)
   q = (p > teta); % P(SNR<X) > teta
   adapt(:,f)=q;

   m1(q) = m(q); nmu(:,f) = m1;
   s1(q) = s(q);     
   v1(q) = v(q); nvar(:,f) = v1;
end


****************************************************************************
*******/

const string AdaptiveNoiseEstimationBlock::type_name = "AdaptiveNoiseEstimation";
const string AdaptiveNoiseEstimationBlock::help_text = "Adaptive noise estimation. To be applied to spectral magnitude data. Assumes normal distribution for noise. The estimates of the mean mu(t) and the variance var(t) of the noise are updated with: mu(t) = memory_factor*mu(t-1) + (1-memory_factor)*noisy(t); s(t) = memory_factor*s(t-1) + (1-memory_factor)*noisy(t)^2; var(t) = s(t) - mu(t)^2. Updates are independent in each individual frequency channel. The estimates are updated only iif P(estimated_SNR&lt;noisy_threshold) &gt; probability_threshold.";

AdaptiveNoiseEstimationBlock::AdaptiveNoiseEstimationBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name ) {

   // inputs
   make_input_sockets(2);
   input_sockets->set_description("in1", "noisy data");
   input_sockets->set_description("in2", "update mask");
   input_sockets->set_optional("in2");

   // outputs
   make_output_sockets(3);
   output_sockets->set_description("out1", "estimated noise mean");
   output_sockets->set_description("out2", "estimated noise variance");
   output_sockets->set_description("out3", "the update mask (non-zero signals that the corresponding mean and variance were updated, compared to the ones in the previous frame)");

   // Set up HAS_DELTAS parameter
   HAS_DELTAS_param = new ParamBool("HAS_DELTAS", false);
   ASSERTIT(HAS_DELTAS_param);
   HAS_DELTAS_param->set_helptext("Set to true if the input has delta features");
   parameters->register_parameter(HAS_DELTAS_param);

   // Set up NFRAMES parameter
   NFRAMES_param = new ParamInt("NFRAMES", 10);
   ASSERTIT(NFRAMES_param);
   NFRAMES_param->set_helptext("The number of frames - taken from the beginning of the signal - from which the initial noise estimate is made.");
   NFRAMES_param->install_validator(new Validator(Validator::VLOWER, 1.0));
   parameters->register_parameter(NFRAMES_param);

   // Set up NOISE_THRESHOLD parameter
   NOISE_THRESHOLD_param = new ParamFloat("NOISE_THRESHOLD", -6.9524);
   ASSERTIT(NOISE_THRESHOLD_param);
   NOISE_THRESHOLD_param->set_helptext("The `noisy_threshold' (in dB) used in P(estimated_SNR&lt;noisy_threshold) &gt; probability_threshold. Check block's documentation.");
   NOISE_THRESHOLD_param->set_unit("dB");
   parameters->register_parameter(NOISE_THRESHOLD_param);

   // Set up PROBABILITY_THRESHOLD parameter
   PROBABILITY_THRESHOLD_param= new ParamFloat("PROBABILITY_THRESHOLD", 0.6);
   ASSERTIT(PROBABILITY_THRESHOLD_param);
   PROBABILITY_THRESHOLD_param->set_helptext("The `probability_threshold' used in P(estimated_SNR&lt;noisy_threshold) &gt; probability_threshold. Check block's documentation");
   parameters->register_parameter(PROBABILITY_THRESHOLD_param);

   // Set up MEMORY_FACTOR parameter
   MEMORY_FACTOR_param= new ParamFloat("MEMORY_FACTOR", 0.995);
   ASSERTIT(MEMORY_FACTOR_param);
   MEMORY_FACTOR_param->set_helptext("The `memory_factor' used for means and variace tracking. Check block's documentation");
   parameters->register_parameter(MEMORY_FACTOR_param);

}

Block* AdaptiveNoiseEstimationBlock::clone(const string &n) const{

   Block *ablock = new AdaptiveNoiseEstimationBlock(n.empty()?getname():n);
   ASSERTIT(ablock);
   return copy_this_block_to(ablock);

}

void AdaptiveNoiseEstimationBlock::reset() {

  Block::reset();
  
  // Check validity of input data
  const DataDescriptor *dd=(*input_sockets)[0]->get_data_descriptor();
  Integer nDims=dd->get_n_dimensions();
  if (nDims!=1) {
    cerr << "Error in network at block: " << getname() << "\n";
    cerr << "Attempting to apply noise estimation to ";
    if (nDims==0)
      cerr << "sample data\n";
    else
      cerr << nDims << "-dimensional frame data.\n";
    cerr << "Noise estimation can only be applied to 1 dimensional frame data.\n";
    throw(CTKError(__FILE__, __LINE__));
  }

  // Check the inputs all have the same shape
  if ((*input_sockets)[1]->connected() && input_shape_check()==false) {
    Integer x1=(*input_sockets)[0]->get_data_descriptor()->get_storage();
    Integer x2=(*input_sockets)[1]->get_data_descriptor()->get_storage();
    cerr << "AdaptiveNoiseEstimation:: Inputs have unequal widths (Data=" << x1 << ", mask=" << x2 << ")" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  // init params
  nframes = NFRAMES_param->get_value();
  if (! (nframes > 0)) {
    cerr<<"NFRAMES parameter in " << getname() << " has been set to " << nframes << ". But NFRAMES must be positive!" << endl;
    throw(CTKError(__FILE__, __LINE__));
  }
  noise_threshold = NOISE_THRESHOLD_param->get_value();
  probability_threshold = PROBABILITY_THRESHOLD_param->get_value();
  memory_factor = MEMORY_FACTOR_param->get_value();
  has_deltas = HAS_DELTAS_param->get_value();
  
  // other vars
  frames_left = nframes;
  frame_buffer.resize(0);

  adapted_mask_in=NULL;
}


// add a new frame to the frame_buffer
void AdaptiveNoiseEstimationBlock::buffer_one_frame() {
  
  CTKVector *invec;
  (*input_sockets)[0]->get_vector(invec);
  frame_buffer.push_back(invec);

}


// compute initial estimates from data in frame_buffer
void AdaptiveNoiseEstimationBlock::compute_initial_estimate() {

  UInteger numframes = frame_buffer.size();
  UInteger vecsize = frame_buffer[0]->size();

  noise_mean.clear();
  noise_mean.resize(vecsize, 0.0);
  
  noise_mean2.clear();
  noise_mean2.resize(vecsize, 0.0);
  
  for (UInteger j=0; j<vecsize; j++) {
    for (UInteger i=0; i<numframes; i++) {
      
      float q = (*frame_buffer[i])[j];
      noise_mean[j] += q;
      noise_mean2[j] += q*q;
      
    }
  }
  
  for (UInteger j=0; j<vecsize; j++) {
     
    noise_mean[j] /= numframes;
    noise_mean2[j] /= numframes;
    
  }
  
  noise_variance.clear();
  noise_variance.resize(vecsize, 0.0);
  
  for (UInteger j=0; j<vecsize; j++) {
    for (UInteger i=0; i<numframes; i++) {
      
      float q = (*frame_buffer[i])[j] - noise_mean[j];
      noise_variance[j] += q*q;
      
    }
  }
  
  for (UInteger j=0; j<vecsize; j++)
    noise_variance[j] /= (numframes-1);
  
}

// the algorithm for adaptive noise tracking
void AdaptiveNoiseEstimationBlock::update_internal_state(CTKVector *pv) {

   float nx = 1+pow(10, noise_threshold/20);

   unsigned int half_size = pv->size()/2;
   
   for (UInteger i=0; i<pv->size(); i++) {

      float y = (*pv)[i];
      float m = memory_factor*noise_mean[i] + (1.0-memory_factor)*y;
      float s = memory_factor*noise_mean2[i] + (1.0-memory_factor)*y*y;
      float v = s - m*m;

      // Assuming additivity, ie noisy speech y=s+n and SNR=s/n:
      // P( SNR<X ) = P( y/n-1 < X ) = P( n > y/(1+X) ) = 1 - 1/2*erf(((y/(1+X)) - \mu)/sqrt(2*\sigma^2) )
      float p;
      if (v<=0)
	p=0.0;
      else
	p= 0.5 - 0.5*erfs( (y/nx - m)/sqrt(2*v) );
      //cout << "p(" << i << ")= " << p << endl;
      ASSERTIT(p>=0.0 && p<=1.0);

      if (adapted_mask_in!=NULL) {
	// If adaptation mask supplied then adapt when the supplied mask dictates ...
	adapted_mask[i]=(*adapted_mask_in)[i];
      } else if (!has_deltas || i<half_size) {
	// ... otherwise adapt when we think we have a good estimate of the current noise ...
	adapted_mask[i] = (p > probability_threshold);
      } else {
	// ... if has deltas then delta means and variances adapt only when static means and variances adapt
	adapted_mask[i] = adapted_mask[i-half_size];
      }
      
      if (adapted_mask[i]) {
	noise_mean[i] = m;
	noise_mean2[i] = s;
	noise_variance[i] = v; 
      }

   }

}


// produce one frame of output
void AdaptiveNoiseEstimationBlock::produce_output() {

   if (inputs_are_all_sample_data) {

      (*output_sockets)[0]->put_sample(noise_mean[0]);
      (*output_sockets)[1]->put_sample(noise_variance[0]);
      (*output_sockets)[2]->put_sample(adapted_mask[0]);

   } else {

      CTKVector *q = new CTKVector(noise_mean);
      ASSERTIT(q);
      (*output_sockets)[0]->put_vector(q);

      CTKVector *qq = new CTKVector(noise_variance);
      ASSERTIT(qq);
      (*output_sockets)[1]->put_vector(qq);

      CTKVector *qqq = new CTKVector(adapted_mask);
      ASSERTIT(qqq);
      (*output_sockets)[2]->put_vector(qqq);

   }
}

float AdaptiveNoiseEstimationBlock::erfs(float x) {
  return 2.0/(1.0+exp(-2.3236*x)) - 1.0;
}

// process new frame
void AdaptiveNoiseEstimationBlock::compute() {

  if ((*input_sockets)[1]->connected())
    (*input_sockets)[1]->get_vector(adapted_mask_in);
  
  
  if (frames_left > 1) {
    
    // still buffering for the initial estimate, no output yet
    buffer_one_frame();
    --frames_left;

  } else if (frames_left == 1) {
    
    // this is the last frame needed to initialise the blocks

    // buffer this one
    buffer_one_frame();
    
    // initialise the block
    compute_initial_estimate();
    
    // allocate mem for 
    adapted_mask.clear();
    adapted_mask.resize(frame_buffer[0]->size());
    
    // run the delayed (buffered) noisy frames thorugh the block
    for (UInteger i=0; i<frame_buffer.size(); i++) {
      
      update_internal_state(frame_buffer[i]);
      produce_output();
      delete frame_buffer[i]; 
    }
    // 
    frames_left = 0;
    
  } else {
    
    // regular update and output
    CTKVector *pv;
    
    (*input_sockets)[0]->get_vector(pv);
    
    update_internal_state(pv);
    produce_output();
    delete pv;
    
  }

}


void AdaptiveNoiseEstimationBlock::close() {

}

/* End of ctk_adaptive_noise_estimation.cpp */
