/******************************************************************************/
/*									      */
/*	ctk_filter_colour.cpp	        	      		                      */
/*									      */
/*	filters colouring eg removes short or puny groups                             */
/*									      */
/*	Author: Martin Cooke, Sheffield University            		      */
/*									      */
/*       CTK VERSION 1.3.5  Apr 22, 2007    	         	      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <cmath>
#include <vector>

#include <algorithm>

#include "ctk_local.hh"
#include "ctk_data_descriptor.hh"
#include "ctk_socket.hh"  
#include "ctk_param.hh"    
#include "ctk_filter_colour.hh"  


/******************************************************************************/
/*                                                                            */
/*       CLASS NAME: FilterColourBlock                                               */
/*                                                                            */
/******************************************************************************/

const string FilterColourBlock::type_name = "FilterColour";
const string FilterColourBlock::help_text = "filters colouring eg removing short/puny groups";

FilterColourBlock::FilterColourBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {

  make_input_sockets(1);  
  make_output_sockets(1);
  input_sockets->set_description("in1", "colouring");
  output_sockets->set_description("out1", "filtered colouring");

  // Set up MIN_FRAMES parameter
  MIN_FRAMES_param= new ParamInt("MIN_FRAMES", PARAM_DEFAULT_MIN_FRAMES);
  MIN_FRAMES_param->set_helptext("The minimum duration for a group. Groups that are shorter than this will be filtered out.\n");
  MIN_FRAMES_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(MIN_FRAMES_param);

  // Set up MIN_MEMBERS parameter
  MIN_MEMBERS_param= new ParamInt("MIN_MEMBERS", PARAM_DEFAULT_MIN_MEMBERS);          
  MIN_MEMBERS_param->set_helptext("The minimum size (in pixels) for a group. Groups that are smaller than this will be filtered out.\n");
  MIN_MEMBERS_param->install_validator(new Validator(Validator::VLOWER, 0.0));
  parameters->register_parameter(MIN_MEMBERS_param);

}

FilterColourBlock::~FilterColourBlock() {
}

Block* FilterColourBlock::clone(const string &n) const{
  Block *ablock = new FilterColourBlock(n.empty()?getname():n);
  return copy_this_block_to(ablock);
}


void FilterColourBlock::reset() {
  Block::reset();  
  
  vector_width=(*input_sockets)[0]->get_data_descriptor()->get_storage();
  min_frames=MIN_FRAMES_param->get_value();
  min_members=MIN_MEMBERS_param->get_value();
  window_size = min_frames+1;
  frame_number = 0;

  // reset colour map
  m.clear();
}



void FilterColourBlock::compute() {

  CTKVector *xv0;
  (*input_sockets)[0]->get_vector(xv0);    

  frame_number++;

  // augment statistics on colours
  for (int i=0; i < vector_width; i++) {
    int colour = (int) (*xv0)[i];
    if (m.count(colour) == 0) { // first time for this colour
      m[colour].count = 1;
      m[colour].first = frame_number;
    } else {
      m[colour].count++;
    }
    m[colour].latest = frame_number;
  }

  frame_buffer.push_back(xv0);
  if (frame_number>window_size) {
    CTKVector *xv= frame_buffer[0];

    // for each group in current frame, check if constraints are met,
    // otherwise replace by group 0
    for (int i=0; i < vector_width; i++) {
      int colour = (int) (*xv)[i];
      if ((m[colour].count) < min_members)
	(*xv)[i] = 0.0;
      else {
	if ((m[colour].latest-m[colour].first+1) < min_frames) {
	  (*xv)[i] = 0.0;
	}
      }
    }
	  
    (*output_sockets)[0]->put_vector(xv);
    frame_buffer.pop_front();
  }


}

void FilterColourBlock::flush_buffer() {

  // Clear internal buffer onto output
  // cerr << "Flushing ctk_colour_mask buffer\n";
  
  while (frame_buffer.size()>0) {
    // clean_up(*frame_buffer[0]);
    (*output_sockets)[0]->put_vector(frame_buffer[0]);
    frame_buffer.pop_front();
  }
  
}


/* End of ctk_filter_colour.cpp */
 
