/******************************************************************************/
/*									      */
/*	ctk_gmask.cpp   			             		      */
/*									      */
/*	Author: Ljubomir Josifovski, Sheffield University              	      	      */
/*									      */
/*     CTK VERSION 1.3.5  Apr 22, 2007                    		      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <cmath>
#include <cassert>

#include "ctk_local.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"

#include "ctk_gmask.hh"

#define ASSERTIT(q) assert(q)

/******************************************************************************/
/* 									       */
/*       CLASS NAME: GMaskBlock 					      */
/*									      */
/******************************************************************************/

/***************************************************************************
********

GMaskBlock is based on the Matlab code (implements snra function):

function [sub,neg,nmu,nvar,adapt,ppres]=snra(noisy,thr,teta)
%
% LBJ 23/JUN/2000

if exist('thr')~=1 | isempty(thr); thr=6.9524; end
if exist('teta')~=1 | isempty(teta); teta=0.5; end
 
[nmu,nvar,adapt]=nest_adaptive(noisy);

nx=1+10^(thr/20);
sqrt2=sqrt(2);
ppres = 0.5+0.5*erfs( (noisy/nx-nmu)./(sqrt2*sqrt(nvar)) ); % P(SNR>X)
neg = (ppres < teta); % P(SNR>X) < teta

sub = noisy-nmu;
neg = neg | (sub<0);


*******************************************************************************/

const string GMaskBlock::type_name = "GMask";
const string GMaskBlock::help_text = "Produce cleaned speech, the present data mask and the probability of the present mask, using the noisy speech and the noise means and variance estimates. This block is intended to be used together with the AdaptiveNoiseEstimation block which produces the estimates of the noise mean and variance. It is assumed that the noisy data is spectral data, that the distribution of the estimated noise is Normal, and that the channels are independent. <p> The probability is P(estimated_SNR&lt;noisy_threshold), and the preset mask is computed as P(estimated_SNR&lt;noisy_threshold) &gt; probability_threshold. The cleaned speech is simply the noisy speech minus the noise mean estimate. <p> <b>WARNING</b>: the cleaned speech may be negative!";

GMaskBlock::GMaskBlock(const string &a_name):CTKObject(a_name),Block(a_name, type_name) {

   // inputs
   make_input_sockets(3);
   input_sockets->set_description("in1", "noisy data");
   input_sockets->set_description("in2", "estimate of the noise mean");
   input_sockets->set_description("in3", "estimate of the noise variance");

   // outputs
   make_output_sockets(3);
   output_sockets->set_description("out1", "`cleaned' speech - ATTENTION - it may be negative!");
   output_sockets->set_description("out2", "present data mask");
   output_sockets->set_description("out3", "probability of the present data");

   // Set up NOISE_THRESHOLD parameter
   NOISE_THRESHOLD_param = new ParamFloat("NOISE_THRESHOLD", 6.9524);
   ASSERTIT(NOISE_THRESHOLD_param);
   NOISE_THRESHOLD_param->set_helptext("The `noisy_threshold' (in dB) used in P(estimated_SNR&lt;noisy_threshold) > probability_threshold. Check block's documentation.");
   NOISE_THRESHOLD_param->set_unit("dB");
   parameters->register_parameter(NOISE_THRESHOLD_param);

   // Set up PROBABILITY_THRESHOLD parameter
   PROBABILITY_THRESHOLD_param= new ParamFloat("PROBABILITY_THRESHOLD",
0.6);
   ASSERTIT(PROBABILITY_THRESHOLD_param);
   PROBABILITY_THRESHOLD_param->set_helptext("The `probability_threshold' used in P(estimated_SNR&lt;noisy_threshold) &gt; probability_threshold. Check block's documentation.");
   parameters->register_parameter(PROBABILITY_THRESHOLD_param);

}

Block* GMaskBlock::clone(const string &n) const{

   Block *ablock = new GMaskBlock(n.empty()?getname():n);
   ASSERTIT(ablock);
   return copy_this_block_to(ablock);

}

void GMaskBlock::reset() {

   Block::reset();

   // init params
   noise_threshold = NOISE_THRESHOLD_param->get_value();
   probability_threshold = PROBABILITY_THRESHOLD_param->get_value();

}

// process new frame
void GMaskBlock::compute() {

  // inputs
  CTKVector *noisy;
  CTKVector *mean; 
  CTKVector *variance;
  
  // get them
  
  (*input_sockets)[0]->get_vector(noisy);
  (*input_sockets)[1]->get_vector(mean);
  (*input_sockets)[2]->get_vector(variance);
   
  UInteger vecsize = noisy->size();
   
  // outputs
  CTKVector *cleaned = new CTKVector(vecsize);
  ASSERTIT(cleaned);
  CTKVector *pmask = new CTKVector(vecsize);
  ASSERTIT(pmask);
  CTKVector *pprob = new CTKVector(vecsize);
  ASSERTIT(pprob);

  // compute them
  float nx = 1+pow(10, noise_threshold/20);
  //float avg_prob = 0.0;
  //float avg_ent= 0.0;
  //cout << "prob= ";
  //float track_prob = 0.0;
  for (UInteger i=0; i<vecsize; i++) {
    
    (*cleaned)[i] = (*noisy)[i] - (*mean)[i];
    (*pprob)[i] = 0.5 + 0.5*gmask_erfs( ((*noisy)[i]/nx - (*mean)[i]) / sqrt(2 * (*variance)[i]) ); // P(SNR>X)
    ASSERTIT((*pprob)[i]>=0.0 && (*pprob)[i]<=1.0);
    //cout << (*pprob)[i] << " ";
    (*pmask)[i] = ( (*pprob)[i] > probability_threshold );
    //avg_prob += (*pprob)[i];
    //avg_ent += -(*pprob)[i] * log((*pprob)[i]+1e-30);
    
   }
  //cout << endl;
  //avg_prob /= vecsize;
  //cout << getname() << ": avg_prob= " << avg_prob << endl;
  //cout << getname() << ": avg_ent= " << avg_ent << endl;
  //float q=0.984; track_prob = q*track_prob + avg_prob;
  //cout << getname() << ": tracked_prob= " << (1-q)*track_prob << endl;
  
  (*output_sockets)[0]->put_vector(cleaned);
  (*output_sockets)[1]->put_vector(pmask);
  (*output_sockets)[2]->put_vector(pprob);

  delete noisy;
  delete mean;
  delete variance;

}

void GMaskBlock::close() {


}


float GMaskBlock::gmask_erfs(float x) {return 2.0/(1.0+exp(-2.3236*x)) - 1.0;};


/* End of ctk_gmask.cpp */
