/******************************************************************************/
/*									      */
/*	ctk_CTK_reader.cpp	    			       		      */
/*									      */
/*	Implementation for CTKReader - for reading .ctk script files   	      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007	        		      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include "ctk_local.hh"

#include "ctk_error.hh"

#include "ctk_function_classes.hh"

#include "ctk_block_headers.hh"

#include "ctk_CTK_reader.hh"

extern string  expand_environment_variables(const string &word);

enum {OP_ERROR=0, OP_BLOCK, OP_ENDBLOCK, OP_ADD, OP_CONNECT, OP_PARAMETER, OP_SET, OP_INPUT, OP_OUTPUT};


/******************************************************************************/
/*									      */
/*	CLASS NAME: UnresolvedParam	       				      */
/*									      */
/******************************************************************************/

// This class is used for representing parameter values that can not be resolved
// until the command line parameter values are know.
// These may be simple variables such as $1, $2 etc or more complex string that
// contain command line parameters as substrings e.g.  file$1_$2.$3   etc


UnresolvedParam::UnresolvedParam(const string &aname, const vector<int> &acomponents, const vector<string> &afixed_components):
  name(aname), components(acomponents), fixed_components(afixed_components) {

  //  cerr << *this;

}

UnresolvedParam::UnresolvedParam(const string &aname, const string &value): name(aname) {

  string element;

  bool in_fixed=true;
  bool last_was_dollar=false;
  
  for (unsigned int i=0; i<value.size(); i++) {
    if (value[i]=='$') {
      add_element(element,in_fixed);
      element.erase();
      element+=value[i];
      last_was_dollar=true;
    } else if (isdigit(value[i])) {
      if (last_was_dollar) in_fixed=false;
      element+=value[i];
      last_was_dollar=false;
    } else {
      last_was_dollar=false;
      if (!in_fixed) {
	add_element(element, false);
	element.erase();
      }
      in_fixed=true;
      element+=value[i];
    }
  }

  add_element(element, in_fixed);
  
  //  cerr << *this;

}


int UnresolvedParam::get_max_param_number() const {
  if (components.size()==0) return 0;
  
  return  *max_element(components.begin(), components.end());
}

bool UnresolvedParam::is_simple() const {
  return (components.size()==1 && fixed_components.size()==0);
}



void UnresolvedParam::add_element(const string &element, bool fixed) {

  if (element.size()==0) return;
  
  if (fixed) {
    fixed_components.push_back(element);
    components.push_back(0);
  } else {
    int param_number;
    sscanf(element.c_str(),"$%d", &param_number);
    components.push_back(param_number);
  }
}




ostream& operator<< (ostream& out, const UnresolvedParam& x) {
  
  out << "UnresolvedParam: \n";
  out << x.name << "\nCommand line components: ";
  for (unsigned int i=0; i<x.components.size(); ++i)
    out << x.components[i] << " ";
  out << "\nFixed components: ";
  for (unsigned int i=0; i<x.fixed_components.size(); ++i)
    out << x.fixed_components[i] << " ";
  out << "\n\n";

  return out;
}




/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKReader			         	       	      */
/*									      */
/******************************************************************************/

CTKReader::CTKReader(string file_name, vector<string> arguments/*=vector<string>()*/) {

  in_block = false;
  
  max_param_number=0;
  
  build_op_map();
  build_constructor_map();
  
  SIt it=lines.end();
  
  insert_file(it, file_name);
  
  process_include_statements();
  
  Integer line=1;
  Integer op;
  string op_string;

  for (SIt it=lines.begin(); it<lines.end(); ++it, ++line) {
    if (!it->is_a_comment()&&!it->is_blank()) {
      CTKScriptElement line_copy=*it;
      op_string=line_copy.get_script_word();

      // Converts strings into Integers and then uses a switch statement.
      
      op=op_translator[op_string];   
      
      if (op==0) {
      	error_string = "ctk_script error - unknown operatrion: " + op_string;
	script_error(*it);
      }
      
      //      else if ((this->*op)(line_copy)==true) {
      //	script_error(it);
      //      }
      
      Boolean error_status=false;
      switch (op) {
      case OP_BLOCK: error_status=interpret_BLOCK(line_copy); break;
      case OP_ENDBLOCK: error_status=interpret_ENDBLOCK(line_copy); break;
      case OP_ADD: error_status=interpret_ADD(line_copy); break;
      case OP_CONNECT: error_status=interpret_CONNECT(line_copy); break;
      case OP_PARAMETER: error_status=interpret_PARAMETER(line_copy); break;
      case OP_SET: error_status=interpret_SET(line_copy); break;
      case OP_INPUT: error_status=interpret_INPUT(line_copy); break;
      case OP_OUTPUT: error_status=interpret_OUTPUT(line_copy); break;
      case OP_ERROR:
      default:
	error_string = "ctk_script error - unknown operatrion: " + op_string;
	script_error(*it);
	break;
      }
      if (error_status) script_error(*it);
      
    } else if (it->is_a_help_comment()) {
      CTKScriptElement line_copy=*it;
      line_copy.get_script_word();  // Remove the "help comment" tag  (#H) 
      help_text+= ( line_copy.get_string() + "\n");
    }
  } 

  /*  
  if (block_list.size()==1)
    cout << " 1 block has been read" << endl;
  else
    cout << block_list.size() << " blocks have been read" << endl;
  */
}

CTKReader::~CTKReader() {
}

BlockList CTKReader::get_block_list() {
  return block_list;
}

const vector <UnresolvedParam> &CTKReader::get_unresolved_parameters() {
  return unresolved_parameters;
}

UInteger CTKReader::get_max_param_number() {
  return max_param_number;
}

Block *CTKReader::get_block(const string &name) {

  for (BlockList::iterator blockp=block_list.begin(); blockp!=block_list.end(); ++blockp)
    if ((*blockp)->am_i_called(name)) return *blockp;
  
  return NULL;
}


//   
//   Private methods
//


void CTKReader::script_error(const CTKScriptElement &element) {
  cerr << error_string << endl;
  cerr << element.get_file_name() << ": line " << element.get_line_number() << ": \"" << element.get_string() << "\"" << endl;
  throw(CTKErrorGCE(__FILE__, __LINE__, error_string.c_str()));
}

void CTKReader::build_op_map() {

  // Converts strings into Integers and then uses a switch statement.
  
  op_translator["BLOCK"]	= OP_BLOCK;
  op_translator["ENDBLOCK"]	= OP_ENDBLOCK;
  op_translator["ADD"]		= OP_ADD;
  op_translator["CONNECT"]	= OP_CONNECT;
  op_translator["PARAMETER"]	= OP_PARAMETER;
  op_translator["SET"]		= OP_SET;
  op_translator["INPUT"]	= OP_INPUT;
  op_translator["OUTPUT"]	= OP_OUTPUT;

}


extern void my_translation_table(map<string, Block*> &constructor_translator);

void CTKReader::build_constructor_map() {
  

  #include "sys/ctk_sys_translation_table.cpp"

  //
  //  Call to user-provided function that supplies translations for user-provided blocks
  //

  my_translation_table(constructor_translator);

}

void CTKReader::process_include_statements() {

  Integer nest_depth=0;
  Boolean did_include;
  
  do {
    did_include=false;
    for (SIt it=lines.begin(); it<lines.end(); ++it) {
      CTKScriptElement line_copy=*it;
      string op_string=line_copy.get_script_word();
      if (op_string==string("INCLUDE")) {
	it=lines.erase(it);  // Remove the include line
	string include_file = line_copy.get_script_word();
	insert_file(it, include_file);
	did_include=true;
      }
    } 
    if (++nest_depth>CTK_MAX_INCLUDE_NEST_DEPTH) {
      throw(CTKErrorGCE(__FILE__, __LINE__, "too many nested includes in script"));
    }
  } while (did_include);
  
}

void CTKReader::insert_file(vector<CTKScriptElement>::iterator &it, string file_name) {

  string expanded_file_name=expand_environment_variables(file_name);
  std::ifstream this_ifstr(expanded_file_name.c_str());
  
  if (this_ifstr==NULL) {
    if (lines.size()==0) {
      throw(FileErrorCFCF(__FILE__, __LINE__, expanded_file_name.c_str()));  // Cannot find main CTK file
    } else {
      throw(FileErrorIFOF(__FILE__, __LINE__, expanded_file_name.c_str()));  // Cannot find some include file
    }
  }
  
  Character buffer[CTK_MAX_SCRIPT_LINE_SIZE];
  int line=1;
  
  while (this_ifstr.getline(buffer,CTK_MAX_SCRIPT_LINE_SIZE,'\n')) {
    it=lines.insert(it, CTKScriptElement(string(buffer), file_name, line++));
    ++it;
  }
  
  it--;
  
}

void CTKReader::do_argument_substitution(vector<string> argument) {
    Character buffer[CTK_MAX_SCRIPT_LINE_SIZE];
  if (argument.size()==0) return;
  
  for (UInteger i=0; i<argument.size(); ++i) {
    string in_word("$");
    sprintf(buffer,"%d",i);
    in_word+=buffer;
    for (UInteger j=0; j<lines.size(); ++j) {
      lines[j].replace_word(in_word, argument[i]);
    }
  }
  
}


Boolean  CTKReader::interpret_BLOCK(CTKScriptElement &s) {
  
  if (in_block==true) {
    error_string = "ctk_script error - nested blocks??";
    return true;
  }
  
  in_block=true;
  help_text.resize(0);  
  string name=s.get_script_word();
  current_block=new Block(name, name, s.get_file_name().c_str());
  return false;
};


Boolean  CTKReader::interpret_ENDBLOCK(CTKScriptElement &s){
    s=s;
  
  if (in_block==false) {
    error_string = "ctk_script error - too many ENDBLOCKS";
    return true;
  }
  
  in_block=false;
  
  block_list.push_back(current_block);
  
  if (help_text.size()!=0)
    current_block->set_helptext(help_text);
  
  return false;
};


Boolean  CTKReader::interpret_ADD(CTKScriptElement &s){
  
  string block_name, block_desc_string;
  string block_type;
  vector<CTKScriptElement*> argument_list;
   
  if (s.interpret_assignment(block_name, block_type)==true) {
    error_string = "ctk_script error - syntax error on ADD line";
    return true; 
  }
  
  CTKScriptElement block_parameters = s.get_braces();
  
  if (block_parameters.interpret_block_parameters(argument_list)==true) {
    error_string = "ctk_script error - syntax error on ADD line";
    return true;
  }
  
  Block* block=constructor_translator[block_type];
  Block* subblock;
  
  if (block==NULL) {
    // Not a built in block, so check the script defined blocks
    block=get_block(block_type);
    if (block==NULL) {
      error_string = "ctk_script error - unidentified block (\"" + block_type + "\") on ADD line";
      return true;
    }
  }
  
  subblock=block->clone(block_name);
  
  vector<CTKScriptElement*>::iterator argp;
  for (argp=argument_list.begin(); argp<argument_list.end(); ++argp) {
   if (do_assignment(subblock, **argp)==true)
      return true;
  }

  sequence_delete(argument_list.begin(), argument_list.end());
  
  try {
    current_block->add_subblock(subblock);
  }
  catch (BlockErrorNUN &error) {
    error.debug_print();
    return true;
  }
  
  return false;
};


Boolean  CTKReader::interpret_CONNECT(CTKScriptElement &s){
  
    string from = s.get_script_word();
  string to = s.get_script_word();
  
  if (to.size()==0) {
    error_string = "ctk_script error - missing item on CONNECT line\n";
    return true;
  }
  
  current_block->connect(from, to);
  
  return false;
};


Boolean  CTKReader::interpret_PARAMETER(CTKScriptElement &s){
  
    string param_name, param_path;
  
  if (s.interpret_assignment(param_name, param_path)==true) {
    error_string = "ctk_script error - syntax error on PARAMETER line\n";
    return true;
  }
  
  current_block->add_parameter(param_name, param_path);

  return false;
};


Boolean  CTKReader::interpret_SET(CTKScriptElement &s){

  return do_assignment(current_block, s);
}


Boolean  CTKReader::interpret_INPUT(CTKScriptElement &s){
  
    string input_name, input_path;
  
  if (s.interpret_assignment(input_name, input_path)==true) {
    error_string = "ctk_script error - syntax error on INPUT line\n";
    return true;
  }
  
  string socket_desc=s.get_script_word();
  
  current_block->add_input_socket(input_name, input_path, socket_desc);
  
  return false;
};


Boolean  CTKReader::interpret_OUTPUT(CTKScriptElement &s){
  
  string output_name, output_path;
  
  if (s.interpret_assignment(output_name, output_path)==true) {
    error_string = "ctk_script error - syntax error on OUTPUT line\n";
    return true;
  }
  
  string socket_desc=s.get_script_word();
  
  current_block->add_output_socket(output_name, output_path, socket_desc);
  
  return false;
};



Boolean CTKReader::do_assignment(Block *block, CTKScriptElement &s) {
  string param_name, param_value;
  Integer n;
  
  if (s.interpret_assignment(param_name, param_value)==true) {
    error_string = "ctk_script error - error attempting parameter assignment:" + s.get_string();
    return true;
  }
  
  // if param contains an unresolved command line parameter then record details - values will be substituted later
  if (contains_unresolved_param(param_value)) {
    
    string param_full_name=block->getfullname();
    param_full_name+=(':'+param_name);

    UnresolvedParam unresolved_param(param_full_name, param_value);
    unresolved_parameters.push_back(unresolved_param);
    
    if (unresolved_param.get_max_param_number()>max_param_number)
	max_param_number=unresolved_param.get_max_param_number();
    
    // True if the param has a simple value (i.e. $1, $2 etc), false if it is a compound (e.g. "$1.$2")
    if (unresolved_param.is_simple())
      block->set_parameter_arg_number(param_name, unresolved_param.get_max_param_number());
    else
      block->set_parameter_unresolved_value(param_name,param_value.c_str()); 
    
    return false;
  }
  
  if (param_name=="NINPUTS") {
    sscanf(param_value.c_str(),"%d",&n);
    block->set_num_inputs(n);
  } else if (param_name=="NOUTPUTS") {
    sscanf(param_value.c_str(),"%d",&n);
    block->set_num_outputs(n);
  } else {
    try {
      block->set_parameter(param_name, param_value.c_str());
    }
    catch (BlockErrorCFP &error) {
      cerr << "ctk_script error - unrecognised parameter name (\"" << param_name << "\")" << endl;
      cerr << "Here are a list of parameters that you might have meant:" << endl;
      block->display_parameters(cerr, "   ");
      error_string = "ctk_script error - unrecognised parameter: " + param_name;
      return true;
    }
  }
  
  return false;
}


bool CTKReader::contains_unresolved_param(const string &param_value) const {

  // Return true if the parameter value contains an unresolved command line parameter e.g. $1, $2 etc

  // Return true if there exists a $ that is immediately followed by a digit
  
  bool was_dollar=false;
  
  for (unsigned int i=0; i<param_value.size(); ++i) {
    if (param_value[i]=='$')
      was_dollar=true;
    else if (isdigit(param_value[i])) {
      if (was_dollar)
	return true;  // $ followed by a numeric value so return true.
    } else
      was_dollar=false;
  }
  
  return false;
}

/* End of ctk_CTK_reader.cpp */
