/******************************************************************************/
/*									      */
/*	ctk_CTK_writer.cpp	    			       		      */
/*									      */
/*	Implementation for CTWriter      - for writing .ctk script files      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		         	      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <fstream>

#include <algorithm>

#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_socket.hh"
#include "ctk_param.hh"
#include "ctk_block.hh"

#include "ctk_CTK_writer.hh"

/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKWriter			         	       	      */
/*									      */
/******************************************************************************/

extern string  expand_environment_variables(const string &word);

const char *INDENT = "  ";

CTKWriter::CTKWriter(){};

void CTKWriter::set_CTKScript_path(int a_ctkscript_path) {
  ctkscript_path=a_ctkscript_path;
}

void CTKWriter::save(Block *a_block, string a_save_file/*="\0"*/) {

  save_file=a_save_file;
  top_block=a_block;
  
  // If a file isn't specified use file in which the block was found
  if (save_file.size()==0) save_file=expand_environment_variables(top_block->get_file_name());

  std::ofstream ofstr(save_file.c_str());

  if (ofstr==NULL) {
    cerr << "Cannot open file: " << save_file << endl;
    throw(CTKError(__FILE__, __LINE__));
  }

  Boolean uses_ctklocal = check_if_system_uses_user_blocks();
  uses_ctklocal = uses_ctklocal || (ctkscript_path==CTKSCRIPT_PATH_ALWAYS_CTKLOCAL);

  write_header(ofstr, uses_ctklocal);  
  ofstr << endl;
 
  // First write all the necessary INCLUDE statements at the top of the file
  written_list.resize(0);  // clear record of the recorded blocktypes
  write_subblocks(ofstr, top_block, INCLUDE_WRITING_MODE);
  ofstr << endl;
  ofstr << endl;

  // Then add all the required infile block definitions
  written_list.resize(0);  // clear record of the recorded blocktypes
  write_subblocks(ofstr, top_block, BLOCK_WRITING_MODE);
  
  ofstr.close();
  
}

extern void my_translation_table(map<string, Block*> &constructor_translator);



Boolean CTKWriter::check_if_system_uses_user_blocks() {

  map<string, Block*> local_constructor_translator;

  Boolean uses_user_blocks=false;
  
  my_translation_table(local_constructor_translator);

  _recurse_check_if_system_uses_user_blocks(top_block, uses_user_blocks, local_constructor_translator);

  return uses_user_blocks;
}

void CTKWriter::_recurse_check_if_system_uses_user_blocks (Block *block, Boolean &uses_user_blocks, map<string, Block*> &local_constructor_translator) {
  
  for (BlockIt blockp=block->subblocks.begin(); blockp!=block->subblocks.end(); ++blockp) {
    _recurse_check_if_system_uses_user_blocks(*blockp, uses_user_blocks, local_constructor_translator);
  }
  
  if (local_constructor_translator[block->get_blocktype()]!=NULL) 
    uses_user_blocks=true;

}


// Writes the CTK script header
//

void CTKWriter::write_header(std::ofstream &ofstr, Boolean use_ctklocal) {

  Boolean error_flag=false;
  const char *path;

  if (use_ctklocal) {
    path = getenv("CTKLOCAL");
    if (path==NULL) {
      error_flag=true;
      cerr << "Problem saving script: $CTKLOCAL is not set." << endl;
      path = getenv("CTKROOT");
    }
  } else {
    path = getenv("CTKROOT");
    if (path==NULL) {
      error_flag=true;
      cerr << "Problem saving script: $CTKROOT is not set." << endl;
    }
  }

  if (path==NULL) {path = "\0";}

  if (error_flag)
    cerr << "First line of script (i.e #!" << path << "/bin/CTKScript) will have to be corrected." << endl; 

 
  ofstr << "#!" << path << "/bin/CTKScript" << endl;
}


// If there are subblocks that are not defined in the same file as the main block we need to
// add INCLUDE lines to the beginning of the CTK file being written


void CTKWriter::write_subblocks(std::ofstream &ofstr, Block *ablock, WritingModeType writing_mode) {

  string type;
  if (has_been_written(type=ablock->get_blocktype())) return;
  written_list.push_back(type);

  // First, recurse through sub-blocks, writing each one...
  string block_file_name = ablock->get_file_name();
  for (BlockIt blockp=ablock->subblocks.begin(); blockp!=ablock->subblocks.end(); ++blockp) {
    string subblock_file_name = (*blockp)->get_file_name();
    if ((*blockp)->subblocks.size()>0 && subblock_file_name==block_file_name) {
      write_subblocks(ofstr, *blockp, writing_mode);
    }
  }

  // Then write the block itself.
  switch (writing_mode) {
  case INCLUDE_WRITING_MODE:  // For writing the 'include statements'
    write_INCLUDE_statements(ofstr, ablock);
    break;
  case BLOCK_WRITING_MODE:   // For writing the 'block statements'
    write_BLOCK(ofstr, ablock);
    break;
  }
  
}



void CTKWriter::write_INCLUDE_statements(std::ofstream &ofstr, Block *ablock) {

  string mainblock_file_name = ablock->get_file_name();
  
  for (BlockIt blockp=ablock->subblocks.begin(); blockp!=ablock->subblocks.end(); ++blockp) {
    string subblock_file_name = (*blockp)->get_file_name();
    if (subblock_file_name.size()!=0 && subblock_file_name!=mainblock_file_name) {
      string type=(*blockp)->get_blocktype();
      if (!has_been_written(type)) {
	written_list.push_back(type);
	ofstr << "INCLUDE " << subblock_file_name << endl;
      }
    }
  }
    
};


void CTKWriter::write_BLOCK(std::ofstream &ofstr, Block *ablock) {
  
  ofstr << "BLOCK " << ablock->get_blocktype().c_str() << "\n\n";
  
  write_helptext(ofstr, ablock);
  
  for (BlockIt blockp=ablock->subblocks.begin(); blockp!=ablock->subblocks.end(); ++blockp) {
    write_ADD_statement(ofstr, *blockp);
  }
  ofstr << endl;
  
  Integer connect_lines_written=0;
  for (BlockIt blockp=ablock->subblocks.begin(); blockp!=ablock->subblocks.end(); ++blockp) {
    connect_lines_written+=write_CONNECT_statements(ofstr, *blockp);
  }
  if (connect_lines_written>0) ofstr << endl;

  write_PARAMETER_statements(ofstr, ablock);
  write_INPUT_OUTPUT_statements(ofstr, ablock->get_input_sockets());
  write_INPUT_OUTPUT_statements(ofstr, ablock->get_output_sockets());
  
  ofstr << "ENDBLOCK\n\n\n";
  
}

// Writes the help text
//

void CTKWriter::write_helptext(std::ofstream &ofstr, Block *ablock) {
  string helptext = ablock->get_helptext();
  if (helptext.size()==0) return;
  
  UInteger length = helptext.size();
  ofstr << "#H ";
  for (UInteger i=0; i<length; ++i) {
    char c=helptext[i];
    if (c=='\n') {
      ofstr << endl;
      if (i!=length-1) ofstr << "#H ";
    } else 
      ofstr << c;
  }
  ofstr << endl;
  ofstr << endl;

}

void CTKWriter::write_ADD_statement(std::ofstream &ofstr, Block *ablock) {
  ofstr << INDENT << "ADD " << ablock->getname() << " = " << ablock->get_blocktype();
  write_parameter_assignments(ofstr, ablock);
  ofstr << endl;
}




void CTKWriter::write_parameter_assignments(std::ofstream &ofstr, Block *ablock) {

  const ParamList *param_list = ablock->get_parameters();

  Boolean first=true;

  write_NINPUTS_NOUTPUTS(ofstr, ablock->get_input_sockets(), first);
  write_NINPUTS_NOUTPUTS(ofstr, ablock->get_output_sockets(), first);

  Integer arg_number;
  for (ParamConstIt param_it=param_list->begin(); param_it!=param_list->end(); ++param_it) {
    const Param* param=*param_it;
    if (should_output_param(param)) {
      ofstr << (first?"(":", ");
      ofstr << param->getname() << "=";
      if ((arg_number=param->get_arg_number())!=0) {
	ofstr << "$" << arg_number;
      } else if (param->has_unresolved_value()) {
	ofstr << "\"" << param->get_unresolved_value() << "\"";
      } else {
	switch (param->type()) {
	  // string parameters need to be enclosed in quotations
	case PARAM_STRING: 
	  ofstr << "\"" << param->get_value_string() << "\"" ;
	  break;
	  // Vector parameter need to be enclosed in curly braces
	case PARAM_VECTOR_BOOL:
	case PARAM_VECTOR_INT:
	case PARAM_VECTOR_FLOAT:
	  ofstr << "{" << param->get_value_string() << "}" ;
	  break;
	  // All other parameter types are written directly
	default:
	  ofstr << param->get_value_string();
	  break;
	}
      }
      first=false;
    }
  }
      
  if (first==false) ofstr << ")";

}

void CTKWriter::write_NINPUTS_NOUTPUTS(std::ofstream &ofstr, const SocketList *sockets, Boolean &first) {
  
  if (sockets->get_dirty_flag()==false) return;
  SocketType socket_type = sockets->type();
  
  ofstr << (first?"(":", ");

  if (socket_type==INPUT_SOCKET_TYPE) {
    ofstr << "NINPUTS=";
  } else {
    ofstr << "NOUTPUTS=";
  }
  Integer nsockets = sockets->size();
  ofstr << nsockets;

  first=false;

}

Integer CTKWriter::number_of_params_to_write(const ParamList *param_list) {

  Integer n=0;
  for (ParamConstIt param_it=param_list->begin(); param_it!=param_list->end(); ++param_it) {
    if (should_output_param(*param_it)) ++n;
  }
  return n;
}

Integer CTKWriter::write_CONNECT_statements(std::ofstream &ofstr, Block *ablock) {

  Integer lines_written=0;
  const OutputSocketList *output_sockets=ablock->get_output_sockets();
  for (int from_socket_number=0; from_socket_number<output_sockets->size(); ++from_socket_number) {
    OutputSocket *socket=(OutputSocket*)(*output_sockets)[from_socket_number];
    if (socket->connected() && !socket->am_i_a_subsocket()) {
      ofstr << INDENT << "CONNECT " << socket->get_socket_name() << " " << socket->get_remote_socket_name() << endl;
      ++lines_written;
    }
  }

  return lines_written;
}



Integer CTKWriter::write_PARAMETER_statements(std::ofstream &ofstr, Block *ablock) {

  const ParamList *param_list = ablock->get_parameters();
  Integer nwritten=0;
  
  for (ParamConstIt param_it=param_list->begin(); param_it!=param_list->end(); ++param_it) {
    const Param* param=*param_it;
    if (param->am_i_a_baseparam()==false) {
      string name = param->getname();
      string subname = param->get_subparam_fullname();
      ofstr << INDENT << "PARAMETER " << name.c_str() << "=" << subname.c_str() << endl;
      ++nwritten;
    }
  }
  
  if (nwritten>0) ofstr << endl;
  
  return nwritten;
}


Integer CTKWriter::write_INPUT_OUTPUT_statements(std::ofstream &ofstr, const SocketList *sockets) {
  SocketType socket_type=sockets->type();
  
  string type_string = (socket_type==INPUT_SOCKET_TYPE)?"INPUT":"OUTPUT";
  Integer nwritten=0;
 
  for (Integer i=0; i<sockets->size(); ++i) {
    string socketname=(*sockets)[i]->get_name();
    string subsocketname=(*sockets)[i]->get_subsocket_name();
    ofstr << INDENT << type_string << " " << socketname.c_str() << "=" << subsocketname.c_str();
    string desc = (*sockets)[i]->get_description();
    if (desc.size()!=0)
      ofstr << " \"" << desc << "\"";
    ofstr << endl;
    ++nwritten;
  }
  
  if (nwritten>0) ofstr << endl;

  return nwritten;
}



Boolean CTKWriter::has_been_written(string type) const {

  return (find(written_list.begin(), written_list.end(), type)!=written_list.end());
  
}

Boolean CTKWriter::should_output_param(const Param *param) {

  // Parameters should be written to the script if:
  //    EITHER They are parameters that are settable via command line arguments, i.e. $1, $2 etc
  //    OR they are visible parameters that have been changed from their default value and are not const
  //  (the 'not const' condition prevents transmitted parameters from being written for each block that they are copied to - i.e. only the source block's transmitted parameter is flagged as not-const, the others should be flagged const as even though they can inherit new values they are not directly manipulable by the user)

  return (param->get_arg_number() > 0 || param->has_unresolved_value() ||
	  (!param->get_const_flag() && param->get_dirty_flag() && !param->get_hidden_flag() && param->get_set_flag()));
}

/* End of ctk_CTK_writer.cpp */
