/******************************************************************************/
/*                                                                            */
/*      ctk_HMM_builder.hh                                                    */
/*                                                                            */
/*      Class declarations for ctk_HMM_builder.cpp                            */
/*                                                                           */
/*      Author: Jon Barker, Sheffield University                              */
/*                                                                            */
/*      CTK VERSION 1.3.5  Apr 22, 2007                              */
/*                                                                            */
/******************************************************************************/

#ifndef CTK_HMM_BUILDER_HH
#define CTK_HMM_BUILDER_HH

#include "ctk_local.hh"
#include "ctk_error.hh"
#include "ctk_HMM.hh"

#include "ctk_HMM_types.hh"

#include <cstdio>

#include <vector>
#include <map>
#include <string>

class SetOfHMMs;
class HMM;
class HMMState;
class HMMMixture;

class HMMBuilder;
class HMMStateBuilder;
class HMMMixtureBuilder;
class HMMVarianceBuilder;
class HMMGlobalBuilder;

typedef bool (HMMGlobalBuilder::* PHMM_global_builder_mem)(FILE* fp);
typedef bool (HMMVarianceBuilder::* PHMM_variance_builder_mem)(FILE* fp);
typedef bool (HMMMixtureBuilder::* PHMM_mixture_builder_mem)(FILE* fp);
typedef bool (HMMStateBuilder::* PHMM_state_builder_mem)(FILE* fp);
typedef bool (HMMBuilder::* PHMM_builder_mem)(FILE* fp);

/******************************************************************************/
/*									      */
/*	CLASS NAME: HMMObjectBuilder   	                 	       	      */
/*									      */
/******************************************************************************/

// Convert a string to upper case
extern void make_upper(string &token);

class HMMObjectBuilder {

private:
  static int vec_size;

  bool finished;   // Flag to indicate that last line of object definition has been read

  SetOfHMMs *hmmset;           // The set of HMMs this mixture belongs to

  string macro_name;

  char read_word_format_string_[MAX_STRING_SIZE];
  
  //
  //
  
public:
  
  HMMObjectBuilder(SetOfHMMs *an_hmmset, int a_vec_size=0);

  virtual ~HMMObjectBuilder(){}
  
  string get_macro_name() const {return macro_name;}

protected:

  bool interpret_IGNORE(FILE *fp);
  bool interpret_StreamInfo(FILE *fp);
  bool interpret_VecSize(FILE *fp);

  const char *read_word_format_string(){return read_word_format_string_;}
  
  template <class T>
  bool read_file(T *thisp,          // pointer to the calling object
		 FILE *fp,          // FILE pointer of file to be interpreted
		 map<string, bool (T::*)(FILE* fp)> op_translator   // token->method translation table
		 )
    {
      
      // Reads string tokens from a file and converts them into calls to handling methods in the calling class
      // according to a supplied translation map. 
      // A handling call can force the read_file to quit before the end of file is reached by setting finished
      // to TRUE.
      
      typedef bool (T::* PT_mem)(FILE* fp);
      
      int last_pos = ftell(fp);
      string token=get_token(fp);
      
      PT_mem op;
      
      thisp->clear_finished();
      while (token.size()>0 && thisp->get_finished()==false) {
	make_upper(token); // Make token into uppercase
	op=op_translator[token];
	if (op==0) {
	  fflush(fp);
	  if (fseek(fp, last_pos, 0)==-1) {
	    cerr << "fseek failed!\n\r";
	    perror("HMMState:");
	    throw(CTKError(__FILE__, __LINE__));
	  }
	  return false;
	} else {
	  if ((thisp->*op)(fp)==false)
	    throw(CTKError(__FILE__, __LINE__));
	}
	if (thisp->get_finished()==false) {
	  last_pos = ftell(fp);
	  token=get_token(fp);
	}
      }

      return true;
    }
  

    
  static int get_vec_size() {return vec_size;}
  
  // Simulate putting last string back into the file to be read again

  void set_finished();
  
  void error_in_HMM_file(FILE *fp);

  bool read_vec_size(FILE *fp);
  bool check_size_consistency(int this_size) const;
    
  SetOfHMMs *get_hmmset() const {return hmmset;}
  
  void set_macro_name(const string &a_macro_name) {macro_name=a_macro_name;}
  
private:
  
  void clear_finished() {finished=false;}
  bool get_finished() const {return finished;}

  string get_token(FILE *fp);
};


/******************************************************************************/
/*									      */
/*	CLASS NAME: HMMBuilder   	                 	       	      */
/*									      */
/******************************************************************************/

class HMMBuilder: public HMMObjectBuilder {

public:
  HMMBuilder(SetOfHMMs *an_hmmset);
  virtual ~HMMBuilder() {};
  
  HMM *getHMMObject(const string &filename, const string &name);
  HMM *getHMMObject(FILE *fp, const string &name);
  
private:
  int num_states;
  int current_state;

  map<string, PHMM_builder_mem> op_translator;

  vector<HMMState*> states;
  Transitions pTrans;

  virtual void build_op_map();
  
  void do_read_file(FILE *fp);

  bool interpret_macro_s(FILE *fp);
  bool interpret_macro_t(FILE *fp);
  
  bool interpret_NumStates(FILE *fp);
  bool interpret_State(FILE *fp);
  bool interpret_TransP(FILE *fp);
  bool interpret_EndHMM(FILE *fp);
  bool interpret_Use(FILE *fp);
};

/******************************************************************************/
/*									      */
/*	CLASS NAME: HMMStateBuilder   	                 	       	      */
/*									      */
/******************************************************************************/

class HMMStateBuilder: public HMMObjectBuilder {

public:
  HMMStateBuilder(SetOfHMMs *an_hmmset, bool in_macro_def=false);
  virtual ~HMMStateBuilder() {};
  
  HMMState *getHMMStateObject(FILE *fp);
  
private:

  int num_mixes;
  int current_mixture;
  map<string, PHMM_state_builder_mem> op_translator;

  
  vector<HMMMixture*> mixture;
  vector<HMMFloat> mix_weights;

  int max_duration;          // max duration parameter
  float voicing;             // Voicing parameter

  void build_op_map(bool macro_def);
  
  bool interpret_NumMixes(FILE *fp);
  bool interpret_Mixture(FILE *fp);
  bool interpret_Voicing(FILE *fp);
  bool interpret_MaxDuration(FILE *fp);
  bool interpret_macro_s(FILE *fp);
  bool interpret_single_mixture(FILE *fp);

  bool get_single_mixture(FILE *fp);
  bool get_mixture(FILE *fp);

  void check_mixtures() const;

};

/******************************************************************************/
/*									      */
/*	CLASS NAME: HMMMixtureBuilder   	                 	       	      */
/*									      */
/******************************************************************************/

class HMMMixtureBuilder: public HMMObjectBuilder {

public:
  HMMMixtureBuilder(SetOfHMMs *an_hmmset, bool in_macro_def=false);
  virtual ~HMMMixtureBuilder() {};
  
  HMMMixture *getHMMMixtureObject(FILE *fp);
  
private:
  
  vector<HMMFloat> mu;          // Means
  vector<HMMFloat> ivar;         // Reciprocal Variances;
  
  bool read_mean;                // Set to true once valid mean values have been read
  bool read_variance;            // Set to true once valid variance values have been read

  int current_mixture;
  map<string, PHMM_mixture_builder_mem> op_translator;

  
  void build_op_map(bool macro_def);
 
  bool interpret_macro_m(FILE *fp);
  bool interpret_Mean(FILE *fp);
  bool interpret_Variance(FILE *fp);
  bool interpret_GConst(FILE *fp);

};

/******************************************************************************/
/*									      */
/*	CLASS NAME: HMMVarianceBuilder   	                 	       	      */
/*									      */
/******************************************************************************/

class HMMVarianceBuilder: public HMMObjectBuilder {

public:
  HMMVarianceBuilder(SetOfHMMs *an_hmmset, bool in_macro_def=false);
  virtual ~HMMVarianceBuilder() {};
  
  vector<HMMFloat> *getHMMVarianceObject(FILE *fp);
  
private:
  
  vector<HMMFloat> ivar;         // Reciprocal Variances;
  
  bool read_variance;            // Set to true once valid variance values have been read

  int current_mixture;
  map<string, PHMM_variance_builder_mem> op_translator;

  
  void build_op_map(bool macro_def);
 
  bool interpret_Variance(FILE *fp);

};


/******************************************************************************/
/*									      */
/*	CLASS NAME: HMMTransitionBuilder   	                 	       	      */
/*									      */
/******************************************************************************/

class HMMTransitionBuilder: public HMMObjectBuilder {

public:
  HMMTransitionBuilder(SetOfHMMs *an_hmmset, bool in_macro_def=false);
  virtual ~HMMTransitionBuilder() {};
  
  Transitions getHMMTransitionObject(FILE *fp);
  
private:
  
  Transitions pTrans;

  map<string, PHMM_variance_builder_mem> op_translator;

  bool interpret_TransP(FILE *fp);

};


/******************************************************************************/
/*									      */
/*	CLASS NAME: HMMGlobalBuilder   	              	       	      */
/*									      */
/******************************************************************************/

class HMMGlobalBuilder: public HMMObjectBuilder {

public:

  HMMGlobalBuilder(SetOfHMMs *an_hmmset, bool in_macro_def=false);
  virtual ~HMMGlobalBuilder() {};
  
  void getHMMGlobalObject(FILE *fp);
  
private:
  
  map<string, PHMM_global_builder_mem> op_translator;
  
  void build_op_map(bool macro_def);
 

};

#endif



/* End of ctk_HMM_builder.hh */
