/******************************************************************************/
/*									      */
/*	ctk_block.cpp	    						      */
/*									      */
/*	Implementation for CTKObject and CTKBlock			      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		        	      */
/*									      */
/******************************************************************************/

#include "ctk-config.h"

#include <cstdio>

#include <algorithm>

#include "boost/iterator_adaptors.hpp"

#include "GUI/images/default_block.xpm"


#include "ctk_local.hh"
#include "ctk_error.hh"

#include "ctk_function_classes.hh"
#include "ctk_block.hh"
#include "ctk_param.hh" 
#include "ctk_socket.hh"
#include "ctk_matlab.hh"
#include "ctk_data_descriptor.hh"


#ifdef _HAS_QT
#include <qapp.h>
#endif

#include "boost/compose.hpp"


#ifdef _HAS_MATLAB
Engine * Block::ep(0);
char *Block::matlab_output_buffer(0);
#endif

Boolean Block::runtime_profiling(false);

/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKObject						      */
/*									      */
/******************************************************************************/


CTKObject::CTKObject():name(), fullname(), parameters(new ParamList(this)), parentobjectp(NULL) {
}

CTKObject::CTKObject(const string &this_name):name(this_name), fullname(),  parameters(new ParamList(this)), parentobjectp(NULL) {

}

CTKObject::~CTKObject(){
  delete parameters;
}


Boolean CTKObject::am_i_called(string aname) const {
  return (name==aname);  
}

string CTKObject::getname() const {return name;}

const char **CTKObject::getIcon() const {return DEFAULT_BLOCK_XPM;}

CTKStatus CTKObject::set_parameter(const string &param_name, const string &value) {
  return parameters->set(param_name, value);
}

CTKStatus CTKObject::set_parameter_unresolved_value(const string &param_name, const string &value) {
  return parameters->set_unresolved_value(param_name, value);
}


CTKStatus CTKObject::set_parameter(const string &param_name, Float value) {
  return parameters->set(param_name, value);
}

CTKStatus CTKObject::set_parameter_arg_number(const string &param_name, Integer arg_number) {
  return parameters->set_arg_number(param_name, arg_number);
}

CTKStatus CTKObject::unset_parameter(const string &param_name) {
  return parameters->unset(param_name);
}

void CTKObject::set_parameter_hidden(const string &param_name) {
  string this_pathname = prefix_object_name(param_name);  
  CTKStatus status = parameters->set_hidden(this_pathname);
  if (status==CTK_FAILURE) throw(BlockErrorCFP(__FILE__, __LINE__, param_name.c_str()));
}

void CTKObject::set_parameter_const(const string &param_name) {
  string this_pathname = prefix_object_name(param_name);  
  CTKStatus status = parameters->set_const(this_pathname);
  if (status==CTK_FAILURE) throw(BlockErrorCFP(__FILE__, __LINE__, param_name.c_str()));
}

void CTKObject::set_parameter_transmitted(const string &param_name) {
  string this_pathname = prefix_object_name(param_name);  
  CTKStatus status = parameters->set_transmitted(this_pathname);
  if (status==CTK_FAILURE) throw(BlockErrorCFP(__FILE__, __LINE__, param_name.c_str()));
}


void CTKObject::unset_parameter_hidden(const string &param_name) {
  string this_pathname = prefix_object_name(param_name);  
  CTKStatus status = parameters->unset_hidden(this_pathname);
  if (status==CTK_FAILURE) throw(BlockErrorCFP(__FILE__, __LINE__, param_name.c_str()));
}

void CTKObject::unset_parameter_const(const string &param_name) {
  string this_pathname = prefix_object_name(param_name);  
  CTKStatus status = parameters->unset_const(this_pathname);
  if (status==CTK_FAILURE) throw(BlockErrorCFP(__FILE__, __LINE__, param_name.c_str()));
}

void CTKObject::unset_parameter_transmitted(const string &param_name) {
  string this_pathname = prefix_object_name(param_name);  
  CTKStatus status = parameters->unset_transmitted(this_pathname);
  if (status==CTK_FAILURE) throw(BlockErrorCFP(__FILE__, __LINE__, param_name.c_str()));
}


string CTKObject::prefix_object_name(const string &s) {
  string prefix=getname();
  if (prefix.empty()) 
    return s;
  else
    return colon_concatenate(prefix, s);
}

string CTKObject::prefix_object_fullname(const string &s) {
  string prefix=getfullname();
  if (prefix.empty()) 
    return s;
  else
    return colon_concatenate(prefix, s);
}

string CTKObject::colon_concatenate(const string &s1, const string &s2) {
  string X=s1;
  X+=':';
  X+=s2;
  return X;
}


void CTKObject::copy_parameters(ParamList *some_parameters) {
  parameters->copy_settings(some_parameters);
}

ParamList *CTKObject::get_parameters() const {return parameters;}
ParamList *CTKObject::get_parameters() {return parameters;}

string CTKObject::getfullname() const {
  fullname.resize(0);
  recursive_build_fullname(fullname);
  return fullname;
}

void CTKObject::recursive_build_fullname(string &this_name) const {
  if (parentobjectp==NULL) {
    this_name=name+this_name;
  } else {
    this_name=string(":")+name+this_name;
    parentobjectp->recursive_build_fullname(this_name);
  }
}

inline Boolean CTKObject::register_parameter(Param* paramp, UInteger parameter_flags/*=0*/){return parameters->register_parameter(paramp, parameter_flags);};

/******************************************************************************/
/*									      */
/*	CLASS NAME: Block						      */
/*									      */
/******************************************************************************/

const string Block::help_text = BLOCK_HELP_TEXT;

Block::Block(): CTKObject() {
  BlockList these_blocks(0);
  make_block(these_blocks);
}
  

Block::Block(const string &this_name, const string &this_block_type, const string &this_file_name): CTKObject(this_name), baseblocks(0), block_type(this_block_type), file_name(this_file_name){ 
  BlockList these_blocks(0);
  make_block(these_blocks);
}

Block::Block(const string &this_name, BlockList these_blocks, const string &this_block_type, const string &this_file_name): CTKObject(this_name), baseblocks(0),  block_type(this_block_type), file_name(this_file_name){
  make_block(these_blocks);
}

Boolean Block::is_my_fullname(string aname) const {
  return aname==getfullname();  
}

// Change filename of any block with filename=old_name to  new_name
//  = use by the GUI interface for doing `saveAs' 
void Block::change_file_name(const string &new_name, const string &old_name) {
  
  if (file_name!=old_name || (subblocks.size()==0 && (getname()!=string("main"))))
    return;

  file_name=new_name;
    
  for (BlockConstIt blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
    (*blockp)->change_file_name(new_name, old_name);
  }

}

void Block::make_block(const BlockList &these_blocks) {

  Block *ablock;
   
  for (BlockConstIt blockp=these_blocks.begin(); blockp<these_blocks.end(); ++blockp) {
    subblocks.push_back(ablock=(*blockp)->clone()); 
    ablock->parentobjectp=this;
  } 

  input_sockets=new InputSocketList(this);
  output_sockets=new OutputSocketList(this);

  // Set up for sample rate parameter
  sample_rate_param= new ParamFloat("SAMPLE_RATE");
  sample_rate_param->set_unit("Hz");
  sample_rate_param->set_helptext("The number of samples per second.");
  parameters->register_parameter(sample_rate_param, CTK_PARAM_TYPE_CONST|CTK_PARAM_TYPE_TRANSMITTED);

  // Set up for verbosity parameter
  verbosity_level_param= new ParamInt("VERBOSITY_LEVEL", PARAM_DEFAULT_VERBOSITY_LEVEL);
  verbosity_level_param->set_helptext("An integer controlling the amount of debugging information to output. <p> 0 = no otput, the larger the number the more detail output. <p> This is only used for some blocks.");  
  parameters->register_parameter(verbosity_level_param, CTK_PARAM_TYPE_HIDDEN);

  
  settable_help_text = "Sorry, no documentation available for this block."; // default user defined help text

  reset_profile_statistics();

}


Block* Block::clone(const string &new_name) const {
   
  Block *b=new Block(new_name.empty()?getname():new_name, block_type, file_name.c_str());
  Block *ablock;

  BlockConstIt blockp;
  
  for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
    b->subblocks.push_back(ablock=(*blockp)->clone());
    ablock->parentobjectp=b;
  }

  b->settable_help_text=settable_help_text;

  delete(b->input_sockets);
  delete(b->output_sockets);
  b->input_sockets= new InputSocketList(b);
  b->output_sockets= new OutputSocketList(b);
  
  b->input_sockets->set_configurable(input_sockets->is_configurable(), input_sockets->get_min_num(), input_sockets->get_max_num());
  b->output_sockets->set_configurable(output_sockets->is_configurable(), input_sockets->get_min_num(), input_sockets->get_max_num());

  for (SocketConstIt socketp=input_sockets->begin(); socketp!=input_sockets->end(); ++socketp) {
    Socket* subsocketp=NULL;
    string subsocket_name=(*socketp)->get_subsocket_name();
    
    for (blockp=b->subblocks.begin(); blockp!=b->subblocks.end(); ++blockp) {
      if ((subsocketp=(*blockp)->find_input_socket(subsocket_name))!=NULL) break;
    }
    
    if (subsocketp==NULL) throw(BlockErrorCFIS(__FILE__, __LINE__, subsocket_name.c_str()));
    
    b->input_sockets->add(**socketp, b, subsocketp);
  }

  for (SocketConstIt socketp=output_sockets->begin(); socketp!=output_sockets->end(); ++socketp) {
    Socket* subsocketp=NULL;
    string subsocket_name=(*socketp)->get_subsocket_name();
    
    for (blockp=b->subblocks.begin(); blockp!=b->subblocks.end(); ++blockp) {
      if ((subsocketp=(*blockp)->find_output_socket(subsocket_name))!=NULL) break;
    }
    
    if (subsocketp==NULL) throw(BlockErrorCFOS(__FILE__, __LINE__, subsocket_name.c_str()));
    
    b->output_sockets->add(**socketp, b, subsocketp);
  }

  parameters->clone(b->parameters, b->subblocks);
  
  for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
    
    for (SocketIt socketp=(*blockp)->output_sockets->begin(); socketp!=(*blockp)->output_sockets->end(); ++socketp) {
      if ((*socketp)->am_i_a_subsocket()==false) { // Only connect top level sockets
 	if (!(*socketp)->get_remote_socket_name().empty()) {
	  b->connect((*socketp)->get_socket_name(),(*socketp)->get_remote_socket_name());
	}
      }
    }
    
  }

  return b;
}



Block::~Block() {

  //  for_each(subblocks.begin(), subblocks.end(), Delete_x<Block *>());
  sequence_delete(subblocks.begin(), subblocks.end());
  
  delete input_sockets;
  delete output_sockets;

}

Block* Block::add_subblock(const Block &subblock, const string &new_name) {
  Block *ablock;

  string xname=subblock.getname();
 
  if (!new_name.empty()) xname=new_name;
  const BlockIt blockp=find_if(subblocks.begin(), subblocks.end(),  bind2nd(mem_fun(&CTKObject::am_i_called), xname));   
  if (blockp!=subblocks.end()) {
    throw(BlockErrorNUN(__FILE__, __LINE__, xname.c_str()));
  }
  
  subblocks.push_back(ablock=subblock.clone(new_name));

  ablock->parentobjectp=this;
  return ablock;
}

Block* Block::add_subblock(const Block *subblock, const string &new_name) {
  Block *ablock;

  string xname=subblock->getname();
  if (!new_name.empty()) xname=new_name;
  const BlockIt blockp=find_if(subblocks.begin(), subblocks.end(),  bind2nd(mem_fun(&CTKObject::am_i_called), xname)); 
  if (blockp!=subblocks.end()) {
    throw(BlockErrorNUN(__FILE__, __LINE__, xname.c_str()));
  }

  subblocks.push_back(ablock=subblock->clone(new_name));

  ablock->parentobjectp=this;
  return ablock;
}

void Block::remove_subblock(const string &aname) {

  _remove_subblock(aname);

  baseblocks.resize(0);
  recursive_flatten(baseblocks);
}

void Block::_remove_subblock(const string &aname) {
  const BlockIt blockp=find_if(subblocks.begin(), subblocks.end(),  bind2nd(mem_fun(&Block::is_my_fullname), aname));
  if (blockp!=subblocks.end()) {
    delete *blockp;
    subblocks.erase(blockp);
  } else {
    for (BlockIt blockp2=subblocks.begin(); blockp2!=subblocks.end(); ++blockp2) 
      (*blockp2)->_remove_subblock(aname);
  }
}

bool Block::initialise() {
  
  if (subblocks.size()==0) return false;   // Main block is empty - return fail
  
  autoconnect();   
  
  // flatten into 1 level structure
  baseblocks.resize(0);
  recursive_flatten(baseblocks);


  if (check_baseblock_connections()==false) throw(BlockErrorLC(__FILE__, __LINE__));
  
  // Parse the structure and calculate which pipes must be
  // primed for the data to flow.
  prime_baseblock_pipes();

  // Make sure the blocks are processed in a consistent order
  
  reset_sockets_all();

  order_operations(baseblocks);

  return true;
}

void Block::close_final_all(){
  for_each(baseblocks.begin(), baseblocks.end(), mem_fun(&Block::close_final));
}

void Block::set_parameter(const string &param_name, const string &value) {

  string this_pathname=prefix_object_fullname(param_name);
  CTKStatus status = _set_parameter(this_pathname, value);
  if (status==CTK_FAILURE) throw(BlockErrorCFP(__FILE__, __LINE__, param_name.c_str()));
}

void Block::set_parameter_unresolved_value(const string &param_name, const string &value) {

  string this_pathname=prefix_object_fullname(param_name);
  CTKStatus status = _set_parameter_unresolved_value(this_pathname, value);
  if (status==CTK_FAILURE) throw(BlockErrorCFP(__FILE__, __LINE__, param_name.c_str()));
}

void Block::set_parameter(const string &param_name, Float value) {

  string this_pathname=prefix_object_fullname(param_name);
  CTKStatus status = _set_parameter(this_pathname, value);
  if (status==CTK_FAILURE) throw(BlockErrorCFP(__FILE__, __LINE__, param_name.c_str()));
}

void Block::unset_parameter(const string &param_name) {

  string this_pathname=prefix_object_fullname(param_name);
  CTKStatus status = _unset_parameter(this_pathname);
  if (status==CTK_FAILURE) throw(BlockErrorCFP(__FILE__, __LINE__, param_name.c_str()));
}

void Block::set_parameter_arg_number(const string &param_name, Integer arg_number) {

  string this_pathname=prefix_object_fullname(param_name);
  CTKStatus status = _set_parameter_arg_number(this_pathname, arg_number);
  if (status==CTK_FAILURE) throw(BlockErrorCFP(__FILE__, __LINE__, param_name.c_str()));
}

CTKStatus Block::_set_parameter(const string &param_name, const string &value) {
  
  if (CTKObject::set_parameter(param_name, value)==CTK_SUCCESS) 
    return CTK_SUCCESS;
  
  for (BlockIt blockp=subblocks.begin(), bp_end=subblocks.end(); blockp!=bp_end; ++blockp) {
    if ((*blockp)->_set_parameter(param_name, value)==CTK_SUCCESS)
      return CTK_SUCCESS;
  }

  return CTK_FAILURE;
}


CTKStatus Block::_set_parameter_unresolved_value(const string &param_name, const string &value) {
  
  if (CTKObject::set_parameter_unresolved_value(param_name, value)==CTK_SUCCESS) 
    return CTK_SUCCESS;
  
  for (BlockIt blockp=subblocks.begin(), bp_end=subblocks.end(); blockp!=bp_end; ++blockp) {
    if ((*blockp)->_set_parameter_unresolved_value(param_name, value)==CTK_SUCCESS)
      return CTK_SUCCESS;
  }

  return CTK_FAILURE;
}

CTKStatus Block::_set_parameter(const string &param_name, Float value) {
  
  if (CTKObject::set_parameter(param_name, value)==CTK_SUCCESS) 
    return CTK_SUCCESS;
  
  for (BlockIt blockp=subblocks.begin(), bp_end=subblocks.end(); blockp!=bp_end; ++blockp) {
    if ((*blockp)->_set_parameter(param_name, value)==CTK_SUCCESS)
      return CTK_SUCCESS;
  }

  return CTK_FAILURE;
}

CTKStatus Block::_unset_parameter(const string &param_name) {
  
  if (CTKObject::unset_parameter(param_name)==CTK_SUCCESS) 
    return CTK_SUCCESS;

  for (BlockIt blockp=subblocks.begin(), bp_end=subblocks.end(); blockp!=bp_end; ++blockp) {
    if ((*blockp)->_unset_parameter(param_name)==CTK_SUCCESS)
      return CTK_SUCCESS;
  }

  return CTK_FAILURE;
}


CTKStatus Block::_set_parameter_arg_number(const string &param_name, Integer arg_number) {
  
  if (CTKObject::set_parameter_arg_number(param_name, arg_number)==CTK_SUCCESS) 
    return CTK_SUCCESS;
  
  for (BlockIt blockp=subblocks.begin(), bp_end=subblocks.end(); blockp!=bp_end; ++blockp) {
    if ((*blockp)->_set_parameter_arg_number(param_name, arg_number)==CTK_SUCCESS)
      return CTK_SUCCESS;
  }

  return CTK_FAILURE;
}

// Automatic connection code: Adjacent blocks will be connected if output block has
// no connections and input block has one free socket

void  Block::autoconnect() {
 
  if (subblocks.size()==0) return;
  
  for_each(subblocks.begin(), subblocks.end(), mem_fun(&Block::autoconnect));
  
  for (BlockConstIt blockp=subblocks.begin(); blockp!=subblocks.end()-1; ++blockp) {
    (*blockp)->autoconnect_to(*(blockp+1));
  }

}


void  Block::autoconnect_to(Block *to_block) {
  
  // Check that my first output socket is free
  if (output_sockets->size()==0) return;  // No sockets

  SocketIt isocketp=to_block->input_sockets->begin();
  SocketIt osocketp=output_sockets->begin();

  // Connect free outputs to free non-optional inputs pairing them in the order which they occur
  for (; isocketp!=to_block->input_sockets->end(); ++isocketp) {
    if (((*isocketp)->connected()==false) && ((*isocketp)->am_i_a_subsocket()==false) && ((*isocketp)->optional()==false)) {
      for (; osocketp!=output_sockets->end(); ++osocketp) {
	if (((*osocketp)->connected()==false) && ((*osocketp)->am_i_a_subsocket()==false) ) {
	  (*osocketp)->make_connection(*isocketp);
	  break;
	}
      }
    }
  }

}


// Just check input connections - unconnected outputs are OK
Boolean  Block::check_baseblock_connections() const {
  Boolean status=true;

  for (BlockConstIt blockp=baseblocks.begin(); blockp!=baseblocks.end(); ++blockp) {
    status=status&((*blockp)->check_my_input_connections());
  }
  return status;
}

Boolean  Block::check_my_output_connections() const {
  Boolean status=true;

  for (SocketIt socketp=output_sockets->begin(); socketp!=output_sockets->end(); ++socketp) {
    if ((*socketp)->connected()==false) {
      cerr << "Unconnected output socket: " << (*socketp)->get_socket_name() << endl;
      status=false;
    }
  }

  return status;
}

Boolean  Block::check_my_input_connections() const {
  Boolean status=true;

  for (SocketIt socketp=input_sockets->begin(); socketp!=input_sockets->end(); ++socketp) {
    if ((*socketp)->connected()==false && (*socketp)->optional()==false) {
      cerr << "Unconnected input socket: " << (*socketp)->get_socket_name() << endl;
      status=false;
    }
  }

  return status;
}



void Block::connect(const string &from_name, const string &to_name){ 

  Socket *outsocketp=NULL;
  Socket *insocketp=NULL;
  
  BlockIt blockp;  
  for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
    if ((outsocketp=(*blockp)->find_output_socket(from_name))!=NULL) break;
  }
  
  if (outsocketp==NULL) {
    // trying fullname
    for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
      if ((outsocketp=(*blockp)->find_output_socket_from_fullname(from_name))!=NULL) break;
    }

    if (outsocketp==NULL) {
      // Couldn't find output socket. Could mean that the socketname wasn't specified as the
      // user knew the block to have only one socket. So see if this is just a subblockname.
      for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
	if ((*blockp)->output_sockets->size()==1 && (*blockp)->am_i_called(from_name)) {
	  outsocketp=(*(*blockp)->output_sockets)[0];
	  break;
	}
      }
      if (outsocketp==NULL) {
	
	throw(BlockErrorCFOS(__FILE__, __LINE__, from_name.c_str()));
	
      }
    }
  }

  
  for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
    if ((insocketp=(*blockp)->find_input_socket(to_name))!=NULL) break;
  }

  if (insocketp==NULL) {
    // trying fullname
    for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
      if ((insocketp=(*blockp)->find_input_socket_from_fullname(to_name))!=NULL) break;
    }
    
    if (insocketp==NULL) {
      
      
      // Couldn't find input socket. Could mean that the socketname wasn't specified as the
      // user knew the block to have only one non-optional input socket.  So see if this is just a subblockname.
      for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
	if ((*blockp)->input_sockets->num_non_optional()==1 && (*blockp)->am_i_called(to_name)) {
	  insocketp=(*(*blockp)->input_sockets)[0];
	  break;
	}
      }
      if (insocketp==NULL) {
	cerr << "Can't find input socket: " << to_name << endl;
	throw(BlockErrorCFIS(__FILE__, __LINE__, to_name.c_str()));
      }
    }
  }
  
  if (outsocketp->make_connection(insocketp)==false) {
    cerr << "cannot connect: " << from_name << " " << to_name << endl;
    throw(BlockErrorCCS(__FILE__, __LINE__, from_name.c_str(), to_name.c_str()));
  }
}

void Block::disconnect(const string &from_name, const string &to_name){ 
  
  Socket *outsocketp=NULL;
  Socket *insocketp=NULL;
  
  BlockIt blockp;  
  for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
    if ((outsocketp=(*blockp)->find_output_socket_from_fullname(from_name))!=NULL) break;
  }
  
  if (outsocketp==NULL) {
    throw(BlockErrorCFOS(__FILE__, __LINE__, from_name.c_str()));
  }
  
  for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
    if ((insocketp=(*blockp)->find_input_socket_from_fullname(to_name))!=NULL) break;
  }
 
  if (insocketp==NULL) {
    throw(BlockErrorCFIS(__FILE__, __LINE__, to_name.c_str()));
  }
 
  if (outsocketp->break_connection(insocketp)==false) {
    cerr << "cannot disconnect: " << from_name << " " << to_name << endl;
    throw(BlockErrorCCS(__FILE__, __LINE__, from_name.c_str(), to_name.c_str()));
  }
}

// inline
Block* Block::get_unfinished_block() {
  if (!(*baseblocks.begin())->is_eod()) return *baseblocks.begin();
  for (BlockList::iterator blockpp = baseblocks.begin(); blockpp!=baseblocks.end(); ++blockpp) {
    if (!(*blockpp)->is_eod()) return *blockpp;
  }
  return NULL;
}


void Block::process(bool a_runtime_profiling /*=  false */) {
  BlockStack proc_stack; 
  Block *ablockp;

  runtime_profiling=a_runtime_profiling;  // Set the global static variable

  timeval time_before, time_after;

  reset_all();

  if (runtime_profiling) {
    gettimeofday(&time_before, NULL);
  }

  user_interrupt=false;

#ifdef _HAS_QT
  int loop_count = 0;
#endif  
  while ((ablockp=get_unfinished_block())!=NULL) {
    proc_stack.push_back(ablockp);
    
    while (proc_stack.size()>0) {
      ablockp=proc_stack.back(); 
      proc_stack.pop_back();
      ablockp->process_me(proc_stack);
      if (proc_stack.size()>10000) {
	cerr << "Problem with data flow. System contains feedback?\n";
	throw(CTKError(__FILE__, __LINE__)); 
	
	break;
      }
    }

#ifdef _HAS_QT
      if (qApp!=NULL) {
	if (++loop_count%10==0) {
	  qApp->processEvents();
	  if (user_interrupt) {
	    user_interrupt=false;
	    break;
	  }
	}
      }
#endif
    
  }

  // Update elapsed time for main block
  if (runtime_profiling) {
    gettimeofday(&time_after, NULL);
    process_time.tv_usec+=(time_after.tv_usec-time_before.tv_usec);
    process_time.tv_sec+=(time_after.tv_sec-time_before.tv_sec);
    process_time_seconds=(process_time.tv_usec/1.0e6)+process_time.tv_sec;
    
    Float total_seconds=0.0;
      
    for (BlockConstIt blockp=baseblocks.begin(); blockp!=baseblocks.end(); ++blockp) {
      (*blockp)->process_time_seconds=((float)(*blockp)->process_time.tv_usec/1.0e6)+(*blockp)->process_time.tv_sec;
      total_seconds+=(*blockp)->process_time_seconds;
    }
    
    for (BlockConstIt blockp=baseblocks.begin(); blockp!=baseblocks.end(); ++blockp) 
      (*blockp)->process_time_percentage=(total_seconds==0.0)?0.0:100.0 * ( (*blockp)->process_time_seconds/total_seconds);

    // For the main block the process time percentage has a slighlty different meaning:
    process_time_percentage=total_seconds*100.0/process_time_seconds;
    
  }

  for_each(baseblocks.begin(), baseblocks.end(), mem_fun(&Block::close));

  for_each(baseblocks.begin(), baseblocks.end(), mem_fun(&Block::close2));


}

// Called after each execution of the system.
void Block::close() {
  // default is to do nothing
}

// Called after close.
void Block::close2() {
  // default is to do nothing
}

// Called after the final execution. 
void Block::close_final() {

}


void Block::reset_all(){
  // Put all baseblocks in a good initial state

  for_each(baseblocks.begin(), baseblocks.end(), mem_fun(&Block::full_reset));
}


// The sequence of operations needed to get the block in a good initial state
void Block::full_reset() {
  // Load input socket data descriptors from the output socket thats feeds them

  SocketIt insocp=input_sockets->begin();
  
  const DataDescriptor *default_dd = NULL;

  while (insocp!=input_sockets->end()) {
    // Fetch descriptor using descriptor from previous socket as default
    default_dd = ((InputSocket*)(*insocp))->fetch_descriptor(default_dd);
    ++insocp;
  }

  // Set the 'inputs are all sample data' flag
  inputs_are_all_sample_data=inputs_are_sample_data_check();
  
  // Execute the 'block specific' reset
  // (This can be overriden to be specific to each block)
  reset();
  
  // Construct output socket data descriptors from input sockets data descriptors
  //  (This can be overriden to be specific to each block)

  build_output_data_descriptors();

}

// Operations necessary to reset the block base class
void Block::reset(){
  verbosity_level=verbosity_level_param->get_value();
  
  inherit_unset_parameters();

  reset_sockets();
}

void Block::build_output_data_descriptors() {
  // The default action is to copy the socket 0 descriptor to each of the output sockets

  SocketIt socketp=input_sockets->begin();
  if (socketp!=input_sockets->end()) {
    const DataDescriptor *dd=(*socketp)->get_data_descriptor();
    if (dd==NULL) {
      cerr << "Internal error - Block::build_output_data_descriptors: no input data desc for building output data desc!" << endl;
      exit(-1);
    }
    for (SocketIt osocketp=output_sockets->begin(); osocketp!=output_sockets->end();  ++osocketp) {
      (*osocketp)->set_data_descriptor(new DataDescriptor(*dd));
    }
  }
    
}


void Block::add_input_socket(const string &aname, const string &subsocket_name, const string &desc/*="\0"*/) {
  Socket *socketp=NULL;
  
  BlockIt blockp;
  for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
    if ((socketp=(*blockp)->find_input_socket(subsocket_name))!=NULL) break;
  }
  
  if (socketp==NULL) {
    // Couldn't find input socket. Could mean that the socketname wasn't specified as the
    // user knew the block to have only one socket.  So see if this is just a subblockname.
    for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
      if ((*blockp)->input_sockets->size()==1 && (*blockp)->am_i_called(subsocket_name)) {
	socketp=(*(*blockp)->input_sockets)[0];
	break;
      }
    }
    if (socketp==NULL) throw(BlockErrorCFIS(__FILE__, __LINE__, subsocket_name.c_str()));
  }

  Socket *input_socket = input_sockets->add_socket(aname, this, socketp);
  if (!desc.empty())
    input_socket->set_description(desc);
}


void Block::add_output_socket(const string &aname, const string &subsocket_name, const string &desc/*="\0"*/) {
  Socket *socketp=NULL;

  BlockIt blockp;
  for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
    if ((socketp=(*blockp)->find_output_socket(subsocket_name))!=NULL) break;
  }
  
  if (socketp==NULL) {
    // Couldn't find output socket. Could mean that the socketname wasn't specified as the
    // user knew the block to have only one socket. So see if this is just a subblockname.
    for (blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
      if ((*blockp)->output_sockets->size()==1 && (*blockp)->am_i_called(subsocket_name)) {
	socketp=(*(*blockp)->output_sockets)[0];
	break;
      }
    }
    if (socketp==NULL)  throw(BlockErrorCFOS(__FILE__, __LINE__, subsocket_name.c_str()));
  }
  
  Socket *output_socket = output_sockets->add_socket(aname, this, socketp);
  if (!desc.empty())
    output_socket->set_description(desc);
  
}

void Block::add_parameter(const string &aname, const string &subparameter_name) {
  Param *paramp=NULL;

  for (BlockIt blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) {
    if ((paramp=(*blockp)->parameters->make_subparameter(subparameter_name, aname))!=NULL) break;
  }
  
  if (paramp==NULL) throw(BlockErrorCFP(__FILE__, __LINE__, subparameter_name.c_str()));
  
  if (parameters->register_parameter(paramp)==false) throw(BlockErrorCRP(__FILE__, __LINE__, aname.c_str()));

}


void Block::set_num_outputs(Integer n) {
  output_sockets->configure(n);
}

void Block::set_num_inputs(Integer n) {
  input_sockets->configure(n);
}

int Block::get_num_outputs() const {
  return output_sockets->size();
}

int Block::get_num_inputs() const {
  return input_sockets->size();
}



int Block::get_min_num_inputs() const {
  return input_sockets->get_min_num();
}

int Block::get_max_num_inputs() const {
  return input_sockets->get_max_num();
}

int Block::get_min_num_outputs() const {
  return output_sockets->get_min_num();
}

int Block::get_max_num_outputs() const {
  return output_sockets->get_max_num();
}

void Block::set_num_outputs(const string &subblock_name, Integer n) {
  Block *block=find_subblock(subblock_name);
  if (block!=NULL) block->set_num_outputs(n);
  else throw(BlockErrorCFSB(__FILE__, __LINE__, subblock_name.c_str()));   
}

void Block::set_num_inputs(const string &subblock_name, Integer n) {
  Block *block=find_subblock(subblock_name);
  if (block!=NULL) block->set_num_inputs(n);
  else throw(BlockErrorCFSB(__FILE__, __LINE__, subblock_name.c_str()));   
}

void Block::reset_all_profile_statistics() {
  reset_profile_statistics();
  for_each(baseblocks.begin(), baseblocks.end(), mem_fun(&Block::reset_profile_statistics));
}

void Block::reset_profile_statistics() {
  process_time.tv_sec=0;
  process_time.tv_usec=0;
  compute_call_count=0;
  process_time_seconds=0.0;
  process_time_percentage=0.0;
}

void Block::write_profiling_report(ostream &outfile) {
  outfile << "Profile Statistics: " << endl;
  
  outfile << getfullname() << ": " << process_time_seconds << endl;
  
  for (BlockConstIt blockp=baseblocks.begin(); blockp!=baseblocks.end(); ++blockp) {
    outfile << (*blockp)->getfullname() << " (" <<  (*blockp)->compute_call_count << ") ";
    outfile << (*blockp)->process_time_seconds << "s [";
    outfile.precision(3);
    outfile << (*blockp)->process_time_percentage<< "%]" << endl;
  }

  outfile << "Network efficiency: ";
  outfile.precision(3);
  outfile << process_time_percentage << "%" << endl;
}

void Block::display(ostream &outfile) {
  reset_all();
  display_me(outfile);
}

void Block::display_parameters(ostream &outfile, const string &space) {
  parameters->display_all(outfile, space);
}

void Block::fprintf_parameters_XML(FILE *fp) {
  parameters->fprintf_XML_all(fp);
}


void Block::display_me(ostream &outfile, const string &space) {
  outfile << space << "BLOCK NAME: " << getname() << endl;
  outfile << space << "BLOCK TYPE: " << block_type << endl;

  outfile << space << "INPUTS:" << *input_sockets << endl;
  outfile << space << "OUTPUTS:" << *output_sockets << endl;

  display_parameters(outfile, space);

  if (subblocks.size()!=0) {
    outfile << space << "SUBBLOCKS: " << endl;
    string new_space= space + "   ";

    for (BlockIt bp=subblocks.begin(); bp!=subblocks.end(); ++bp)
      (*bp)->display_me(outfile, new_space);

  }
  outfile << endl << endl;

}

Socket* Block::find_output_socket(const string &aname) {
  SocketIt socketp=find_if(output_sockets->begin(), output_sockets->end(),  bind2nd(mem_fun(&Socket::am_i_called),aname));
  return (socketp==output_sockets->end())?NULL:*socketp;
}

Socket* Block::find_input_socket(const string &aname) {
  SocketIt socketp=find_if(input_sockets->begin(), input_sockets->end(),  bind2nd(mem_fun(&Socket::am_i_called),aname));
  return (socketp==input_sockets->end())?NULL:*socketp;
}

Socket* Block::find_output_socket_from_fullname(const string &aname) {
  SocketIt socketp=find_if(output_sockets->begin(), output_sockets->end(),  bind2nd(mem_fun(&Socket::is_my_fullname),aname));
  return (socketp==output_sockets->end())?NULL:*socketp;
}

Socket* Block::find_input_socket_from_fullname(const string &aname) {
  SocketIt socketp=find_if(input_sockets->begin(), input_sockets->end(),  bind2nd(mem_fun(&Socket::is_my_fullname),aname));
  return (socketp==input_sockets->end())?NULL:*socketp;
}

Block* Block::find_subblock(const string &aname) {
  BlockIt blockp=find_if(subblocks.begin(), subblocks.end(),  bind2nd(mem_fun(&CTKObject::am_i_called),aname));
  return (blockp==subblocks.end())?NULL:*blockp;
}

Boolean Block::is_eod() const {return eod;}

void Block::set_eod() {eod=true;}

void Block::set_current_count(int x){current_count=x;}

void Block::process_me(BlockStack &proc_stack) {

  Integer socket;

  if (input_sockets->size()==0 && is_eod()) {
    for_each(output_sockets->begin(), output_sockets->end(), mem_fun(&Socket::set_end_of_data));  
    return;
  }
  
  if (input_sockets->size()==0 && current_count>0) {
    --current_count;
    if (current_count==0) set_eod();
  }
  
  if (will_run(socket)==false) {
    
    if ((*input_sockets)[socket]->get_end_of_data()==true) {
      if (!is_eod())
	flush_buffer(); // allow block to finish processing any data stored internally
      set_eod();

      for_each(output_sockets->begin(), output_sockets->end(), mem_fun(&Socket::set_end_of_data));

      
      for (SocketIt socketp=output_sockets->begin(); socketp!=output_sockets->end(); ++socketp) {
	if ((*socketp)->connected()) {
	  Block *b = (*socketp)->get_remote_baseblock();
	  if (find(proc_stack.begin(), proc_stack.end(), b)==proc_stack.end())
	    proc_stack.push_back(b);   
	}
      }
      
    } else {
      Block *b = (*input_sockets)[socket]->get_remote_baseblock();
      if (find(proc_stack.begin(), proc_stack.end(), b)==proc_stack.end())
	proc_stack.push_back(b);   
    }
    return;
  }
  
  process_input_data();
  
  for (SocketIt socketp=output_sockets->begin(); socketp!=output_sockets->end(); ++socketp) {
    if ((*socketp)->connected()) {
      Block *b = (*socketp)->get_remote_baseblock();
      if (find(proc_stack.begin(), proc_stack.end(), b)==proc_stack.end())
	proc_stack.push_back(b);
    }
  }

  
}


// Virtual function - defaults to doing nothing, but should be overwritten by blocks that
// need to process data stored in internal buffers before closing down

void Block::flush_buffer() {

}



void Block::process_input_data() {
  Integer dummy_arg;

  // Repeatedly call compute until one or more
  // input buffers are exhausted.

  // This method is overriden for source blocks - ie. blocks with no inputs

  
  bool has_inputs = input_sockets->size()>0;
  
  if (runtime_profiling==false) {
    do {
      compute();
    } while (will_run(dummy_arg) && has_inputs);
  } else {
    cerr << "Process: " << getname() << "\n";
    do {
      compute_with_runtime_profiling_wrapper();
    } while (will_run(dummy_arg) && has_inputs);
  }
  
}

void Block::compute_with_runtime_profiling_wrapper() {
  timeval time_before, time_after;

  gettimeofday(&time_before, NULL);
  compute();
  gettimeofday(&time_after, NULL);

  // Update accumulated statistics
  ++compute_call_count;
  process_time.tv_usec+=(time_after.tv_usec-time_before.tv_usec);
  process_time.tv_sec+=(time_after.tv_sec-time_before.tv_sec);
  
}

Boolean Block::will_run(Integer &return_socket) {
  
  SocketIt socketp=find_if(input_sockets->begin(), input_sockets->end(), boost::compose_f_gx(logical_not<bool>(), mem_fun(&Socket::data_ready)));
  if (socketp==input_sockets->end()) {
    return true;
  }
  else {
    return_socket=socketp-input_sockets->begin();
    return false;
  }
}

Boolean Block::ready_to_fire() const {
  if (input_sockets->size()==0) {
    if (!trigger_source) return false;
    return true;
  } else {
    Boolean trigger=true;
    
    for (SocketIt socketp=input_sockets->begin(); socketp!=input_sockets->end(); ++socketp) {
      if ((*socketp)->connected())
	trigger=trigger&(*socketp)->has_fired();
    }
    return trigger;
  }
}

Boolean  Block::order_operations(BlockList &blocklist) {

  if (!ready_to_fire()) return false;
  
  for (BlockIt blockp=blocklist.begin(); blockp!=blocklist.end(); ++blockp) {
    if ((*blockp)->order_operations((*blockp)->subblocks)==false) {
      Integer pos=blockp-blocklist.begin();
      blocklist.push_back(*blockp);
      blockp=blocklist.erase(blocklist.begin()+pos)-1;
    }
  }
  
  fire();  // clear inputs, trigger outputs
  
  return true;
}


void Block::fire() {

  trigger_source=false;
  for_each(input_sockets->begin(), input_sockets->end(), mem_fun(&Socket::reset_fire));
  for_each(output_sockets->begin(), output_sockets->end(), mem_fun(&Socket::fire));
}

Boolean Block::prime_my_pipes() {

  if (!ready_to_fire()) return false;
  
  fire();  // clear inputs, trigger outputs

  return true;
}

Boolean Block::make_input_sockets(Integer n, Boolean configurable/*=false*/, Integer min_value/*=0*/, Integer max_value/*=CTK_MAX_SOCKETS*/) {
  if (input_sockets->size()!=0) {
    cerr << "calling Block::make_input_sockets with n outside min:max range" << endl;
    return true;
  }

  if (configurable && ((n<min_value)||(n>max_value))) return true;
  
  input_sockets->set_configurable(configurable, min_value, max_value);

  char new_name[CTK_CHAR_BUFFER_SIZE];
  for (Integer i=0; i<n; ++i) {
    sprintf(new_name,"in%d",i+1);
    input_sockets->add_socket(new_name, this, NULL);
  }
  return false;
}


Boolean Block::make_output_sockets(Integer n, Boolean configurable/*=false*/, Integer min_value/*=0*/, Integer max_value/*=CTK_MAX_SOCKETS*/) {
  if (output_sockets->size()!=0) {
    cerr << "calling Block::make_output_sockets with n outside min:max range" << endl;
    return true;
  }
  
  if (configurable && ((n<min_value)||(n>max_value))) return true;

  output_sockets->set_configurable(configurable, min_value, max_value);
  
  char new_name[CTK_CHAR_BUFFER_SIZE];
  for (Integer i=0; i<n; ++i) {
    sprintf(new_name,"out%d",i+1);
    output_sockets->add_socket(new_name, this, NULL);
  }

  return false;
}

Block* Block::copy_this_block_to(Block *ablock) const {
  ablock->copy_parameters(parameters);
  if (input_sockets->is_configurable()) ablock->set_num_inputs(input_sockets->size());
  if (output_sockets->is_configurable()) ablock->set_num_outputs(output_sockets->size());
  return ablock;
}

// Returns true if all inputs are of sample type - i.e. 0-d
Boolean Block::inputs_are_sample_data_check() {
  Boolean flag=true;
  for (SocketIt socketp=input_sockets->begin(); socketp!=input_sockets->end(); ++socketp) {
    if ((*socketp)->connected()) {
      if ((*socketp)->get_data_descriptor()->is_sample_data()==false) {
	flag=false;
	break;
      }
    }
  }
  return flag;
}

 
// Returns true if all inputs have the same shape (i.e. size and dimension)
Boolean Block::input_shape_check() {
  Boolean flag=true;
  const DataDescriptor *dd0=(*input_sockets->begin())->get_data_descriptor();
  
  for (SocketIt socketp=input_sockets->begin()+1; socketp!=input_sockets->end(); ++socketp) {
    if ((*socketp)->connected()) {
      if (dd0->matches_shape_of((*socketp)->get_data_descriptor())==false) {
	flag=false;
	break;
      }
    }
  }
  
  return flag;
}

// Returns true if all inputs are either 1D frames or simple samples
Boolean Block::inputs_1D_or_less_check() {

  for (SocketIt socketp=input_sockets->begin(); socketp!=input_sockets->end(); ++socketp) {
    if ((*socketp)->connected()) {
      if ((*socketp)->get_data_descriptor()->get_n_dimensions() > 1) return false;
    }
  }
  
  return true;
}

// Finds the total storage of all the input data
Integer Block::sum_of_input_storage() {
  Integer total=0;
  
  for (SocketIt socketp=input_sockets->begin(); socketp!=input_sockets->end(); ++socketp) {
    if ((*socketp)->connected()) {
      total+=(*socketp)->get_data_descriptor()->get_storage();
    }
  }
  return total;
}


// This method parses the network and tries to calculate which
// pipes must be primed in order for it to flow

void Block::prime_baseblock_pipes() {

  Integer tries=0;
  
  if (subblocks.size()==0) return;
  
  while (++tries<CTK_DEPENDENCY_RESOLUTION_ATTEMPTS) {
    
    UInteger trigger_count=0, new_count=0;

    reset_sockets_all();
    
    do {
      trigger_count=new_count;
      new_count+=count_if(baseblocks.begin(), baseblocks.end(), mem_fun(&Block::prime_my_pipes));
    } while (new_count!=trigger_count);
    
    if (trigger_count!=baseblocks.size()) {
      for_each(baseblocks.begin(), baseblocks.end(), mem_fun(&Block::update_prime_list));
    } else break;
  }

  if (tries==CTK_DEPENDENCY_RESOLUTION_ATTEMPTS)
    throw(BlockErrorDRE(__FILE__, __LINE__));
}


void Block::recursive_flatten(BlockList &these_baseblocks) {
  
  //  if (subblocks.size()==0) {
  //   return;
  //  }
  
  _recursive_flatten(these_baseblocks);
}


void Block::_recursive_flatten(BlockList &these_baseblocks) {
  if (subblocks.size()==0) {
    these_baseblocks.push_back(this);
    return;
  }
  
  for (BlockIt blockp=subblocks.begin(); blockp!=subblocks.end(); ++blockp) 
    (*blockp)->recursive_flatten(these_baseblocks);
}


void Block::compute(){
};


void Block::update_prime_list() {
  Integer prime_count=count_if(input_sockets->begin(),input_sockets->end(), mem_fun(&Socket::has_fired));
  
  if (prime_count>0 && prime_count<input_sockets->size()) {
    for (SocketIt socketp=input_sockets->begin(); socketp!=input_sockets->end(); ++socketp) {
      if ((*socketp)->connected()) {
	if (!(*socketp)->has_fired()) {
	  (*socketp)->make_primed_on_reset();
	}
      }
    }
  }
}



void Block::reset_sockets_all(){
  for_each(baseblocks.begin(), baseblocks.end(), mem_fun(&Block::reset_sockets));  
  trigger_source=true;
}

void Block::reset_sockets() {
  eod=false;
  for_each(output_sockets->begin(), output_sockets->end(), mem_fun(&Socket::reset));  
  for_each(input_sockets->begin(), input_sockets->end(), mem_fun(&Socket::reset));  
  trigger_source=true;
}

void Block::inherit_unset_parameters(){ 
  ParamList *plistp;
  Block *blockp;

  if (input_sockets->size()!=0)
    parameters->clear_transmitted_parameters(); // unset all transmitted parameter
  
  for (SocketIt socketp=input_sockets->begin(); socketp!=input_sockets->end(); ++socketp) {
    if ((*socketp)->connected()) {
      blockp=(*socketp)->get_remote_baseblock();
      plistp=blockp->get_parameters();

      ParamIt dodgy_parameter;
      CTKStatus status = parameters->copy_transmitted_settings(plistp, dodgy_parameter);
      if (status==CTK_FAILURE) {
	cerr << "Inconsistent transmitted parameters: " << getfullname() << ": " << (*dodgy_parameter)->getname() << endl;
	throw(CTKError(__FILE__, __LINE__));
      }
    }
  }
}

/******************************************************************************/
/*									      */
/* The Matlab Interface         					      */
/*									      */
/******************************************************************************/

#ifdef _HAS_MATLAB
void Block::matlab_startup() {
  // start MATLAB
  if (ep==NULL) {
    cerr << "Attempting to start MATLAB engine...";
    if(!(ep=engOpen("\0"))){
      cerr << "Can't start MATLAB engine." << endl;
      throw(CTKError(__FILE__, __LINE__));
    }
    cerr << "OK" << endl;
    matlab_output_buffer=new char[MATLAB_OUTPUT_BUFFER_SIZE];
    engOutputBuffer(ep, matlab_output_buffer, MATLAB_OUTPUT_BUFFER_SIZE);
  }
}

void Block::matlab_shutdown(Boolean no_warning/*=false*/) {
  if (ep!=NULL) {
    if (no_warning==false) {
      cerr << "Hit return to continue" << endl;
      fgetc(stdin);
    }
    engEvalString(ep, "close;");
    engClose(ep);
    ep=NULL;
    delete []matlab_output_buffer;
  }
}

void Block::dispatch_matlab_command(const char *command) {
  if (ep==NULL) {
    matlab_startup();
  }
  engEvalString(ep, command);

  // Flush output buffer

  Integer i=MATLAB_PROMPT_LENGTH;
  char *bp=&matlab_output_buffer[i];
  // First few characters of buffer are a spurious prompt
  while (i++<MATLAB_OUTPUT_BUFFER_SIZE) {
    if (*bp==0) break;
    cerr << *bp;
    
    if (*bp=='\n') cerr << '\r';  // JON - need this line until curses is sorted out

    *bp++=0;
  }

}
#endif  // _HAS_MATLAB

/******************************************************************************/
/*									      */
/*	CLASS NAME: SourceBlock						      */
/*									      */
/******************************************************************************/

const string SourceBlock::help_text = SOURCE_BLOCK_HELP_TEXT;

SourceBlock::SourceBlock():CTKObject(), Block(), duration_param() {
  do_constructor();
}  

SourceBlock::SourceBlock(const string &a_name, const string &a_block_type): CTKObject(a_name), Block(a_name, a_block_type) {
  do_constructor();
    
  // Set up for sample rate parameter
  if (!a_name.empty()) {
    unlock_parameters(); 
    unset_parameter_const("SAMPLE_RATE");
    relock_parameters();
  }
}

void SourceBlock::do_constructor() {

  // Set up for DURATION parameter
  duration_param = new ParamFloat("DURATION");
  duration_param->set_unit("S");
  duration_param->set_helptext("The number of seconds of signal to generate. If set to 0 the block will run indefinitely.");
  parameters->register_parameter(duration_param);

}

SourceBlock::~SourceBlock(){
  //  cerr << "In SourceBlock Destructor " << endl;
};

void SourceBlock::build_output_data_descriptors() {
  // Make a default 'sample data' descriptor for each output socket

  for (SocketIt osocketp=output_sockets->begin(); osocketp!=output_sockets->end();  ++osocketp) {
      (*osocketp)->set_data_descriptor(new DataDescriptor());
  }
    
}

void SourceBlock::reset() {
  Block::reset();
  if (sample_rate_param->get_set_flag()) 
    set_current_count((int)(duration_param->get_value()*sample_rate_param->get_value()));
}


/* End of ctk_block.cpp  */

