/******************************************************************************/
/*									      */
/*	ctk_block.hh	    						      */
/*									      */
/*	Class declarations for ctk_block.cpp				      */
/*									      */
/*	Author: Jon Barker, Sheffield University			      */
/*									      */
/*      CTK VERSION 1.3.5  Apr 22, 2007		         	      */
/*									      */
/******************************************************************************/

#ifndef CTK_BLOCK_HH
#define CTK_BLOCK_HH

#include <sys/time.h>  // Included for run-time profiling

#include <vector>
#include <stack>
#include <string>
#include <queue>

#include "ctk_local.hh"
#include "ctk_types.hh"    // For CTKVector and CTKSample 

#include "ctk_help_text.hh"
#include "ctk_param.hh"

#ifdef _HAS_MATLAB
#include "engine.h"
#endif


class InputSocketList;
class OutputSocketList;
class Socket;
class Block;
class ParamList;
class Param;
class ParamFloat;
class ParamInt;

/** DECLARATION AND DEFINITION OF BLOCK CLASSES */

// Number of iterations before giving up when trying to resolve feedback problems
const Integer CTK_DEPENDENCY_RESOLUTION_ATTEMPTS = 500;  

const Float PARAM_DEFAULT_SAMPLE_RATE = 16000.0;
const Integer PARAM_DEFAULT_VERBOSITY_LEVEL = 0;

/******************************************************************************/
/*									      */
/*	Local Type Definitions						      */
/*									      */
/******************************************************************************/

typedef vector<Block*> BlockList;
typedef BlockList::iterator BlockIt;
typedef BlockList::const_iterator BlockConstIt;

//typedef stack<Block*, deque<Block*> > BlockStack;
//typedef queue<Block*> BlockStack;

typedef vector<Block*> BlockStack;

/******************************************************************************/
/*									      */
/*	CLASS NAME: CTKObject						      */
/*									      */
/******************************************************************************/

class CTKObject {

private:

  /// Stores the name of the block
  string name;

  mutable string fullname;
 

protected:

  ParamList *parameters;

  /* Pointer to the parent block if this is a subblock */
  CTKObject *parentobjectp;

public:
  
  CTKObject();

  CTKObject(const string &this_name);

  virtual ~CTKObject();

  Boolean am_i_called(string saname) const;
  
  string getname() const;

  // Return the CTKObject's icon in xpm format - used by the GUI
  virtual const char **getIcon() const;

  void change_name(const string &new_name) {
    name=new_name;
  };
  

  void copy_parameters(ParamList *some_parameters);
  
  ParamList *get_parameters() const;
  ParamList *get_parameters();

  string getfullname() const;

  // Returns    "name:s"
  string prefix_object_name(const string &s);
  string prefix_object_fullname(const string &s);


protected:

  CTKStatus set_parameter(const string &param_name, const string &value);

  CTKStatus set_parameter_unresolved_value(const string &param_name, const string &value);

  CTKStatus set_parameter(const string &param_name, Float value);

  CTKStatus set_parameter_arg_number(const string &param_name, Integer arg_number);

  CTKStatus unset_parameter(const string &param_name);

  
  void set_parameter_const(const string &param_name);

  void set_parameter_hidden(const string &param_name);

  void set_parameter_transmitted(const string &param_name);

  void unset_parameter_const(const string &param_name);

  void unset_parameter_hidden(const string &param_name);

  void unset_parameter_transmitted(const string &param_name);
  
  inline void unlock_parameters(){parameters->unlock();};

  inline void relock_parameters(){parameters->relock();};
  
  inline Boolean register_parameter(Param* paramp, UInteger parameter_flags=0);


private:

  void recursive_build_fullname(string &this_name) const;

  // Returns "s1:s2"
  string colon_concatenate(const string &s1, const string &s2);

};


/******************************************************************************/
/*									      */
/*	CLASS NAME: Block						      */
/*									      */
/******************************************************************************/

const int MATLAB_OUTPUT_BUFFER_SIZE = 8192;
const int MATLAB_PROMPT_LENGTH = 3;
const int CTK_MAX_SOCKETS = 8;

class Block: public virtual CTKObject {

  friend class CTKWriter;
  
private:  

  /** The list of subblocks which this block is composed of.
    This list will be empty if the block is a base-level block. */
  BlockList subblocks;

  /** This is the list of base-level blocks built by initialise() by 
    recursive expansion of the blocks sub-blocks. */
  BlockList baseblocks;

  /* Character buffer for matlab output */
  static char *matlab_output_buffer;

  const string block_type;

  static const string help_text;
  string settable_help_text;     //  User definable help texts - used for blocks that aren't inbuilt
  
  /* The name of the file in which the block's definition is stored - if it is not an inbuilt block*/
  string file_name;

  Integer verbosity_level;

  Boolean eod;  // End of data

  /// The remaining number of triggers. 
  Integer current_count;

  Boolean trigger_source;
  
  
  // RUN TIME PROFILING
  static Boolean  runtime_profiling;   // TRUE if Run time profiling is activated
  timeval process_time;           // Accumulated process time   - the time spent in the compute() method
  Float process_time_seconds;    //                - same time expressed in seconds
  Float process_time_percentage;  // process_time as a percentage of all total compute time for network  
  Integer compute_call_count;   // The number of times compute() has been called for this block

  bool user_interrupt;     // User interrupt signal - set true to stop a running system
  
protected:
  
  /** The data inputs. If the block is not a base-level block these will infact
    be input_super_sockets which connect to base-level block input sockets */
  //  InputSocketVector input_sockets;
  InputSocketList *input_sockets;
  
  /** The data outputs. If the block is not a base-level block these will infact
    be output_super_sockets which connect to base-level block output sockets */
  //  OutputSocketVector output_sockets;
  OutputSocketList *output_sockets;

  ParamFloat *sample_rate_param;
  ParamInt *verbosity_level_param;
  
  /* Set to true if all the input sockets are passing in sample data */
  Boolean inputs_are_all_sample_data;

  int get_frames_remaining() const {return current_count;}
  
#ifdef _HAS_MATLAB
  /* The Matlab engine pointer. Set to NULL when Matlab not in use */
  static Engine *ep; 
#endif

  //
  //
  //
  
public:
  
  /*name Construction and Destruction. */

  Block();

  Block(const string &this_name, const string &this_block_type, const string &this_file_name=string());

  /// Construct a block with a name and a list of subblocks
  Block(const string &this_name, BlockList these_blocks, const string &this_block_type, const string &this_file_name=string());

  /// Copy a block and give it a new name. i.e. a virtual copy constructor
  virtual Block* clone(const string &new_name=string()) const;
  
 
  virtual ~Block();

  
  /// Add a block to the subblock list
  Block* add_subblock(const Block &subblock, const string &new_name=string());

  /// Add a block to the subblock list
  Block* add_subblock(const Block *subblockp, const string &new_name=string());
    
  /// Remove a named block from the subblock list
  void remove_subblock(const string &name);

  /** This must be called after construction abd before process().
    It performs the following tasks.
    \begin{itemize}
    \item{Construction of the list of the blocks base-blocks.}
    \item{Checks that the network of baseblocks is fully connected.}
    \item{Detects dataflow lock-ups and sets the prime status of pipes to resolve them.}
    \item{Calculates the dependency order of the base-blocks and rearranges the block list
    accordingly (this is not necessary for the stack based data flow control, but necessary 
    for simpler control strategies)}
    \end{itemize}
    */
  bool initialise();

  /// Connect a subblock's output to another subblock's input.
  void connect(const string &from_name, const string &to_name); 

  /// Disconnect a subblock's output from another subblock's input.
  void disconnect(const string &from_name, const string &to_name); 

  /// Run the block.
  void process(bool runtime_profiling=false);   // if set to true then run time profiling is enabled
    
  // This is called after process to shutdown the block system
  void close_final_all(); 

  /// Define an input socket to a super-block.
  void add_input_socket(const string &aname, const string &subsocket_name, const string &socket_desc="\0");
  
  /// Define an output socket for a super-block.
  void add_output_socket(const string &aname, const string &subsocket_name, const string &socket_desc="\0"); 

  void add_parameter(const string &aname, const string &subparameter_name);

  void set_parameter(const string &param_name, const string &value);

  void set_parameter_unresolved_value(const string &param_name, const string &value);

  void set_parameter(const string &param_name, Float value);

  void unset_parameter(const string &param_name);

  void set_parameter_arg_number(const string &param_name, Integer arg_number);

  int get_min_num_inputs() const;
  
  int get_max_num_inputs() const;
  
  int get_min_num_outputs() const;
  
  int get_max_num_outputs() const;  

  int get_current_count() const {return current_count;}
  
  void set_num_outputs(Integer n);
  void set_num_inputs(Integer n);

  int get_num_outputs() const;
  int get_num_inputs() const;
  
  void set_num_outputs(const string &subblock_name, Integer n);
  void set_num_inputs(const string &subblock_name, Integer n);
  
  void display(ostream &outfile);
  
  void display_parameters(ostream &outfile, const string &space);
  void fprintf_parameters_XML(FILE *fp);

  void reset_all_profile_statistics();
  void write_profiling_report(ostream &outfile);    // Display a summary of the networks profiling statistics
  
  const BlockList &get_baseblocks() const {return baseblocks;}
  const BlockList &get_subblocks() const {return subblocks;}
  const InputSocketList* get_input_sockets() const {return input_sockets;}
  const OutputSocketList* get_output_sockets() const{return output_sockets;}

  const string &get_blocktype() const {return block_type;}
  const string &get_file_name() const {return file_name;}
  void change_file_name(const string &new_name, const string &old_name);

  virtual const string &get_helptext() const {return settable_help_text;}
  void set_helptext(string text) {settable_help_text=text;}

  Boolean is_my_fullname(string aname) const;
  
  Integer get_verbosity_level() const {return verbosity_level;}
  
  void autoconnect();
  
  float get_profile_process_time() const {return process_time_seconds;}
  float get_profile_process_time_percentage() const {return process_time_percentage;}
  int get_profile_call_count() const {return compute_call_count;};

  // The Matlab interface
#ifdef _HAS_MATLAB
  static void matlab_startup();
  
  static void matlab_shutdown(Boolean no_warning=false);
  
  static void dispatch_matlab_command(const char *command);
#endif

  // Called by GUI to signal a requested interrupt of a running block signal
  void set_user_interrupt () {user_interrupt=true;}

protected:
  
  /// Write description of block 
  virtual void display_me(ostream &outfile, const string &space=string());

  /// Close the block - will be called after the processing for each example is complete.
  virtual void close();

  /// Close the block again - will be called after the close method ... handy for dealing with dodgy static variables that can only be reset when close has been called for each block. e.g. clearing the word record memory in the decoder when multiple decoders are present
  virtual void close2();

  /// Close the block - will be called after processing for an entire list is complete.
  virtual void close_final();

  /// Reset the block. Must be called before a block is rerun.
  virtual void reset();

  /// Use input data descriptors to set output data descriptors
  virtual void build_output_data_descriptors();
  
  // Should be overwritten by blocks that need to process and output data stored
  // in internal buffers before they close down.
  virtual void flush_buffer();



  
  void set_current_count(int x);
  
  Boolean make_input_sockets(Integer n, Boolean configurable=false, Integer min=0, Integer max=CTK_MAX_SOCKETS);
  
  Boolean make_output_sockets(Integer n, Boolean configurable=false, Integer min=0, Integer max=CTK_MAX_SOCKETS);

  Block* copy_this_block_to(Block *ablock) const;

  // Various checks that can be run on the input data descriptors
  Boolean input_shape_check();
  Boolean inputs_are_sample_data_check();
  Boolean inputs_1D_or_less_check();

  // Sum of the storage for all the inputs
  Integer sum_of_input_storage();


  /// Puts a wrapper around compute for measuring runtime statistics
  void compute_with_runtime_profiling_wrapper();

  Boolean get_runtime_profiling() const {return runtime_profiling;}
  
  void set_eod();
  

private:

  /// The method that does the work. Needed by all baseblocks. Empty for superblocks.
  virtual void compute();




  
  /// Called by process() on each subblock of a block.
  void process_me(BlockStack &proc_stack);

  void reset_sockets();
  
  void process_input_data();
  
  /// Return End Of Data. i.e. Has the input been consumed?
  Boolean is_eod() const;
  
  /// Arrange subblocks in their correct dependency order.
  Boolean order_operations(BlockList &blocklist);
  
  /// Obscure method needed for detecting lock-ups
  Boolean prime_my_pipes();

  /// Check that block has sufficient data in buffers to run
  Boolean will_run(Integer &return_socket);

  //  void set_file_name(const string &this_file_name);
  void make_block(const BlockList &these_blocks);

  // Return the first block for which eod is not set
  inline Block* get_unfinished_block();
  /// 

  //
  void reset_all();
  
  /// Trigger block. Reset fire status on inputs and set on outputs.
  void fire();

  /** Return a pointer to a input socket with a given name. */
  Socket* find_input_socket(const string &aname);

  /** Return a pointer to an output socket with a given name. */
  Socket* find_output_socket(const string &aname);

  /** Return a pointer to a input socket with a given FULL name. */
  Socket* find_input_socket_from_fullname(const string &aname);

  /** Return a pointer to an output socket with a given FULL name. */
  Socket* find_output_socket_from_fullname(const string &aname);


  /** Return a pointer to a subblock with a given name. */
  Block *find_subblock(const string &aname);

  /// Mark input pipes as primed on reset. Called by the lock-up resolving code.
  void prime_baseblock_pipes();
  
  /// Build the baseblock list.
  void recursive_flatten(BlockList &these_baseblocks);
    
 
  /** When a lock-up occurs in the pipe priming mechanism this method calculates which pipe
    must be made primed-on-reset to solve the problem */
  void update_prime_list();

  /// Check that all my baseblock sockets have been connected to something.
  Boolean check_baseblock_connections() const;

    
  /// Check that all my input sockets have been connected to something.
  Boolean check_my_input_connections() const;

  /// Check that all my output sockets have been connected to something.
  Boolean check_my_output_connections() const;


  /// Perform the full sequence of operations necessary to get block into good initial state
  void full_reset();

  CTKStatus _unset_parameter(const string &param_name);
  CTKStatus _set_parameter(const string &param_name, const string &value);
  CTKStatus _set_parameter_unresolved_value(const string &param_name, const string &value);
  CTKStatus _set_parameter(const string &param_name, Float value);
  CTKStatus _set_parameter_arg_number(const string &param_name, Integer arg_number);

  void autoconnect_to(Block *to_block);
  
  void reset_sockets_all();

  virtual void inherit_unset_parameters();

  void _remove_subblock(const string &name);
  void _recursive_flatten(BlockList &these_baseblocks);

  void reset_profile_statistics();

  Boolean ready_to_fire() const;  // Returns true if block is ready to fire - i.e. all inputs have fired 

}///
;


/******************************************************************************/
/*									      */
/*	CLASS NAME: SourceBlock						      */
/*									      */
/******************************************************************************/

/** 
  This is a virtual class from which all 'source' blocks should be derived.
  A source block is one which generates data and has no input sockets. 
  
  source_block reset sets up an execution count mechanism this is used
  to limit the amount of data that a source provides. */


class SourceBlock: public Block {

private:

  static const string help_text;  // Static help texts - one for each inbuilt block class

  /// The number of times the input can be triggered. Set to <0 for infinite.
  ParamFloat *duration_param;

  //
  //
  //
  
public:  

  // name Construction and Destruction. 
 
  /// Construct an empty source block with a given trigger count.
  SourceBlock();
  SourceBlock(const string &a_name, const string &a_block_type);

  virtual ~SourceBlock();

  /// Copy a source block and assign it a new name
  virtual Block* clone(const string &new_name) const =0;
 
  
  virtual const string &get_helptext() const {return help_text;}

  /// Reset the source block. Called by the top-block reset.

  virtual void reset();

  
private:
  
  /// A pure virtual compute method - to be supplied by the child
  virtual void compute()=0;

  /// Use input data descriptors to set output data descriptors
  virtual void build_output_data_descriptors();


  // The bit that is common to all the constructors
  void do_constructor();



}///
;



//
//
//
#endif       

/* End of ctk_block.hh */
